
#ifndef HEP_MC_MY_GEN_PARTICLE_H
#define HEP_MC_MY_GEN_PARTICLE_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// particle within an event coming in/out of a vertex
// particle is the basic building block or unit of the event record
//////////////////////////////////////////////////////////////////////////
//
// example:
//      GenParticle* p = new GenParticle( HepLorentzVector(1,1,1,3), 11, 1 );
// creates a particle with 4-vector (p,E)=1,1,1,3 - with pdg id 11 (electron)
// and give this particle status =1.
//
// The definition of a HepLorentzVector scope resolution operator allows for
//  the use of 4 vector algebra with GenParticles.
//  i.e. if two particles are defined:
//   HepMC::GenParticle p_electron( HepLorentzVector(0,0,5,5), 11, 1 );
//   HepMC::GenParticle p_positron( HepLorentzVector(0,5,0,5), -11, 1 );
//  then you can find their cms 4 vector:
//   HepLorentzVector v_cms = (HepLorentzVector)p_electron 
//                            + (HepLorentzVector)p_positron;
//
// the pointers to end/production vertices can only be set by the
//  vertices themselves - thus to set the production vertex for a particle,
//  you add the particle to that vertex with GenVertex::add_particle_out()
//
// We decide not to have a separate 4 vector for the momentum 
//  at decay time (which MC++ includes to allow dE/dX losses etc). 
//  If you want that, just add a decay vertex with the
//  same particle (modified momentum) going out
//


namespace HepMC {
  class MyGenParticle; //  Fwd Dec
};

#include "HepMC/GenParticleBaseT.h"
#include "HepMC/MyGenEvent.h"
#include "HepMC/MyGenVertex.h"
#include "HepMC/MyFlow.h"
#include "HepMC/Polarization.h"
#include "CLHEP/Vector/LorentzVector.h"
#include <iostream>

namespace HepMC {

  class MyGenParticle: 
    public GenParticleBaseT<MyGenEvent,MyGenParticle,MyGenVertex> {

    
    friend class MyGenEvent; // To let MyGenEvent access the static counter() method.
    friend std::ostream& operator<<( std::ostream&, const MyGenParticle& );

  public:
    MyGenParticle( const HepLorentzVector& momentum, int pdg_id,
		   int status = 0, 
		   const MyFlow& itsflow = MyFlow(),
		   const Polarization& polar = Polarization(0,0) );
    MyGenParticle( const MyGenParticle& inparticle ); // shallow copy.
    virtual ~MyGenParticle();
	
    MyGenParticle& operator=( const MyGenParticle& inparticle ); // shallow.
    bool         operator==( const MyGenParticle& ) const;
    bool         operator!=( const MyGenParticle& ) const;

    // dump this particle's full info to ostr
    void       print( std::ostream& ostr = std::cout ) const; 

    operator HepLorentzVector() const; // conversion operator

    ////////////////////
    // access methods //
    ////////////////////

    HepLorentzVector     momentum() const;
    int                  pdg_id() const;
    int                  status() const;
    Polarization         polarization() const;

    void   set_momentum( const HepLorentzVector& vec4 );
    void   set_pdg_id( int id );
    void   set_status( int status = 0 );
    void   set_polarization( const Polarization& polarization =
			     Polarization(0,0) );

  protected: // for internal use only by friend MyGenVertex class
    static unsigned int counter(); // temporary for debugging

  private:
    HepLorentzVector m_momentum;          // 4 vector in GeV
    int              m_pdg_id;            // id according to PDG convention
    int              m_status;            // As defined for HEPEVT
    Polarization     m_polarization;

    static unsigned int s_counter;
    
  };

  //////////////
  // INLINES  //
  //////////////
  inline MyGenParticle::operator HepLorentzVector() const 
  { return m_momentum; }
  
  inline HepLorentzVector MyGenParticle::momentum() const 
  { return m_momentum; }
  
  inline int MyGenParticle::pdg_id() const { return m_pdg_id; }
  
  inline int MyGenParticle::status() const { return m_status; }
  
  inline Polarization MyGenParticle::polarization() const 
  { return m_polarization; }

  inline void MyGenParticle::set_momentum( const HepLorentzVector& vec4 )
  { m_momentum = vec4; }
  
  inline void MyGenParticle::set_pdg_id( int id ) { m_pdg_id = id; }
  
  inline void MyGenParticle::set_status( int status ) { m_status = status; }
  
  inline void MyGenParticle::set_polarization( const Polarization& polar )
  { m_polarization = polar; }

};

#endif

