
#ifndef HEP_MC_MY_GEN_EVENT_H
#define HEP_MC_MY_GEN_EVENT_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// Event record for MC generators (for use at any stage of generation)
//////////////////////////////////////////////////////////////////////////
//
// This class is intended as both a "container class" ( to store a MC
//  event for interface between MC generators and detector simulation )
//  and also as a "work in progress class" ( that could be used inside
//  a generator and modified as the event is built ).
//
// Iterators are provided which allow the user to easily obtain a
//  list of particles or vertices in an event --- this list can be filled
//  subject to some sort of selection criteria. Examples are given below
//  ( see HepMC::copy_if and std::copy )

#include "HepMC/Utils.h"

// Since a container of all vertices in the event is maintained, the time
//  required to loop over all vertices (or particles) is very fast -- and 
//  the user does not gain much by first making his own list.
//  (this is not true for the GenVertex:: versions of these iterators, which
//   allow you to specify the vertex starting point and range)

// Data Members:
// signal_process_id()   The integer ID that uniquely specifies this signal
//                       process, i.e. MSUB in Pythia. It is necessary to
//                       package this with each event rather than with the run
//                       because many processes may be generated within one
//                       run.
// event_number()        Strictly speaking we cannot think of any reason that
//                       an event would need to know its own event number, it
//                       is more likely something that would be assigned by
//                       a database. It is included anyway (tradition?) since
//                       we expect it may be useful for debugging. It can
//                       be reset later by a database.
// signal_process_vertex() pointer to the vertex containing the signal process
// weights()             Vector of doubles which specify th weight of the evnt,
//                       the first entry will be the "event weight" used for
//                       hit and miss etc., but a general vector is used to
//                       allow for reweighting etc. We envision a list of
//                       WeightTags to be included with a run class which
//                       would specify the meaning of the Weights .
// random_states()       Vector of integers which specify the random number 
//                       generator's state for this event. It is left to the
//                       generator to make use of this. We envision a vector of
//                       RndmStatesTags to be included with a run class which
//                       would specify the meaning of the random_states.
//
///////////////////////
// Memory allocation //
///////////////////////
// -When a vertex (particle) is added to a event (vertex), it is "adopted" 
//  and becomes the responsibility of the event (vertex) to delete that 
//  particle. 
// -objects responsible for deleting memory:
//    -events delete included vertices
//    -each vertex deletes its outgoing particles which do not have decay
//     vertices
//    -each vertex deletes its incoming particles which do not
//     have creation vertices 
//
////////////////////////
// About the Barcodes //
////////////////////////
// - each vertex or particle has a barcode, which is just an integer which
//   uniquely identifies it inside the event (i.e. there is a one to one
//   mapping between particle memory addresses and particle barcodes... and 
//   the same applied for vertices)
// - The value of a barcode has NO MEANING and NO ORDER!
//   For the user's convenience, when an event is read in via an IO_method
//   from an indexed list (like the HEPEVT common block), then the index will
//   become the barcode for that particle.
// - particle barcodes are always positive integers
//   vertex barcodes are always negative integers
//   The barcodes are chosen and set automatically when a vertex or particle
//   comes under the ownership of an event (i.e. it is contained in an event).
// - You can tell when a particle or vertex is owned, because its 
//   parent_event() return value will return a pointer to the event which owns
//   it (or null if its an orphan).
// 

namespace HepMC {
  class MyGenEvent; //  Fwd Dec
};

#include "HepMC/GenEventBaseT.h"
#include "HepMC/MyGenParticle.h"
#include "HepMC/MyGenVertex.h"
#include "HepMC/WeightContainer.h"
#include <map>
#include <vector>
#include <algorithm>
#include <iostream>

namespace HepMC {

  class MyGenEvent: 
    public GenEventBaseT<MyGenEvent,MyGenParticle,MyGenVertex> {
  public:
    MyGenEvent( int signal_process_id = 0, int event_number = 0,
	      MyGenVertex* signal_vertex = 0,
	      const WeightContainer& weights = std::vector<double>(),
	      const std::vector<long int>& randomstates
	      = std::vector<long int>() );
    MyGenEvent( const MyGenEvent& inevent );          // deep copy
    MyGenEvent& operator=( const MyGenEvent& inevent ); // deep.
    virtual ~MyGenEvent(); //deletes all vertices/particles in this evt
    
    void print( std::ostream& ostr = std::cout ) const; // dumps to ostr

    ////////////////////
    // access methods //
    ////////////////////

    int signal_process_id() const;
    int event_number() const;
    double event_scale() const;
    double alphaQCD() const;
    double alphaQED() const;
    MyGenVertex* signal_process_vertex() const;

    // direct access to the weights container is allowed. 
    // Thus you can user myevt.weights()[2];
    // to access element 2 of the weights.
    // or use myevt.weights().push_back( mywgt ); to add an element.
    // and you can set the weights with myevt.weights() = myvector;
    WeightContainer&        weights();
    const WeightContainer&  weights() const;

    std::vector<long int> random_states() const;

    void set_signal_process_id( int id );
    void set_event_number( int eventno );
    void set_event_scale( double scale );
    void set_alphaQCD( double a );
    void set_alphaQED( double a );
    void set_signal_process_vertex( MyGenVertex* );
    void set_random_states( const std::vector<long int>& randomstates );

  protected:
    static unsigned int counter(); //num MyGenEvent objects in memory

  private: // data members
    int                   m_signal_process_id;
    int                   m_event_number;  
    double                m_event_scale;// GeV, see hep-ph/0109068
    double                m_alphaQCD;   // QCD coupling, see hep-ph/0109068
    double                m_alphaQED;   // QED coupling, see hep-ph/0109068
    MyGenVertex*            m_signal_process_vertex;
    WeightContainer       m_weights; // weights for this event first weight
    // is used by default for hit and miss
    std::vector<long int> m_random_states; // container of rndm num 
    // generator states

    static unsigned int   s_counter;

  };

    ///////////////////////////
    // INLINE Access Methods //
    ///////////////////////////

    inline int MyGenEvent::signal_process_id() const 
    { return m_signal_process_id; }

    inline int MyGenEvent::event_number() const { return m_event_number; }

    inline double MyGenEvent::event_scale() const { return m_event_scale; }

    inline double MyGenEvent::alphaQCD() const { return m_alphaQCD; }

    inline double MyGenEvent::alphaQED() const { return m_alphaQED; }
 
    inline MyGenVertex* MyGenEvent::signal_process_vertex() const {
	// returns a (mutable) pointer to the signal process vertex
	return noteworthy_vertex();
    }  

    inline WeightContainer& MyGenEvent::weights() { return m_weights; }

    inline const WeightContainer& MyGenEvent::weights() const 
    { return m_weights; }

    inline std::vector<long int> MyGenEvent::random_states() const 
    { return m_random_states; }

    inline void MyGenEvent::set_signal_process_id( int id )
    { m_signal_process_id = id; }

    inline void MyGenEvent::set_event_number( int eventno )
    { m_event_number = eventno; }


    inline void MyGenEvent::set_event_scale( double sc ) { m_event_scale = sc; }

    inline void MyGenEvent::set_alphaQCD( double a ) { m_alphaQCD = a; }

    inline void MyGenEvent::set_alphaQED( double a ) { m_alphaQED = a; }

    inline void MyGenEvent::set_signal_process_vertex( MyGenVertex* vtx ) {
      set_noteworthy_vertex(vtx);
    }

    inline void MyGenEvent::set_random_states( const std::vector<long int>&
					     randomstates )
    { m_random_states = randomstates; }

};

#endif

