
#ifndef HEP_MC_MY_GEN_PARTICLE_H
#define HEP_MC_MY_GEN_PARTICLE_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// particle within an event coming in/out of a vertex
// particle is the basic building block or unit of the event record
//////////////////////////////////////////////////////////////////////////
//
// example:
//      GenParticle* p = new GenParticle( HepLorentzVector(1,1,1,3), 11, 1 );
// creates a particle with 4-vector (p,E)=1,1,1,3 - with pdg id 11 (electron)
// and give this particle status =1.
//
// The definition of a HepLorentzVector scope resolution operator allows for
//  the use of 4 vector algebra with GenParticles.
//  i.e. if two particles are defined:
//   HepMC::GenParticle p_electron( HepLorentzVector(0,0,5,5), 11, 1 );
//   HepMC::GenParticle p_positron( HepLorentzVector(0,5,0,5), -11, 1 );
//  then you can find their cms 4 vector:
//   HepLorentzVector v_cms = (HepLorentzVector)p_electron 
//                            + (HepLorentzVector)p_positron;
//
// the pointers to end/production vertices can only be set by the
//  vertices themselves - thus to set the production vertex for a particle,
//  you add the particle to that vertex with GenVertex::add_particle_out()
//
// We decide not to have a separate 4 vector for the momentum 
//  at decay time (which MC++ includes to allow dE/dX losses etc). 
//  If you want that, just add a decay vertex with the
//  same particle (modified momentum) going out
//


namespace HepMC {
  class GenParticle; //  Fwd Dec
};

#include "HepMC/GenParticleBaseT.h"
#include "HepMC/GenEvent.h"
#include "HepMC/GenVertex.h"
#include "HepMC/Flow.h"
#include "HepMC/Polarization.h"
#include "CLHEP/Vector/LorentzVector.h"
#include <iostream>

namespace HepMC {

  class GenParticle: 
    public GenParticleBaseT<GenEvent,GenParticle,GenVertex> {

    
    friend class GenEvent; // To let GenEvent access the static counter() method.
    friend std::ostream& operator<<( std::ostream&, const GenParticle& );

  public:
    GenParticle( const HepLorentzVector& momentum, int pdg_id,
		   int status = 0, 
		   const Flow& itsflow = Flow(),
		   const Polarization& polar = Polarization(0,0) );
    GenParticle( const GenParticle& inparticle ); // shallow copy.
    virtual ~GenParticle();
	
    GenParticle& operator=( const GenParticle& inparticle ); // shallow.
    bool         operator==( const GenParticle& ) const;
    bool         operator!=( const GenParticle& ) const;

    // dump this particle's full info to ostr
    void       print( std::ostream& ostr = std::cout ) const; 

    operator HepLorentzVector() const; // conversion operator

    ////////////////////
    // access methods //
    ////////////////////

    HepLorentzVector     momentum() const;
    int                  pdg_id() const;
    int                  status() const;
    Polarization         polarization() const;

    void   set_momentum( const HepLorentzVector& vec4 );
    void   set_pdg_id( int id );
    void   set_status( int status = 0 );
    void   set_polarization( const Polarization& polarization =
			     Polarization(0,0) );

  protected: // for internal use only by friend GenVertex class
    static unsigned int counter(); // temporary for debugging

  private:
    HepLorentzVector m_momentum;          // 4 vector in GeV
    int              m_pdg_id;            // id according to PDG convention
    int              m_status;            // As defined for HEPEVT
    Polarization     m_polarization;

    static unsigned int s_counter;
    
  };

  //////////////
  // INLINES  //
  //////////////
  inline GenParticle::operator HepLorentzVector() const 
  { return m_momentum; }
  
  inline HepLorentzVector GenParticle::momentum() const 
  { return m_momentum; }
  
  inline int GenParticle::pdg_id() const { return m_pdg_id; }
  
  inline int GenParticle::status() const { return m_status; }
  
  inline Polarization GenParticle::polarization() const 
  { return m_polarization; }

  inline void GenParticle::set_momentum( const HepLorentzVector& vec4 )
  { m_momentum = vec4; }
  
  inline void GenParticle::set_pdg_id( int id ) { m_pdg_id = id; }
  
  inline void GenParticle::set_status( int status ) { m_status = status; }
  
  inline void GenParticle::set_polarization( const Polarization& polar )
  { m_polarization = polar; }

};

#endif

