//--------------------------------------------------------------------------
#ifndef HEPMC_GEN_VERTEX_BASE_T_H
#define HEPMC_GEN_VERTEX_BASE_T_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// GenVertexBaseT within an event
// A vertex is indirectly (via particle "edges") linked to other 
//   vertices ("nodes") to form a composite "graph"
//////////////////////////////////////////////////////////////////////////

// --> HANDLE COMPILER INCONSISTENCIES
// This pre-compiler directive is included (2002-01-16) to allow compatibility
// with several compilers.
// This code was originally written w/ gcc 2.95 which uses a forward_iterator.
// forward_iterator is non-standard, and so neither visual c++ not gcc > 3.x
// have it.
// For it to work on windows you need to define the _WIN32 variable 
// when compiling.

namespace HepMC {
  template <class Ev,class Pa,class Ve>
  class GenVertexBaseT; // Fwd Dec
};

//#ifdef _WIN32        // Platform: Windows MS Visual C++
//#define MISSING_FORWARD_ITERATOR
//#elif __GNUC__ && (__GNUC__ > 2) // Linux gcc 3.x
//#define MISSING_FORWARD_ITERATOR
//#define USE_OPENMODE_INSTEAD_OF_OPEN_MODE

//#elif __SUNPRO_CC    // Solaris CC 5.2
//#define MISSING_FORWARD_ITERATOR
//#define NEED_SOLARIS_FRIEND_FEATURE
//#endif // Platform

//#ifdef MISSING_FORWARD_ITERATOR    // redefine forward_iterator for 
//#define forward_iterator iterator  //  those compilers that don't have one.
//#endif // MISSING_FORWARD_ITERATOR

//#ifdef USE_OPENMODE_INSTEAD_OF_OPEN_MODE
//#define open_mode openmode
//#endif // USE_OPENMODE_INSTEAD_OF_OPEN_MODE


#include <iostream>
#include <iterator>
#include <vector>
#include <set>
#include <algorithm>
#include "HepMC/IteratorRange.h"

namespace HepMC {

    template <class Ev,class Pa,class Ve>
    class GenParticleBaseT;

    template <class Ev,class Pa,class Ve>
    class GenEventBaseT;

    template <class Ev,class Pa,class Ve>
    class GenVertexBaseT {

	friend std::ostream& operator<< <>( std::ostream&, const GenVertexBaseT<Ev,Pa,Ve>& );
	friend class GenEventBaseT<Ev,Pa,Ve>;

#ifdef NEED_SOLARIS_FRIEND_FEATURE
	// This bit of ugly code is only for CC-5.2 compiler. 
	// M.Dobbs 2002/02/19 
	// It is not needed by linux gcc, nor Windows Visual C++.
    public: 
	class vertex_iterator;
	friend class vertex_iterator;
	class particle_iterator;
	friend class particle_iterator;
#endif // NEED_SOLARIS_FRIEND_FEATURE

    public:
	GenVertexBaseT( );
	GenVertexBaseT( const GenVertexBaseT<Ev,Pa,Ve>& invertex );            // shallow copy
	virtual    ~GenVertexBaseT();

	GenVertexBaseT<Ev,Pa,Ve>& operator= ( const GenVertexBaseT<Ev,Pa,Ve>& invertex ); // shallow
	bool       operator==( const GenVertexBaseT<Ev,Pa,Ve>& a ) const;
	bool       operator!=( const GenVertexBaseT<Ev,Pa,Ve>& a ) const;
	void       print( std::ostream& ostr = std::cout ) const;

	void       add_particle_in( Pa * inparticle );
	void       add_particle_out( Pa * outparticle );
	// remove_particle finds *particle in the in and/or out list and
	//  removes it from these lists ... it DOES NOT DELETE THE PARTICLE 
	//  or its relations. You could delete the particle too as follows:
	//      delete vtx->remove_particle( particle );
	Pa * remove_particle( Pa * particle );

	////////////////////
	// access methods //
	////////////////////

	Ev * parent_event() const;
	
	// The barcode is the vertex's reference number, every vertex in the
	//  event has a unique barcode. Vertex barcodes are negative numbers,
	//  particle barcodes are positive numbers.
	// In general there is no reason to "suggest_barcode", if a vertex is
	//  added to the event without a suggested barcode, the event will
	//  assign one for it.
	int                     barcode() const;
	bool                    suggest_barcode( int the_bar_code );
	
	////////////////////
	// Iterators      // users should use prefer to use particle_iterator
	////////////////////    

    public:
	//typedef GenParticleBaseT<Ev,Pa,Ve> erm1;
	//typedef erm1 * erm2;
	//typedef std::set<erm2> erm3;
	//typedef typename erm3::const_iterator particles_in_const_iterator;
	typedef typename std::set<Pa *>::const_iterator
	particles_in_const_iterator;
	typedef typename std::set<Pa * >::const_iterator 
	particles_out_const_iterator;
	particles_in_const_iterator         particles_in_const_begin() const;
	particles_in_const_iterator         particles_in_const_end() const;
	particles_out_const_iterator        particles_out_const_begin() const;
	particles_out_const_iterator        particles_out_const_end() const;
	int                                 particles_in_size() const;
	int                                 particles_out_size() const;

    protected:
	static unsigned int     counter(); // temporary for debugging

	// only the GenEventBaseT (friend) is allowed to set the parent_event,
	//  and barcode. It is done automatically anytime you add a 
	//  vertex to an event
	void                    set_parent_event_( Ev * evt );
	void                    set_barcode_( int the_bar_code );

	/////////////////////////////
	// edge_iterator           // (protected - for internal use only)
	/////////////////////////////
	// If the user wants the functionality of the edge_iterator, he should
	// use particle_iterator with IteratorRange = family, parents, children
	//
	class edge_iterator :
	// public std::forward_iterator<Ve *,ptrdiff_t> {
	public std::iterator<std::forward_iterator_tag, Pa*> {  // ANSI C++ FIX LESTER SPECIAL: Note change from Ve* to Pa* ... is this correct?
	    // iterate over the family of edges connected to m_vertex begins 
	    // with parents (incoming particles) then children (outgoing)
	    // This is not a recursive iterator ... it is a building block
	    // for the public iterators and is intended for internal use only.
	    // The acceptable Iterator Ranges are: family, parents, children
	public:
	    edge_iterator();
	    edge_iterator( const Ve & vtx, IteratorRange range =family );
	    edge_iterator( const edge_iterator & p );
	    virtual        ~edge_iterator();
	    edge_iterator & operator=( const edge_iterator & p );
	    Pa *            operator*(void) const;
	    edge_iterator & operator++(void); // Pre-fix increment 
	    edge_iterator   operator++(int);   // Post-fix increment
	    bool            operator==( const edge_iterator& a ) const;
	    bool            operator!=( const edge_iterator& a ) const;
	    bool            is_parent() const; // true if parent of root vtx
	    bool            is_child() const;  // true if child of root vtx
	    const Ve *  vertex_root() const;
	private:
	    const Ve *  m_vertex;
	    IteratorRange  m_range;
	    typename std::set<Pa *>::const_iterator m_set_iter;
	    bool           m_is_inparticle_iter;
	    bool           m_is_past_end;
	};
	friend class edge_iterator;
	int              edges_size( IteratorRange range = family ) const;
	edge_iterator    edges_begin( IteratorRange range = family) const;
	edge_iterator    edges_end( IteratorRange /* dummy_range */ ) const;

    public:
	///////////////////////////////
	// vertex_iterator           //
	///////////////////////////////
	class vertex_iterator :
	//	    public std::forward_iterator<Ve *,ptrdiff_t> {
	public std::iterator<std::forward_iterator_tag, Ve*> {  // ANSI C++ FIX LESTER
	    // Iterates over all vertices connected via a graph to this vertex.
	    // this is made friend to that it can access protected edge
	    // iterator the range can be IteratorRange= ( parents, children, 
	    // family, ancestors, descendants, relatives )
	    // example for range=descendants the iterator 
	    // will return all vertices
	    // which are children (connected by an outgoing particle edge),
	    // grandchildren, great-grandchildren, etc. of this vertex
	    // In all cases the iterator always returns this vertex
	    // (returned last).
	    // The algorithm is accomplished by converting the graph to a tree
	    // (by "chopping" the edges connecting to an already visited
	    // vertex) and returning the vertices in POST ORDER traversal.
	    //
	public:
	    vertex_iterator();
	    vertex_iterator( Ve & vtx_root, IteratorRange range );
	    // next constructor is intended for internal use only
	    vertex_iterator( Ve & vtx_root, IteratorRange range,
			     std::set<const Ve *> & visited_vertices );
	    vertex_iterator( const vertex_iterator& v_iter );
	    virtual             ~vertex_iterator();
	    vertex_iterator&    operator=( const vertex_iterator& );
	    Ve *                operator*(void) const;
	    vertex_iterator&    operator++(void);  //Pre-fix increment 
	    vertex_iterator     operator++(int);   //Post-fix increment
	    bool                operator==( const vertex_iterator& ) const;
	    bool                operator!=( const vertex_iterator& ) const;
	    Ve *                vertex_root() const;
	    IteratorRange       range() const;
	    void                copy_with_own_set( const vertex_iterator& 
						   v_iter,
						   std::set<const Ve *>& 
						   visited_vertices );
	protected:                  // intended for internal use only
	    GenVertexBaseT<Ev,Pa,Ve>* follow_edge_(); // non-null if recursive iter. created
	    void    copy_recursive_iterator_( const vertex_iterator* 
					      recursive_v_iter );
	private:
	    Ve *       m_vertex;   // the vertex associated to this iter
	    IteratorRange    m_range;
	    std::set<const Ve*> * m_visited_vertices;
	    bool             m_it_owns_set;  // true if it is responsible for 
	                                     // deleting the visited vertex set
	    edge_iterator    m_edge; // particle edge pointing to return vtx
	    vertex_iterator* m_recursive_iterator;
	};	
	friend class vertex_iterator;
	vertex_iterator     vertices_begin( IteratorRange range = relatives );
	vertex_iterator     vertices_end( IteratorRange /* dummy_range */ );
 
    public:
	///////////////////////////////
	// particle_iterator         //
	///////////////////////////////
	class particle_iterator :
	//	    public std::forward_iterator<Ve*,ptrdiff_t>{
	public std::iterator<std::forward_iterator_tag, Pa*> {  // ANSI C++ FIX LESTER SPECIAL: Note change from Ve* to Pa* ... is this correct?
	    // Iterates over all particles connected via a graph.
	    // by iterating through all vertices in the m_range. For each
	    // vertex it returns orphaned parent particles 
	    // (i.e. parents without production vertices) 
	    // then children ... in this way each particle is associated
	    // to exactly one vertex and so it is returned exactly once.
	    // Is made friend so that it can access protected edge iterator
	public:
	    particle_iterator();
	    particle_iterator( Ve & vertex_root, IteratorRange range );
	    particle_iterator( const particle_iterator& );
	    virtual             ~particle_iterator();
	    particle_iterator&  operator=( const particle_iterator & );
	    Pa *                operator*(void) const;
	    particle_iterator&  operator++(void);  //Pre-fix increment 
	    particle_iterator   operator++(int);   //Post-fix increment
	    bool                operator==( const particle_iterator& ) const;
	    bool                operator!=( const particle_iterator& ) const;
	protected:
	    Pa * advance_to_first_();
	private:
	    vertex_iterator     m_vertex_iterator;
	    edge_iterator       m_edge;     // points to the return
	};
	friend class particle_iterator;
	particle_iterator       particles_begin( IteratorRange range 
						 = relatives );
	particle_iterator       particles_end( IteratorRange 
					       /* dummy_range */ );

	////////////////////////////////////////////////
    protected: // for internal use only
	void delete_adopted_particles();
	
    private: // GenVertexBaseT data members
	std::set<Pa *>  m_particles_in;  //all incoming particles
	std::set<Pa *>  m_particles_out; //all outgoing particles
	Ev * m_event;
	int                  m_barcode;   // unique identifier in the event

	static unsigned int  s_counter;
    };  

    ////////////////////////////
    // INLINES access methods //
    ////////////////////////////

    template <class Ev,class Pa,class Ve>
    inline Ev * GenVertexBaseT<Ev,Pa,Ve>::parent_event() const { return m_event; }

    template <class Ev,class Pa,class Ve>
    inline int  GenVertexBaseT<Ev,Pa,Ve>::barcode() const { return m_barcode; }
    template <class Ev,class Pa,class Ve>
    inline void GenVertexBaseT<Ev,Pa,Ve>::set_barcode_( int bc ) { m_barcode = bc; }

    //////////////
    // INLINES  //
    //////////////

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particles_in_const_iterator 
    GenVertexBaseT<Ev,Pa,Ve>::particles_in_const_begin() const { 
	return m_particles_in.begin(); 
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particles_in_const_iterator 
    GenVertexBaseT<Ev,Pa,Ve>::particles_in_const_end() const { 
	return m_particles_in.end(); 
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particles_out_const_iterator 
    GenVertexBaseT<Ev,Pa,Ve>::particles_out_const_begin() const { 
	return m_particles_out.begin();
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particles_out_const_iterator 
    GenVertexBaseT<Ev,Pa,Ve>::particles_out_const_end() const {	
	return m_particles_out.end(); 
    }

    template <class Ev,class Pa,class Ve>
    inline int GenVertexBaseT<Ev,Pa,Ve>::particles_in_size() const {
	return m_particles_in.size(); 
    }

    template <class Ev,class Pa,class Ve>
    inline int GenVertexBaseT<Ev,Pa,Ve>::particles_out_size() const {
	return m_particles_out.size(); 
    }	

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::edge_iterator::operator==( 
	const edge_iterator& a ) const { 
	return **this == *a; 
    }

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::edge_iterator::operator!=(
	const edge_iterator& a ) const { 
	return !(**this == *a); 
    }

    template <class Ev,class Pa,class Ve>
    inline const Ve * GenVertexBaseT<Ev,Pa,Ve>::edge_iterator::vertex_root() const {
	return m_vertex;
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::edge_iterator GenVertexBaseT<Ev,Pa,Ve>::edges_begin( IteratorRange 
						      range ) const {
	return GenVertexBaseT<Ev,Pa,Ve>::edge_iterator(*(dynamic_cast<const Ve*>(this)), range);
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::edge_iterator GenVertexBaseT<Ev,Pa,Ve>::edges_end( IteratorRange 
						    /* dummy_range */ ) const {
	return GenVertexBaseT<Ev,Pa,Ve>::edge_iterator();
    }

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator::operator==( 
	const vertex_iterator& a ) const {
	return **this == *a; 
    }

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator::operator!=( 
	const vertex_iterator& a ) const {
	return !(**this == *a); 
    }

    template <class Ev,class Pa,class Ve>
    inline Ve * GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator::vertex_root() const {
	return m_vertex; 
    }

    template <class Ev,class Pa,class Ve>
    inline IteratorRange GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator::range() const {
	return m_range; 
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator GenVertexBaseT<Ev,Pa,Ve>::vertices_begin( 
	IteratorRange range ){
	// this is not const because the it could return itself
	return vertex_iterator( *this, range );
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::vertex_iterator GenVertexBaseT<Ev,Pa,Ve>::vertices_end( 
	IteratorRange /* dummy_range */ ) {
	return vertex_iterator();
    }

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::particle_iterator::operator==( 
	const particle_iterator& a ) const {
	return **this == *a; 
    }

    template <class Ev,class Pa,class Ve>
    inline bool GenVertexBaseT<Ev,Pa,Ve>::particle_iterator::operator!=( 
	const particle_iterator& a ) const {
	return !(**this == *a); 
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particle_iterator GenVertexBaseT<Ev,Pa,Ve>::particles_begin( 
	IteratorRange range ) {
	return particle_iterator( *(dynamic_cast<Ve*>(this)), range );
    }

    template <class Ev,class Pa,class Ve>
    inline typename GenVertexBaseT<Ev,Pa,Ve>::particle_iterator GenVertexBaseT<Ev,Pa,Ve>::particles_end(
	IteratorRange /* dummy_range */ ){
	return particle_iterator();
    }

} // HepMC

#include "HepMC/GenVertexBaseT.icc"

#endif  // HEPMC_GEN_VERTEX_H
//--------------------------------------------------------------------------




