//--------------------------------------------------------------------------
#ifndef HEPMC_GEN_PARTICLE_BASE_T_H
#define HEPMC_GEN_PARTICLE_BASE_T_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// particle within an event coming in/out of a vertex
// particle is the basic building block or unit of the event record
//////////////////////////////////////////////////////////////////////////
//
// example:
//      GenParticleBaseT* p = new GenParticleBaseT( HepLorentzVector(1,1,1,3), 11, 1 );
// creates a particle with 4-vector (p,E)=1,1,1,3 - with pdg id 11 (electron)
// and give this particle status =1.
//
// The definition of a HepLorentzVector scope resolution operator allows for
//  the use of 4 vector algebra with GenParticleBaseTs.
//  i.e. if two particles are defined:
//   HepMC::GenParticleBaseT p_electron( HepLorentzVector(0,0,5,5), 11, 1 );
//   HepMC::GenParticleBaseT p_positron( HepLorentzVector(0,5,0,5), -11, 1 );
//  then you can find their cms 4 vector:
//   HepLorentzVector v_cms = (HepLorentzVector)p_electron 
//                            + (HepLorentzVector)p_positron;
//
// the pointers to end/production vertices can only be set by the
//  vertices themselves - thus to set the production vertex for a particle,
//  you add the particle to that vertex with GenVertexBaseT::add_particle_out()
//
// We decide not to have a separate 4 vector for the momentum 
//  at decay time (which MC++ includes to allow dE/dX losses etc). 
//  If you want that, just add a decay vertex with the
//  same particle (modified momentum) going out
//

namespace HepMC {
  template <class Ev,class Pa,class Ve>
    class GenParticleBaseT; //Fwd Dec
};

#include "HepMC/FlowBaseT.h"
#include <iostream>


namespace HepMC {

  template <class Ev,class Pa,class Ve>
    class GenVertexBaseT;
  template <class Ev,class Pa,class Ve>
    class GenEventBaseT; 
  
  template <class Ev,class Pa,class Ve>
    class GenParticleBaseT {
    
    friend class GenVertexBaseT<Ev,Pa,Ve>; // so vertex can set decay/production vertexes
    friend class GenEventBaseT<Ev,Pa,Ve>;  // so event can set the barCodes
    friend std::ostream& operator<< <>( std::ostream&, const GenParticleBaseT<Ev,Pa,Ve>& );
    private:
        typedef FlowBaseT<Pa,Ve> UglyKludge;  // Internal use only! I can't figure out why I need this typedef.  Can anyone get rid of it?
    public:
        GenParticleBaseT(
		     const FlowBaseT<Pa,Ve> & itsflow =  
		     UglyKludge()
			);
	GenParticleBaseT( const GenParticleBaseT<Ev,Pa,Ve>& inparticle ); // shallow copy.
	virtual ~GenParticleBaseT();
	
	GenParticleBaseT<Ev,Pa,Ve>& operator=( const GenParticleBaseT<Ev,Pa,Ve>& inparticle ); // shallow.
	bool         operator==( const GenParticleBaseT<Ev,Pa,Ve>& ) const;
	bool         operator!=( const GenParticleBaseT<Ev,Pa,Ve>& ) const;

	// dump this particle's full info to ostr
	void       print( std::ostream& ostr = std::cout ) const; 


	////////////////////
	// access methods //
	////////////////////

        FlowBaseT<Pa,Ve> flow() const;
	int              flow( int code_index ) const;
	Ve *           production_vertex() const;
	Ve *           end_vertex() const;
        Ev *           parent_event() const;

	//
	// The barcode is the particle's reference number, every vertex in the
	//  event has a unique barcode. Particle barcodes are positive numbers,
	//  vertex barcodes are negative numbers.
	// In general there is no reason to "suggest_barcode", if a particle is
	//  added to the event without a suggested barcode, the event will
	//  assign one for it.
	int                  barcode() const;
	bool                 suggest_barcode( int the_bar_code );

        void set_flow( const FlowBaseT<Pa,Ve>& f );
        void set_flow( int code_index, int code = 0 );

    protected: // for internal use only by friend GenVertexBaseT class

        static unsigned int counter(); // temporary for debugging

	void   set_production_vertex_( Ve * productionvertex = 0);
	void   set_end_vertex_( Ve * decayvertex = 0 );
	void   set_barcode_( int the_bar_code ); // for use by GenEventBaseT only

    private:
	FlowBaseT<Pa,Ve> m_flow;
	Ve *       m_production_vertex; // null if vacuum or beam
	Ve *       m_end_vertex;        // null if not-decayed
	int              m_barcode;           // unique identifier in the event

        static unsigned int s_counter;
    };  

    //////////////
    // INLINES  //
    //////////////

    template <class Ev,class Pa,class Ve>
    inline Ve * GenParticleBaseT<Ev,Pa,Ve>::production_vertex() const 
    { return m_production_vertex; }

    template <class Ev,class Pa,class Ve>
    inline Ve * GenParticleBaseT<Ev,Pa,Ve>::end_vertex() const { return m_end_vertex; }

    template <class Ev,class Pa,class Ve>
    inline FlowBaseT<Pa,Ve> GenParticleBaseT<Ev,Pa,Ve>::flow() const { return m_flow; }

    template <class Ev,class Pa,class Ve>
    inline int GenParticleBaseT<Ev,Pa,Ve>::flow( int code_index ) const
       { return m_flow.icode( code_index ); }

    template <class Ev,class Pa,class Ve>
    inline void GenParticleBaseT<Ev,Pa,Ve>::set_flow( const FlowBaseT<Pa,Ve>& f ) { m_flow = f; }

    template <class Ev,class Pa,class Ve>
    inline void GenParticleBaseT<Ev,Pa,Ve>::set_flow( int code_index, int code ) 
      {
	if ( code == 0 ) { 
	  m_flow.set_unique_icode( code_index );
	} else { 
	  m_flow.set_icode( code_index, code );
	}
      }
    
    template <class Ev,class Pa,class Ve>
    inline int  GenParticleBaseT<Ev,Pa,Ve>::barcode() const { return m_barcode; }

    template <class Ev,class Pa,class Ve>
    inline void GenParticleBaseT<Ev,Pa,Ve>::set_barcode_( int bc ) { m_barcode = bc; }

} // HepMC

#include "HepMC/GenParticleBaseT.icc"

#endif  // HEPMC_GEN_PARTICLE_H
//--------------------------------------------------------------------------

