//--------------------------------------------------------------------------
#ifndef HEPMC_GEN_EVENT_BASE_T_H
#define HEPMC_GEN_EVENT_BASE_T_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, September 1999, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// Event record for MC generators (for use at any stage of generation)
//////////////////////////////////////////////////////////////////////////
//
// This class is intended as both a "container class" ( to store a MC
//  event for interface between MC generators and detector simulation )
//  and also as a "work in progress class" ( that could be used inside
//  a generator and modified as the event is built ).
//
// Iterators are provided which allow the user to easily obtain a
//  list of particles or vertices in an event --- this list can be filled
//  subject to some sort of selection criteria. Examples are given below
//  ( see HepMC::copy_if and std::copy )

namespace HepMC {
  template <class Ev,class Pa,class Ve>
  class GenEventBaseT; //Fwd Dec
};

#include "HepMC/Utils.h"

// Since a container of all vertices in the event is maintained, the time
//  required to loop over all vertices (or particles) is very fast -- and 
//  the user does not gain much by first making his own list.
//  (this is not true for the GenVertexBaseT:: versions of these iterators, which
//   allow you to specify the vertex starting point and range)

// Data Members:
// signal_process_id()   The integer ID that uniquely specifies this signal
//                       process, i.e. MSUB in Pythia. It is necessary to
//                       package this with each event rather than with the run
//                       because many processes may be generated within one
//                       run.
// event_number()        Strictly speaking we cannot think of any reason that
//                       an event would need to know its own event number, it
//                       is more likely something that would be assigned by
//                       a database. It is included anyway (tradition?) since
//                       we expect it may be useful for debugging. It can
//                       be reset later by a database.
// signal_process_vertex() pointer to the vertex containing the signal process
// weights()             Vector of doubles which specify th weight of the evnt,
//                       the first entry will be the "event weight" used for
//                       hit and miss etc., but a general vector is used to
//                       allow for reweighting etc. We envision a list of
//                       WeightTags to be included with a run class which
//                       would specify the meaning of the Weights .
// random_states()       Vector of integers which specify the random number 
//                       generator's state for this event. It is left to the
//                       generator to make use of this. We envision a vector of
//                       RndmStatesTags to be included with a run class which
//                       would specify the meaning of the random_states.
//
///////////////////////
// Memory allocation //
///////////////////////
// -When a vertex (particle) is added to a event (vertex), it is "adopted" 
//  and becomes the responsibility of the event (vertex) to delete that 
//  particle. 
// -objects responsible for deleting memory:
//    -events delete included vertices
//    -each vertex deletes its outgoing particles which do not have decay
//     vertices
//    -each vertex deletes its incoming particles which do not
//     have creation vertices 
//
////////////////////////
// About the Barcodes //
////////////////////////
// - each vertex or particle has a barcode, which is just an integer which
//   uniquely identifies it inside the event (i.e. there is a one to one
//   mapping between particle memory addresses and particle barcodes... and 
//   the same applied for vertices)
// - The value of a barcode has NO MEANING and NO ORDER!
//   For the user's convenience, when an event is read in via an IO_method
//   from an indexed list (like the HEPEVT common block), then the index will
//   become the barcode for that particle.
// - particle barcodes are always positive integers
//   vertex barcodes are always negative integers
//   The barcodes are chosen and set automatically when a vertex or particle
//   comes under the ownership of an event (i.e. it is contained in an event).
// - You can tell when a particle or vertex is owned, because its 
//   parent_event() return value will return a pointer to the event which owns
//   it (or null if its an orphan).
// 

#include "HepMC/GenVertexBaseT.h"
#include "HepMC/GenParticleBaseT.h"
#include <map>
#include <vector>
#include <algorithm>
#include <iostream>

namespace HepMC {
  
  template <class Ev,class Pa,class Ve>
    class GenEventBaseT {
    friend class GenParticleBaseT<Ev,Pa,Ve>;
    friend class GenVertexBaseT<Ev,Pa,Ve>;  
    public:
        GenEventBaseT(Ve * noteworthy_vertex=0);
	GenEventBaseT( const GenEventBaseT<Ev,Pa,Ve>& inevent );          // deep copy
	GenEventBaseT<Ev,Pa,Ve>& operator=( const GenEventBaseT<Ev,Pa,Ve>& inevent ); // deep.
	virtual ~GenEventBaseT(); //deletes all vertices/particles in this evt
    
	void print( std::ostream& ostr = std::cout ) const; // dumps to ostr

	Pa * barcode_to_particle( int barCode ) const;
	Ve *   barcode_to_vertex( int barCode ) const;

	////////////////////
	// access methods //
	////////////////////

	Ve *    noteworthy_vertex() const;
	void    set_noteworthy_vertex( Ve * );

	int     particles_size() const;
	bool    particles_empty() const;
	int     vertices_size() const;
	bool    vertices_empty() const;

	bool    add_vertex   ( Ve * vtx ); // adds to evt and adopts
	bool    remove_vertex( Ve * vtx ); // erases vtx from evt, 

    public:
	///////////////////////////////
	// vertex_iterators          //
	///////////////////////////////
	// Note:  the XXX_iterator is "resolvable" as XXX_const_iterator, but 
	//  not the reverse, which is consistent with STL, 
	//  see Musser, Derge, Saini 2ndEd. p. 69,70.
	class vertex_const_iterator :
	// public std::forward_iterator<Ve *,ptrdiff_t>{
	public std::iterator<std::forward_iterator_tag, Ve *> { // ANSI C++ FIX LESTER
	  // Iterates over all vertices in this event
	public:
	    vertex_const_iterator(
		const typename
		std::map<int,Ve *,std::greater<int> >::const_iterator& i)
		: m_map_iterator(i) {}
	    vertex_const_iterator() {}
	    vertex_const_iterator( const vertex_const_iterator& i )
		{ *this = i; }
	    virtual ~vertex_const_iterator() {}
	    vertex_const_iterator&  operator=( const vertex_const_iterator& i )
		{ m_map_iterator = i.m_map_iterator; return *this; }
	    Ve * operator*(void) const { return m_map_iterator->second; }
	    vertex_const_iterator&  operator++(void)  //Pre-fix increment 
		{ ++m_map_iterator; return *this; }
	    vertex_const_iterator   operator++(int)   //Post-fix increment
		{ vertex_const_iterator out(*this); ++(*this); return out; }
	    bool  operator==( const vertex_const_iterator& a ) const
		{ return m_map_iterator == a.m_map_iterator; }
	    bool  operator!=( const vertex_const_iterator& a ) const
		{ return !(m_map_iterator == a.m_map_iterator); }
	protected:
	    typename std::map<int,Ve *,std::greater<int> >::const_iterator 
	                                                        m_map_iterator;
	};
	friend class vertex_const_iterator;
	vertex_const_iterator      vertices_begin() const
	    { return GenEventBaseT<Ev,Pa,Ve>::vertex_const_iterator( 
		m_vertex_barcodes.begin() ); }
	vertex_const_iterator      vertices_end() const
	    { return GenEventBaseT<Ev,Pa,Ve>::vertex_const_iterator(
		m_vertex_barcodes.end() ); }

	class vertex_iterator :
	//public std::forward_iterator<Ve *,ptrdiff_t>{
	public std::iterator<std::forward_iterator_tag, Ve *> { // ANSI C++ FIX LESTER
	  // Iterates over all vertices in this event
	public:
	    vertex_iterator( 
		const typename 
		std::map<int,Ve *,std::greater<int> >::iterator& i )
		: m_map_iterator( i ) {}
	    vertex_iterator() {}
	    vertex_iterator( const vertex_iterator& i ) { *this = i; }
	    virtual ~vertex_iterator() {}
	    vertex_iterator&  operator=( const vertex_iterator& i ) {
		m_map_iterator = i.m_map_iterator;
		return *this;
	    }
	    operator vertex_const_iterator() const
		{ return vertex_const_iterator(m_map_iterator); }
	    Ve *        operator*(void) const
		{ return m_map_iterator->second; }
	    vertex_iterator&  operator++(void)  //Pre-fix increment 
		{ ++m_map_iterator;     return *this; }
	    vertex_iterator   operator++(int)   //Post-fix increment
		{ vertex_iterator out(*this); ++(*this); return out; }
	    bool              operator==( const vertex_iterator& a ) const
		{ return m_map_iterator == a.m_map_iterator; }
	    bool              operator!=( const vertex_iterator& a ) const
		{ return !(m_map_iterator == a.m_map_iterator); }
	protected:
	    typename std::map<int,Ve *,std::greater<int> >::iterator 
	                                                       m_map_iterator;
	};
	friend class vertex_iterator;
	vertex_iterator            vertices_begin() 
	    { return GenEventBaseT<Ev,Pa,Ve>::vertex_iterator( 
		m_vertex_barcodes.begin() ); }
        vertex_iterator            vertices_end()
	    { return GenEventBaseT<Ev,Pa,Ve>::vertex_iterator(
		m_vertex_barcodes.end() ); }

    public:
	///////////////////////////////
	// particle_iterator         //
	///////////////////////////////
	// Example of iterating over all particles in the event:
	//      for ( GenEventBaseT<Ev,Pa,Ve>::particle_const_iterator p = particles_begin();
	//            p != particles_end(); ++p ) {
	//         (*p)->print();
	//      }
	//
	class particle_const_iterator :
	//public std::forward_iterator<Pa *,ptrdiff_t> {
	public std::iterator<std::forward_iterator_tag, Pa *> { // ANSI C++ FIX LESTER
	  // Iterates over all vertices in this event
	public:
	    particle_const_iterator(
		const typename std::map<int,Pa *>::const_iterator& i )
		: m_map_iterator(i) {}
	    particle_const_iterator() {}
	    particle_const_iterator( const particle_const_iterator& i )
		{ *this = i; }
	    virtual ~particle_const_iterator() {}
	    particle_const_iterator& operator=(
		const particle_const_iterator& i )
		{ m_map_iterator = i.m_map_iterator; return *this; }
	    Pa *        operator*(void) const
		{ return m_map_iterator->second; }
	    particle_const_iterator&  operator++(void)  //Pre-fix increment 
		{ ++m_map_iterator; return *this; }
	    particle_const_iterator   operator++(int)   //Post-fix increment
		{ particle_const_iterator out(*this); ++(*this); return out; }
	    bool  operator==( const particle_const_iterator& a ) const
		{ return m_map_iterator == a.m_map_iterator; }
	    bool  operator!=( const particle_const_iterator& a ) const
		{ return !(m_map_iterator == a.m_map_iterator); }
	protected:
	    typename std::map<int,Pa *>::const_iterator m_map_iterator;
	};	
	friend class particle_const_iterator;
	particle_const_iterator      particles_begin() const
	    { return GenEventBaseT<Ev,Pa,Ve>::particle_const_iterator( 
		m_particle_barcodes.begin() ); }
	particle_const_iterator      particles_end() const
	    { return GenEventBaseT<Ev,Pa,Ve>::particle_const_iterator(
		m_particle_barcodes.end() ); }

	class particle_iterator :
	//public std::forward_iterator<Pa *,ptrdiff_t>{
	public std::iterator<std::forward_iterator_tag, Pa *> { // ANSI C++ FIX LESTER
	    // Iterates over all vertices in this event
	public:
	    particle_iterator( const typename std::map<int,Pa *>::iterator& i )
		: m_map_iterator( i ) {}
	    particle_iterator() {}
	    particle_iterator( const particle_iterator& i ) { *this = i; }
	    virtual ~particle_iterator() {}
	    particle_iterator&  operator=( const particle_iterator& i ) {
		m_map_iterator = i.m_map_iterator;
		return *this;
	    }
	    operator particle_const_iterator() const
		{ return particle_const_iterator(m_map_iterator); }
	    Pa *        operator*(void) const
		{ return m_map_iterator->second; }
	    particle_iterator&  operator++(void)  //Pre-fix increment 
		{ ++m_map_iterator;     return *this; }
	    particle_iterator   operator++(int)   //Post-fix increment
		{ particle_iterator out(*this); ++(*this); return out; }
	    bool              operator==( const particle_iterator& a ) const
		{ return m_map_iterator == a.m_map_iterator; }
	    bool              operator!=( const particle_iterator& a ) const
		{ return !(m_map_iterator == a.m_map_iterator); }
	protected:
	    typename std::map<int,Pa *>::iterator m_map_iterator;
	};
	friend class particle_iterator;
	particle_iterator particles_begin() 
	    { return GenEventBaseT<Ev,Pa,Ve>::particle_iterator(
		m_particle_barcodes.begin() ); }
        particle_iterator particles_end()
	    { return GenEventBaseT<Ev,Pa,Ve>::particle_iterator(
		m_particle_barcodes.end() ); }

	////////////////////////////////////////////////
    protected:
	//
	// Following methods intended for use by GenParticleBaseT<Ev,Pa,Ve>/Vertex classes:
	// In general there is no reason they should be used elsewhere.
	bool         set_barcode( Pa * p, int suggested_barcode =0 );
	bool         set_barcode( Ve * v, int suggested_barcode =0 );
        void         remove_barcode( GenParticleBaseT<Ev,Pa,Ve> * p ); // BARR
	void         remove_barcode( GenVertexBaseT<Ev,Pa,Ve> * v ); // BARR
      
	static unsigned int counter(); //num GenEventBaseT<Ev,Pa,Ve> objects in memory
   	void delete_all_vertices();

    private: // data members
	Ve *            m_noteworthy_vertex;
	std::map< int,Ve *,std::greater<int> >   m_vertex_barcodes;
	std::map< int,Pa *,std::less<int> >    m_particle_barcodes;

	static unsigned int   s_counter;
    };

    ///////////////////////////
    // INLINE Access Methods //
    ///////////////////////////


    template <class Ev,class Pa,class Ve>
    inline void GenEventBaseT<Ev,Pa,Ve>::remove_barcode( GenParticleBaseT<Ev,Pa,Ve> * p )
    { m_particle_barcodes.erase( p->barcode() ); }

    template <class Ev,class Pa,class Ve>
    inline void GenEventBaseT<Ev,Pa,Ve>::remove_barcode( GenVertexBaseT<Ev,Pa,Ve> * v )
    { m_vertex_barcodes.erase( v->barcode() ); }

    template <class Ev,class Pa,class Ve>
    inline Pa * GenEventBaseT<Ev,Pa,Ve>::barcode_to_particle( int barCode ) const
    { 
	typename std::map<int,Pa *>::const_iterator i 
	    = m_particle_barcodes.find(barCode);
	return ( i != m_particle_barcodes.end() ) ? (*i).second : 0;
    }

    template <class Ev,class Pa,class Ve>
    inline Ve * GenEventBaseT<Ev,Pa,Ve>::barcode_to_vertex( int barCode ) const
    {
	typename std::map<int,Ve *,std::greater<int> >::const_iterator i 
	    = m_vertex_barcodes.find(barCode);
	return ( i != m_vertex_barcodes.end() ) ? (*i).second : 0;
    }

    template <class Ev,class Pa,class Ve>
    inline Ve * GenEventBaseT<Ev,Pa,Ve>::noteworthy_vertex() const {
	// returns a (mutable) pointer to the signal process vertex
	return m_noteworthy_vertex;
    }  

    template <class Ev,class Pa,class Ve>
    inline void GenEventBaseT<Ev,Pa,Ve>::set_noteworthy_vertex( Ve * vtx ) {
	m_noteworthy_vertex = vtx;
	if ( m_noteworthy_vertex ) add_vertex( m_noteworthy_vertex );
    }

    template <class Ev,class Pa,class Ve>
    inline int GenEventBaseT<Ev,Pa,Ve>::particles_size() const {
	return (int)m_particle_barcodes.size();
    }
    template <class Ev,class Pa,class Ve>
    inline bool GenEventBaseT<Ev,Pa,Ve>::particles_empty() const {
	return (bool)m_particle_barcodes.empty();
    }
    template <class Ev,class Pa,class Ve>
    inline int GenEventBaseT<Ev,Pa,Ve>::vertices_size() const {
	return (int)m_vertex_barcodes.size();
    }
    template <class Ev,class Pa,class Ve>
    inline bool GenEventBaseT<Ev,Pa,Ve>::vertices_empty() const {
	return (bool)m_vertex_barcodes.empty();
    }

} // HepMC

#include "HepMC/GenEventBaseT.icc"

#endif  // HEPMC_GEN_EVENT_H

//--------------------------------------------------------------------------



