//--------------------------------------------------------------------------
#ifndef HEPMC_FLOW_BASE_T_H
#define HEPMC_FLOW_BASE_T_H

//////////////////////////////////////////////////////////////////////////
// Matt.Dobbs@Cern.CH, January 2000, refer to:
// M. Dobbs and J.B. Hansen, "The HepMC C++ Monte Carlo Event Record for
// High Energy Physics", Computer Physics Communications (to be published).
//
// particle's flow object
// keeps track of an arbitrary number of flow patterns within a graph 
// (i.e. color flow, charge flow, lepton number flow, ...) 
// Flow patterns are coded with an integer, in the same manner as in Herwig.
// Note: 0 is NOT allowed as code index nor as flow code since it
//       is used to indicate null.
//////////////////////////////////////////////////////////////////////////

// This class can be used to keep track of flow patterns within 
//  a graph. An example is color flow. If we have two quarks going through
//  an s-channel gluon to form two more quarks:
//
//  \q1       /q3   then we can keep track of the color flow with the
//   \_______/      HepMC::Flow class as follows: 
//   /   g   \. 
//  /q2       \q4
//
//  lets say the color flows from q2-->g-->q3  and q1-->g-->q4
//  the individual colors are unimportant, but the flow pattern is.
//  We can capture this flow by assigning the first pattern (q2-->g-->q3)
//  a unique (arbitrary) flow code 678 and the second pattern (q1-->g-->q4)
//  flow code 269  ( you can ask HepMC::Flow to choose
//  a unique code for you using Flow::set_unique_icode() ).
//  The first two code indices are reserved for color codes, so we store 
//  these codes with the particles as follows:
//    q2->flow().set_icode(1,678);
//    g->flow().set_icode(1,678);
//    q3->flow().set_icode(1,678);
//    q1->flow().set_icode(1,269);
//    g->flow().set_icode(2,269);
//    q4->flow().set_icode(1,269);
//  later on if we wish to know the color partner of q1 we can ask for a list
//  of all particles connected via this code to q1 which do have less than 
//  2 color partners using:
//    set<Pa*> result=q1->dangling_connected_partners(q1->icode(1),1,2);
//  this will return a list containing q1 and q4.
//    set<Pa*> result=q1->connected_partners(q1->icode(1),1,2);
//  would return a list containing q1, g, and q4.
//

#include <iostream>
#include <map>
#include <set>

namespace HepMC {
  
  template<class Pa, class Ve>
    class FlowBaseT {
    
    friend std::ostream& operator<<<>( std::ostream& ostr, const FlowBaseT<Pa,Ve>& f );
    
    public:
    FlowBaseT( Pa* particle_owner = 0 );
    FlowBaseT( const FlowBaseT<Pa,Ve>& );
    virtual         ~FlowBaseT();
    FlowBaseT<Pa,Ve>& operator=( const FlowBaseT<Pa,Ve>& );
    bool            operator==( const FlowBaseT<Pa,Ve>& a ) const; //compares only flow
    bool            operator!=( const FlowBaseT<Pa,Ve>& a ) const; //patterns not owner

    void            print( std::ostream& ostr = std::cout ) const;

    // returns all connected particles which have "code" in any  of the 
    //  num_indices beginning with index code_index.
    std::set<Pa*> connected_partners( int code, int code_index =1,
				      int num_indices = 2 ) const;
    // same as connected_partners, but returns only those particles which
    //  are connected to <=1 other particles (i.e. the flow line "dangles"
    //  at these particles)
    std::set<Pa*> dangling_connected_partners( int code, 
					       int code_index = 1, int num_indices = 2 ) const;

    ////////////////////
    // access methods //
    ////////////////////
    const Pa* particle_owner() const;
    int             icode( int code_index = 1 ) const;
    FlowBaseT  set_icode( int code_index, int code );
    FlowBaseT  set_unique_icode( int code_index = 1 );

    //////////////////////
    // container access //
    //////////////////////

    bool            empty() const;
    int             size() const;
    void            clear();
    bool            erase( int code_index );

    typedef std::map<int,int>::iterator       iterator;
    typedef std::map<int,int>::const_iterator const_iterator;
    iterator            begin();
    iterator            end();
    const_iterator      begin() const;
    const_iterator      end() const;

    protected: // intended for internal use only
    void            connected_partners( std::set<Pa*>* output, 
					int code,
					int code_index,
					int num_indices ) const;
    void            dangling_connected_partners( std::set<Pa*>* 
						 output, 
						 std::set<Pa*>*
						 visited_particles, 
						 int code, int code_index, 
						 int num_indices ) const; 
    private:
    Pa*         m_particle_owner;
    std::map<int,int> m_icode; // stores flow patterns as(code_index,icode)
  };  

  ///////////////////////////
  // INLINE Access Methods //
  ///////////////////////////

  template<class Pa, class Ve>
    inline const Pa* FlowBaseT<Pa,Ve>::particle_owner() const {
    return m_particle_owner;
  }
  template <class Pa,class Ve>
    inline int FlowBaseT<Pa,Ve>::icode( int code_index ) const {
    std::map<int,int>::const_iterator a = m_icode.find(code_index);
    return a==m_icode.end() ? 0 : (*a).second;
  }
  template<class Pa, class Ve>
    inline FlowBaseT<Pa,Ve> FlowBaseT<Pa,Ve>::set_icode( int code_index, int code ) {
    m_icode[code_index] = code;
    return *this;
  }
  template<class Pa, class Ve>
    inline FlowBaseT<Pa,Ve> FlowBaseT<Pa,Ve>::set_unique_icode( int flow_num ) {
    // use this method if you want to assign a unique flow code, but
    // do not want the burden of choosing it yourself
    m_icode[flow_num] = int(this);
    return *this;
  }

  template<class Pa, class Ve>
  inline bool FlowBaseT<Pa,Ve>::empty() const { return (bool)m_icode.empty(); }
  template<class Pa, class Ve>
    inline int FlowBaseT<Pa,Ve>::size() const { return (int)m_icode.size(); }
  template<class Pa, class Ve>
    inline void FlowBaseT<Pa,Ve>::clear() { m_icode.clear(); }
  template<class Pa, class Ve>
    inline bool FlowBaseT<Pa,Ve>::erase( int code_index ) {
    return (bool)m_icode.erase( code_index );
  }
  template<class Pa, class Ve>
    inline typename FlowBaseT<Pa,Ve>::iterator FlowBaseT<Pa,Ve>::begin() { return m_icode.begin(); }
  template<class Pa, class Ve>
    inline typename FlowBaseT<Pa,Ve>::iterator FlowBaseT<Pa,Ve>::end() { return m_icode.end(); }
  template<class Pa, class Ve>
    inline typename FlowBaseT<Pa,Ve>::const_iterator FlowBaseT<Pa,Ve>::begin() const { return m_icode.begin(); }
  template<class Pa, class Ve>
    inline typename FlowBaseT<Pa,Ve>::const_iterator FlowBaseT<Pa,Ve>::end() const { return m_icode.end(); }

  ///////////////////////////
  // INLINE Operators      //
  ///////////////////////////

  template<class Pa, class Ve>
    inline bool FlowBaseT<Pa,Ve>::operator==( const FlowBaseT<Pa,Ve>& a ) const {
    // equivalent flows have the same flow codes for all flow_numbers 
    // (i.e. their m_icode maps are identical), but they need not have the
    // same m_particle owner
    return (m_icode == a.m_icode);
  }
  template<class Pa, class Ve>
    inline bool FlowBaseT<Pa,Ve>::operator!=( const FlowBaseT<Pa,Ve>& a ) const {
    return !( *this == a );
  }
  template<class Pa, class Ve>
    inline FlowBaseT<Pa,Ve>& FlowBaseT<Pa,Ve>::operator=( const FlowBaseT<Pa,Ve>& inflow ) {
    // copies only the m_icode ... not the particle_owner
    // this is intuitive behaviour so you can do
    // oneparticle->flow() = otherparticle->flow()
    //
    m_icode = inflow.m_icode;
    return *this;
  }

} // HepMC

#include "HepMC/FlowBaseT.icc"

#endif  // HEPMC_FLOW_H
//--------------------------------------------------------------------------

