// ====================================
//  NAME    : FWUKL1 Library
//
//  VERSION : 2
//
//  REVISION: 0
//
//  DATE    : 2006/10/12
//
//  RELEASE : 2007/10/10
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Gareth Rogers
//
//  EMAIL   : rogers@hep.phy.cam.ac.uk
// ====================================

/* \mainpage UKL1 Library Documentation
 *
 * This library is used to control the UKL1 boards.
 */

// ====================
//  #USES DECLARATIONS
// ====================

/*! fwUkl1ExcepionHandling library is used to store and display errors that occur with fwUkl1 function calls. */
#uses "fwUkl1ExceptionHandling.ctl"

/*! fwHw library is used for recipe and data point manipulation. */
#uses "fwHw.ctl"

/*! fwCcpc library is used for accessing the UKL1 registers. */
#uses "fwCcpc.ctl"


// =========================
//  CONSTANTs & GLOBAL VARs
// =========================

/** @defgroup SectionConstants Constants
 *  List of constants that can be used by the user.
 *  @{
 */

/*! Defines the name of the system that the project has been instealled under. */
const string FWUKL1_SYS_NAME = "R2DAQL1";

/*! Defines the name of the top level control unit (CU) that is used in the L1 FSM tree. */
const string FWUKL1_CU = "L1";
/*! Defines the name of the data point type of the UKL1 hardware type. All UKL1 board data points are created of this type. */
const string FWUKL1_HW_TYPE = "UKL1";

/*! Defines the name of the recipe type that is associated with the configure action. */
const string FWUKL1_CONFIGURE_TYPE = "UKL1_Configure";

/*! Defines the name of the recipe type that is associated with the start and stop actions. */
const string FWUKL1_START_TYPE = "UKL1_Start";

/*! Defines the name of the data point type that holds the UKL1 exception log. */
const string FWUKL1_EXCEPTION_LOG_TYPE = "FwUkl1ExceptionLog";
/*! Defines the name of the data point that holds the UKL1 exception log. */
const string FWUKL1_EXCEPTION_LOG = "Ukl1Exceptions.exceptionLog";

/*! Broadcast FE FPGA number. Value used to write to all the FE FPGAs. */
const unsigned FWUKL1_BROADCAST_FPGA = 5;
/*! Broadcast FE FPGA channel number. Value used to write to all of a FE FPGA's channels. */
const unsigned FWUKL1_BROADCAST_CHANNEL = 10;
/*! Number of Front end (ingress) FPGAs on the RICH L1 board. */
const unsigned FWUKL1_FRONT_FPGAS = 4;
/*! Number of channels that are on each front end FPGA. */
const unsigned FWUKL1_FPGA_CHANNELS = 9;
/*! Number of channels that are on the RICH L1 board. */
const unsigned FWUKL1_BOARD_CHANNELS = 36;
/*! Number of ports on the gigabit ethernet card. */
const unsigned FWUKL1_GBE_PORTS = 4;
/*! Number of temperature sensors present on the board. */
const unsigned FWUKL1_TEMPERATURE_SENSORS = 3;

/*! This numeric constant is used to identify a production UKL1 board. */
const unsigned FWUKL1_PRODUCTION = 2;
/*! This numeric constant is used to identify a prototype UKL1 board. */
const unsigned FWUKL1_PROTOTYPE = 1;

/*! Constant that defines the size of a 32-bit data word. */
const unsigned FWUKL1_32BIT_WORD = 32;
/*! Constant that defines the size of a 16-bit data word. */
const unsigned FWUKL1_16BIT_WORD = 16;
/*! Constant that defines the size of an 8-bit data word. */
const unsigned FWUKL1_8BIT_WORD = 8;

//@}


// ==================
//  UKL1 BOARD SETUP
// ==================

/** @defgroup SectionSetup Board setup
 *    The functions in this section are responsible for setting up the software object that represents a UKL1 board.
 *    It provides a function that is capable of dynamically adding or removing the UKL1 boards from the system. By
 *    placing in an infinite loop it can be used to add/remove UKL1 boards from the system automatically. This should
 *    be used with some caution as it will regenerate the FSM tree everytime a board is added or removed from the system.\n
 *    The ability to manually add/delete UKL1 boards from the system is also provided, by the User must take care that
 *    they are adding/removing a valid UKL1 board to or from the system.\n
 *    UKL1 boards can be identified via their ID number and a function is provided that is capable of identifying UKL1
 *    production and prototype boards. It is via this function that the fwUkl1_findAndAddUkl1Boards() identifies UKL1
 *    boards from all the CCPCs present in the DIM server.
 *  @{
 */

/*!
 * Gets the names off all the CCPCs found in the DIM server and then identifies the UKL1 boards based on their ID number.
 * It will then add boards that are found to the system. It is also capable of removing boards that are found in the system
 * but no longer present in DIM.
 *
 * \param  refresh If TRUE the list existing list of UKL1 boards will be updated with any new boards found and those
 *           no longer present will be removed. FALSE all existing boards will be deleted along with their associated
 *           settings and the whole list created from new.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings.
 * \param  forceReconfigure At present this only affects the DP used to interact with the hardware and not the FSM tree.
 *           If TRUE it will recreate the default settings from those hardcoded into the file and reconfigures the
 *           harware data point from these settings. If FALSE the settings are just loaded from the defaults.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * There are several modes of operation of this function depending on the values of refresh, deleteExisting and forceReconfigure.
 * Full details of how the HW DP and the FSM tree are affected by these can be found in the documentation for _fwUkl1_createHwTypeCCPCUKL1
 * and _fwUkl1_createUkl1FsmTreeDU. A summary of the most useful modes are described.
 * \li refresh=TRUE, deleteExisting=FALSE, forcereconfigure=FALSE - The most efficient mode, this finds those UKL1 boards that are
 *       not in the system and creates them loading from defaults, it deletes UKL1 boards no longer in the DNS node and will
 *       perform no updates on the boards that already fully exist in the system. This is the recommend mode of operation for standard use.
 * \li refresh=FALSE, deleteExisting=FALSE, forceReconfigure=FALSE - This first clears all the boards from the system before adding all
 *       all those identified in the DNS node from defaults. Useful if the settings for existing UKL1 boards are believed to be corrupt.
 * \li Any mode with deleteExisting=TRUE will cause the exising software objects to be deleted and before recreated, much like refresh=TRUE.
 * \li forceReconfigure=TRUE is meant to be used only by experts as it is will reload from settings hardcoded into this library.
 *       Only required when an expert makes a change to the datapoint structure and then the library must be updated to match.
 *       Can also help if the defaults are believed to be corrupt or when first installed in a system.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_findAndAddUkl1Boards(bool refresh, bool deleteExisting, bool forceReconfigure, dyn_string& exceptionInfo) {
  //Note the size of the exception info.
  int exInfoSize = dynlen(exceptionInfo);
  //Holds the list of CCPC names retrieved.
  dyn_string ccpcList;
  //First get a list of presently registered CCPCs in the DIM server, without the system name present.
  fwCcpc_get(ccpcList, FWUKL1_SYS_NAME + ":");
  //Save the total number found for later use.
  const int numCcpc = dynlen(ccpcList);
  //Filter the CCPC names to determine, which of these are actually UKL1 boards.
  //Holds the names of the found UKL1 boards.
  dyn_string toBeAddedUkl1List;
  //Holds the version number of the found UKL1 boards.
  dyn_uint versionList;
  //Keep track of the number of UKL1 boards added to the list, start from 1 due to PVSS array silliness.
  unsigned numUkl1Added = 1;
  //Loop over all the found CCPCs.
  for (unsigned ccpc = 1; ccpc <= numCcpc; ++ccpc) {
    //Name of the CCPC currently being checked.
    const string ccpcName = ccpcList[ccpc];
    //Check to see if it is a production or prototype.
    //Ignore the return as it will probably just indicate a prototype anyway...
    int isUkl1;
    string serialNumber;
    fwUkl1_isUkl1Board(ccpcName, isUkl1, serialNumber, exceptionInfo);
    if ( exInfoSize != dynlen(exceptionInfo) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Adding board without a serial number to the system.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }
    //At the moment we can ignore any errors as if they do occur it means that we probably have the prototype board.
    //Eventually this will need to be checked properly.
    if ( 0 != isUkl1 ) {
      //Add the name of the board to the list of UKL1 names.
      toBeAddedUkl1List[numUkl1Added] = ccpcName;
      //Add constant to declare whether we are dealing with a production or prototype board.
      versionList[numUkl1Added] = isUkl1;
      //Increment the number added.
      ++numUkl1Added;
      //Indicate our progress.
      fwShowProgressBar("Identified " + FWUKL1_SYS_NAME + ":" + ccpcName + " with serial number 0x" + serialNumber + " as a UKL1 board.");
    }//if(0!=isUkl1) Don't do anything for else as we are just not interested in this board.
  }//CCPC for

  //Produce a warning message if we failed to find any UKL1s or even CCPCs.
  if ( 0 == numCcpc ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): No CCPCs found in the DIM server.", "0");
    exInfoSize = dynlen(exceptionInfo);
  }
  else if ( 0 == dynlen(toBeAddedUkl1List) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): No UKL1 boards found, but found " + numCcpc + " CCPCs in the DIM server.", "0");
    exInfoSize = dynlen(exceptionInfo);
  }
  //If we didn't find anything then the rest of this section is doesn't really need to be done, but it won't do anything anyway.

  //Now we have a list of UKL1 boards in the system lets deal with them.
  //The variable toBeAddedList holds the names of the UKL1 boards that need to be added to the library.
  //This always need to contain all the found boards as we always need to check the datapoints for consistency
  //and ensure the relevant DIM and FSM functionality is established.
  //The function fwUkl1_AddUkl1Board will ensure that the minimum possible is done depending on calling arguments.

  //Holds the names of the UKL1 boards that need to be removed from the library. It will either be those that are
  //no longer present after a refresh or all of them in the case of a reload.
  dyn_string toBeDeletedUkl1List;
  //Just find the new boards in the system and those that have been lost.
  if (refresh) {
    //Reset the number added.
    numUkl1Added = 1;
    //Look through the list of existing names and check for those that need to be deleted.
    //Get the list.
    const dyn_string currentList = fwUkl1_getUkl1NamesList(exceptionInfo);
    const int curListSize = dynlen(currentList);
    for (int board = 1; board <= curListSize; ++board) {
      //Check all the names in the library and check if they are still present in the new server list.
      if ( 0 == dynContains(toBeAddedUkl1List, currentList[board]) ) {
	//If they are not present they need to be deleted.
	toBeDeletedUkl1List[numUkl1Added] = currentList[board];
        fwShowProgressBar("Deleting " + currentList[board] + ".");
      }
    }
  }// if (refresh).
  else {
    //In this case we are starting afresh and can just remove any existing settings from 
    toBeDeletedUkl1List = fwUkl1_getUkl1NamesList(exceptionInfo);
    fwShowProgressBar("Deleting " + toBeDeletedUkl1List);
    //Will return an empty list in the event of an error. We will just have to see why later.
  }// else (refresh)

  //We will also delete any incomplete boards that are found in the system. They can always go.
  //Give all the incomplete names to the delete FSM DU and HW DP functions as they can handle a name
  //not exisiting.
  //Just return empty arrays in the event of failure so don't worry about errors.
  dynAppend(toBeDeletedUkl1List, fwUkl1_getUkl1HwDpNamesOnlyList(exceptionInfo));
  dynAppend(toBeDeletedUkl1List, fwUkl1_getUkl1FsmTreeNamesOnlyList(exceptionInfo));

  //Delete those boards that need to be deleted.
  if ( 0 != dynlen(toBeDeletedUkl1List) ) {
    //Exception info won't be checked after this call as we will continue anyway, just let the panel deal with the exceptions.
    fwUkl1_deleteUkl1Boards(toBeDeletedUkl1List, exceptionInfo);
    if ( exInfoSize != dynlen(exceptionInfo) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Failed to delete all the requested boards from the system.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }
  }

  //Add those boards that need to be added and add them in ascending order so they look nicer in the FSM.
  dynSortAsc(toBeAddedUkl1List);
  if ( 0 != dynlen(toBeAddedUkl1List) ) {
    //Exception info won't be checked after this call as we will continue anyway, just let the panel deal with the exceptions.
    fwUkl1_addUkl1Boards(toBeAddedUkl1List, versionList, deleteExisting, forceReconfigure, exceptionInfo);
    if ( exInfoSize != dynlen(exceptionInfo) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_findAndAddBoards(): Failed to all the requested boards to the system.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }
  }

  //Found and added all the boards!
  fwShowProgressBar("Create UKL1 list.");
  return;
}

/*!
 * Checks the given CCPC to see if it attached to UKL1 board. It identifies the board by the serial number accessible via
 * the I2C bus.
 *
 * \param  ccpcName Name of the CCPC as found in the DIM server.
 * \param  isUkl1 Returned by reference, there are three possible values:
 *   \li FWUKL1_PRODUCTION The board is a UKL1 production board.
 *   \li FWUKL1_PROTOTYPE The board is a UKL1 prototype board.
 *   \li 0 The board is not a UKL1.
 * \param  serialNumber Returned by reference, the serial number of the found board. Returns:
 *   \li Serial number of found board, whether UKL1 or not.
 *   \li Prototype in the event that it is a prototype without a working PROM. This will occur in preference to fail in the event of an error!
 *   \li Fail if an error occurred.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * There exists a UKL1 prototype that did not have a working PROM and as such no serial number can be read. In this case the
 * an exception is returned, but isUkl1 will indicate that it is a prototype and will return `PROTOTYPE' for the serial number.
 * This should be treated with suspicion especially if it is known that there are no prototype boards in the system. It will
 * eventually be `retired' and this function will need to be updated.
 */
void fwUkl1_isUkl1Board(string ccpcName, unsigned& isUkl1, string& serialNumber, dyn_string& exceptionInfo) {
  //Attempt to read the serial number from the given CCPC. All boards should have this identifier in the same place.
  //This cannot be done through the fwUkl1_getUkl1SerialNumber function as that assumes the existence of the relevant
  //hardware data point. We can't assume it exists as we want to check if the board is a UKL1 before creating it!
  //Use the fwCcpc library I2C read function. It should be on bus 0 at address 0x50, sub-address 0. It is 32-bits big (4-bytes)
  //and we will assume an infinite page size as it doesn't really matter.
  //The bus is not constant as when dealing with prototypes it could be 1!
  int bus=0;
  const int address=0x50, subAddress=0, size=4, pageSize=-1;
  //We wish the negative acknowledge on address to be enabled.
  const bool nack = TRUE;
  //Holds the read data.
  dyn_char dcData;
  int status1 = fwCcpc_I2CReadSub(ccpcName, bus, address, 0, size, pageSize, FWCCPC_I2C_COMBINED, nack, dcData);
  if ( 0 == status1 ) {
    //Successfully read the serial number from the CCPC. Is it an UKL1?
    string tmpstr = fwUkl1_convertByteToHexLittleEndian(dcData, exceptionInfo);
    const unsigned ccpcSerNum = fwCcpc_convertHexToDec(tmpstr, exceptionInfo );
    //If the word is byte swapped then the revision will occupy the 3rd nibble.
    //This we don't care about when identifying the board, hence mask it.
    if ( (0x601 == (ccpcSerNum>>20)) || (0x1060 == (ccpcSerNum&0xF0FF)) ) {
      //Check whether the serial number indicates that it is a production or prototype board.
      if ( (0x00020000 == (ccpcSerNum&0x000F0000)) || (0x0200 == (ccpcSerNum&0x0F00)) ) {
	//Production.
	isUkl1 = FWUKL1_PRODUCTION;
      } else {
	//Prototype
	isUkl1 = FWUKL1_PROTOTYPE;
      }
      //Convert the serial number to a string.
      serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
    } else {
      //The CCPC is not attached to a UKL1, so we must tell everyone.
      isUkl1 = 0;
      serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
    }
  }//if(0==status1)
  else {
    //We failed to read from the I2C bus at the appropriate location.
    //This could be because some of the prototype boards had a problem that meant it was moved to bus 1.
    //Try to read again, but this time on bus 1.
    bus = 1;
    dynClear(dcData);
    int status2 = fwCcpc_I2CReadSub(ccpcName, bus, address, subAddress, size, pageSize, FWCCPC_I2C_COMBINED, nack, dcData);
    if ( 0 == status2 ) {
      //Successfully read the serial number from the CCPC. Is it an UKL1?
      const unsigned ccpcSerNum = fwCcpc_convertByteToDec(dcData);
      if ( (0x601 == (ccpcSerNum>>20)) || (0x1060 == (ccpcSerNum&0xF0FF)) ) {
	//Check whether the serial number indicates that it is a production or prototype board.
	if ( (0x00020000 == (ccpcSerNum&0x000F0000)) || (0x0200 == (ccpcSerNum&0x0F00)) ) {
	  //Production.
	  isUkl1 = FWUKL1_PRODUCTION;
	} else {
	  //Prototype
	  isUkl1 = FWUKL1_PROTOTYPE;
	}
	//Convert the serial number to a string.
	serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
      } else {
	//The CCPC is not attached to a UKL1, so we must tell everyone.
	isUkl1 = 0;
	serialNumber = fwCcpc_convertDecToHex(ccpcSerNum);
      }
    }//if(0==status2)
    else {
      //This could be a broken prototype board, indicate that we think that and return the appropriate serial number,
      //but do flag it as an error.
      isUkl1 = FWUKL1_PROTOTYPE;
      serialNumber = "Prototype";
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_isUkl1Board(): " + ccpcName + " did not have a PROM at I2C bus address 0x50 and therefore was identified as a prototype UKL1 board.","2");
    }//else(0==status2)
  }//else(0==status1)

  return;
}

/*!
 * Adds a list of UKL1 boards to the system. It creates the hardware datapoint, subscribing it to the DIM server;
 * and the finite statemachine datapoint, connecting the callback function for state changes. The names of those
 * added are put in the library list of DIM server names.
 *
 * \param  toBeAddedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be added to the system. If a name is already found in the system it will be removed and then readded.
 * \param  versionList FWUKL1_PRODUCTION if the UKL1 board to be added is a production version, FWUKL1_PROTOTYPE
 *           if it is a prototype version.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings. Default: FALSE.
 * \param  forceReconfigure If TRUE it will force the settings to be manually set and not to be loaded from the defaults
 *           for the fwHw datapoint type. FALSE and the settings are just loaded from the defaults. Default: FALSE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_addUkl1Boards(dyn_string toBeAddedUkl1List, dyn_uint versionList, bool deleteExisting, bool forceReconfigure, dyn_string& exceptionInfo) {
  //Current length of the exception info, used to determine if more errors have appeared.
  int exInfoSize = dynlen(exceptionInfo);

  //Store the number of elements in the list.
  const unsigned numToBeAdded = dynlen(toBeAddedUkl1List);
  //If the list was empty return!
  if ( 0 == numToBeAdded )
    return;

  //Loop over each board adding them to the system.
  for (unsigned board = 1; board <= numToBeAdded; ++board) {
    const string ukl1Name = toBeAddedUkl1List[board];
    const unsigned ukl1Version = versionList[board];
    //Create the hardware type datapoint and subscribe it to the DIM server.
    //We want to just load the new datapoint from the HW type defaults and if it does exist then lets delete it.
    _fwUkl1_createHwTypeCCPCUKL1Datapoint(ukl1Name, ukl1Version, deleteExisting, forceReconfigure, exceptionInfo);
    if ( exInfoSize < dynlen(exceptionInfo) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to setup " + ukl1Name + "'s connection with the DIM server. Communication with this UKL1 is not possible.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }

    fwShowProgressBar("Added HW DP for " + ukl1Name + ".");
  }//for each board.

  //Add the boards as device units in the FSM tree.
  _fwUkl1_createUkl1FsmTreeDU(toBeAddedUkl1List, deleteExisting, exceptionInfo);
  if ( exInfoSize < dynlen(exceptionInfo) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_addUkl1Boards(): Failed to add the boards to the FSM tree list.", "2");
  }
  
  //Done.
  return;
}//fwUkl1_addUkl1Boards()

/*!
 * Deletes a list of UKL1 boards from the system. It removes the hardware datapoint, unsubscribing it from the DIM server;
 * and the finite statemachine datapoint, disconnecting the callback function for state changes. The names of those
 * added are removed from the list of UKL1 names stored in the library.
 *
 * \param  toBeDeletedUkl1List A dyn_string of names of the UKL1 boards as they appear in the DIM server that need to
 *           be removed from the system.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * This function reports its progress via fwShowProgressBar() and thus fwOpenProgressBar() can be called to monitor its progress.
 * It does not submit a percentage completion however and can only be used with types 1 and 3 with fwOpenProgressBar().
 */
void fwUkl1_deleteUkl1Boards(dyn_string toBeDeletedUkl1List, dyn_string& exceptionInfo) {
  //Store the number of elements in the list.
  const unsigned numToBeDeleted = dynlen(toBeDeletedUkl1List);
  //If the list was empty return!
  if ( 0 == numToBeDeleted )
    return;

  //Loop over each board removing them from the system.
  for (unsigned board = 1; board <= numToBeDeleted; ++board) {
    const string ukl1Name = toBeDeletedUkl1List[board];

    //Remove the hardware type datapoint and unsubscribes it from the DIM server.
    //Only remove it if it is present in the list of hardware.
    dyn_string hwList;
    bool bStatus = fwHw_get("FWUKL1_HW_TYPE", hwList);
    //If we weren't successful in retrieving the list of hardware then try and delete it anyway.
    //If we were successful only try and delete it if it is present in the system.
    if ( !bStatus || dynContains(hwList, ukl1Name) ) {
      bStatus = fwHw_remove(FWUKL1_HW_TYPE, ukl1Name);
      if ( !bStatus )
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_deleteUkl1Board(): Failed to remove " + ukl1Name + "'s DIM server subscriptions.", "2");

      fwShowProgressBar("Removed HW DP for " + ukl1Name + ".");
    }//if UKL1 is present in list.
    else
      fwShowProgressBar("HW DP for " + ukl1Name + " not present in the system to remove.");
  }//for each board.

  //Remove the list of boards to the FSM tree.
  const int exInfoSize = dynlen(exceptionInfo);
  _fwUkl1_deleteUkl1FsmTreeDU(toBeDeletedUkl1List, exceptionInfo);
  if ( exInfoSize < dynlen(exceptionInfo) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_deleteUkl1Boards(): Failed to delete the boards from the FSM tree list.", "2");
  }
  
  //Done.
  return;
}//fwUkl1_deleteUkl1Boards()

/*!
 * Returns the full list of names of UKL1 boards that are present in both the FSM tree and as a HWTypeCCPCUKL1 data point
 * and hence are fully accessible in the system.
 *
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_string List of all the names of the UKL1 board that has been retrieved.
 */
dyn_string fwUkl1_getUkl1NamesList(dyn_string& exceptionInfo) {
  //This will hold the list to that is returned.
  dyn_string namesList;

  //Get the list of HW DPs in the system.
  dyn_string hwList;
  if ( fwHw_get(FWUKL1_HW_TYPE, hwList) ) {
    //Now we have the hardware list check to see if it is a node. We don't care if it is local or remote, both are >0 so just check that.
    const int numHw = dynlen(hwList);
    //This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int hw = 1; hw <= numHw; ++hw) {
      if ( 0 < fwFsmTree_isNode(hwList[hw]) ) {
	//Add it to the list.
	namesList[nameIndex] = hwList[hw];
	++nameIndex;
      }//if isNode.
      //No need for else we don't keep track here of incomplete names.
    }//for hw.
  } else {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_getUkl1NamesList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");
  }

  //Put what ever was retrieved in ascending order to make it look nice.
  dynSortAsc(namesList);
  //Done.
  return namesList;
}

/*!
 * Returns number of UKL1 board names that are currently in the system.
 *
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return unsigned Number of UKL1 boards that are stored in the list, -1 in the event of an error.
 */
int fwUkl1_getNamesListSize(dyn_string& exceptionInfo) {
  const int exInfoSize = dynlen(exceptionInfo);
  const int size = dynlen(fwUkl1_getUkl1NamesList(exceptionInfo));
  if ( (exInfoSize < dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    //Indicate something went wrong via a message.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getNamesListSize(): Error occurred while retrieving names list.", "1");
    //Set the size to -1 to indicate an error.
    size = -1;
  }
  else if ( size == -1 ) {
    //Indicate PVSS error.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getNamesListSize(): PVSS encountered an error while determining the size of the list.", "1");
    //Size is -1 so don't worry about it.
  }
  return size;   
}

/*!
 * Returns the names of the UKL1 board names that are current stored as hardware data points but are not present in the FSM tree.
 *
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_string List of all the names of the UKL1 board that are present only as hardware data points. Empty if there are none.
 */
dyn_string fwUkl1_getUkl1HwDpNamesOnlyList(dyn_string& exceptionInfo) {
  //This holds the list that is to be returned.
  dyn_string namesList;

  //Get the list of HW DPs in the system.
  dyn_string hwList;
  if ( fwHw_get(FWUKL1_HW_TYPE, hwList) ) {
    //Now we have the hardware list check to see if it is a node. 0 indicates that it is not.
    const int numHw = dynlen(hwList);
    //This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int hw = 1; hw <= numHw; ++hw) {
      if ( 0 == fwFsmTree_isNode(hwList[hw]) ) {
	//Add it to the list.
	namesList[nameIndex] = hwList[hw];
	++nameIndex;
      }//if is not node.
      //No need for else we don't keep track here of complete names.
    }//for hw.
  } else {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_getUkl1HwDpNamesOnlyList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");
  }

  //Hopefully this is empty.
  return namesList;
}

/*!
 * Returns the names of the UKL1 board names that are current present in the FSM tree, but are not stored as hardware data points.
 *
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_string List of all the names of the UKL1 board that are present only as hardware data points. Empty if there are none.
 */
dyn_string fwUkl1_getUkl1FsmTreeNamesOnlyList(dyn_string& exceptionInfo) {
  //This holds the list that is to be returned.
  dyn_string namesList;

  //Get the list of children in the FSM tree node.
  //The function also returns by reference the type of each child, CU, LU or DU. We don't care, should only ever be a DU.
  dyn_int junkChildrenTypes;
  dyn_string fsmList = fwCU_getChildren(junkChildrenTypes, FWUKL1_CU);
  //There is a special FSM child called `FWUKL1_CU'_confDB, which should not be returned.
  //It will never exist in the hardware, but should always exist in the FSM tree.
  int confDbIndex = dynContains(fsmList, (FWUKL1_CU + "_ConfDB"));
  dynRemove(fsmList, confDbIndex);
  //Now get the list of HW DPs for comparison with.
  dyn_string hwList;
  if ( fwHw_get(FWUKL1_HW_TYPE, hwList) ) {
    //How many FSM children are there? Required to know if we are to loop over them.
    const int numChildren = dynlen(fsmList);
    //This keeps track of the index in the names list that we wish to add elements for.
    int nameIndex = 1;
    for (int child = 1; child <= numChildren; ++child) {
      if ( 0 == dynContains(hwList, fsmList[child]) ) {
	//The FSM child is not present as a harware DP.
	namesList[nameIndex] = fsmList[child];
	++nameIndex;
      }//if is not in the hardware list.
      //No need for else we don't keep track here of complete names.
    }//for children.
  } else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1FsmTreeNamesOnlyList(): Failed to retrieve list of UKL1 hardware datapoints in the system, assuming there are none.", "2");
  }

  //Hopefully this is empty.
  return namesList;
}

// ============
//   INTERNAL
// ============

/*!
 * This is used to add list of UKL1 boards to the appropriate FSM tree. It will add the UKL1 boards as device units (DUs) under
 * the control unit (CU) defined by FWUKL1_CU.
 *
 * \param  ukl1List Names of UKL1 boards to be added to the FSM tree. Should be as they appear in the DIM server.
 * \param  deleteExisting If TRUE then if the DU is found in the tree it will be deleted before it is readded. If FALSE then the settings
 *           for the DU will just be reset. Default: FALSE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * The DUs added to the tree will appear as their DIM server name, with the panel fwUkl1DUOverview.pnl added for
 * User interaction with the hardware. The label for the DUs will be the name of the UKL1 board as it appears in the DIM server.\n
 * The FSM CU node may have to be stopped for this function to complete and the FWUKL1_CU node will be restarted once the
 * operation is complete. No other nodes in the tree will be affected.\n
 * The behaviour of this function is dependent upon the value of deleteExisting, the behaviour is described below:
 *   \li Creation - If the DU does not exist it will be created with the settings as described above.
 *         In this case the FSM tree node is modified and requires it to be stopped and restarted.
 *   \li Hard recreation - DU exists, deleteExisting=TRUE. The DU is first deleted and then the function then proceeds as for creation.
 *         This requires the FSM tree node to be stopped and restarted.
 *   \li Soft recreation - DU exists, deleteExisting=FALSE. This does not require the FSM tree node to be stopped or restarted, it will
 *         simply ensure that the User panel and label are set correctly. Default: FALSE.
 *
 * Note that if the CU (as defined by FWUKL1_CU) does not exist then it will be created and the function will proceed as for creation.
 * The function will also refresh the FSM tree list in the device editor and navigator (DEN) and also will display the CU panel which
 * will refresh it if the list changed.
 */
void _fwUkl1_createUkl1FsmTreeDU(dyn_string ukl1List, bool deleteExisting, dyn_string& exceptionInfo) {
  //Don't do anything if the list is empty.
  if ( 0 == dynlen(ukl1List) ) {
    return;
  }

  //It is possible to call this function and for it to have no effect on the FSM tree node. In this case there is
  //no point in stopping, regenerating and restarting the FSM tree node. This variable is set to true if a restart
  //is required.
  bool restart = FALSE;
  //This flag is used to determine if we should create the DU or not.
  bool createDu = FALSE;
  //This flag is used to signal if there is no CU.
  bool noCU = FALSE;
  //First we must determine if the FWUKL1_CU node exists in the FSM tree.
  const int nodeExists = fwFsmTree_isNode(FWUKL1_CU);
  //If the node does not exist then we must create it.
  if ( 0 == nodeExists ) {
    //Stop the FSM tree to do this. Can't have been stopped by this stage.
    fwFsmTree_stopTree();
    //Tree was stopped will need to regenerate and restart.
    restart = TRUE;
    //Create the node as a control unit (last arg=1) and put it under the root node `FSM'.
    const string createdNode = fwFsmTree_addNode("FSM", FWUKL1_CU, "DAQ_Domain_v1", 1);
    if ( "" != createdNode ) {
      //It created the node and now we must set the label and UI panel.
      fwFsmTree_setNodeLabel(FWUKL1_CU, "RICH L1");
      fwFsmTree_setNodePanel(FWUKL1_CU, "fwUkl1/fwUkl1CUOverview.pnl");
    } else {
      //Tell everyone we have no CU.
      noCU = TRUE;
      //Failed to create the node, update the status value and tell everyone.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): Failed to create the control unit in the FSM state machine.", "1");
    }
  }//if(0==nodeExists)

  //The next section can be done regardless of whether the node exist section was called or not,
  //but if it failed then we don't have a CU FSM tree node to add the DUs to so check the status is still good.
  if ( !noCU ) {
    //Loop over all the names given in the ukl1List and determine what is the appropriate behaviour for each.
    const int numUkl1s = dynlen(ukl1List);
    for (int ukl1 = 1; ukl1 <= numUkl1s; ++ukl1) {
      //Get the name of the UKL1 board as it will appear in the FSM.
      const string ukl1FsmName = ukl1List[ukl1];
      //Check whether the DU exists.
      const int duExists = fwFsmTree_isNode(ukl1FsmName);
      if ( 0 == duExists ) {
	//If the node does not exist then we must create it.
	createDu = TRUE;
      }//if(0==duExists)
      else {
	//Determine whether or not to delete the DU before it is recreated.
	if ( deleteExisting ) {
	  //Remove the DU, which will require the node to be stopped first if it has not already been stopped.
	  if (!restart) {
	    //Must always stop everything.
	    fwFsmTree_stopTree();
	    //We are going to have to restart now.
	    restart = TRUE;
	  }
	  //Remove the DU from the tree, accept the default of removing a node recursively from the tree as the DU shouldn't have any children!
	  fwFsmTree_removeNode(FWUKL1_CU, ukl1FsmName);
	  //Check the DU was actually deleted.
	  const int duDeleted = fwFsmTree_isNode(ukl1FsmName);
	  if ( 0 == duDeleted ) {
	    //DU was deleted so create it.
	    createDu = TRUE;
	  } else {
	    //The DU still exists in the tree when it shouldn't. Warn everyone and update the status.
	    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_createUkl1FsmTreeDU(): Failed to delete " + ukl1Name + " from the FSM tree. UKL1 boards may exist in the list that no longer exist in the system.", "2");
	  }
	}//if(deleteExisting)
      }//else(0==duExists)
	
      //If we need to create the DU then we best had.
      if ( createDu ) {
	//We need to be stopped by here if we are not already.
	if (!restart) {
	  //Must always stop everything.
	  fwFsmTree_stopTree();
	  //We are going to have to restart now.
	  restart = TRUE;
	}
	//Add the board to the tree as a HwTypeCCPCUKL1 type and as a device unit (last arg=0).
	fwFsmTree_addNode(FWUKL1_CU, ukl1FsmName, "HwTypeCCPCUKL1", 0);
      }//if(createDu)
	
      //Check to see if the node was added. Done for all cases.
      const int duPresent = fwFsmTree_isNode(ukl1FsmName);
      if ( 0 != duPresent ) {
	//Added the board :) Now set the label and the User panel.
	fwFsmTree_setNodeLabel(ukl1FsmName, ukl1List[ukl1]);
	fwFsmTree_setNodePanel(ukl1FsmName, "objects/fwUkl1/fwUkl1DUOverview.pnl");
        fwShowProgressBar("Added " + ukl1FsmName + " to the FSM.");
      } else {
	//Failed to add the DU. Warn people and update the status appropriately.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createUkl1FsmTreeDU(): Failed to add " + ukl1Name + " to the FSM tree. Communication with this board will not be possible via FSM.", "2");
      }
    }//for(numUkl1s)
  }//if(noCU) Don't bother with an else we already have the appropriate status and will have signaled the error.

  //We should only regenerate and restart the FSM tree node if it has actually been stopped and modified.
  if (restart) {
    //Commented functions should not need to be performed left here for reference...
    //fwFsmTree_generateAll();
    //Generate the FSM types.
    //fwFsmTree_generateFsmTypes();
    //Generate the FSM for CU
    fwFsmTree_generateTreeNode(FWUKL1_CU);
    // Start FSM processes for the CU sub-tree
    fwFsmTree_startTree();
  }

  //Refresh the Tree in case the DEN is running, this can be done regardless. It ensures that the tree display is correct.
  fwFsmTree_refreshTree();
  //Don't try and view the CU panel as if it is run in a script in the console it has no concept of a UI and ths can't start it.

  //Now we are done.
  return;
}//_fwUkl1_createUkl1FsmTreeDU()

/*!
 * This is used to delete a list UKL1 boards from the appropriate FSM tree. It will look in the tree FWUKL1_CU
 *
 * \param  ukl1List List of names of UKL1 boards to be deleted from the FSM tree. Names should be as they appear in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_deleteUkl1FsmTreeDU(dyn_string ukl1List, dyn_string& exceptionInfo) {
  if ( 0 == dynlen(ukl1List) ) {
    //Nothing to do so just exit.
    return;
  }

  //First we must determine if the FWUKL1_CU node exists in the FSM tree.
  const int nodeExists = fwFsmTree_isNode(FWUKL1_CU);
	
  //Does the node we wish to delete from exist?
  if ( 0 != nodeExists ) {
    //The node exists, either locally or remotely.
    //Stop it so we can edit it, accept the default of stopping all children.
    fwFsmTree_stopTree();
    //Loop over all the names given in the ukl1List and determine what is the appropriate behaviour for each.
    const int numUkl1s = dynlen(ukl1List);
    for (int ukl1 = 1; ukl1 <= numUkl1s; ++ukl1) {
      //Convert the ukl1Name into the name as it will appear in the FSM.
      const string ukl1FsmName = ukl1List[ukl1];
      //Check whether the DU exists.
      const int duExists = fwFsmTree_isNode(ukl1FsmName);
      if ( 0 != duExists ) {
	//Remove the DU from the tree, accept the default of removing a node recursively from the tree as the DU shouldn't have any children!
	fwFsmTree_removeNode(FWUKL1_CU, ukl1FsmName);
	//Check the DU was actually deleted.
	const int duDeleted = fwFsmTree_isNode(ukl1FsmName);
	if ( 0 == duDeleted ) {
          fwShowProgressBar("Removed " + ukl1FsmName + " from FSM.");
        }//if(0==duDeleted)
        else {
	  //The DU still exists in the tree when it shouldn't. Warn everyone and update the status.
	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_deleteUkl1FsmTreeDU(): Failed to delete " + ukl1Name + " from the FSM tree. There is no physical board in the crate, but it will still appear in the FSM tree.", "2");
	}//else(0==duDeleted)
      }//if(0==duExists)
      else
        fwShowProgressBar(ukl1FsmName + " is not present in the FSM tree to remove.");
    }//for(numUkl1s)

    //Only any point of doing this if the node exists.
    //Generate the FSM for CU.
    fwFsmTree_generateTreeNode(FWUKL1_CU);
    //Start FSM processes for the CU sub-tree. We could have deleted only a couple of boards.
    fwFsmTree_startTreeNode(FWUKL1_CU);
    //Refresh the Tree in case the DEN is running, this can be done regardless.
    fwFsmTree_refreshTree();
    //Don't try and reopen the CU, we might not be able to achieve that if we are running in a console script.
  }//if(0!=nodeExists)
  else {
    //The node does not exist so we cannot delete any boards. Warn people, it will happen on a fresh install though.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_deleteUkl1FsmTreeDU(): UKL1 control unit `" + FWUKL1_CU + "' not found. UKL1 FSM tree does not exist, cannot remove UKL1 boards from a none existant FSM tree.", "1");
  }//else(0!=nodeExists)

  //Done.
  return;
}//_fwUkl1_deleteUkl1FsmTreeDU()

/*!
 * This will create a datapoint from the HwTypeCCPCUKL1 datapoint type, which represents a UKL1 board, with all
 * the appropriate settings and subscribe these registers to the DIM server.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version If FWUKL1_PRODUCTION it is a production board, if it is FWUKL1_PROTOTYPE it is a prototype board,
 *           this affects the I2C bus that needs to be used.
 * \param  deleteExisting If the datapoint is found to exist it will be deleted and then recreated with the relevant
 *           settings.
 * \param  forceReconfigure If TRUE it will force the settings to be set from hardcoded settings and not to be loaded from
 *           the defaults for the fwHw datapoint type, it will also reset the default settings to those hardcoded in the library.
 *	     FALSE and the settings are just loaded from the defaults.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * This function has the following possible ways of operating:
 *    \li Hard creation - Datapoint does not exist and the datapoint is created, the settings are loaded from the library
 *          and saved to defaults, all registers are subscribed to the DIM server. forceReconfigure=TRUE, deleteExisting has
 *          no relevance in this case.
 *    \li Soft creation - Datapoint does not exist and the datapoint is created, the settings are loaded from the
 *          defaults, all registers are subscribed to the DIM server. forceReconfigure=FALSE, deleteExisting has
 *          no relevance in this case.
 *    \li Full hard recreation - Datapoint exists, deleteExisting=TRUE, forceReconfigure=TRUE the datapoint registers
 *          are unsubscribed from the DIM server and then the datapoint is deleted. It then proceeds as for
 *          hard creation.
 *    \li Full soft recreation - Datapoint exists, deleteExisting=TRUE, forceReconfigure=FALSE the datapoint registers
 *          are unsubscribed from the DIM server and then the datapoint is deleted. It then proceeds as for
 *          soft creation.
 *    \li Hard recreation - Datapoint exists, deleteExisting=FALSE, forceReconfigure=TRUE, all the settings and their
 *          defaults are reset to the settings as for a full creation and DIM server subscriptions are recreated.
 *          The datapoint is never deleted.
 *    \li Soft recreation - Datapoint exists, deleteExisting=FALSE, forceReconfigure=FALSE, all the settings are reloaded
 *          from their default values. The datapoint is never deleted and the DIM server subscriptions are not recreated.
 *
 * The name of the created datapoint will be the concatantion of FWUKL1_DPNAME_PREFIX and ukl1Name. The setting placed in this
 * datapoint set the name of the UKL1 board as in the DIM server and the address of all the registers that
 * the datapoint can access. Only status registers are set up for monitoring and the datapoints do not contain
 * any values, hence none are written to the board. Note that no inverse is provided for this function
 * i.e._fwUkl1_deleteHwTypeCCPCUKL1Datapoint as this is done via fwHw_remove.\n\n
 */
void _fwUkl1_createHwTypeCCPCUKL1Datapoint(string ukl1Name, unsigned ukl1Version, bool deleteExisting, bool forceReconfigure, dyn_string& exceptionInfo) {
  if ( "" == ukl1Name ) {
    //Nothing to delete.
    return;
  }

  //Some datapoint and datapoint type related constants...
  //Name of the datapoint type we are creating from.
  const string dptName = FWHW_TYPE_DPT + FWHW_CCPC_DESCRIPTION + FWUKL1_HW_TYPE;
  //Name of the default settings data point.
  const string defaultSettingsDpName = FWHW_DEFAULT_SETTINGS_DP + FWHW_CCPC_DESCRIPTION + FWUKL1_HW_TYPE;

  //Check to see if the default settings datapoint actually exists. If it doesn't then we should add it
  //to the system, otherwise it will cause much confusion later... especially as there is no error checking
  //associated with loading/saving to the default values!
  if ( !dpExists(defaultSettingsDpName) ) {
    //Potentially override the User choice, but then we must...
    forceReconfigure = TRUE;
    //Create the hardware datapoint, but make sure we don't try and load the default settings as we are creating them.
    if ( fwHw_create(FWUKL1_HW_TYPE, defaultSettingsDpName, FALSE) ) {
      //Successfully recreated the defaults datapoint, better warning people it happened as it is a bit odd.
      //Actually this will happen during the install and will be normal, don't signal it as an error.
      //fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Had to recreate the UKL1 hardware data point's default settings. This indicates that the install may not have proceeded correctly, this specific error should not cause any further problems it may indicate a more serious problem however.", "2");
    } else {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to recreate the default settings for the UKL1 hardware data point. This should not have to have needed to be done in the first place (install of fwUkl1 component may not have completed successfully) and failure to create it indicates a more serious problem.", "2");
    }
  }

  //Now check to see if the datapoint exists.
  if ( dpExists(ukl1Name) ) {
    //Datapoint exists. There are three possible cases that need to be handled here...

    //Soft recreation.
    if ( !deleteExisting && !forceReconfigure ) {
      //No real need to add anything to the exception here.
      _fwUkl1_configureHwTypeCCPCUKL1Settings(ukl1Name, ukl1Version, FALSE, exceptionInfo);

      //Ensure the DIM services are setup. If they are it will be ignored otherwise they will be set up.
      if ( !fwHw_subscribe(ukl1Name) )
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to subscribe " + ukl1Name + " to the DIM server, communication with this UKL1 board will not be possible.", "2");

    }//if ( !deleteExisting && !forceReconfigure )

    //Hard recreation.
    else if ( !deleteExisting && forceReconfigure ) {
      //Reconfigure all the settings from fwUkl1 lib values and save as defaults.
      _fwUkl1_configureHwTypeCCPCUKL1Settings(ukl1Name, ukl1Version, TRUE, exceptionInfo);

      //Now resubscribe to the server.
      if ( !fwHw_subscribe(ukl1Name) )
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to subscribe " + ukl1Name + " to the DIM server, communication with this UKL1 board will not be possible.", "2");

    }//else if ( !deleteExisting && forceReconfigure )

    //Full hard&soft recreation, hardness determined by forceReconfigure during recreation.
    else if ( deleteExisting ) {
      //Here we want to delete the existing datapoint and recreate it.
      //fwHw_remove deletes the hardware data point and completely unsubscribes it from the DIM server.
      bool status = fwHw_remove(FWUKL1_HW_TYPE, ukl1Name);
      if ( !status )
	fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to delete " + ukl1Name + "'s hardware data point. Attempting to reconfiguring and use the existing data point.", "2");

      //Whether we deleted the data point or not we should try and recreate it by calling ourselves again.
      //Behaviour depends on forceReconfigure, we definately don't want to try and delete the data point.
      //If we do try and delete the data point we could potentially get into an infinite loop if some how it isn't removed.
      //By not asking to delete it, if it isn't removed then we can just reconfigure it and won't get to this stage again.
      //Used to check no exceptions occurred during this function call.
      const int exInfoSize = dynlen(exceptionInfo);
      _fwUkl1_createHwTypeCCPCUKL1Datapoint(ukl1Name, ukl1Version, FALSE, forceReconfigure, exceptionInfo);
      if ( exInfoSize < dynlen(exceptionInfo) )
	fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Problems were encountered while recreating " + ukl1Name + "'s hardware datapoint. Check previous errors for details.", "2");

      //Just return as we have done everything we want to.
      return;
    }// else if ( deleteExisting )

  }//if ( dpExists(ukl1Name) )

  else {
    //Hard creation.
    if ( forceReconfigure ) {
      //DP doesn't exist so create it, but don't reload from defaults we need to set those up first.
      if ( fwHw_create(FWUKL1_HW_TYPE, ukl1Name, FALSE) ) {
	//Set all the values on the datapoint to something sensible, it will create a detailed enough exception tree.
	_fwUkl1_configureHwTypeCCPCUKL1Settings(ukl1Name, ukl1Version, TRUE, exceptionInfo);
      } else {
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to create the hardware data point for " + ukl1Name + ", communication with this board is not possible.", "1");
      }
    }// if ( forceReconfigure )

    //Soft creation
    else {
      //Use the hardware tool to create the datapoint and don't save to defaults.
      if ( fwHw_create(FWUKL1_HW_TYPE, ukl1Name, FALSE) ) {
	//Set all the values on the datapoint to something sensible, it will create a detailed enough exception tree.
	_fwUkl1_configureHwTypeCCPCUKL1Settings(ukl1Name, ukl1Version, FALSE, exceptionInfo);
      } else {
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to create the hardware data point for " + ukl1Name + ", communication with this board is not possible.", "1");
      }
    }// else (forceReconfigure)

    //Now subscribe all the registers to the DIM server. DIM server will not do anything if the registers are already found to be subscribed.
    if ( !fwHw_subscribe(ukl1Name) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_createHwTypeCCPCUKL1Datapoint(): Failed to subscribe " + ukl1Name + " to the DIM server, communication with this board is not possible.", "2");
    }

  }// else ( dpExists(ukl1Name) )

  //If we got this far then everything must have been created and subscribed without issue.
  return;
}//_fwUkl1_createHwTypeCCPCUKL1Datapoint()

/*!
 * This is used to configure the register settings of the HwTypeCCPCUKL1 datapoint type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server. The appropriate
 *           datapoint to be set will be found from this.
 * \param  ukl1Version Verision number of the UKL1 board, either FWUKL1_PRODUCTION or FWUKL1_PROTOTYPE.
 * \param  saveAsDefaults If true then the settings loaded into the datapoint are also saved as
 *           its default values.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1Settings(string ukl1Name, unsigned ukl1Version, bool saveAsDefaults, dyn_string& exceptionInfo) {
  //Return status of the function.
  int status = 0;

  //Decide how best to load the settings.
  //This can be used to mark that we failed to load from the default settings and should try to load from hardware defaults.
  bool failedFromDefaults = FALSE;
  //This is the case if we wish to load from the default settings.
  //Can only be done for production boards, when we don't want to save the settings to the defaults.
  if ( !saveAsDefaults && (FWUKL1_PRODUCTION == ukl1Version) ) {
    //In this case we can just reload from the default settings.
    if ( !fwHw_applyDefaultSettings(FWUKL1_HW_TYPE, ukl1Name) ) {
      failedFromDefaults = TRUE;
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "Failed to apply default settings to " + ukl1Name + ", attempting to configure the board from hardcoded defaults.", "2");
    }
  }

  //Write the common settings to the all the datapoints on the board, this doesn't have a default
  //setting so it must always be done.
  _fwUkl1_configureHwTypeCCPCUKL1CommonSettings(ukl1Name, exceptionInfo);
  //Initialise the general settings, such as the FSM state and exception log. Don't have defaults associated with them.
  _fwUkl1_configureHwTypeCCPCUKL1GeneralSettings(ukl1Name, ukl1Version, exceptionInfo);
    
  //This case is when we wish to reload the default settings or we wish to configure a prototype board.
  //The prototypes require a different I2C bus number to that stored in the default settings.
  //If we wish to reset the defaults to the hardcoded settings then we can't load from them!
  if ( saveAsDefaults || (FWUKL1_PROTOTYPE == ukl1Version) || failedFromDefaults ) {
    //Even if these functions fail we will try to configure the rest of the settings and can check the error
    //log higher up to see if enough failed to indicate a major problem.
    //This notes the size of the exception now so we can check if any more appeared in the log after all these function calls.
    const int exInfoSize = dynlen(exceptionInfo);

    //Now we must do the specific settings for all of the hardware type included in the UKL1 type.
    //GBE
    _fwUkl1_configureHwTypeCCPCUKL1GBESpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //TTCrx
    _fwUkl1_configureHwTypeCCPCUKL1TTCRXSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //Ingress FPGA
    _fwUkl1_configureHwTypeCCPCUKL1IngressFpgaSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //Egress FPGA
    _fwUkl1_configureHwTypeCCPCUKL1EgressFpgaSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //UKL1 resets
    _fwUkl1_configureHwTypeCCPCUKL1ResetsSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //UKL1 PROM
    _fwUkl1_configureHwTypeCCPCUKL1PROMSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
    //UKL1 temperature sensors.
    _fwUkl1_configureHwTypeCCPCUKL1TempSensorSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);

    //Check to see if any exceptions were risen.
    if ( exInfoSize >= dynlen(exceptionInfo) ) {
      //No new exceptions were generated so set them as the defaults, if we are asked to.
      //Having done all that do we want to save them as the defaults?
      if (saveAsDefaults) {
	//Will get the settings that we have just set and save them to the default values.
	//At present it appears that this function just returns 1, update this to check for errors once it can actually show them!
	if ( 1 != fwHw_saveSettingsAsDefault(FWUKL1_HW_TYPE, ukl1Name) )
	  fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_configureHwTypeCCPCUKL1Settings(): " + ukl1Name + "'s default settings were not saved correctly, this could mean they are corrupt and the data point should not be configured from the default settings until this problem is fixed.","2");
      }
    } else {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureHwTypeCCPCUKL1Settings(): " + ukl1Name + " will operate with limited functionality as some (or all) of the registers were not configured correctly. DIM communication will not be possible for those that configuration failed.", "2");
      if (saveAsDefaults)
	fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureHwTypeCCPCUKL1Settings(): " + ukl1Name + "'s default settings were not saved as the data point from which the settings would be saved from was not configured correctly hence corrupt defaults could be created.", "2");
    }

  }

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1Settings(

/*!
 * This will set the common settings for the fwHw type UKL1 and all included fwHw types.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureHwTypeCCPCUKL1CommonSettings(string ukl1Name, dyn_string& exceptionInfo) {
  //The common setting to be written.
  dyn_string commonSettings;
  commonSettings[1] = ukl1Name;
  //Write the common setting, this is done recursively so all the subtypes have their common setting
  //updated to.
  if ( !fwHw_setCommonSettings(ukl1Name, commonSettings, TRUE) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1CommonSettings(): Failed to set the CCPC name for " + ukl1Name + ", this will prevent correct communication with this UKL1.", "1");
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1CommonSettings()

/*!
 * This will set the specific settings for the status data point element that the UKL1 HW type has.
 * It controls the FSM interactions. It should be ensured that the ukl1Name HW data point exists before calling this function.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureHwTypeCCPCUKL1GeneralSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name and add the name of the DPE to the end.
  string dpeName = ukl1Name + ".status";
  //Set the board to NOT_READY by default.
  if ( 0 != dpSet(dpeName, "NOT_READY") )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1GeneralSettings(): Failed to initialise the UKL1 FSM state to NOT_READY for, problems accessing the DP were encountered. Board will be in a UNKNOWN state when FSM is started.", "1");
  //Done.
}//_fwUkl1_configureHwTypeCCPCUKL1GeneralSettings()

/*!
 * This will set the specific settings for the fwHw type GBE that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureHwTypeCCPCUKL1GBESpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".GBE";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "JTAGID";
  registerList[2]  = "MACSoftReset";
  registerList[3]  = "RxFifoPrtReset";
  registerList[4]  = "TxFifoPrtReset";
  registerList[5]  = "PortEnable";
  registerList[6]  = "RxFifoEnable";
  registerList[7]  = "Mode";
  registerList[8]  = "Clock";
  registerList[9]  = "SPI3ConfigTrnmtGlobal";
  registerList[10] = "SPI3ConfigRcv";
  registerList[11] = "RxStatus";
  registerList[12] = "PHYCtrl";
  registerList[13] = "PHYData";
  registerList[14] = "MDIOControl";
  //This contains the address of all the registers to be set.
  dyn_uint regAddrList;
  regAddrList[1]  = 0x50c;
  regAddrList[2]  = 0x505;
  regAddrList[3]  = 0x59e;
  regAddrList[4]  = 0x620;
  regAddrList[5]  = 0x500;
  regAddrList[6]  = 0x5b3;
  regAddrList[7]  = 0x501;
  regAddrList[8]  = 0x794;
  regAddrList[9]  = 0x700;
  regAddrList[10] = 0x701;
  regAddrList[11] = 0x793;
  regAddrList[12] = 0x680;
  regAddrList[13] = 0x681;
  regAddrList[14] = 0x683;
  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of 32-bit words for command.
  //3rd element is the check for change rate, set to 2 to match Tell1 change rate and to be compatible with their panels.
  //4th element is send on change only setting.
  dyn_int specificSetting = makeDynAnytype(0x00, 1, 2, 1);
  dyn_dyn_int specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Now write all these to the datapoint.
  //Set the general GBE port settings.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1GBESpecificSettings(): Failed to setup the specific settings for the GBE ethernet registers for, the GBE cannot be configured and status information will not be available.", "1");

  //Now setup the PROM, this will indicate its own errors don't need to add anything here.
  _fwUkl1_configureHwTypeCCPCUKL1GBEPROMSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);
  //Now setup the four ports, this will indicate its own errors don't need to add anything here.
  _fwUkl1_configureHwTypeCCPCUKL1GBEPORTSpecificSettings(ukl1Name, exceptionInfo);

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1GBESpecificSettings()

/*!
 * This will set the specific settings for the fwHw GBE PORM type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureHwTypeCCPCUKL1GBEPROMSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".GBE.PROM";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1] = "Id";
  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the bus number that the PROM is on, always either 0 or 1 depending on the version.
  int bus = 0;
  if ( FWUKL1_PROTOTYPE == ukl1Version )
    bus = 1;
  //2nd element is the address of the PROM on the bus, always 0x57.
  //3rd element is the address of the PROM registers at register 0x57, always 0.
  //4th element is the number of bytes that should be written to that register, 46 for compatibility with Tell1.
  //5th element is the page size, again 46 for compatibility with Tell1.
  //6th element is the the type of I2C interface to use, always FWCCPC_I2C_COMBINED for the PROM.
  //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
  //8th element is the refresh rate, always 2.
  //9th element is the on data change settings always 1.
  dyn_int specificSetting = makeDynAnytype(bus, 0x57, 0, 46, 46, FWCCPC_I2C_COMBINED, 1, 2, 1);
  dyn_dyn_int specificSettings;
  specificSettings[1] = specificSetting;

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
   fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1GBEPROMSpecificSettings(): Failed to setup the specific settings for the GBE ethernet PROM, communication with the PROM is not possible.", "1");
  
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1GBEPROMSpecificSettings()

/*!
 * This will set the specific settings for the fwHw GBE PORT type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1GBEPORTSpecificSettings(string ukl1Name, dyn_string& exceptionInfo) {
  //Loop over the four ports.
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = ukl1Name + ".GBE.PORT" + port;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1]  = "Duplex";
    registerList[2]  = "MAC";
    registerList[3]  = "Config";
    registerList[4]  = "TXFifoThreshold";
    registerList[5]  = "TXFifoLowWtrmrk";
    registerList[6]  = "TXFifoHighWtrmrk";
    registerList[7]  = "PHYControl";
    registerList[8]  = "PHYStatus";
    registerList[9]  = "TXStat";
    registerList[10] = "RXStat";

    //This contains the address of all the registers to be set.
    dyn_uint regAddrList;
    //Calculate the start address in the block that contains the per port settings.
    const unsigned offset = (0x80*(port));
    regAddrList[1]  = offset + 0x02;
    regAddrList[2]  = offset + 0x10;
    regAddrList[3]  = offset + 0x18;
    regAddrList[4]  = 0x614 + port;
    regAddrList[5]  = 0x60a + port;
    regAddrList[6]  = 0x600 + port;
    regAddrList[7]  = offset + 0x60;
    regAddrList[8]  = offset + 0x61;
    regAddrList[9]  = offset + 0x40;
    regAddrList[10] = offset + 0x20;

    //This contains the size in bytes of the register to be read.
    dyn_uint regSizeList;
    regSizeList[1]  = 1;
    regSizeList[2]  = 1;
    regSizeList[3]  = 1;
    regSizeList[4]  = 1;
    regSizeList[5]  = 1;
    regSizeList[6]  = 1;
    regSizeList[7]  = 1;
    regSizeList[8]  = 1;
    regSizeList[9]  = 25;
    regSizeList[10] = 26;

    //Now we create the settings, these are actually almost the same for each except for the address.
    //1st element is the address to be filled in later.
    //2nd element is the number of 32-bit words for command.
    //3rd element is the check for change rate, 4th is send on change only setting.
    dyn_int specificSetting = makeDynAnytype(0x00, 1, 2, 1);
    dyn_dyn_int specificSettings;
    //Loop over all the registers
    for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
      //Put the appropriate register address in the settings array.
      specificSetting[1] = regAddrList[count];
      //Put the appropriate size of the register in.
      specificSetting[2] = regSizeList[count];
      //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
      specificSettings[count] = specificSetting;
    }
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1GBEPORTSpecificSettings(): Failed to setup the specific settings for the GBE ethernet port " + port + ", port cannot be configured nor is the status available.", "1");
  }//for port
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1GBEPORTSpecificSettings()
  
/*!
 * This will set the specific settings for the fwHw IngressFPGA type that has been inserted into the
 * UKL1 fwHw type.
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1IngressFpgaSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Check the size of exceptionInfo.
  int exInfoSize = dynlen(exceptionInfo);
  
  //Loop over the Ingress FPGAs.
  for (unsigned inFpga = 0; inFpga < FWUKL1_FRONT_FPGAS; ++inFpga) {
    //Setup the status registers.
    _fwUkl1_configureHwTypeCCPCUKL1IngressFpgaStatusSpecificSettings(ukl1Name, inFpga, ukl1Version, exceptionInfo);
    if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureHwTypeCCPCUKL1IngressFpgaSpecificSettings(): Failed to configure Ingress FPGA " + inFpga + "'s status register settings, will not be able to view these nor the configuration settings.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception info
    
    //Now setup the nine channels.
    _fwUkl1_configureHwTypeCCPCUKL1ChannelSpecificSettings(ukl1Name, inFpga, exceptionInfo);
    if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureHwTypeCCPCUKL1IngressFpgaSpecificSettings(): Failed to configure Ingress FPGA " + inFpga + "'s input channels, will not be able to configure these while running.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception info
  }//for inFpga
  
  //Done.
  return;
}

/*!
 * This will set the specific settings for the fwHw IngressFPGAStatus type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  inFpga Number of the Ingress FPGA to configure the status for.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureHwTypeCCPCUKL1IngressFpgaStatusSpecificSettings(string ukl1Name, int inFpga, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".IngressFpga" + inFpga + ".IngressFpgaStatus";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1] = "StatusFifo";

  //This contains the address of all the registers to be set.
  dyn_int regAddrList;
  regAddrList[1] = _fwUkl1_fpgaizeAddress(0x28+inFpga);
  
  //Now create the settings for the FIFO.
  //1st element is the address of the first word in the FIFO, dependent on Ingress number.
  //2nd element is the type, which is 1 for FIFOs.
  //3rd element is the width of the FIFO, which is the size of each word that is to be read out in bytes, 2 bytes as we have 16-bit registers.
  //4rd element is the size of the FIFO, the total number of words in the FIFO. 76 words (4 general status and 9*8 words for the channels.).
  //5th to 10th elements can be ignored, these are the hexpar and par settings which are irrelevant.
  //11th element is the rate at which it checks for change, 0s as we cannot read back from this register, it makes no sense.
  //12th element is send on change only setting, set this.
  dyn_int specificSetting = makeDynInt(0x00, 1, 2, 76, 0, 0, 0, 0, 0, 0, 0, 1);
  dyn_dyn_int specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }//for count

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1IngressFpgaStatusSpecificSettings(): Failed to setup the specific settings for INGRESS FPGA " + inFpga + "'s status.", "1");
  
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1IngressFpgaStatusSpecificSettings()

/*!
 * This will set the specific settings for the fwHw InputChannel type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  inFpga Number of the Ingress FPGA that the channels are being configured for.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1ChannelSpecificSettings(string ukl1Name, unsigned inFpga, dyn_string& exceptionInfo) {
  //Loop over the input channels.
  for (unsigned channel = 0; channel < FWUKL1_FPGA_CHANNELS; ++channel) {
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = ukl1Name + ".IngressFpga" + inFpga + ".Channel" + channel;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1] = "ConfigurationFifo";
    registerList[2] = "PixelMasksFifo";

    //Now we create the specific settings for the registers. These are dependent on the register.
    dyn_dyn_int specificSettings;
    //Now create the settings for the configuration FIFO.
    //1st element is the register address of the FIFO, register 0x1f or 31.
    //2nd element is the type, which is 1 for FIFOs.
    //3rd element is the width of the FIFO, which is the size of each word that is to be read out in bytes, 2 bytes as we have 16-bit registers.
    //4rd element is the size of the FIFO, the total number of words in the FIFO.
    //5th to 10th elements can be ignored, these are the hexpar and par settings which are irrelevant.
    //11th element is the rate at which it checks for change, 0s as we cannot read back from this register, it makes no sense.
    //12th element is send on change only setting, set this.
    specificSettings[1] = makeDynInt(_fwUkl1_fpgaizeAddress(0x1f), 1, 2, 2, 0, 0, 0, 0, 0, 0, 0, 1);
    //Same settings as for the configuration FIFO except now the register is 16 words big.
    //This is the maximum number of words the UKL1 will transfer at a time from the FIFO.
    specificSettings[2] = makeDynInt(_fwUkl1_fpgaizeAddress(0x1f), 1, 2, 16, 0, 0, 0, 0, 0, 0, 0, 1);
    
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1ChannelSpecificSettings(): Failed to setup the specific settings for Ingress FPGA " + inFpga + ", channel " + channel + ", configuration of this channel is not possible.", "1");
  }//for channel
  
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1ChannelSpecificSettings()

/*!
 * This will set the specific settings for the fwHw EgressFPGA type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1EgressFpgaSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".EgressFpga";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "MacDestinationAddress";
  registerList[2]  = "MacSourceAddress";
  registerList[3]  = "IpDestinationAddress";
  registerList[4]  = "IpSourceAddress";
  registerList[5]  = "Protocol";
  registerList[6]  = "TypeOfService";
  registerList[7]  = "TimeToLive";
  registerList[8]  = "Type";
  registerList[9]  = "Version";
  registerList[10] = "FixedMepDestinationAddress";
  registerList[11] = "DontWaitForMepDestBrdcst";
  registerList[12] = "Throttle";
  registerList[13] = "ThrottlePolarity";
  registerList[14] = "TfcDecoder";
  registerList[15] = "PollGbePorts";
  registerList[16] = "ManualGbePortNumber";
  registerList[17] = "UseManualGbePort";
  registerList[18] = "PartitionId";
  registerList[19] = "MepPackingFactor";
  registerList[20] = "ThrottleMode";
  registerList[21] = "Latency";
  registerList[22] = "L1Id";
  registerList[23] = "IngressDisable";
  registerList[24] = "IngressConfigMailboxControl";
  registerList[25] = "DataFragmentSize";
  registerList[26] = "MepTruncationHighWtrmrk";

  //This contains the address of all the registers to be set.
  dyn_int regAddrList;
  regAddrList[1]  = _fwUkl1_fpgaizeAddress(0);
  regAddrList[2]  = _fwUkl1_fpgaizeAddress(3);
  regAddrList[3]  = _fwUkl1_fpgaizeAddress(6);
  regAddrList[4]  = _fwUkl1_fpgaizeAddress(8);
  regAddrList[5]  = _fwUkl1_fpgaizeAddress(10);
  regAddrList[6]  = _fwUkl1_fpgaizeAddress(11);
  regAddrList[7]  = _fwUkl1_fpgaizeAddress(12);
  regAddrList[8]  = _fwUkl1_fpgaizeAddress(13);
  regAddrList[9]  = _fwUkl1_fpgaizeAddress(14);
  regAddrList[10] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[11] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[12] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[13] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[14] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[15] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[16] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[17] = _fwUkl1_fpgaizeAddress(15);
  regAddrList[18] = _fwUkl1_fpgaizeAddress(16);
  regAddrList[19] = _fwUkl1_fpgaizeAddress(18);
  regAddrList[20] = _fwUkl1_fpgaizeAddress(19);
  regAddrList[21] = _fwUkl1_fpgaizeAddress(22);
  regAddrList[22] = _fwUkl1_fpgaizeAddress(23);
  regAddrList[23] = _fwUkl1_fpgaizeAddress(24);
  regAddrList[24] = _fwUkl1_fpgaizeAddress(24);
  regAddrList[25] = _fwUkl1_fpgaizeAddress(29);
  regAddrList[26] = _fwUkl1_fpgaizeAddress(30);

  //This contains a list of the number of 16-bit words per register.
  dyn_uint wordNumList;
  wordNumList[1]  = 3;
  wordNumList[2]  = 3;
  wordNumList[3]  = 2;
  wordNumList[4]  = 2;
  wordNumList[5]  = 1;
  wordNumList[6]  = 1;
  wordNumList[7]  = 1;
  wordNumList[8]  = 1;
  wordNumList[9]  = 1;
  wordNumList[10] = 1;
  wordNumList[11] = 1;
  wordNumList[12] = 1;
  wordNumList[13] = 1;
  wordNumList[14] = 1;
  wordNumList[15] = 1;
  wordNumList[16] = 1;
  wordNumList[17] = 1;
  wordNumList[18] = 2;
  wordNumList[19] = 1;
  wordNumList[20] = 1;
  wordNumList[21] = 1;
  wordNumList[22] = 1;
  wordNumList[23] = 1;
  wordNumList[24] = 1;
  wordNumList[25] = 1;
  wordNumList[26] = 1;

  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of bits in the register.
  //3rd element is the number of words for command.
  //4th element is the check for change rate (2secs).
  //5th is send on change only setting.
  dyn_uint specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 2, 1);
  dyn_dyn_uint specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Put the appropriate size of register into the array.
    specificSetting[3] = wordNumList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }

  //Setup the BE FPGA status registers, don't worry about the errors from this they will be logged appropriately.
  _fwUkl1_configureHwTypeCCPCUKL1EgressFpgaStatusSpecificSettings(ukl1Name, ukl1Version, exceptionInfo);

  //Now write all the read write register settings to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1EgressFpgaSpecificSettings(): Failed to setup the specific settings for the Egress FPGA, the Egress configuration settings cannot be accessed.", "1");

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1EgressFpgaSpecificSettings()

/*!
 * This will set the specific settings for the fwHw EgressFPGAStatus type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1EgressFpgaStatusSpecificSettings(string ukl1Name, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".EgressFpga.EgressFpgaStatus";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "TtcId";
  registerList[2]  = "TfcFifoRequest";
  registerList[3]  = "TtcReady";
  registerList[4]  = "EventMergerReadState";

  registerList[5]  = "GigabitTransciever0Disable";
  registerList[6]  = "GigabitTransciever1Disable";
  registerList[7]  = "IngressDisable";
  registerList[8]  = "GigabitTransciever2Disable";
  
  registerList[9]  = "SPI3TxBusy";
  registerList[10] = "MepFifoFull";
  registerList[11] = "Throttled";
  registerList[12] = "GigabitTranscieverFifoFull";
  registerList[13] = "GigabitTranscieverHighTxReady";
  registerList[14] = "GigabitTranscieverLowTxReady";

  registerList[15] = "TfcL0Reset";
  registerList[16] = "TfcL1Reset";
  registerList[17] = "TfcBShort";
  registerList[18] = "Tpa";
  registerList[19] = "FragmentReady";
  registerList[20] = "MepReady";
  registerList[21] = "FragmentRequest";
  registerList[22] = "EventQValid";

  registerList[23] = "FragmenterState";

  registerList[24] = "EthernetQValid";
  registerList[25] = "SequenceError";
  registerList[26] = "MepOccupancy";
  registerList[27] = "GigabitTranscieverFifoTxWait";
  registerList[28] = "TfcFifoOccupancy";
  
  registerList[29] = "TfcBLong";


  //This contains the address of all the registers to be set.
  dyn_int regAddrList;
  regAddrList[1]  = _fwUkl1_fpgaizeAddress(32);
  regAddrList[2]  = _fwUkl1_fpgaizeAddress(32);
  regAddrList[3]  = _fwUkl1_fpgaizeAddress(32);
  regAddrList[4]  = _fwUkl1_fpgaizeAddress(32);
  
  regAddrList[5]  = _fwUkl1_fpgaizeAddress(33);
  regAddrList[6]  = _fwUkl1_fpgaizeAddress(33);
  regAddrList[7]  = _fwUkl1_fpgaizeAddress(33);
  regAddrList[8]  = _fwUkl1_fpgaizeAddress(33);

  regAddrList[9]  = _fwUkl1_fpgaizeAddress(34);
  regAddrList[10] = _fwUkl1_fpgaizeAddress(34);
  regAddrList[11] = _fwUkl1_fpgaizeAddress(34);
  regAddrList[12] = _fwUkl1_fpgaizeAddress(34);
  regAddrList[13] = _fwUkl1_fpgaizeAddress(34);
  regAddrList[14] = _fwUkl1_fpgaizeAddress(34);

  regAddrList[15] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[16] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[17] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[18] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[19] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[20] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[21] = _fwUkl1_fpgaizeAddress(35);
  regAddrList[22] = _fwUkl1_fpgaizeAddress(35);

  regAddrList[23] = _fwUkl1_fpgaizeAddress(36);

  regAddrList[24] = _fwUkl1_fpgaizeAddress(37);
  regAddrList[25] = _fwUkl1_fpgaizeAddress(37);
  regAddrList[26] = _fwUkl1_fpgaizeAddress(37);
  regAddrList[27] = _fwUkl1_fpgaizeAddress(37);
  regAddrList[28] = _fwUkl1_fpgaizeAddress(37);

  regAddrList[29] = _fwUkl1_fpgaizeAddress(38);

  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the address to be filled in later.
  //2nd element is the number of bits in the register.
  //3rd element is the number of words for command.
  //4th element is the check for change rate (2secs).
  //5th is send on change only setting.
  dyn_uint specificSetting = makeDynAnytype(0x00, FWCCPC_LBUS_BITS_32, 1, 2, 1);
  dyn_dyn_uint specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[1] = regAddrList[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }//for count

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1EgressFpgaStatusSpecificSettings(): Failed to setup the specific settings for the Egress FPGA, the Egress status values will be unavailble.", "1");

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1EgressFpgaSpecificSettings()

/*!
 * This will set the specific settings for the fwHw UKL1Resets type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1ResetsSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".Resets";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "IngressFpgaStatusBufferPointers";
  registerList[2]  = "SPI3RxReadPointer";
  registerList[3]  = "PartialL1";
  registerList[4]  = "DCMs";

  //Now we create the settings, they are all the same as all the registers exist on the same address.
  //1st element is the address, all 0x15 (21).
  //2nd element is the number of bits in the register, set to 32-bits like everyone else.
  //3rd element is the number of words for command.
  //4th element is the check for change rate (2secs).
  //5th is send on change only setting.
  dyn_uint specificSetting = makeDynAnytype(_fwUkl1_fpgaizeAddress(0x15), FWCCPC_LBUS_BITS_32, 1, 2, 1);
  dyn_dyn_uint specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(registerList); ++count) {
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }//for count

  //Now write all the read write register settings to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUK1ResetsSpecificSettings(): Failed to setup the specific settings for the UKL1 resets, will not be able to perform certain specific resets. General UKL1 reset still available.", "1");
  
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1ResetsSpecificSettings()

/*!
 * This will set the specific settings for the fwHw EEPORM type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 *           documentation) are used.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1PROMSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".PROM";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1] = "Id";
  //The bus number that the PROM is on is determined by whether it is production or prototype.
  //Zero for production
  int bus = 0;
  //One for prototype.
  if ( FWUKL1_PROTOTYPE == ukl1Version )
    bus = 1;
  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the bus number that the PROM is on, dependent on whether it is a production or prototype board.
  //2nd element is the address of the PROM on the bus, always 0x50.
  //3rd element is the address of the PROM registers at register 0x50, always 0.
  //4th element is the number of bytes that should be written to that register, 4 which is the number of bytes stored in the PROM.
  //5th element is the page size, 4 the size of the number of bytes. Probably.
  //6th element is the the type of I2C interface to use, always FWCCPC_I2C_COMBINED for the PROM.
  //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
  //8th element is the refresh rate, always 2. Monitoring must be enabled to use this setting.
  //9th element is the on data change settings always 1. Monitoring must be enabled to use this setting.
  dyn_int specificSetting = makeDynAnytype(bus, 0x50, 0, 4, 4, FWCCPC_I2C_COMBINED, 1, 2, 1);
  dyn_dyn_int specificSettings;
  specificSettings[1] = specificSetting;
  
  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1PROMSpecificSettings(): Failed to setup the specific settings for the UKL1 PROM, communication with the PROM is not possible.", "1");

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1PROMSpecificSettings()

/*!
 * This will set the specific settings for the fwHw TEMPSENS types that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 *           documentation) are used.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1TempSensorSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Loop over all the available temperature sensors and configure them.
  for (int tempSens = 1; tempSens <= FWUKL1_TEMPERATURE_SENSORS; ++tempSens) {
    //Convert the UKL1 board name into the datapoint name.
    const string dpName = ukl1Name + ".Temp" + tempSens;

    //Set the specific settings.
    //This contains a list of all the register names that are to be set.
    dyn_string registerList;
    registerList[1] = "TempRead";
    //1st element is the bus number that the PROM is on, dependent on whether it is a production or prototype board.
    //Zero for production
    int bus = 0;
    //One for prototype.
    if ( FWUKL1_PROTOTYPE == ukl1Version )
      bus = 1;
    //2nd element is the address of the temperature sensor on the bus, varies depending on which sensor it is:
    unsigned address;
    if ( 1 == tempSens )
      address = 0x48;
    else if ( 2 == tempSens )
      address = 0x4a;
    else if ( 3 == tempSens )
      address = 0x4c;
    //3rd element is the address of the temperature data on the register, always 0.
    //4th element is the number of bytes that should be read from the register, 1 as the temperature is stored as an 8-bit number.
    //5th element is the page size, -1 which is infinite and we can ignore it.
    //6th element is the the type of I2C interface to use, always FWCCPC_I2C_COMBINED for the temperature sensor.
    //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
    //8th element is the refresh rate, always 2. Monitoring must be enabled to use this setting.
    //9th element is the on data change settings always 1. Monitoring must be enabled to use this setting.
    dyn_int specificSetting = makeDynAnytype(bus, address, 0, 1, -1, FWCCPC_I2C_COMBINED, 1, 2, 1);
    dyn_dyn_int specificSettings;
    specificSettings[1] = specificSetting;
  
    //Now write all these to the datapoint.
    if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1TempSensorSpecificSettings(): Failed to setup the specific settings for the temperature sensor number " + tempSens + ", communication with this sensor is not possible.", "1");
  }//for tempSensor
  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1TempSensorSpecificSettings()

/*!
 * This will set the specific settings for the fwHw TTCRX type that has been inserted into the
 * UKL1 fwHw type.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  ukl1Version Indicates whether we have a production or prototype board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void _fwUkl1_configureHwTypeCCPCUKL1TTCRXSpecificSettings(string ukl1Name, unsigned ukl1Version, dyn_string& exceptionInfo) {
  //Convert the UKL1 board name into the datapoint name.
  const string dpName = ukl1Name + ".TTCRX";

  //Set the specific settings.
  //This contains a list of all the register names that are to be set.
  dyn_string registerList;
  registerList[1]  = "IACId";
  registerList[2]  = "I2CId";
  registerList[3]  = "FineDelay1";
  registerList[4]  = "FineDelay2";
  registerList[5]  = "CoarseDelay";
  registerList[6]  = "Control";
  registerList[7]  = "Status";
  registerList[8]  = "Config1";
  registerList[9]  = "Config2";
  registerList[10] = "Config3";
  registerList[11] = "SingleErrCnt";
  registerList[12] = "DoubleErrCnt";
  registerList[13] = "SEUErrCnt";
  registerList[14] = "BunchCnt";
  registerList[15] = "EventCnt";
  //This contains the address of all the registers to be set.
  dyn_uint regAddrList;
  regAddrList[1]  = 16;
  regAddrList[2]  = 18;
  regAddrList[3]  = 0;
  regAddrList[4]  = 1;
  regAddrList[5]  = 2;
  regAddrList[6]  = 3;
  regAddrList[7]  = 22;
  regAddrList[8]  = 19;
  regAddrList[9]  = 20;
  regAddrList[10] = 21;
  regAddrList[11] = 8;
  regAddrList[12] = 10;
  regAddrList[13] = 11;
  regAddrList[14] = 24;
  regAddrList[15] = 26;
  //This contains the number of bytes that need to be set for the each of the registers.
  dyn_uint numBytes;
  numBytes[1]  = 2;
  numBytes[2]  = 1;
  numBytes[3]  = 1;
  numBytes[4]  = 1;
  numBytes[5]  = 1;
  numBytes[6]  = 1;
  numBytes[7]  = 1;
  numBytes[8]  = 1;
  numBytes[9]  = 1;
  numBytes[10] = 1;
  numBytes[11] = 2;
  numBytes[12] = 1;
  numBytes[13] = 1;
  numBytes[14] = 2;
  numBytes[15] = 3;

  //Now we create the settings, these are actually almost the same for each except for the address.
  //1st element is the bus number that the TTCrx is on,determined by whether it is a prototype or not.
  //Zero for production
  int bus = 0;
  //One for prototype.
  if ( FWUKL1_PROTOTYPE == ukl1Version )
    bus = 1;
  //2nd element is the address of the TTCrx on the bus, again always 0x42 the address the register addresses should be written to.
  //3rd element is the address of the TTCrx registers on the TTCrx chip, variable.
  //4th element is the number of bytes that should be written to that register, again variable.
  //5th element is the page size, don't really know but if -1 it is infinite and we can ignore it.
  //6th element is the the type of I2C interface to use, always FWCCPC_I2C_SEPARATED for the TTCrx chip.
  //7th element is the negative acknowledge, NACK, leave it enabled as I think it tells us if it failed. Always 1. 
  //8th element is the refresh rate, always 2. Have to enable monitoring to eactually use this setting.
  //9th element is the on data change settings always 1. Have to enable monitoring to eactually use this setting.
  dyn_int specificSetting = makeDynAnytype(bus, 0x42, 0, 0, -1, FWCCPC_I2C_SEPARATED, 1, 2, 1);
  dyn_dyn_int specificSettings;
  //Loop over all the registers
  for (unsigned count = 1; count <= dynlen(regAddrList); ++count) {
    //Put the appropriate register address in the settings array.
    specificSetting[3] = regAddrList[count];
    specificSetting[4] = numBytes[count];
    //Now put the specific settings in the appropriate element of the array containing all the settings for each register.
    specificSettings[count] = specificSetting;
  }//for count

  //Now write all these to the datapoint.
  if ( !fwHw_setSpecificSettings(dpName, registerList, specificSettings) )
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHwTypeCCPCUKL1TtcrxSpecificSettings(): Failed to setup the specific settings for the UKL1 TTCrx chip, communication with the TTCrx is not possible.", "1");

  //Done.
  return;
}//_fwUkl1_configureHwTypeCCPCUKL1TtcrxSpecificSettings()

//@}


// ================
//  UKL1 INTERFACE
// ================

/** @defgroup SectionInterface Read/write interface
 *    This section provides functions for communicating with the hardware on a specified UKL1
 *    board, with the board always identified by its name as found in the DIM server.
 *    It provides the ability to access all registers on the hardware and to perform other
 *    hardware related tasks such as resetting or reloading the UKL1s. It also provides
 *    access to the recipes saved for the UKL1 hardware type. This is done via the UKL1
 *    board name as found in the DIM server and the name of the specific recipe that is being
 *    address. Recipe and hardware access are setup such that they are almost identical function
 *    prototypes and it is only the addition of the recipe name and choice of verifying the hardware
 *    writes that causes them to be different. This hides the subtlies of writing to the hardware
 *    of database from the User.\n
 *
 *    There are two methods for reading the values a register. It can either be done directly
 *    or via monitoring.
 *    In the case of direct reading the function fwCcpc_read() or fwHw_getRecipe() is used and the
 *    values returned by this function are parsed by fwUkl1_parseReadRegister()to remove the relevant
 *    bits from the register. It removes a continuous sections of bits and returns it as a hex string,
 *    this can then be converted into an appropriate display value. Further multiple sections of the
 *    defined size can be removed, via fwUkl1_parseReadRegisterSections().
 *    In the case of monitoring callback functions can be setup in order to parse the data of a
 *    given data point only when it changes and again fwUkl1_parseReadRegister can be used to
 *    manipulate the register data to get at the desired information or fwUkl1_parseReadRegisterSections()
 *    if data is to be removed for example in the case of FE FPGA broadcast data. A read, using fwCcpc_read,
 *    can be used to force a reset of all monitored value, in this case the data returned by
 *    fwCcpc_read can just be ignored.
 *    This method is recommended for dealing with status registers.
 *    The library provides a function to read all the registers for a given area of the UKL1
 *    e.g. all the BE FPGA register, and then returns them in a predefined formatted state.
 *    This is to be used in the case that monitoring is not required or the User does not wish
 *    to know or does not know the data is organised across the registers. The values returned
 *    will be in strings will often contain a description of the state rather than just a number.
 *    Again values for similar element e.g. the input channel disable states can be returned
 *    as comma separated values.
 *    This method is recommended for reading from configuration registers.\n
 *
 *    The library provides a mechanism for writing to the registers on the UKL1. It provides a
 *    wrapper for the fwCcpc_write mechanism however it provides a layer of abstraction that does
 *    not require the User to understand the layout of the registers and which registers contain
 *    multiple settings. It takes a string that typically describes the registers state and is
 *    identical to the value read in the case of the read/write configuration registers. This
 *    function is advised to be used when writing to the configuration registers.
 *
 *    All errors are returned by the framework defined exception handling mechanism taking an
 *    argument exceptionInfo by reference, which will contain details of an errors that occurred.
 *  @{
 */

// ==========================
//   DATA PARSING FUNCTIONS
// ==========================

// =============
//    GENERAL
// =============

/*!
 * The purpose of this function is to take the data read from the UKL1 registers and to allow multiple groups of continuous
 * bits to be return in the form of a integer array. The function poses the ability to deal with registers of any size e.g. the
 * 32-bit registers from the GBE or FPGAs, or the 8-bit registers from the I2C. It is also capable of dealing with the format
 * of the FPGA registers as read from the UKL1s and a boolean switch it used to indicate that these must be handled. Typically
 * this will be used with monitored registers.
 *
 * \param  dcData Data read from the register as a byte array.
 * \param  numSections This is the number of times the defined section of data is to be read. It should be be defined by the
 *           the library constants that define the number of components on the board. For example FWUKL1_FRONT_FPGAS to retrieve
 *           the same setting for all the FE FPGAs when stored in the same register. This is set to 1 when only a single section
 *           need be removed. Each section is returned as an element in returned array. The first element contains the bits
 *           at startBit then subsequent elements contain the sections starting at the next section of bits.
 * \param  sectionGap The number of bits between the last bit in a section and the first bit in the next section.
 * \param  startBit Number of the first bit to be read from the register, where bit 0 is the least significant bit.
 * \param  length Number of bits to be read, set to 1 if only startBit is to be read. Maximum 32.
 * \param  fpga Boolean value to indicate whether we are parsing FPGA registers or not. This must be set correctly otherwise
 *           the data will not be properly interpreted.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return int Unsigned integer representation of the bits if the function is successful or -1 if it fails.
 *
 * This function primarily designed to be use with registers that contain multiple settings in a single register that are a constant
 * number of bits apart and are the same number of bits in size. This is most effective when dealing with the FE FPGA or channel
 * registers to implement a broadcast mode.\n
 * The reason for the FPGA flag is that the UKL1 FPGA registers are read as if they were 32-bit registers and not the 16-bit registers
 * that they are physically (this is because sequential addresses are 32-bits apart and not 16!). This function will remove the extra two
 * junk words that are read and parse only the remaining data in this case. Do not remove these extra words manually before hand.
 * This function can return sections that are a maximum of 32-bits in length. The only registers that are bigger than this are the
 * MAC addresses and these have their own dedicated convertion functions. The input size of the register is unlimited.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  //Fictional UKL1 board name, as found in the DIM server.
 *  const string ukl1Name = "pcukl101";
 *  //Register name needs to be the full DP name, hence requires the HW DP name prefix defined by this library.
 *  dyn_string registerNames;
 *  registerNames[1] = FWUKL1_HW_DP_NAME_PREFIX + ukl1Name + ".IngressFpga0.disable"
 *  //Stores the read data.
 *  dyn_char readData;
 *  //Holds errors.
 *  dyn_int callStatusList;
 *  //Read and save the status.
 *  const bool readStatus = fwCcpc_read(registerNames, ddcData, callStatusList);
 *  //Check for errors...
 *  if (!readStatus)
 *    (...)
 *
 *  //Data read from the disable register on the FE FPGA contains the disable settings for all the FE FPGA disable setting.
 *  //we split this data up into 4 settings, thus requiring only one read to get all the data.
 *  //Number of sections wanted is the number of FE FPGAs.
 *  const unsigned numSections = FWUKL1_FRONT_FPGAS;
 *  //The settings are in adjacent bits.
 *  const unsigned sectionGap = 0;
 *  //The data is a single bit, the first one starting at bit zero.
 *  const unsigned startBit = 0;
 *  const unsigned length   = 1;
 *  //The it is a setting on an FPGA register.
 *  const bool fpga = TRUE;
 *  //Check for errors.
 *  dyn_string exceptionInfo;
 *  //Convert the data.
 *  const dyn_int inFpgaDisableData = fwUkl1_parseReadRegisterSections(ddcData[1], numSections, sectionGap, startBit, length, fpga, exceptionInfo);
 *  //Check for errors.
 *  if ( 0 < dynlen(exceptionInfo) )
 *    (...)
 *
 *  //Let us deal with all four settings.
 *  for (int inFpga = 1; inFpga <= FWUKL1_FRONT_FPGAS; ++inFpga) {
 *    if ( 1 == inFpgaDisableData[inFpga] )
 *      (...disabled code...)
 *    else if ( 0 == inFpgaDisableData[inFpga] )
 *      (...enabled code...)
 *    else
 *      (...-1 would be seen if an error occur,
 *          which should have been handled after function call,
 *          need to check anyway otherwise it might be dealt with
 *          by an above case...)
 *
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
 */
dyn_int fwUkl1_parseReadRegisterSections(const dyn_char& dcData, unsigned numSections, unsigned sectionGap, unsigned startBit, unsigned length, bool fpga, dyn_string& exceptionInfo) {
  //This will hold the output data.
  dyn_int output;
  //Loop over the number of sections that we wish to get.
  for (int sec = 1; sec <= numSections; ++sec) {
    //First get the requested bits.
    const int exInfoSize = dynlen(exceptionInfo);
    output[sec] = fwUkl1_parseReadRegister(dcData, startBit, length, fpga, exceptionInfo);
    if ( (exInfoSize < dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
      //Inform us about which section was lost.
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseReadRegisterSections(): Failed to retrieve section " + sec + " out of " + numSections + ".", "2");
    }

    //Move the start bit to the start of the next section.
    //Add each time the total number of bits that need to be read and skipped.
    //Again minus 1 because of different bases, 0 and 1, for bit position and number of bits.
    startBit += (length-1 + sectionGap);
  }

  //Not going to concern ourselves with errors as the parseReadRegister() will tag them appropriately.

  //Return the data.
  return output;
}//fwUkl1_parseReadRegisterSections()

/*!
 * The purpose of this function is to take the data read from the UKL1 registers and to allow a specific group of continuous
 * bits to be returned as a integer. The function poses the ability to deal with registers of any size e.g. the
 * 32-bit registers from the GBE or FPGAs, or the 8-bit registers from the I2C. It is also capable of dealing with the format
 * of the FPGA registers as read from the UKL1s and a boolean switch it used to indicate that these must be handled. Typically
 * this will be used with monitored registers.
 *
 * \param  dcData Data read from the register as a byte array.
 * \param  startBit Number of the first bit to be read from the register, where bit 0 is the least significant bit.
 * \param  length Number of bits to be read, set to 1 if only startBit is to be read. Maximum 32.
 * \param  fpga Boolean value to indicate whether we are parsing FPGA registers or not. This must be set correctly otherwise
 *           the data will not be properly interpreted.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return int Unsigned integer representation of the bits if the function is successful or -1 if it fails.
 *
 * The reason for the FPGA flag is that the UKL1 FPGA registers are read as if they were 32-bit registers and not the 16-bit registers
 * that they are physically (this is because sequential addresses are 32-bits apart and not 16!). This function will remove the extra two
 * junk words that are read and parse only the remaining data in this case. Do not remove these extra words manually before hand.
 * This function can return sections that are a maximum of 32-bits in length. The only registers that are bigger than this are the
 * MAC addresses and these have their own dedicated convertion functions. The input size of the register is unlimited.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  //Fictional UKL1 board name, as found in the DIM server.
 *  const string ukl1Name = "pcukl101";
 *  //Register name needs to be the full DP name, hence requires the HW DP name prefix defined by this library.
 *  dyn_string registerNames;
 *  registerNames[1] = FWUKL1_HW_DP_NAME_PREFIX + ukl1Name + ".EgressFpga.ZeroSuppression"
 *  //Stores the read data.
 *  dyn_char readData;
 *  //Holds errors.
 *  dyn_int callStatusList;
 *  //Read and save the status.
 *  const bool readStatus = fwCcpc_read(registerNames, ddcData, callStatusList);
 *  //Check for errors...
 *  if (!readStatus)
 *    (...)
 *
 *  //Data read from the zero suppression register contains also the HPD pixel mode data which we are not interested in,
 *  //hence this function will return only the zero suppression data.
 *  //Zero suppression data spans 4 bits, starting with the least significant bit.
 *  const unsigned startBit = 0;
 *  const unsigned length   = 4;
 *  //The zero suppression setting is on a FPGA register.
 *  const bool fpga = TRUE;
 *  //Check for errors.
 *  dyn_string exceptionInfo;
 *  //Convert the data.
 *  const int zeroSuppData = fwUkl1_parseReadRegister(ddcData[1], startBit, length, fpga, exceptionInfo);
 *  //Check for errors.
 *  if ( 0 < dynlen(exceptionInfo) )
 *    (...)
 *
 *  //If zero suppression is enabled then 0xf would be returned.
 *  if ( 0xf == zeroSuppData )
 *    (...zero suppressed code...)
 *  else if ( 0 == zeroSuppData )
 *    (...none zero suppressed code...)
 *  else
 *    (...-1 would be seen if an error occur,
 *        which should have been handled after function call,
 *        need to check anyway otherwise it might be dealt with
 *        by an above case...)
 *
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
 */
int fwUkl1_parseReadRegister(dyn_char dcData, unsigned startBit, unsigned length, bool fpga, dyn_string& exceptionInfo) {
  //Check there is actually some data.
  if ( 0 >= dynlen(dcData) ) {
    fwException_raise(exceptionInfo, "ERROR", "fwUkl1_parseReadRegister(): No data given to function, cannot continue.", "1");
    return -1;
  }//if(0>=dynlen(dcData))
  
  //Check that we don't gain any more exception data...
  int exInfoSize = dynlen(exceptionInfo);

  //First we must remove the extra bits.
  //If we are dealing with an FPGA the we need to convert from 16- to 32-bit words.
  if (fpga) {
    fwUkl1_convert32To16BitWords(dcData, exceptionInfo);
    if ( exInfoSize < dynlen(exceptionInfo) ) {
      //Return that we failed, indicate that it was this function that called the byte removal function before it failed.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseReadRegiser(): Failed to properly parse the FPGA register data.", "1");
      return -1;
    }//if converted from 32 to 16 bits successfully.
  }//if(fpga)

  if ( 32 < length ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseReadRegister(): Requested number of bits was " + length + ", returning a 32-bit in hence most significant bits will be lost.", "2");
    length = 32;
  }//if(32<length)

  //In order to be able to span byte boundaries we need to know where we are going to start and finish in terms
  //of the appropriate byte. These constants will ease the navigation of the bit-field.

  //Determine how many bytes we have been given.
  const int numBytes = dynlen(dcData);
  if ( -1 == numBytes ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseReadRegister(): Failed to determine number of bytes in the input array.", "2");
    return -1;
  }//if(-1==numBytes)
  //Determine how many bits we have, there are eight bits in each element as each element contains a char.
  const unsigned numBits  = 8 * numBytes;

  //Perform a sanity check on the starting bit.
  if ( (numBits+(unsigned)1) < startBit ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseReadRegister(): Requested bit to start from, " + startBit + ", is greater than the number of bits given, " + numBits + ". Setting start to bit numBits-length, " + (numBits-length) + ".", "2");
    startBit = numBits-length;
  }//if(0>startBit)
  
  //The last bit is the number of bits we wish to remove offset by the starting bit number.
  //We subtract 1 as if we just remove 1 bit then the start and end bit are the same.
  unsigned endBit = startBit + length - 1;
  if ( numBits <= endBit ) {
    //Can't end at a point passed the number of bits that we have.
    //The last bit is 1 less than the number of bits in the register as we start counting at 0.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseReadRegiser(): Reducing last bit to be read to " + (numBits-1) + " from " + endBit + ", most significant bits will be lost. There are only " + (numBits-1) + " available.", "2");
    //numBits is starts at 1, whereas endBit starts at 0.
    endBit = numBits -1;
  }//if(numBits<=endBit)
  
  //The first byte that we need to read from. Remembering that we are using a big endian numbering scheme.
  const unsigned startByte = numBytes - (unsigned)(startBit/8);
  //Which is the first bit we need to read from in the starting byte.
  const unsigned startBitInByte = (unsigned)startBit%8;
  //This is the last byte that we need to read from.
  const unsigned endByte   = numBytes - (unsigned)endBit/8;
  //This is the last bit in the last byte that we need to read from.
  const unsigned endBitInByte = (unsigned)endBit%8;
  
  //This are used to enable us to write to the output data word.
  //The word that will be returned.
  unsigned uData = 0;
  //Marks where the next region of bits should start from in the word.
  unsigned nextBitInWord = 0;

  //Now we need to loop from the startByte to the endByte, decrement as we are big endian.
  for (unsigned byte = startByte; byte >= endByte; --byte) {
    //There are three cases that we need to consider.
    if ( startByte == endByte ) {
      //In this case we know are working within the same byte so we just align the desired start bit to the start of the byte.
      //Then create a mask of 1's of the numbers of bits we wish to extract and the operate on the bit aligned byte.
      uData |= (dcData[byte]>>startBitInByte) & ~(~0 << length);
    }//if(startByte==endByte)
    else if ( byte == startByte ) {
      //In this case we need to add from the startBit to the end of the byte to the output word.
      //Align the startBit with the beginning of the byte, don't need to mask as we want the rest of the byte to be output.
      uData |= (dcData[byte]>>startBitInByte) << nextBitInWord;
      //Advance the nextBit so where we know where to OR the next words into.
      nextBitInWord += (8-startBitInByte);
    }//else if(byte==startByte)
    else if ( byte == endByte ) {
      //In this case we need to add from the first bit in the byte to the endBit.
      //Create a mask that has 1's covering the bits from the start bit in the byte to the endBitInByte and then operate on the bit.
      //Plus 1 because the shift is 1 based and the bit counting 0.
      uData |= (dcData[byte] & ~(~0 << (endBitInByte+1))) << nextBitInWord;
    }//else if(byte==endByte)
    else {
      //This case we can just take the whole byte and write it into the appropriate place in our word.
      //First convert is to a 32-bit number so that is can be bit shifted appropriately i.e. to bits higher than 8.
      uData |= (unsigned)dcData[byte] << nextBitInWord;
      //Have advanced 8 bits.
      nextBitInWord += 8;
    }//else
  }//for byte
  
  //Done.
  return uData;
}//fwUkl1_parseReadRegister()

/*!
 * Takes a colon delimited hex MAC address, as a string, and removes the colons
 * to return a continuous hex string. It also performs checks on the data format.
 *
 * \param  macAddress The string that contains the MAC address to be parsed. Should be of the form XX:XX:XX:XX:XX:XX
 *           or XX:XX:XX:XX. The final two bytes can be omitted and should be for the source MAC address
 *           and then in the result hex MAC address they will be replaced with zeros in the resulting hex string.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string MAC address without the colons. If the MAC address contains no colons
 *           it will be returned as it was. If the MAC address does not look like a MAC
 *           address `Fail' will be returned.
 *
 * Expects the MAC address to be of the form 00:0e:0c:b0:2d:19 or 000e0cb02d19
 * i.e. 12 characters long.
 * \b Example
 * \code
 *   (...)
 *   //A colon delimited MAC address that we wish to convert.
 *   const string macAddress1 = 00:0e:0c:b0:2d:19;
 *   //A colonless MAC address.
 *   const string hexAddress1 = fwUkl1_convertMacToHex(macAddress1, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Will not have any error...
 *   DebugTN(hexAddress1);  //000e0cb02d19
 *
 *   //A colon delimited MAC address that we wish to convert.
 *   const string macAddress2 = 00:0e:0c:b0;
 *   //A colonless MAC address.
 *   const string hexAddress2 = fwUkl1_convertMacToHex(macAddress1, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Will not have any error...
 *   DebugTN(hexAddress2);  //000e0cb00000
 *
 *   //Invalid colon separated MAC address.
 *   const string macAddress3 = 00:0e:0c:b0:2d;
 *   const string hexAddress3 = fwUkl1_convertMacToHex(macAddress2, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle error...
 *   DebugTN(hexAddress3);  //Fail
 *   (...)
 *  \endcode
 */
string fwUkl1_convertMacToHex(const string& macAddress, dyn_string& exceptionInfo) {
  //Check that it contains the appropriate number of characters, either with or without colons.
  if ( (12 != strlen(macAddress)) && (17 != strlen(macAddress)) &&  //Full MAC address lengths w/out colons.
       ( 8 != strlen(macAddress)) && (11 != strlen(macAddress)) ) { //Missing remaining byte MAC address lengths w/out colons.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertMacToHex(): MAC address is " + strlen(macAddress) + " characters long. Should be 8, 11, 12 or 17 characters.", "1");
    return "Fail";
  }

  //Will hold the MAC address once it has been parsed.
  string parsedMacAddress;
  //Try to split the string using a colon as a delimiter.
  dyn_string tmpMac = strsplit(macAddress, ":");
  //Check it has found the appropriate number of parts.
  const int lenMac = dynlen(tmpMac);
  if ( 1 == lenMac ) {
    //String contained no colons, it is the right lenght so it can be returned as it was given.
    parsedMacAddress = macAddress;
  }
  else if ( 4 == lenMac ) {
    //Contains 4 parts, put them all into one string and then pad the remaining four characters with 0s.
    int status = sprintf(parsedMacAddress, "%02s%02s%02s%02s0000", tmpMac[1], tmpMac[2], tmpMac[3], tmpMac[4]);
    //Check for errors.
    if ( -1 == status ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertMacToHex(): Failed to reconstruct MAC address as hex string.", "1");
      parsedMacAddress = "Fail";
    }
  }
  else if ( 6 == lenMac ) {
    //Contains 6 parts, put them all into one string.
    int status = sprintf(parsedMacAddress, "%02s%02s%02s%02s%02s%02s", tmpMac[1], tmpMac[2], tmpMac[3], tmpMac[4], tmpMac[5], tmpMac[6]);
    //Check for errors.
    if ( -1 == status ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertMacToHex(): Failed to reconstruct MAC address as hex string.", "1");
      parsedMacAddress = "Fail";
    }
  }
  else {
    //Not properly formed.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertMacToHex(): Found only " + lenMac + " colon delimited sections, should contain 1, 4 or 6 for a valid MAC address.", "1");
    parsedMacAddress = "Fail";
  }
  
  //Done.
  return parsedMacAddress;
}//fwUkl1_convertMacToHex()

/*!
 * Takes an array of bytes and returns them as a colon delimited MAC address.
 *
 * \param  macAddress The array containing the bytes to be parsed.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string Expects a 6 element array containing 3 elements with 16-bit words of data in each.
 *           Returns `Fail' if the input data is misformed.
 *
 * This code requires that the extra bytes generated from a fwCcpc_read() are first removed, via fwUkl1_convert32To16BitWords().
 * Generally it is expected that this function will only be used internally and fwUkl1_parseEgressFpgaConfigRegistersForGet() will be used if dealing
 * with the BE FPGA registers. It is not defined an internal function as it mirrors the convert MAC to byte function, which is not internal.
 *
 * \b Example
 * \code
 *   (...)
 *   //Fictional UKL1 board name, as found in the DIM server.
 *   const string ukl1Name = "pcukl101";
 *   //Register name needs to be the full DP name, hence requires the HW DP name prefix defined by this library.
 *   dyn_string registerNames;
 *   registerNames[1] = FWUKL1_HW_DP_NAME_PREFIX + ukl1Name + ".EgressFpga.MacDestinationAddress"
 *   //Stores the read data.
 *   dyn_char readData;
 *   //Holds errors.
 *   dyn_int callStatusList;
 *   //Read and save the status.
 *   const bool readStatus = fwCcpc_read(registerNames, ddcData, callStatusList);
 *   //Check for errors...
 *   if (!readStatus)
 *     (...)
 *
 *   fwUkl1_convert32To16BitWord(ddcData[1], exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle errors, should be none if read successful...
 *
 *   const string macAddress = fwUkl1_convertByteToMac(ddcData[1], exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle errors, should be none if read successful...
 *   DebugTN(macAddress); //Colon delimited MAC address e.g. 00:0E:0C:B0:2D:19
 *
 *   (...)
 *  \endcode
 */
string fwUkl1_convertByteToMac(const dyn_char& macAddress, dyn_string& exceptionInfo) {
  //If the masks are in the data then they will be at the front i.e. lowest element numbers.
  //Determine the number of elements in the array and count backwards from there.
  //This will allow absolute indexing whether the masks are present or not.
  const int size = dynlen(macAddress);
  //Check that it contains the appropriate number of elements.
  //Technically if it came from the configuration database it could contain the write mask tagged on the end.
  //This doesn't cause a problem as we use absolute indexes later, just require there to be 6 elements minimum (and the MAC to be in those).
  if ( 6 > size ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToMac(): Input array contains " + size + " elements. Should be 6 or 12 elements big.", "1");
    return "Fail";
  }

  //Now we need to put the data into the appropriate format.
  string output;
  //This should be in a big endian order once it reaches here.
  output =  fwCcpc_convertByteToHex(macAddress[size - 5]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 4]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 3]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 2]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size - 1]);
  output += ":";
  output += fwCcpc_convertByteToHex(macAddress[size]);

  //Done.
  return output;
}//fwUkl1_convertByteToMac()

/*!
 * Takes a standard formed IP(4) address, as a string, and convert it to a continuous hex
 * string. Also performs syntacs checking on the given string.
 *
 * \param  ipAddress The string that contains the IP address to be parsed. Should be of the form:
 *           XXX.XXX.XXX.XXX, where X represents a decimal number, e.g. 192.168.192.192. It is possible
 *           to miss the final byte i.e. 192.168.192, this is required for the source IP address.
 *           In this case where the final byte is missing it will be replaced with zeros in the resulting hex string.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string If the IP address does not look like a IP address `Fail' will be returned, otherwise the hex string.
 *
 * Expects the MAC address to be of the form 192.168.255.255 or 192.168.255
 * \b Example
 * \code
 *   (...)
 *   //An IP address that we wish to convert.
 *   const string ipAddress1 = 192.168.255.1;
 *   //The equivalent hex string.
 *   const string hexAddress1 = fwUkl1_convertIpToHex(ipAddress1, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Will not have any error...
 *   DebugTN(hexAddress1);  //c0a8ff01
 *
 *   //An IP address that we wish to convert.
 *   const string ipAddress2 = 192.168.255;
 *   //The equivalent hex string.
 *   const string hexAddress2 = fwUkl1_convertIpToHex(ipAddress1, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Will not have any error...
 *   DebugTN(hexAddress2);  //c0a8ff00, note the last byte are zeros.
 *
 *   //Invalid IP address.
 *   const string ipAddress3 = 192.168.;
 *   const string hexAddress3 = fwUkl1_convertIpToHex(ipAddress2, exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle error...
 *   DebugTN(hexAddress3);  //Fail
 *   (...)
 *  \endcode
 */
string fwUkl1_convertIpToHex(const string& ipAddress, dyn_string& exceptionInfo) {
  //Check that this does not contain more than the maximum number of bits that can be present in an IP(4) address.
  //Remember that an IP address length is very variable 1.1.1.1 compared to 255.255.255.255
  if ( 16 <= strlen(ipAddress) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertIpToHex(): IP address is " + strlen(ipAddress) + " characters long. Can be a maximum of 15.", "1");
    return "Fail";
  }

  //Holds the resulting IP address in hex.
  string output;
  //Try to split the string using a . as a delimiter.
  dyn_string tmpIp = strsplit(ipAddress, ".");
  //Check it has found the appropriate number of parts.
  const int lenIp = dynlen(tmpIp);
  if ( 4 == lenIp ) {
    //Contains 4 parts, put them all into one string.
    for (unsigned element = 1; element <= dynlen(tmpIp); ++element) {
      //Take a section of the IP and convert it to a hex string.
      string tmpSection = fwCcpc_convertDecToHex(tmpIp[element]);
      //If it does not contain 2 characters pad the string.
      if ( 2 != strlen(tmpSection) )
	tmpSection = "0" + tmpSection;
      output += tmpSection;
    }
  }
  else if ( 3 == lenIp ) {
    //Contains 3 parts, put them all into one string and then pad the last two characters to make up the byte.
    for (unsigned element = 1; element <= dynlen(tmpIp); ++element) {
      //Take a section of the IP and convert it to a hex string.
      string tmpSection = fwCcpc_convertDecToHex(tmpIp[element]);
      //If it does not contain 2 characters pad the string.
      if ( 2 != strlen(tmpSection) )
	tmpSection = "0" + tmpSection;
      output += tmpSection;    }
    //Add zeros for the remaining two characters.
    output += "00";
  }
  else {
    //Not properly formed.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertIpToHex(): Found " + lenIp + " full stop delimited sections, require 3 or 4.", "1");
    output = "Fail";
  }

  //Done.
  return output;
}//fwUkl1_convertIpToHex()

/*!
 * Takes an array of bytes, and returns them as a full stop delimited IP address.
 *
 * \param  ipAddress The array containing the bytes to be parsed.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string Expects a 4 element array. Returns `Fail' if the input data is missformed.
 *
 * This code requires that the extra bytes generated from a fwCcpc_read() are first removed, via fwUkl1_convert32To16BitWords().
 * Generally it is expected that this function will only be used internally and fwUkl1_parseEgressFpgaConfigRegistersForGet() will be used if dealing
 * with the BE FPGA registers. It is not defined an internal function as it mirrors the convert IP to byte function, which is not internal.
 *
 * \b Example
 * \code
 *   (...)
 *   //Fictional UKL1 board name, as found in the DIM server.
 *   const string ukl1Name = "pcukl101";
 *   //Register name needs to be the full DP name, hence requires the HW DP name prefix defined by this library.
 *   dyn_string registerNames;
 *   registerNames[1] = FWUKL1_HW_DP_NAME_PREFIX + ukl1Name + ".EgressFpga.IpDestinationAddress"
 *   //Stores the read data.
 *   dyn_char readData;
 *   //Holds errors.
 *   dyn_int callStatusList;
 *   //Read and save the status.
 *   const bool readStatus = fwCcpc_read(registerNames, ddcData, callStatusList);
 *   //Check for errors...
 *   if (!readStatus)
 *     (...)
 *
 *   fwUkl1_convert32To16BitWord(ddcData[1], exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle errors, should be none if read successful...
 *
 *   const string ipAddress = fwUkl1_convertByteToIp(ddcData[1], exceptionInfo);
 *   if ( 0 != dynlen(exceptionInfo) )
 *     //...Handle errors, should be none if read successful...
 *   DebugTN(ipAddress); //Colon delimited MAC address e.g. 192.168.255.0
 *
 *   (...)
 *  \endcode
 */
string fwUkl1_convertByteToIp(const dyn_char& ipAddress, dyn_string& exceptionInfo) {
  //If the masks are in the data then they will be at the front i.e. lowest element numbers.
  //Determine the number of elements in the array and count backwards from there.
  //This will allow absolute indexing whether the masks are present or not.
  const int size = dynlen(ipAddress);
  //Check that it contains the appropriate number of elements.
  //Technically if it came from the configuration database it could contain the write mask tagged on the end.
  //This doesn't cause a problem as we use absolute indexes later, just require there to be 4 elements minimum (and the IP to be in those).
  if ( 4 > size ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToIp(): Input array contains " + size + " elements. Should be 4 elements big.", "1");
    return "Fail";
  }

  //Now we need to put the data into the appropriate format.
  string output;
  output += fwCcpc_convertByteToDec(ipAddress[size - 3]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size - 2]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size - 1]);
  output += ".";
  output += fwCcpc_convertByteToDec(ipAddress[size]);

  //Done.
  return output;
}//fwUkl1_convertByteToIp()

/*!
 * This takes the data that is read by from the L1 board FPGAs (LBUS), which is read as a series of 32-bit words,
 * and converts it to a series of 16-bit words can then be decoded to return the register value.
 *
 * \param  dcData The dyn_char array that contains the bytes read back from the hardware, will have the upper 2-bytes
 *           removed from the array. Assume byte ordering that matches that done read by the FW CCPC library. Return
 *           by reference.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_char The dyn_char that now has its bytes in an order that can be decoded by the fwCcpc library
 *           to return a string representing the hex number.
 *
 * The function performs two tasks. First it considers the byte array as a group of 32-bit numbers and removes
 * the upper two bytes to reduce them to a 16-bit number. The resulting array must then be word swapped, however
 * the words must maintain their internal byte ordering.
 */
void fwUkl1_convert32To16BitWords(dyn_char& dcData, dyn_string& exceptionInfo) {
  //Convert the read data from a series of 32-bit words to a series of 16-bit words.

  //Check that we have a multiple of 32-bits.
  if ( 0 != (dynlen(dcData)%4) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convert32To16BitWords(): Provided array does not contain enough data to be a 32-bit word, contains " + dynlen(dcData) + " bytes.", "1");
    return;
  }

  //The loop must run over the number of 32-bit words in the read data.
  const unsigned total32Words = dynlen(dcData)/4;
  for (unsigned word = 0; word < total32Words; ++word) {
    //We need to remove the first two elements in that word.
    //This will follow the sequence of odd numbers.
    //After we delete the first element the second element becomes the first, so we always delete the same one.
    dynRemove(dcData, 2*word+1);
    dynRemove(dcData, 2*word+1);
  }

  //In the event that we have read back more than one word then we will need to swap the ordering of the 16-bit
  //words contained within the array.
  const unsigned total16Words = dynlen(dcData)/2;
  if (1 < total16Words) {
    //Copy the data into a temporary array.
    const dyn_char tmpData = dcData;
    //Loop over all the 16-bit words of the array and put them back into the original in the reverse order.
    const unsigned totalElements = dynlen(dcData);
    for (unsigned word = 0; word < totalElements; word+=2) {
      //Need to keep the byte ordering within the word, but word swap the array. 16-bit words.
      dcData[word+1] = tmpData[totalElements-word-1];
      dcData[word+2] = tmpData[totalElements-word];
    }
  }
  
  //Done.
  return;
}//fwUkl1_convert32To16BitWords()

/*!
 * Converts an array of bytes into a hex string and treats the dyn_char array as if it were little endian.
 *
 * \param  dcData The array of bytes to be converted.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string Hex string of the dyn_char. Will return `Fail' if the data could not be converted.
 */
string fwUkl1_convertByteToHexLittleEndian(const dyn_char& dcData, dyn_string& exceptionInfo) {
  //We must reverse the byte ordering of the array.
  //For this we will need to know its size.
  const int numBytes = dynlen(dcData);
  //This stores the reversed bytes.
  dyn_char swappedDcData;
  if ( 0 < numBytes ) {
    //Have a none zero number of bytes.
    //Loop through them swapping the first element for last, 2nd for 2nd to last etc..
    for (int byte = 1; byte <= numBytes; ++byte) {
      swappedDcData[numBytes-(byte-1)] = dcData[byte];
    }
  }
  else if ( 0 == numBytes )
    return "";
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_convertByteToHexLittleEndian(): PVSS error, failed to determine size of byte array, could not convert to hex string.", "1");
    return "Fail";
  }

  return fwCcpc_convertByteToHex(swappedDcData);
}//fwUkl1_convertByteToHexLittleEndian()


/*!
 * Takes a string and ensures that it contains enough characters for the number of bytes specified
 * by numBytes. Adds `0' to the beginnig if it is too short and removes characters from the beginning
 * if it is too long. Must be in hex.
 *
 * \param  sData The string that contains the data to be checked. Returned by reference.
 * \param  numBytes The number of bytes that the returned character string should return.
 * \return void
 *
 * This is for use with strings that represent a hex number that should be a set number of bits long.
 * It assumes that the lowest order bit is the last (reading left to right) character and hence its behaviour is
 * to move the last character always towards the final character in the padded string (the 2nd, 4th, 6th etc character).
 * This is achieved by padding or removing at the beginning of the string.
 *
 * \b Example:
 * \code
 *   (...)
 *   //Declare some data e.g. User input from a panel.
 *   const string hexData1 = "1234";
 *   //Wish to use the fwCcpc library to write this to a 32-bit registers hence is needs to be zero padded
 *   //to be 4-bytes big (8-characters).
 *   fwUkl1_padString(hexData1, 4);
 *   DebugTN(hexData1); // "00001234"
 *
 *   //Declare some more data e.g. User input from a panel.
 *   const string hexData2 = "1234";
 *   //Wish to use the fwCcpc library to write this to an 8-bit registers hence it need to be 1-byte (2-characters) big.
 *   fwUkl1_padString(hexData2, 1);
 *   //Here the data is misformed and it is assumed that the least significant bit (last character) in the given number
 *   //should remain the least significant bit.
 *   DebugTN(hexData2); // "34"
 *   (...)
 * \endcode
 */
void fwUkl1_padString(string& sData, unsigned numBytes) {
  //First determine the length.
  const int datalen = strlen(sData);
  //Create a temporary string to store the original string.
  const string tmpOrig = sData;
  //Determine the number of characters that must be in the final string,
  //each character represents half a byte or nibble.
  const int numChar = numBytes*2;
  //Perform the checking and padding/trimming.
  if ( numChar > datalen ) {
    //The data string is too short so increase its length by padding at the beginning, highest order bytes.
    //A temporary string to hold the amount of padding.
    string tmpPad;
    for (int character = 1; character <= (numChar-datalen); ++character) {
      tmpPad += "0";
    }
    //Now insert the padding into the data string.
    sData = tmpPad + tmpOrig;
  }
  else if ( numChar < datalen ) {
    //Remove the initial characters of the string until it reaches the appropriate length, highest order bytes.
    sData = substr(sData, (datalen-numChar), datalen);
  } else {
    //Do nothing, it is the right length.
  }

  //Return the resulting string.
  return;
}//fwUkl1_padString()

/*!
 * This takes a array of string where each element of the array contains the relevant settings for either
 * the FE FPGAs, the channels on a FE FPGA or the GBE ports and converts these to an array of strings.
 * The returned result is an array of array of strings that contain all the relevant settings.
 *
 * \param  origData The dyn_string that contains the data to be parsed. Each element should contain a
 *           comma separated list of settings.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_dyn_string Each element contains a dyn_string, where each element contains the setting for
 *           an individual unit. In the event of an empty input an empty dyn_dyn_string is returned and this is
 *           signalled via exceptionInfo. In the event there is no comma delimited  or it could not be parsed
 *           an empty dyn_string is returned for the corresponding element in the returned dyn_dyn_string.
 *           As it is not possible to know how many elements the User was expecting from the parsed data
 *           it is not possible to populate the array with a sensible amount of empty strings hence the size
 *           of the returned dyn_string in each element of the returned dyn_dyn_string must be checked.
 *           It is gaurantied in that if no exceptionInfo is generated by this function the returned dyn_dyn_string
 *           will contain at least an empty dyn_string for each element of the input data.
 *
 * The input strings must contain a comma separated list of the unit settings. The function checks that
 * there are either four or nine settings found, if there are none then the data format is considered to be
 * invalid and for that specific element an empty string is return. It will continue to parse the rest of
 * the data if an error is found. The ordering of the output data is the first element in the array corresponds
 * to the first entry in the string reading left to right.
 */
dyn_dyn_string fwUkl1_parseBroadcastData(const dyn_string& origData, dyn_string& exceptionInfo) {
  //Holds the result that we will be returning.
  dyn_dyn_string parsedData;

  //First determine the number strings we will need to parse.
  const unsigned totalLength = dynlen(origData);
  if ( -1 == totalLength ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseBroadcastData(): Failed to determine number of elements in input array, PVSS error.", "1");
    //Parsed data should be empty in this case.
    return parsedData;
  }

  //Loop over the array and deal with each element.
  //If total length is zero then we will just return with an emtpy array.
  for (unsigned element = 1; element <= totalLength; ++element) {
    //Holds the result of the string parsing.
    //Split the string up using comma's as a deliminator.
    parsedData[element] = strsplit(origData[element], ',');
    //Check we have enough elements for either all the FE FPGAs, GBE ports (both 4) or FE FPGA
    //channels (9).
    const unsigned foundParts = dynlen(parsedData[element]);
    if ( (1 != foundParts) && (4 != foundParts) && (9 != foundParts) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseBroadcastData(): Found " + foundParts + " sections in the broadcast string, must have 1, 4 or 9.", "2");
      //Parsed data wasn't good so return an empty array.
      parsedData[element] = makeDynString();
    }
  }
  //Finally return the parsed strings.
  return parsedData;
}//fwUkl1_parseBroadcastData()


// ===========
//    FPGAS
// ===========

/*!
 * This function takes the data for the various Egress FPGA registers described by a string and converts it to data that can
 * used with _fwUkl1_write() or _fwUkl1_saveRecipe(). It allows data to be given in a more human readable format and converts
 * it to the byte array required for writting to the CCPC. It also returns the appropriate mask that must be used when
 * writing the data. If being used with _fwUkl1_saveRecipe() then the mask will have to be appended to the data array.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The appropriate name prefix is added and
 *           returned by reference. The following values are permitted (all case sensitive):
 *   \li MacDestinationAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or it may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li MacSourceAddress An eight character string that gives the first 4 bytes of the MAC address. The remaining
 *         bytes are not User configurable and are set during a fwUkl1_configure(). There are two possible forms either a continuous
 *         string of the address or it may posses a colon delimiter every two characters,
 *         e.g. 000e0cb0 or 00:0e:0c:b0.
 *   \li IpDestinationAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li IpSourceAddress A string that contains either the first three bytes of the address (reading left to right) in hex format
 *         e.g. c0a0c0 or in a typical IP4 format e.g. 192.168.192 . The remaining byte is not User configurable and set during
 *         fwUkl1_configure().
 *   \li Protocol Default format described below.
 *   \li TypeOfService Default format described below.
 *   \li TimeToLive Default format described below.
 *   \li Type Default format described below.
 *   \li Version Default format described below.
 *   \li PartitionId Must contain an 8 character string representing the two 16-bit words of data for the partition ID.
 *   \li L1Id An 8-bit value, 2-character string. This will be present in the output data packet and MUST be unique for each L1.
 *         This is set correctly by the FSM and generally should not be User defined, except maybe when the User is an expert.
 *   \li ThrottleMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         of the HPD pixel chips and correspond to the settings used in the Ingress FPGA channels.
 *   \li Latency Default format described below.
 *   \li MepPackingFactor Default format described below.
 *   \li FixedMepDestinationAddress `Enable' or `Disable' to enable or disable a fixed multi-event packet destination address.
 *   \li DontWaitForMepDestBrdcst `Don't wait' or `Wait' to enable or disable the waiting for a multi-event packet destination
 *         address broadcast, respectively.
 *   \li Throttle `Enable' or `Disable' to enable or disable the throttle.
 *   \li ThrottlePolarity `Invert' or `Normal' to select the polarity of the throttle output.
 *   \li TfcDecoder `Enable' or `Disable' to enable or disable the decoding of TFC commands. The board will not operate without
 *         this setting enabled.
 *   \li PollGbePorts `Enable' or `Disable' to enable the Egress FPGA to send data to the GBE card for transmission.
 *   \li UseManualGbePort `Enable' or `Disable' to disable automatic detection of available GBE ports and instead enable the
 *         use of a single fixed GBE port defined by ManualGbePortNumber. This port must be enabled via the GBE control.
 *   \li ManualGbePortNumber The GBE port that MEPs should be sent out on. Only a single port can be set, if data is to be sent
 *         out over multiple ports then UseManualGbePort should be set to false. This does not affect the enable states of the 
 *         GBE ports as set by in the GBE control. Range: 0 to 3.
 *   \li DataFragmentSize A decimal number specifying the size of the data packet that is transmitted via the GBE card.
 *         It is the size of the fragment of the multi-event packet that is transmitted. Max 9000.
 *   \li MepTruncationHighWtrmrk Decimal number, max 65355, that represents the number of 32-bit words that can be put into
 *         a MEP before it is truncated.
 *   \li IngressConfigMailboxControl Default format described below. Note that the mask returned allows all bits to be set
 *         related to mailbox control (note the Ingress inhibit bits protected by the mask) so all settings must be
 *         considered when writing.
 *   \li IngressDisable A comma separated list of the inhibit setting for the Ingress FPGA. Accepts `Enable' or `Disable'
 *         of which there should be four entries in a comma separated list. All Ingress FPGAs must be set at a time.
 *         For example "Enable,Disable,Disable,Enable". Ingress FPGA 0 is the left most entry in the string.
 * \param sData Data to be written to the registers in the form of a string. The default format to be used is a hex string,
 *          lower or upper case that need not be zero padded to the appropriate register size.
 * \param ddcData Data converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param masks Masks that are associated with the write data also to be used with the fwCcpc_write function.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * Improperly formatted data will be removed from the list of write registers and will be indicated by an increased size of the
 * exceptionInfo array. The returned ddcData is not quite suitable for writting to the database and the masks must first
 * be appeneded to the end of element in the ddcData, this must be done by the User.
 */
void fwUkl1_parseEgressFpgaConfigRegistersForSet(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".EgressFpga.";

  //Add prefix to each name and convert the data into a byte array.
  //This indicates if there was a formatting error and we should remove those elements from the list or if they can be parsed.
  bool formatError;
  //Evaluate the size each time incase there is a problem with any of the data formating and an element is removed.
  //PVSS will update the size accordingly and this way we won't go out of bounds.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //Generally we will succeed so just mark it good here and it can be marked back if necessary.
    //Masks should only be created if no error is found.
    formatError = FALSE;

    //Check for destination MAC address being written.
    if ("MacDestinationAddress" == registerNames[element]) {
      //This is potentially has colons in it that may need to be removed.
      sData[element] = fwUkl1_convertMacToHex(sData[element], exceptionInfo);
      if ( "Fail" != sData[element] ) {
	//Need a mask for a 96-bit word write, 3 32-bit words.
        //All bytes in the dest MAC addr are configuration unlike the src addr.
	masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
	//Need to make sure the three 16-bit words that make up the data are padded to 32-bit words.
	//This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
	//It is explained in the main library documentation.
	sData[element] = "0000" + substr(sData[element], 8, 4) + "0000" + substr(sData[element], 4, 4) +
	  "0000" + substr(sData[element], 0, 4);
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, removed from list of writes.", "2");
      }
    }//else if MAC dest addr.
    
    else if ("MacSourceAddress" == registerNames[element]) {
      //This is potentially has colons in it that may need to be removed.
      sData[element] = fwUkl1_convertMacToHex(sData[element], exceptionInfo);
      if ( "Fail" != sData[element] ) {
	//Need a mask for a 96-bit word write, 3 32-bit words.
        //The first 32-bit word (which is the right most two bytes when reading the address e.g. 00:11:22:33:XX:XX)
        //are reserved and not User configurable. These are written during fwUkl1_configure().
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
	//Need to make sure the three 16-bit words that make up the data are padded to 32-bit words.
	//This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
	//It is explained in the main library documentation.
	sData[element] = "000000000000" + substr(sData[element], 4, 4) +
	  "0000" + substr(sData[element], 0, 4);
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, removed from list of writes.", "2");
      }
    }//else if MAC src addr.
    
    //Check for the IP addresses.
    else if ( "IpDestinationAddress" == registerNames[element] ) {
      //This is potentially in decimal with full stop delimiters.
      sData[element] = fwUkl1_convertIpToHex(sData[element], exceptionInfo);
      if ( "Fail" != sData[element] ) {
	//Need a 48-bit mask for this.
        //All bytes are User configurable, of the 16-bit words anyway.
	masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
	//Need to make sure the two 16-bit words that make up the data are padded to 32-bit words.
	//This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
	//It is explained in the main library documentation.
	sData[element] = "0000" + substr(sData[element], 4, 4) + "0000" + substr(sData[element], 0, 4);
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, removed from list of writes.", "2");
      }
    }//else if IP dest addr
    
    //Check for the IP addresses.
    else if ( "IpSourceAddress" == registerNames[element] ) {
      //This is potentially in decimal with full stop delimiters.
      sData[element] = fwUkl1_convertIpToHex(sData[element], exceptionInfo);
      if ( "Fail" != sData[element] ) {
	//Need a 48-bit mask for this.
        //The last 8-bits are reserved and not User configurable. These are written during fwUkl1_configure().
        //This is the second written byte, which corresponds to the last section of the IP e.g. 192.168.1.XXX.
	masks[element] = makeDynChar(0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0xff, 0xff);
	//Need to make sure the two 16-bit words that make up the data are padded to 32-bit words.
	//This relies on a deep knowledge of the way the fwCcpc library and L1 hardware works.
	//It is explained in the main library documentation.
	sData[element] = "0000" + substr(sData[element], 4, 2) + "000000" + substr(sData[element], 0, 4);
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, removed from list of writes.", "2");
      }
    }//else if IP src addr
    
    //Check for the partition ID.
    else if ( "PartitionId" == registerNames[element] ) {
      //Must be 8 characters big to be a valid ID, which is 4 bytes. This function will just make it fit if it is too big.
      fwUkl1_padString(sData[element], 4);
      //This needs to be written to two 32-bit registers with the two 16-bits words of the given partition ID
      //in each. Need to pad the beginning with four 0 characters and between the 1st and 2nd words.
      sData[element] = "0000" + substr(sData[element], 4, 4) + "0000" + substr(sData[element], 0, 4);
      //Here we need a 64-bit mask.
      masks[element] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0xff, 0xff);
    }//else if partition ID
    
    //Check for a change in throttle mode.
    else if ( "ThrottleMode" == registerNames[element] ) {
      //Convert the requested setting into a padded hex string.
      if ( "ALICE" == sData[element] ) {
	//For this we need to write to only bit 4.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x10);
	sData[element] = "00000010";
      } else if ( "LHCb" == sData[element] ) {
	//For this we need to write to only bits 4 to 7.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x10);
	sData[element] = "00000000";
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `ALICE' or `LHCb'. Removed from list of writes.", "2");
      }
    }//else if throttle mode
    
    //Check for fixing MEP destination address.
    else if ( "FixedMepDestinationAddress" == registerNames[element] ) {
      if ( "Disable" == sData[element] ) {
	//Write to bit 1.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x01);
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	//Write to bit 1.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x01);
	sData[element] = "00000001";
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Disable' or `Enable'. Removed from list of writes.", "2");
      }
    }//else if fixed MEP dest addr
    
    //Check for MEP destination broadcast wait setting.
    else if ( "DontWaitForMepDestBrdcst" == registerNames[element] ) {
      if ( "Don't wait" == sData[element] ) {
	//Write to bit 2.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x02);
	sData[element] = "00000002";
      } else if ( "Wait" == sData[element] ) {
	//Write to bit 2.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x02);
	sData[element] = "00000000";
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Don't wait' or `Wait'. Removed from list of writes.", "2");
      }
    }//else if don't wiat for MEP dest brdcst
    
    //Check for throttle enabling.
    else if ( "Throttle" == registerNames[element] ) {
      if ( "Disable" == sData[element] ) {
	//Write to bits 3 and 4.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x0c);
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	//Write to bits 3 and 4.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x0c);
	sData[element] = "0000000c";
      } else {
	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Disable' or `Enable'. Removed from list of writes.", "2");
      }
    }//else if throttle
    
    //Check for throttle invertion.
    else if ( "ThrottlePolarity" == registerNames[element] ) {
      if ( "Invert" == sData[element] ) {
	//Write to bits 5 and 6.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x30);
	sData[element] = "00000030";
      } else if ( "Normal" == sData[element] ) {
	//Write to bits 5 and 6.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x30);
	sData[element] = "00000000";
      } else {
 	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Invert' or `Normal'. Removed from list of writes.", "2");
     }
    }//else if throttle polarity
    
    //Check if TFC decoder is enabled.
    else if ( "TfcDecoder" == registerNames[element] ) {
      if ( "Disable" == sData[element] ) {
	//Write to bit 6.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x40);
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	//Write to bit 6.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x40);
	sData[element] = "00000040";
      } else {
 	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Invert' or `Normal'. Removed from list of writes.", "2");
      }
    }//else if TFC decoder
    
    //Check if transmission to the GBE output ports is enabled.
    else if ( "PollGbePorts" == registerNames[element] ) {
      if ( "Disable" == sData[element] ) {
	//Write to bit 7.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x80);
	sData[element] = "00000000";
      } else if ( "Enable" == sData[element] ) {
	//Write to bit 7.
	masks[element] = makeDynChar(0x00, 0x00, 0x00, 0x80);
	sData[element] = "00000080";
      } else {
 	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Disable' or `Enable'. Removed from list of writes.", "2");
      }
    }//else if Poll GBE ports.
    
    //Check to see if we are automatically detecting available GBE ports or using a User defined port.
    else if ( "UseManualGbePort" == registerNames[element] ) {
      if ( "Disable" == sData[element] ) {
        //Write to bit 10.
        masks[element] = makeDynChar(0x00, 0x00, 0x04, 0x00);
        sData[element] = "00000000";
      }
      else if ( "Enable" == sData[element] ) {
        //Write to bit 10.
        masks[element] = makeDynChar(0x00, 0x00, 0x04, 0x00);
        sData[element] = "00000400";
      }
      else {
 	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, expected `Disable' or `Enable'. Removed from list of writes.", "2");
      }
    }//else if use manual GBE port.
    
    //Sets the port that the MEPs are to be sent to if the UseManualGbePort flag is set.
    else if ( "ManualGbePortNumber" == registerNames[element] ) {
      unsigned port = sData[element];
      if ( 4 > port ) {
        //Set the port number to bits 8 and 9.
        port <<= 8;
        //Now convert it to a string.
        sData[element] = fwCcpc_convertDecToHex(port);
        //Pad the string.
        fwUkl1_padString(sData[element], 4);
        //Create the mask.
        masks[element] = makeDynChar(0x00, 0x00, 0x03, 0x00);
      }
      else {
 	formatError = TRUE;
	//Indicate who was lost before we remove the register name from the list.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " due to an incorrect data format, valid GBE port numbers are 0,1,2 or 3. Removed from list of writes.", "2");
      }
    }//else if set the manual GBE port number.
        
    //Data fragment size and MEP truncation high watermark are in decimal.
    else if ( ("DataFragmentSize" == registerNames[element]) || ("MepTruncationHighWtrmrk" == registerNames[element]) ) {
      //Convert from decimal to hex.
      sData[element] = fwCcpc_convertDecToHex(sData[element]);
      //Pad the string.
      fwUkl1_padString(sData[element], 4);
      //Create a 16-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0xff, 0xff);
    }//else if MTU/data fragment size and MEP trunc high wtrmrk
    
    //L1 ID is only 2-characters big.
    else if ( "L1Id" == registerNames[element] ) {
      //Create an 8-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0x00, 0xff);
      //Pad the string.
      fwUkl1_padString(sData[element], 4);
    }//else if L1 ID
    
    //Disabling the Ingress FPGA.
    else if ( "IngressDisable" == registerNames[element] ) {
      //Parse the data which will be in broadcast format.
      int exInfoSize = dynlen(exceptionInfo);
      dyn_dyn_string sParsedData = fwUkl1_parseBroadcastData(sData[element], exceptionInfo);
      if ( (dynlen(exceptionInfo) == exInfoSize) && (-1 != exInfoSize) ) {
        //Holds the mask. Start with a mask to write to all the Ingress control bits, but remove them if there is an error.
        unsigned mask = 0xf;
        //Holds the data to be written as we determine it.
        unsigned disable = 0;
        //Now loop over the Ingress FPGAs.
        for (unsigned loopFpga = 0; loopFpga < FWUKL1_FRONT_FPGAS; ++loopFpga) {
          //Create the data to write. In the event that we are only writing to 1 FPGA then we must always take
          //the data from the 1st element, as this is the only element that will contain any data.
          if ( "Disable" == sParsedData[element][loopFpga+1] ) {
            //Set the relevant bit high.
            disable |= 1<<loopFpga;
          }//if disable FPGA
          else if ( "Enable" == sParsedData[element][loopFpga+1] ) {
            //Do nothing the bit is already zero.
          }//else if enable FPGA
          else {
            //Indicate who was lost, don't use the format error here, just don't write to the appropriate bit via the mask.
            fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Failed to parse " + registerNames[element] + " for Ingress FPGA " + loopFpga + " due to an incorrect data format, expected `Disable' or `Enable'. Removed from list of writes.", "2");
            //Alter the mask so this FPGA is not written.
            mask &= ~(1<<loopFpga);
          }//else don't know how to handle data to write.
        }//for loopFpga
        //Convert the mask data to a string.
        string sTmpMask = fwCcpc_convertDecToHex(mask);
        fwUkl1_padString(sTmpMask,4);
        masks[element] = fwCcpc_convertHexToByte(sTmpMask);
        //Convert the register data to a string.
        sData[element] = fwCcpc_convertDecToHex(disable);
        fwUkl1_padString(sData[element], 4);
      }//if fwUkl1_parseBroadcastData() successfull
      else {
        //Mark the problem with the formatting.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForSet(): Could not parse broadcast data for Ingress FPGA inhibit settings. Removing from the list of writes.", "");
        formatError = TRUE;
      }//else fwUkl1_parseBroadcastData() unsuccessful
    }//else if IngressDisable
      
    //All the others just require a 16-bit mask and padding to 32-bit word.
    else {
      //Need a 16-bit mask.
      masks[element]   = makeDynChar(0x00, 0x00, 0xff, 0xff);
      //Pad the string, it still must be 32-bits long, the mask ensures we ignore the upper 16-bits.
      fwUkl1_padString(sData[element], 4);
    }//else

    if ( !formatError ) {
      //Put the data and register names in a format that the write can use.
      //Add the prefix to convert it to a datapoint name.
      registerNames[element] = prefix + registerNames[element];
      //Convert from a hex string to a byte array.
      ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
    }//if(!formatError)
    else {
      //Failed to parse the register, exception has already been raised...
      //Remove from register list.
      dynRemove(registerNames, element);
      //Remove the original data so it everything is kept in the same order as the registerNames
      dynRemove(sData, element);
      //Masks and ddcData aren't written so they are not affected.
      //Decrement the element number as we need to start this loop at the same number as before.
      --element;
    }//else(!formatError)

  }//for element

  //Done.
  return;
}//fwUkl1_parseEgressFpgaConfigRegistersForSet()

/*!
 * Takes the data read from the Egress FPGA configuration registers using fwCcpc_read or from a recipe and converts them to
 * strings to provide a description of the register setting. It also splits settings that occupy the same register address
 * into their own description.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the names and the BE FPGA registers to be read from as named in the UKL1 hardware
 *           datapoint. They can either be the full name or just the register name. The following registers can be parsed
 *           (their names are given in the short form, just the DP element) and also a description of the return value given by sData,
 *           all are case sensitive:
 *   \li MacDestinationAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li MacSourceAddress A 12 character string that gives the hex number for the MAC address, can either be a continuous
 *         string of the address or may posses a colon delimiter every two characters,
 *         e.g. 000e0cb02d19 or 00:0e:0c:b0:2d:19.
 *   \li IpDestinationAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li IpSourceAddress A string that contains either the address in hex format e.g. c0a0c0c0 or in a typical IP4 format
 *         e.g. 192.168.192.192.
 *   \li Protocol Default format described below.
 *   \li TypeOfService Default format described below.
 *   \li TimeToLive Default format described below.
 *   \li Type Default format described below.
 *   \li Version Default format described below.
 *   \li PartitionId Must contain an 8 character string representing the two 16-bit words of data for the partition ID.
 *   \li L1Id An 8-bit value, 2-character string. This will be present in the output data packet and MUST be unique for each L1.
 *         This is set correctly by the FSM and generally should not be User defined, except maybe when the User is an expert.
 *   \li ThrottleMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         of the pixel chips. Should match the Ingress FPGA channel setting.
 *   \li Latency Default format described below.
 *   \li MepPackingFactor Default format described below.
 *   \li FixedMepDestinationAddress `Enable' or `Disable' to enable or disable a fixed multi-event packet destination address.
 *   \li DontWaitForMepDestBrdcst `Don't wait' or `Wait' to enable or disable the waiting for a multi-event packet destination
 *         address broadcast, respectively.
 *   \li Throttle `Enable' or `Disable' to enable or disable the throttle.
 *   \li ThrottlePolarity `Invert' or `Normal' to select the polarity of the throttle output.
 *   \li TfcDecoder `Enable' or `Disable' to enable or disable the decoding of TFC commands. The board will not operate without
 *         this setting enabled.
 *   \li PollGbePorts `Enable' or `Disable' to enable the Egress FPGA to send data to the GBE card for transmission.
 *   \li UseManualGbePort `Enable' or `Disable' to disable automatic detection of available GBE ports and instead enable the
 *         use of a single fixed GBE port defined by ManualGbePortNumber. This port must be enabled via the GBE control.
 *   \li ManualGbePortNumber The GBE port that MEPs should be sent out on. Only a single port can be set, if data is to be sent
 *         out over multiple ports then UseManualGbePort should be set to false. This does not affect the enable states of the 
 *         GBE ports as set by in the GBE control. Range: 0 to 3.
 *   \li DataFragmentSize A decimal number specifying the size of the data packet that is transmitted via the GBE card.
 *         It is the size of the fragment of the multi-event packet that is transmitted. Max 9000.
 *   \li MepTruncationHighWtrmrk Decimal number, max 65355, that represents the number of 32-bit words that can be put into
 *         a MEP before it is truncated.
 *   \li IngressConfigMailboxControl Default format described below.
 *   \li IngressDisable A comma separated list of the inhibit setting for the Ingress FPGA. Accepts `Enable' or `Disable'
 *         of which there should be four entries in a comma separated list. All Ingress FPGAs must be set at a time.
 *         For example "Enable,Disable,Disable,Enable". Ingress FPGA 0 is the left most entry in the string.
 * \param  ddcData This contains the raw data that has been read back from the PVSS datapoint.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           By default a hex representation, zero padded to the nearest byte boundary of the settings.
 *           In the event that the read failed it will contain the text `Fail'.
 * \param  callStatusList This is the status list returned by the fwCcpc_read function and if the data to be parsed was generated
 *           by the use of this function then a callStatusList should be passed in order that errors can be checked for.
 *           It is possible, for example via monitoring, that the fwCcpc_read function is not used and in this case an empty
 *           array should be passed and it will be ignored. Any errors found in this array will be reported via the exceptionInfo.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_parseEgressFpgaConfigRegistersForGet(string ukl1Name, const dyn_string& registerNames, dyn_dyn_char ddcData, dyn_string& data, const dyn_int& callStatusList, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that if the full DPE name is given we match to it.
  const string prefix = ukl1Name + ".EgressFpga.";
  //Determine the number of registers for which we are reading the data.
  const unsigned numRegs = dynlen(registerNames);
  //Determine whether we should be checking the status list. If it is empty don't look in it.
  const bool checkStatus = (0!=dynlen(callStatusList));
  //Loop over all the given registers and put the data into a single string for return.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //This series of `if' statements determines how to handle the possible raw status register data we may receive.
    //Check to see if we have actually been given any data.
    if ( 0 == dynlen(ddcData) ) {
      //Fill the return array with null strings.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): No data read back to parse.", "2");
    }
    //Check to see if the specific element has any data.
    else if ( 0 == dynlen(ddcData[element]) ) {
      //Fill the returns with null strings the equivalent of no data.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): No data read for register " + registerNames[element] + ".", "2");
    }
    //Check for errors. As we are using AND the second case will only be evaluated provided checkStatus is not empty
    //and hence we should avoid out of bounds errors.
    //Check for both the case when the register names has the appropriate prefix and for the case where it doesn't.
    else if ( checkStatus && (0 != callStatusList[element]) ) {
      //Construct the error message.
      string errMsg = "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " encountered ";
      if (callStatusList[element] > 0) {
	errMsg += "a problem writing to the CCPC server, return code " + callStatusList[element] + ".";
      } else if (callStatusList[element] < 0) {
	errMsg += "a problem with PVSS.";
      } else {
	errMsg += "an unknown error.";
      }
      //Now raise the exception.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
      //Mark the error.
      data[element] = "Fail";
    }
    //We have some valid data, deal with it.
    else {
      //This series of `if' statements determines how to handle the possible raw status register data we may receive.
      //Convert the data from a series of 32-bit words to a series of 16-bit words.
      const int exInfoSize = dynlen(exceptionInfo);
      fwUkl1_convert32To16BitWords(ddcData[element], exceptionInfo);

      //Need to know the size of the read data to allow absolute indexing of the given data.
      //It can be used to account for whether the mask data is in the lower elements if it comes from a recipe.
      const int size = dynlen(ddcData[element]);

      //Check that the data went through the first stage of parsing correctly.
      if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): Failed to read " + registerNames[element] + ".", "2");
        data[element] = "Fail";
      }
      //Parse the MAC addresses.
      else if ( ("MacDestinationAddress" == registerNames[element]) || ("MacSourceAddress" == registerNames[element]) ||
  	      (prefix + "MacDestinationAddress" == registerNames[element]) || (prefix + "MacSourceAddress" == registerNames[element]) ) {
        //Now convert the remaining 48-bits into a MAC address.
        //Don't worry about errors the exceptionInfo should be appropriately filled if there was an error plus the string reads `Fail'.
        data[element] = fwUkl1_convertByteToMac(ddcData[element], exceptionInfo);
      }//else if MAC destination or source address
      
      //Parse the IP addresses.
      else if ( ("IpDestinationAddress" == registerNames[element]) || ("IpSourceAddress" == registerNames[element]) ||
  	      (prefix + "IpDestinationAddress" == registerNames[element]) || (prefix + "IpSourceAddress" == registerNames[element]) ) {
        //Convert the given data into a full stop delimited IP.
        //Don't worry about exceptions the convertion function will ensure `Fail' is returned in an error and describe what went wrong.
        data[element] = fwUkl1_convertByteToIp(ddcData[element], exceptionInfo);
      }//else if IP destination or source address
      
      //Parse throttle mode settings.
      else if ( ("ThrottleMode" == registerNames[element]) || (prefix + "ThrottleMode" == registerNames[element]) ) {
        //Convert the requested setting into a padded hex string.
        if ( 0x10 == (ddcData[element][size] & 0x10) ) {
  	  data[element] = "ALICE";
        } else if ( 0x00 == (ddcData[element][size] & 0x10) ) {
  	  data[element] = "LHCb";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if throttle mode
      
      //Parse enable forced fixed MEP destination address settings.
      else if ( ("FixedMepDestinationAddress" == registerNames[element]) || (prefix + "FixedMepDestinationAddress" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x01 == (ddcData[element][size] & 0x01) ) {
    	  data[element] = "Enable";
        } else if ( 0x00 == (ddcData[element][size] & 0x01) ) {
  	  data[element] = "Disable";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if fixed MEP destination address
      
      //Parse don't wait for MEP destination broadcast settings.
      else if ( ("DontWaitForMepDestBrdcst" == registerNames[element]) || (prefix + "DontWaitForMepDestBrdcst" == registerNames[element]) ) {
        //Is it waiting for not.
        if ( 0x02 == (ddcData[element][size] & 0x02) ) {
  	  data[element] = "Don't wait";
        } else if ( 0x00 == (ddcData[element][size] & 0x02) ) {
  	  data[element] = "Wait";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if don't wait for MEP destination broadcast
      
      //Check to see if the throttle is enabled.
      else if ( ("Throttle" == registerNames[element]) || (prefix + "Throttle" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x0c == (ddcData[element][size] & 0x0c) ) {
  	  data[element] = "Enable";
        } else if ( 0x00 == (ddcData[element][size] & 0x0c) ) {
  	  data[element] = "Disable";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if throttle
      
      //Check polarity of throttle.
      else if ( ("ThrottlePolarity" == registerNames[element]) || (prefix + "ThrottlePolarity" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x30 == (ddcData[element][size] & 0x30) ) {
  	  data[element] = "Invert";
        } else if ( 0x00 == (ddcData[element][size] & 0x30) ) {
  	  data[element] = "Normal";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if throttle polarity
      
      //Check to see if the TFC is enabled.
      else if ( (prefix + "TfcDecoder" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x40 == (ddcData[element][size] & 0x40) ) {
          data[element] = "Enable";
        } else if ( 0x00 == (ddcData[element][size] & 0x40) ) {
          data[element] = "Disable";
        } else {
          fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
          data[element] = "Fail";
        }
      }//else if TFC decoder
      
      //Check to see if the transmission of data to the GBE is enabled.
      else if ( ("PollGbePorts" == registerNames[element]) || (prefix + "PollGbePorts" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x80 == (ddcData[element][size] & 0x80) ) {
  	  data[element] = "Enable";
        } else if ( 0x00 == (ddcData[element][size] & 0x80) ) {
  	  data[element] = "Disable";
        } else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if poll gbe ports.
      
      //Check to see if we are automatically detecting available GBE ports or using a User defined port.
      else if ( ("UseManualGbePort" == registerNames[element]) || (prefix + "UseManualGbePort" == registerNames[element]) ) {
        //Is it enabled or disabled.
        if ( 0x04 == (ddcData[element][size-1] & 0x04) )
          data[element] = "Enable";
        else if ( 0x00 == (ddcData[element][size-1] & 0x04) )
          data[element] = "Disable";
        else {
          fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
          data[element] = "Fail";
        }
      }//else if use manual GBE port.
    
      //Sets the port that the MEPs are to be sent to if the UseManualGbePort flag is set.
      else if ( ("ManualGbePortNumber" == registerNames[element])  || (prefix + "ManualGbePortNumber" == registerNames[element])) {
        //Get the number.
        int exInfoSize = dynlen(exceptionInfo);
        //We have already converted from 32- to 16-bits so don't do it again i.e. fgpa = false
        int port = fwUkl1_parseReadRegister(ddcData[element], 8, 2, false, exceptionInfo);
        if ( (-1 != port) && (dynlen(exceptionInfo) == exInfoSize) )
          data[element] = port;
        else {
  	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
  	  data[element] = "Fail";
        }
      }//else if set the manual GBE port number.
        
      //Both the data fragment size and MEP truncation high watermark should be returned in decimal.
      else if ( ("DataFragmentSize" == registerNames[element]) || (prefix + "DataFragmentSize" == registerNames[element]) ||
                ("MepTruncationHighWtrmrk" == registerNames[element]) || (prefix + "MepTruncationHighWtrmrk" == registerNames[element])) {
        if ( size == 4 ) {
  	  //Value is twice as big as we expect, hence the mask must be present. Zero the first two elements to cure this problem.
  	  ddcData[element][1] = 0;
  	  ddcData[element][2] = 0;
        }//No else as it is just right in any other case.
        data[element] = fwCcpc_convertByteToDec(ddcData[element]);
      }//else if data fragment size or MEP truncation high watermark.
      
      //This is only 8-bits big.
      else if ( ("L1Id" == registerNames[element]) || (prefix + "L1Id" == registerNames[element]) ) {
        //Zero the elements until only the last on is left as this is the actual L1 ID.
        const unsigned maxCount = dynlen(ddcData[element]);
        //Notice that the loop deliberately goes to less than element count. This is to ensures that the whole array is not zerod.
        for (unsigned count = 1; count < maxCount; ++count) {
  	  ddcData[element][count] = 0;
        }
        //Also remove the leading zeros.
        data[element] = strltrim(fwCcpc_convertByteToHex(ddcData[element]), "0");
        //Check we shouldn't be returning zero!
        if ("" == data[element])
  	  data[element] = "0";
      }//else if L1Id
      
      //Ingerss inhibit register.
      else if ( ("IngressDisable" == registerNames[element]) || ((prefix + "IngressDisable") == registerNames[element]) ) {
        //Is it enabled or disabled?
        //First convert it to a single number.
        unsigned fpgaStates = fwCcpc_convertByteToDec(ddcData[element]);
        //This is to mark whether it is the first time round the loop.
        bool firstTime = TRUE;
        //Now loop over the Ingress FPGAs.
        for (unsigned loopFpga = 0; loopFpga < FWUKL1_FRONT_FPGAS; ++loopFpga) {
          //If it is not the first time round the loop we should add a comma before we write the next setting.
          if ( !firstTime )
            data[element] += ",";
          if ( 0x01 == ((fpgaStates>>loopFpga) & 0x01) )
            data[element] += "Disable";
          else if ( 0x00 == ((fpgaStates>>loopFpga) & 0x01) )
            data[element] += "Enable";
          else {
            fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseEgressFpgaConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
            data[element] += "Fail";
          }//else failed to recognise the data.
          //If it is the first time, it isn't any more.
          if ( firstTime )
            firstTime = FALSE;
        }//for loopFpga
      }//else if IngressDisable
      
      //Can just return the read data to a hex string as the extra words have already been removed.
      else {
        //Need to deal with the mask being present here......
        if ( (size == 4) && ("PartitionId" != registerNames[element]) && (prefix + "PartitionId" != registerNames[element]) ) {
  	  //In this case we must have a 16-bit register and the mask from the recipe.
  	  //Only the partition ID is 32-bits big and the register name is not that.
  	  //Zero the first two elements to get ride of the mask.
  	  ddcData[element][1] = 0;
  	  ddcData[element][2] = 0;
        }
        if ( size == 8 ) {
  	  //In this case it is the partition ID and we can delete the first 4 elements.
  	  ddcData[element][1] = 0;
  	  ddcData[element][2] = 0;
  	  ddcData[element][3] = 0;
  	  ddcData[element][4] = 0;
        }
        //The other case is 2 elements, which is just a 16-bit register or 4 (32-bits) and it is the partition ID.
        //In both of these cases the data can be left alone.
        data[element] = strltrim(fwCcpc_convertByteToHex(ddcData[element]), "0");
        //Check we shouldn't be returning zero!
        if ("" == data[element])
  	  data[element] = "0";
      }
    }//parse valid data else.
  }//for elements

  //Done.
  return;
}//fwUkl1_parseEgressFpgaConfigRegistersForGet()

/*!
 * This function takes the data for the various Ingress FPGA channel registers described by a string and converts it to data that can
 * used with _fwUkl1_write() or _fwUkl1_saveRecipe(). It allows data to be given in a more human readable format and converts
 * it to the byte array required for writting to the CCPC. It also returns the appropriate mask that must be used when
 * writing the data. If being used with fwHw_saveRecipe() then the mask will have to be appended to the data array.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on the Ingress FPGA that is to be written to. Range: 0 to 8.
 * \param  inFpga Number of the Ingress FPGA that is to be written. Range:0 to 3.
 * \param  registerNames List of Ingress FPGA channel register names that are to be written to. Returned with the appropriate name
 *           prefixed by reference. The allowed values are:
 *   \li Disable Accepts the text `Disable' or `Enable' to disable or enable the Ingress FPGA channel(s).
 *   \li ZeroSuppression String should be either `Enable' or `Disable' to enable or disable zero suppression
 *         for the desired Ingress FPGA channel.
 *   \li HpdPixelMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         required from the pixel chips.
 *   \li PixelMasks `Force on' or `Force off'. This determines how the pixel masks are applied, they can either be
 *         forced on which means that pixel will always be active in the data; or forced off which means the pixel
 *         is always inactive in the data.
 *   \li Emulator Accepts the text `Disable' or `Enable' to disable or enable input data emulation on the Ingress FPGA channel(s).
 *         This setting disables the input data from the optical inputs on the associated channel(s) and then sends emulated
 *         data through the L1 data processing chain.
 *   \li EmulatorHpdPixelMode Accepts `LHCb' or `ALICE'. The type of emulated data that will be sent. This will not automatically
 *         enable the event emulation.
 *   \li EmulatorEvent Accepts `Send' or `Don't send' to enable the sending of emulated events. This can only occur if the emulator is
 *         enabled via the `Emulator' setting. `Don't send' will prevent events being sent through the chain even if the emulator is
 *         enabled.
 * \param  sData Data to be written to the registers in the form of a hex string. There is no default value, on the
 *           data described under the specific register name will be accepted.
 * \param  dcData Data converted into a dyn_char for use with the _fwUkl1_write function. Returned by reference.
 *           As all the settings are written into a FIFO the appropriately ordered bytes are returned.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * All of the listed register names will be present otherwise the parsing of the registers will fail. There
 * are no masks associated with a FIFO write.
 */
void fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(string ukl1Name, unsigned channel, unsigned inFpga, const dyn_string& registerNames, dyn_string sData, dyn_char& dcData, dyn_string& exceptionInfo) {
  //Convert the data into a byte array.
  //This indicates if there was a formatting error. In the event of an error give up!
  bool formatError;
  //This holds the FIFO data as 32-bit word. It will be later converted a byte array.
  //The channel number and type are in the lower 16-bits, the type is zero so can be left.
  //Due to the crazy edianness of the fwCcpc library we have to write the lower word in the upper word!
  //and vice versa.
  unsigned fifo = channel << 12+16;
  //As we don't remove registers in the event of an error we can determine the size of the names list once.
  const int numRegs = dynlen(registerNames);
  for (unsigned element = 1; element <= numRegs; ++element) {
    if ( "Disable" == registerNames[element] ) {
      //Check to see what to do.
      if ( "Disable" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<0;
      }//if disable channel
      else if ( "Enable" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//if("Disable"==registerNames[element])
    else if ( "HpdPixelMode" == registerNames[element] ) {
      //Check to see what to do.
      if ( "ALICE" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<1;
      }//if disable channel
      else if ( "LHCb" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `ALICE' or `LHCb'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("HpdPixelMode"==registerNames[element])
    else if ( "ZeroSuppression" == registerNames[element] ) {
      //Check to see what to do.
      if ( "Enable" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<2;
      }//if disable channel
      else if ( "Disable" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("ZeroSuppression"==registerNames[element])
    else if ( "Emulator" == registerNames[element] ) {
      //Check to see what to do.
      if ( "Disable" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<4;
      }//if disable channel
      else if ( "Enable" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("Emulator"==registerNames[element])
    else if ( "EmulatorHpdPixelMode" == registerNames[element] ) {
      //Check to see what to do.
      if ( "ALICE" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<5;
      }//if disable channel
      else if ( "LHCb" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("EmulatorHpdPixelMode"==registerNames[element])
    else if ( "EmulatorEvent" == registerNames[element] ) {
      //Check to see what to do.
      if ( "Send" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<7;
      }//if disable channel
      else if ( "Don't send" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("EmulatorEvent"==registerNames[element])
    else if ( "PixelMasks" == registerNames[element] ) {
      //Check to see what to do.
      if ( "Force on" == sData[element] ) {
        //Set the relevant bit high.
        fifo |= 1<<8;
      }//if disable channel
      else if ( "Force off" == sData[element] ) {
        //Do nothing the bit is already zero.
      }//else if enable channel
      else {
        //Indicate who was lost and mark the format error so we quit.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse " + registerNames[element] + " for channel " + channel + ". Expected `Disable' or `Enable'. Parsing aborted.", "1");
      }//else unrecognised option.
    }//elseif("PixelMasks"==registerNames[element])
    else {
      //Indicate that something went wrong. We could just remove this register and not abort, the rest of the registers could be present...
      formatError = TRUE;
      //Indicate who was unrecognised. Don't indicate that one of the required registers was parsed.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(): Failed to parse reigster `" + registerNames[element] + "' as the register is unknown. Parsing aborted.", "1");
    }//else unknown register
    
    //Check for format errors
    if (formatError) {
      //Quit in the event of a format error as there won't be enough data to write.
      break;
    }//else(formatError)

  }//for element

  if ( !formatError ) {
    //Put the data and register names in a format that the write can use.
    //Convert from a hex string to a byte array.
    string hexFifo = fwCcpc_convertDecToHex(fifo);
    fwUkl1_padString(hexFifo, 4);
    dcData = fwCcpc_convertHexToByte(hexFifo);
  }//if(!formatError)
  //Done.
  return;
}//fwUkl1_parseIngressFpgaChannelConfigRegisterForSet()

/*!
 * Takes the configuration settings read from the Ingress status FIFO and returns the values as a string along with
 * a descriptive name for the register. This can take data either from a recipe or from the hardware, but if taken
 * from the hardware it is assumed that the data has already been through the first stage of parsing and contains
 * only the configuration settings for the specific channel.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the names of the settings that are in the corresponding elements
 *           in the sData array. It is not, as with the other parsing functions that name of the register that it
 *           was read from. In this case all data is read from a single register. The following registers are return parsed
 *           and can be returned, all are case sensitive:
 *   \li Disable Returns the text `Disable' or `Enable' for a disabled or enabled the Ingress FPGA channel.
 *   \li ZeroSuppression String should be either `Enable' or `Disable' if the zero suppression is enabled or disabled
 *         for the desired Ingress FPGA channel.
 *   \li HpdPixelMode String should be either `ALICE' or `LHCb', which should correspond to the mode of operation
 *         required from the pixel chips.
 *   \li PixelMasks `Force on' or `Force off' depending on the action of the pixel masks. Force on indicates that
 *         the pixel masks will enable that pixel in the data and force off forces the disabling of that pixel
 *         in data.
 *   \li Emulator Returns the text `Disable' or `Enable' depending on whether the emulator is disabled or enabled.
 *         This setting disables the input data from the optical inputs on the associated channel and then sends emulated
 *         data through the L1 data processing chain.
 *   \li EmulatorHpdPixelMode Returns `LHCb' or `ALICE'. The type of emulated data that will be sent. This will not automatically
 *         enable the event emulation.
 *   \li EmulatorEvent Accepts `Send' or `Don't send' to indicate whether emulated events are being sent or not. This can only occur if the emulator is
 *         enabled via the `Emulator' setting. `Don't send' will prevent events being sent through the chain even if the emulator is
 *         enabled.
 * \param  dcData This contains the raw data that has been read back from the hardware or recipe. Note it is a 
 *           dyn_char and not dyn_dyn_char as with the other functions of this type. This function can only handle one register input
 *           and hence requires only a single character array for input data.
 * \param  sData dyn_string returned by reference that contains a string representing the value in the registers.
 *           By default a hex representation, zero padded to the nearest byte boundary of the settings.
 *           In the event that the read failed it will contain the text `Fail'.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_parseIngressFpgaChannelConfigRegisterForGet(string ukl1Name, const dyn_string& registerNames, dyn_char dcData, dyn_string& sData, dyn_string& exceptionInfo) {
  //Keep track of the amount of exception information that is generated.
  int exInfoSize = dynlen(exceptionInfo);
  //Determine the number of registers that we are reading.
  const unsigned numRegs = dynlen(registerNames);
  //We can check for valid data outside the register loop as we are only check one set of data.
  //This series of `if' statements determines how to handle the possible raw status register data we may receive.
  //Check to see if we have actually been given any data.
  if ( 0 == dynlen(dcData) ) {
    //Fill the returns with null strings the equivalent of no data.
    sData = makeDynString("","","","","","","");
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): No data read for register for the Ingress FPGA channel configuration.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if no data.
  //Parse the data it must be valid!
  else {
    //We have some valid data, deal with it.
    //Loop over all the registers and see which bits we want.
    for (int element = 1; element <= numRegs; ++element) {
      //This is used by all case statements to store the relevant bit for the setting we are investigating.
      int interestingBit = -1;
      //Note we only will ever get the short form of the names as they don't exist in the hardware or the
      //recipe in this form.
      //Channel inhibit
      if ( "Disable" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 0, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "Disable";
          else
            sData[element] = "Enable";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//if channel inhibit
      
      //Zero suppression
      else if ( "ZeroSuppression" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 2, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "Enable";
          else
            sData[element] = "Disable";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if zero suppression
      
      //HPD pixel mode
      else if ( "HpdPixelMode" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 1, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "ALICE";
          else
            sData[element] = "LHCb";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if HPD pixel mode
      
      //Pixel masks
      else if ( "PixelMasks" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 8, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "Force on";
          else
            sData[element] = "Force off";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if pixel masks
      
      //Emulator
      else if ( "Emulator" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 4, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "Disable";
          else
            sData[element] = "Enable";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if emulator
      
      //EmulatorEvents
      else if ( "EmulatorEvent" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 7, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "Send";
          else
            sData[element] = "Don't send";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if emulator events
      
      //EmulatorHpdPixelMode
      else if ( "EmulatorHpdPixelMode" == registerNames[element] ) {
        interestingBit = fwUkl1_parseReadRegister(dcData, 5, 1, FALSE, exceptionInfo);
        if  ( (dynlen(exceptionInfo) <= exInfoSize) && (-1 != exInfoSize) ) {
          if ( 1 == interestingBit )
            sData[element] = "ALICE";
          else
            sData[element] = "LHCb";
        }//if got interesting bit.
        else {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to extract the relevant setting bit for the " + registerNames[element] + " setting.", "2");
          exInfoSize = dynlen(exceptionInfo);
          sData[element] = "Fail";
        }//else failed to get interesting bit.
      }//else if Emulator HPD pixel mode

      //Unknown register
      else {
        fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_parseIngressFpgaChannelConfigRegistersForGet(): Failed to recognise the register name `" + registerNames[element] + "' no data returned for this setting.", "2");
        exInfoSize = dynlen(exceptionInfo);
        sData[element] = "Fail";
      }//else unknown reg
    }//for element
  }//parse valid data else.

  //Done.
  return;
}//fwUkl1_parseIngressFpgaChannelConfigRegistersForGet()


// =========
//    GBE
// =========

/*!
 * This function takes the data for the various GBE registers described by a string and converts it to data that can
 * used with fwCcpc_write() or fwHw_saveRecipe(). It allows data to be given in a more human readable format and converts
 * it to the byte array required for writting to the CCPC. It also returns the appropriate mask that must be used when
 * writing the data. If being used with fwHw_saveRecipe() then the mask will have to be appended to the data array.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames List of GBE register names that are to be written to. Returned with the appropriate name
 *           prefix by reference. The allowed values are:
 *   \li JTAGID Default data format, see below.
 *   \li MACSoftReset Default data format, see below.
 *   \li RxFifoPrtReset Default data format, see below.
 *   \li TxFifoPrtReset Default data format, see below.
 *   \li PortEnable `Enable' or `Disable' in a comma delimited format. Values for all four ports must be present.
 *   \li RxFifoEnable Default data format, see below.
 *   \li Mode Default data format, see below.
 *   \li Clock Default data format, see below.
 *   \li SPI3ConfigTrnmtGlobal Default data format, see below.
 *   \li SPI3ConfigRcv Default data format, see below.
 *   \li RxStatus Default data format, see below.
 *   \li PHYCtrl Default data format, see below.
 *   \li PHYData Default data format, see below.
 *   \li MDIOControl Default data format, see below.
 * \param  sData Data to be written to the registers in the form of a hex string. The default value is an 8-character
 *           hex string, other values are accepted for specific registers as described above.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write or fwHw_saveRecipe function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * Improperly formatted data will be removed from the list of write registers and will be indicated by an increased size of the
 * exceptionInfo array.
 */
void fwUkl1_parseGbeConfigRegistersForSet(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".GBE.";

  //Add prefix to each name and convert the data into a byte array.
  //This indicates if there was a formatting error and we should remove those elements from the list or if they can be parsed.
  bool formatError;
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //This way we only need to indicate if something went wrong.
    formatError = FALSE;
    //Deal with the different formats.
    if ( "PortEnable" == registerNames[element] ) {
      //Just parse this data as broadcast data, all the other data can be left alone.
      //Technically this function takes a dyn_string, however attempting to makeDynString causes the function
      //to fail horribly. It is probably something to do with temporaries and const &.
      dyn_dyn_string sParsedData = fwUkl1_parseBroadcastData(sData[element], exceptionInfo);
      if ( (0 < dynlen(sParsedData)) && (0 < dynlen(sParsedData[1])) ) {
        //Holds the data to be written as it is determined.
        unsigned disable = 0;
        //Now loop over the appropriate range of GBE ports.
        for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
          //Create the data to write.
          if ( "Enable" == sParsedData[1][port+1] ) {
            //Set the relevant bit high.
            disable |= 1<<port;
          }//if enable
          else if ( "Disable" == sParsedData[1][port+1] ) {
            //Do nothing the bit is already zero.
          }//else if disable
          else {
            formatError = TRUE;
            //Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
            fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForSet(): Failed to parse " + registerNames[element] + ", expected `Disable' or `Enable'. Removed from list of writes.", "2");
          }//else don't understand data format
        }//for port
        //Now convert the int data back to a string.
        sData[element] = fwCcpc_convertDecToHex(disable);
      }//if(0<dynlen(sParsedData)&&0<dynlen(sParsedData[1]))
      else {
        formatError = TRUE;
        //Indicate who was lost before we remove the register name from the list, details of failure indicated by the parse IP address function.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForSet(): Failed to parse " + registerNames[element] + ", no data was found to write. Removed from list of writes.", "2");
      }//else(0<dynlen(sParsedData)&&0<dynlen(sParsedData[1]))
    }//if port enable
    //There is no else for generic registers as they are handled by the if(!formatError) below.
    
    if ( !formatError ) {
      //Put the data and register names in a format that the write can use.
      //Add the prefix to convert it to a datapoint name.
      registerNames[element] = prefix + registerNames[element];
      //Pad the string.
      fwUkl1_padString(sData[element], 4);
     //Convert from a hex string to a byte array.
      ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
      //Everything needs a 32-bit mask.
      masks[element]   = makeDynChar(0xff, 0xff, 0xff, 0xff);
    } else {
      //Failed to parse the register, exception has already been raised...
      //Remove from register list.
      dynRemove(registerNames, element);
      //Remove the original data so it everything is kept in the same order as the registerNames
      dynRemove(sData, element);
      //Masks and ddcData aren't written so they are not affected.
      //Decrement the element number as we need to start this loop at the same number as before.
      --element;
    }
  }
  
  //Done.
  return;
}//fwUkl1_parseGbeConfigRegistersForSet()

/*!
 * This takes the data read from the GBE registers in the format as read from fwCcpc_read() or fwHw_getRecipe() and
 * returns the values as a decriptive string.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint elements to be
 *           read from, which corresponds to the appropriate hardware register. The accepted names are:
 *   \li JTAGID Default data format, see below.
 *   \li MACSoftReset Default data format, see below.
 *   \li RxFifoPrtReset Default data format, see below.
 *   \li TxFifoPrtReset Default data format, see below.
 *   \li PortEnable `Enable' or `Disable' in a comma delimited format. Values for all four ports must be present.
 *   \li RxFifoEnable Default data format, see below.
 *   \li Mode Default data format, see below.
 *   \li Clock Default data format, see below.
 *   \li SPI3ConfigTrnmtGlobal Default data format, see below.
 *   \li SPI3ConfigRcv Default data format, see below.
 *   \li RxStatus Default data format, see below.
 *   \li PHYCtrl Default data format, see below.
 *   \li PHYData Default data format, see below.
 *   \li MDIOControl Default data format, see below.
 * \param  ddcData This contains the raw data that has been read back from the PVSS datapoint.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will either be a hex representation, zero padded to the nearest byte boundary or a text description
 *           of the settings. In the event that the read failed it will contain the text `Fail'.
 * \param  callStatusList This is the status list returned by the fwCcpc_read function and if the data to be parsed was generated
 *           by the use of this function then a callStatusList should be passed in order that errors can be checked for.
 *           It is possible, for example via monitoring or reading from a recipe, that the fwCcpc_read function is not used and in
 *           this case an empty array should be passed and it will be ignored. Any errors found in this array will be reported
 *           via the exceptionInfo.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_parseGbeConfigRegistersForGet(string ukl1Name, const dyn_string& registerNames, dyn_dyn_char ddcData, dyn_string& data, const dyn_int& callStatusList, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that if we are given the full DPE name we can match against it.
  const string prefix = ukl1Name + ".GBE.";
  //Determine the number of registers for which we are reading the data.
  const unsigned numRegs = dynlen(registerNames);
  //Determine whether we should be checking the status list. If it is empty don't look in it.
  const bool checkStatus = (0 != dynlen(callStatusList));
  //Used to check whether the function executed without a problem.
  int status = 0;

  //Loop over all the given registers and put the data into a single string for return.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //This series of `if' statements determines how to handle the possible raw status register data we may receive.
    //Check to see if we have actually been given any data.
    if ( 0 == dynlen(ddcData) ) {
      //Fill the return array with null strings.
      if ( ("PortEnable" != registerNames[element]) || ((prefix + "PortEnable") != registerNames[element]) ) {
        data[element] = "";
      } else {
      	data[element] = ",,,";
      }
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForGet(): No data read back to parse.", "2");
    }
    //Check to see if the specific element has any data.
    else if ( 0 == dynlen(ddcData[element]) ) {
      if ( ("PortEnable" != registerNames[element]) || ((prefix + "PortEnable") != registerNames[element]) ) {
        data[element] = "";
      } else {
      	data[element] = ",,,";
      }
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForGet(): No data read for register " + registerNames[element] + ".", "2");
    }
    //Check for errors. As we are using AND the second case will only be evaluated provided checkStatus is not empty
    //and hence we should avoid out of bounds errors.
    //Check for both the case when the register names has the appropriate prefix and for the case where it doesn't.
    else if ( (checkStatus && (0 != callStatusList[element])) ) {
      //Construct the error message.
      string errMsg = "fwUkl1_parseGbeConfigRegistersForGet(): " + registerNames[element] + " encountered ";
      if (callStatusList[element] > 0) {
	errMsg += "a problem writing to the CCPC server, return code " + callStatusList[element] + ".";
	status = 1;
      } else if (callStatusList[element] < 0) {
	errMsg += "a problem with PVSS.";
	status = -1;
      } else {
	errMsg += "an unknown error.";
	status = -1;
      }
      //Now raise the exception.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
      //Mark the error.
      if ( ("PortEnable" != registerNames[element]) || ((prefix + "PortEnable") != registerNames[element]) ) {
        data[element] = "Fail";
      } else {
      	data[element] = "Fail,Fail,Fail,Fail";
      }
    }
    else if ( 0 == dynlen(ddcData[element]) ) {
      //No data, raise an exception.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForGet(): Failed to read " + registerNames[element] + ".", "2");
    }
    //Deal with the port enable settings.
    else if (("PortEnable" == registerNames[element]) || ((prefix + "PortEnable") == registerNames[element]) ) {
      //Is it enabled or disabled?
      //First convert it to a string and trim off the last 8 characters if it is from the recipe, otherwise there will be no change.
      //First convert it to a single number.
      const string sTmp = substr(fwCcpc_convertByteToHex(ddcData[element]), 0, 8);
      unsigned portStates = fwCcpc_convertHexToDec(sTmp);
      //This is to mark whether it is the first time round the loop.
      bool firstTime = TRUE;
      //Now loop over the appropriate range of channels.
      for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
	//If it is not the first time round the loop we should add a comma before we write the next setting.
	if ( !firstTime )
	  data[element] += ",";
	if ( 0x01 == ((portStates>>port) & 0x01) ) {
	  data[element] += "Enable";
	} else if ( 0x00 == ((portStates>>port) & 0x01) ) {
	  data[element] += "Disable";
	} else {
	  fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbeConfigRegistersForGet(): " + registerNames[element] + " has unrecognised value `" + fwCcpc_convertByteToHex(ddcData[element]) + "'.", "2");
	  data[element] += "Fail";
	}
	//If it is the first time, it isn't any more.
	if ( firstTime )
	  firstTime = FALSE;
      }
    }
    else {
      //Can just return the read data to a hex string as the extra words have already been removed.
      //Trim off the mask in the case of the recipe data, otherwise the string will only be that big.
      data[element] = substr(fwCcpc_convertByteToHex(ddcData[element]), 0, 8);
    }
  }//for element

  //Done.
  return;
}//fwUkl1_parseGbeConfigRegistersForGet()

/*!
 * This function takes the data for the various GBE registers described by a string and converts it to data that can
 * used with fwCcpc_write() or fwHw_saveRecipe(). It allows data to be given in a more human readable format and converts
 * it to the byte array required for writting to the CCPC. It also returns the appropriate mask that must be used when
 * writing the data. If being used with fwHw_saveRecipe() then the mask will have to be appended to the data array.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  registerNames List of GBE register names that are to be written to. Returned with the appropriate name
 *           prefix by reference. The allowed values are:
 *   \li Duplex Default data format, see below.
 *   \li MAC Default data format, see below.
 *   \li Config Default data format, see below.
 *   \li TXFifoThreshold Decimal number.
 *   \li TXFifoLowWtrmrk Decimal number.
 *   \li TXFifoHighWtrmrk Decimal number.
 *   \li PHYControl Default data format, see below.
 *   \li PHYStatus Default data format, see below.
 *   \li TXStat Default data format, see below.
 *   \li RXStat Default data format, see below.
 * \param  sData Data to be written to the registers in the form of a hex string. The default value is an 8-character
 *           hex string, other values are accepted for specific registers as described above.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * Improperly formatted data will be removed from the list of write registers and will be indicated by an increased size of the
 * exceptionInfo array.
 */
void fwUkl1_parseGbePortConfigRegistersForSet(string ukl1Name, unsigned port, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".GBE.PORT" + port + ".";
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //Need a 32-bit mask, the same for all.
    masks[element]   = makeDynChar(0xff, 0xff, 0xff, 0xff);
    //Deal with the different formats.
    if ( ("TXFifoThreshold" == registerNames[element]) ||
	 ("TXFifoLowWtrmrk" == registerNames[element]) ||
	 ("TXFifoHighWtrmrk" == registerNames[element]) ) {
      //These will be decimal numbers.
      sData[element] = fwCcpc_convertDecToHex(sData[element]);
    }
    //Everything needs to be padded the string.
    fwUkl1_padString(sData[element], 4);
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];

    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
    
  }

  //Done.
  return;
}//fwUkl1_parseGbePortConfigRegistersForSet()

/*!
 * This takes the data read from the GBE port registers in the format as read from fwCcpc_read() or fwHw_getRecipe() and
 * returns the values as a decriptive string.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint elements to be
 *           read from, which corresponds to the appropriate hardware register. The accepted names are:
 *   \li Duplex Default data format, see below.
 *   \li MAC Default data format, see below.
 *   \li Config Default data format, see below.
 *   \li TXFifoThreshold Decimal number.
 *   \li TXFifoLowWtrmrk Decimal number.
 *   \li TXFifoHighWtrmrk Decimal number.
 *   \li PHYControl Default data format, see below.
 *   \li PHYStatus Default data format, see below.
 *   \li TXStat Default data format, see below.
 *   \li RXStat Default data format, see below.
 * \param  ddcData This contains the raw data that has been read back from the PVSS datapoint.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will either be a hex representation, zero padded to the nearest byte boundary or a text description
 *           of the settings. In the event that the read failed it will contain the text `Fail'.
 * \param  callStatusList This is the status list returned by the fwCcpc_read function and if the data to be parsed was generated
 *           by the use of this function then a callStatusList should be passed in order that errors can be checked for.
 *           It is possible, for example via monitoring or reading from a recipe, that the fwCcpc_read function is not used and in
 *           this case an empty array should be passed and it will be ignored. Any errors found in this array will be reported
 *           via the exceptionInfo.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_parseGbePortConfigRegistersForGet(string ukl1Name, unsigned port, const dyn_string& registerNames, dyn_dyn_char ddcData, dyn_string& data, const dyn_int& callStatusList, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that if we are given the full DPE name we can match against it.
  const string prefix = ukl1Name + ".GBE.PORT" + port + ".";
  //Determine the number of registers for which we are reading the data.
  const unsigned numRegs = dynlen(registerNames);
  //Determine whether we should be checking the status list. If it is empty don't look in it.
  const bool checkStatus = (0!=dynlen(callStatusList));
  //Used to check whether the function executed without a problem.
  int status = 0;

  //Loop over all the given registers and put the data into a single string for return.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //This series of `if' statements determines how to handle the possible raw status register data we may receive.
    //Check to see if we have actually been given any data.
    if ( 0 == dynlen(ddcData) ) {
      //Fill the return array with null strings.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbePortConfigRegistersForGet(): No data read back to parse.", "2");
    }
    //Check to see if the specific element has any data.
    else if ( 0 == dynlen(ddcData[element]) ) {
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseGbePortConfigRegistersForGet(): No data read for register " + registerNames[element] + ".", "2");
    }
    //Check for errors. As we are using AND the second case will only be evaluated provided checkStatus is not empty
    //and hence we should avoid out of bounds errors.
    //Check for both the case when the register names has the appropriate prefix and for the case where it doesn't.
    else if ( (checkStatus && (0 != callStatusList[element])) ) {
      //Construct the error message.
      string errMsg = "fwUkl1_parseGbePortConfigRegistersForGet(): " + registerNames[element] + " encountered ";
      if (callStatusList[element] > 0) {
	errMsg += "a problem writing to the CCPC server, return code " + callStatusList[element] + ".";
	status = 1;
      } else if (callStatusList[element] < 0) {
	errMsg += "a problem with PVSS.";
	status = -1;
      } else {
	errMsg += "an unknown error.";
	status = -1;
      }
      //Now raise the exception.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
      //Mark the error.
      data[element] = "Fail";
    }
    //Deal with the different formats.
    else if ( ( (prefix + "TXFifoThreshold" == registerNames[element]) ) || ("TXFifoThreshold" == registerNames[element]) ||
	      ( (prefix + "TXFifoLowWtrmrk") == registerNames[element] ) || ("TXFifoLowWtrmrk" == registerNames[element]) ||
	      ( (prefix + "TXFifoHighWtrmrk") == registerNames[element] ) || ("TXFifoHighWtrmrk" == registerNames[element]) ) {
      //These want to be decimal numbers. Only ever interested in the last 8 characters, from a recipe the mask would be there.
      const string sTmp = substr(fwCcpc_convertByteToHex(ddcData[element]), 0, 8);
      data[element] = fwCcpc_convertHexToDec(sTmp);
    }
    else {
      //Can just return the read data to a hex string.
      data[element] = substr(fwCcpc_convertByteToHex(ddcData[element]), 0, 8);
    }

  }//for element
  
  //Done.
  return;
}//fwUkl1_parseGbePortConfigRegistersForGet()

// =========
//    I2C
// =========

/*!
 * This function takes the data for the various TTCrx registers described by a string and converts it to data that can
 * used with fwCcpc_write() or fwHw_saveRecipe(). It allows data to be given in a more human readable format and converts
 * it to the byte array required for writting to the CCPC. It also returns the appropriate mask that must be used when
 * writing the data. If being used with fwHw_saveRecipe() then the mask will have to be appended to the data array.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames List of GBE register names that are to be written to. Returned with the appropriate name
 *           prefix by reference. The allowed values are:
 *   \li IACId Requires a two byte word.
 *   \li I2CId Default data format, see below.
 *   \li FineDelay1 Default data format, see below.
 *   \li FineDelay2 Default data format, see below.
 *   \li CoarseDelay Default data format, see below.
 *   \li Control Default data format, see below.
 *   \li Status Default data format, see below.
 *   \li Config1 Default data format, see below.
 *   \li Config2 Default data format, see below.
 *   \li Config3 Default data format, see below.
 *   \li SingleErrCnt Default data format, see below.
 *   \li DoubleErrCnt Default data format, see below.
 *   \li SEUErrCnt Requires a two byte word, read only.
 *   \li BunchCnt Requires a two byte word, read only.
 *   \li EventCnt Requires a three byte word, read only.
 * \param  sData Data to be written to the registers in the form of a hex string. The default value is an 2-character
 *           hex string, other values are accepted for specific registers as described above.
 * \param  ddcData sData converted into a dyn_dyn_char for use with the fwCcpc_write function. Returned by reference.
 * \param  masks Masks that are associated with the write data.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * Improperly formatted data will be removed from the list of write registers and will be indicated by an increased size of the
 * exceptionInfo array.
 */
void fwUkl1_parseTtcrxConfigRegistersForSet(string ukl1Name, dyn_string& registerNames, dyn_string sData, dyn_dyn_char& ddcData, dyn_dyn_char& masks, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".TTCRX.";

  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= dynlen(registerNames); ++element) {
    //The execptions are those that require more than one byte and hence a larger mask.
    if ( "IACId" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      fwUkl1_padString(sData[element], 2);
    }
    else if ( "SingleErrCnt" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      fwUkl1_padString(sData[element], 2);
    }
    else if ( "BunchCnt" == registerNames[element] ) {
      //This requires a two byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff);
      fwUkl1_padString(sData[element], 2);
    }
    else if ( "EventCnt" == registerNames[element] ) {
      //This requires a three byte mask and padding.
      masks[element]   = makeDynChar(0xff, 0xff, 0xff);
      fwUkl1_padString(sData[element], 3);
    }
    else {
      //The standard only require an 8-bit mask.
      masks[element]   = makeDynChar(0xff);
      //Pad the string, such that it is only 1-byte big.
      fwUkl1_padString(sData[element], 1);
    }
    
    //Put the data and register names in a format that the write can use.
    //Add the prefix to convert it to a datapoint name.
    registerNames[element] = prefix + registerNames[element];
    //Convert from a hex string to a byte array.
    ddcData[element] = fwCcpc_convertHexToByte( sData[element] );
  }
  
  //Done.
  return;
}//fwUkl1_parseTtcrxConfigRegistersForSet()

/*!
 * This takes the data read from the TTCrx registers in the format as read from fwCcpc_read() or fwHw_getRecipe() and
 * returns the values as a decriptive string.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint elements to be
 *           read from, which corresponds to the appropriate hardware register. The accepted names are:
 *   \li IACId Requires a two byte word.
 *   \li I2CId Default data format, see below.
 *   \li FineDelay1 Default data format, see below.
 *   \li FineDelay2 Default data format, see below.
 *   \li CoarseDelay Default data format, see below.
 *   \li Control Default data format, see below.
 *   \li Status Default data format, see below.
 *   \li Config1 Default data format, see below.
 *   \li Config2 Default data format, see below.
 *   \li Config3 Default data format, see below.
 *   \li SingleErrCnt Default data format, see below.
 *   \li DoubleErrCnt Default data format, see below.
 *   \li SEUErrCnt Requires a two byte word, read only.
 *   \li BunchCnt Requires a two byte word, read only.
 *   \li EventCnt Requires a three byte word, read only.
 * \param  ddcData This contains the raw data that has been read back from the PVSS datapoint.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will either be a hex representation, zero padded to the nearest byte boundary or a text description
 *           of the settings. In the event that the read failed it will contain the text `Fail'.
 * \param  callStatusList This is the status list returned by the fwCcpc_read function and if the data to be parsed was generated
 *           by the use of this function then a callStatusList should be passed in order that errors can be checked for.
 *           It is possible, for example via monitoring or reading from a recipe, that the fwCcpc_read function is not used and in
 *           this case an empty array should be passed and it will be ignored. Any errors found in this array will be reported
 *           via the exceptionInfo.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_parseTtcrxConfigRegistersForGet(string ukl1Name, const dyn_string& registerNames, dyn_dyn_char ddcData, dyn_string& data, const dyn_int& callStatusList, dyn_string& exceptionInfo) {
  //Define the string to be added to each of the register names such that if we are given the full DPE name we can match against it.
  const string prefix = ukl1Name + ".TTCRX.";
  //Determine the number of registers for which we are reading the data.
  const unsigned numRegs = dynlen(registerNames);
  //Determine whether we should be checking the status list. If it is empty don't look in it.
  const bool checkStatus = (0!=dynlen(callStatusList));
  //Used to check whether the function executed without a problem.
  int status = 0;

  //Loop over all the given registers and put the data into a single string for return.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //This series of `if' statements determines how to handle the possible raw status register data we may receive.
    //Check to see if we have actually been given any data.
    if ( 0 == dynlen(ddcData) ) {
      //Fill the return array with null strings.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseTtcrxConfigRegistersForGet(): No data read back to parse.", "2");
    }
    //Check to see if the specific element has any data.
    else if ( 0 == dynlen(ddcData[element]) ) {
      //Fill the return array with null strings.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseTtcrxConfigRegistersForGet(): No data read for register " + registerNames[element] + ".", "2");
    }
    //Check for errors. As we are using AND the second case will only be evaluated provided checkStatus is not empty
    //and hence we should avoid out of bounds errors.
    //Check for both the case when the register names has the appropriate prefix and for the case where it doesn't.
    else if ( (checkStatus && (0 != callStatusList[element])) ) {
      //Construct the error message.
      string errMsg = "fwUkl1_parseTtcrxConfigRegistersForGet(): " + registerNames[element] + " encountered ";
      if (callStatusList[element] > 0) {
	errMsg += "a problem writing to the CCPC server, return code " + callStatusList[element] + ".";
	status = 1;
      } else if (callStatusList[element] < 0) {
	errMsg += "a problem with PVSS.";
	status = -1;
      } else {
	errMsg += "an unknown error.";
	status = -1;
      }
      //Now raise the exception.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
      //Mark the error.
      data[element] = "Fail";
    }
    else if ( 0 == dynlen(ddcData[element]) ) {
      //No data, raise an exception.
      data[element] = "";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_parseTtcrxConfigRegistersForGet(): No data read to parse for " + registerNames[element] + ".", "2");
    }
    else {
      //Currently no exceptions, all either 1 or 2 bytes big. Just return their hex values.
      //These want to be decimal numbers. Only ever interested in the last 8 characters, from a recipe the mask would be there.
      data[element] = substr(fwCcpc_convertByteToHex(ddcData[element]), 0, 8);
    }

  }//for element.

  //Done.
  return;
}//fwUkl1_parseTtcrxConfigRegistersForGet()

// =============================
//   HARDWARE ACCESS FUNCTIONS
// =============================

// ===========
//    FPGAS
// ===========

/*!
 * Writes to a given set of UKL1 registers. Data should be in a form acceptable to the fwUkl1_parseEgressFpgaConfigRegistersForSet() as it will
 * first be passed through this function before being written to the hardware.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  writeRecipe If TRUE then the function will write the given register settings to the configurationDB recipe given by
 *           recipe name. FALSE it will write the register settings to the UKL1 board.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 *          This array is only required if writeRecipe=TRUE, otherwise it can be empty.
 * \param  registerNames dyn_string containing the names of the registers that need to be read. It need not contain the full
 *           path of the register i.e. the datapoint name and hardware chip (EgressFpga, GBE) must be neglected. Care must be
 *           taken that the registers requested exist.
 * \param  data Details of the settings for each register can be found in the fwUkl1_parseEgressFpgaConfigRegisterForWrite() documentation.
 * \param  verifyWrite If this is set to TRUE write will be verified to be successful. Not used if writeDb=TRUE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * If the recipe name does not exist it will be created. If the recipe type does not exist then the recipe may lose its type and
 * all data associated with it. No checks are performed on the recipe type at present.
 */
void fwUkl1_setEgressFpgaConfigRegisters(const string& ukl1Name, bool writeRecipe, const dyn_string& recipeNameAndType, dyn_string registerNames, const dyn_string& data, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setEgressFpgaConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Check that the register names and data arrays are the same size.
  //Save only the length of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setEgressFpgaConfigRegisters(): Number of registers to write does not match amount of data to be written.", "1");
    return;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  int exInfoSize = dynlen(exceptionInfo);
  fwUkl1_parseEgressFpgaConfigRegistersForSet(ukl1Name, registerNames, data, ddcData, masks, exceptionInfo);
  //Check that we were actually able to parse the data.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_setEgressFpgaConfigRegisters(): Failed to parse input data, not all the data was written.", "2");
  }//if exception info present.

  //The data and registers have been parsed
  if (writeRecipe) {
    //Our save recipe function requires the masks to have already been appended to the data.
    const int numElements = dynlen(ddcData);
    for (int element = 1; element <= numElements; ++element) {
      dynAppend(ddcData[element], masks[element]);
    }//for element
    //Now save the recipe
    _fwUkl1_saveRecipe(ukl1Name, recipeNameAndType, registerNames, ddcData, "FWHW_LBUS", exceptionInfo);
    //and check for errors.
    //We know exInfoSize hasn't changed since its size was first found and that it isn't -1 to get to this stage.
    if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == dynlen(exceptionInfo)) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setEgressFpgaConfigRegisters(): Failed to save recipe for " + ukl1Name + ".", "1");
  }//if(writeRecipe)
  else {
    //The data and registers have been parsed, so let us write to the hardware.
    _fwUkl1_write(registerNames, ddcData, masks, verifyWrite, exceptionInfo);
    if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == dynlen(exceptionInfo)) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setEgressFpgaConfigRegisters(): Failed to write to registers for " + ukl1Name + ".", "1");
  }//else(writeRecipe)

  //Done.
  return;
}//fwUkl1_setEgressFpgaConfigRegisters()

/*!
 * This reads from a given set of registers on the Egress FPGA.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  readRecipe If TRUE then the function will read the settings from the given recipe name and return the settings
 *           contained within. FALSE and the returned values will come directly from the hardware registers.
 * \param  recipeName Name of the recipe to save the configuration information to if writeDb=TRUE, not used if writeDb=FALSE.
 *           The recipe must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE is type of run that the recipe is to be loaded
 *           e.g PHYSICS, CALIBRATION, can take any value, but should be descriptive of the type of run the recipe settings represent;
 *           FsmAction is the action in the FSM when these settings should be loaded e.g. Configure if the recipe is to be used to moved to
 *           the READY state, typically only the first letter is upper case.
 * \param  registerNames dyn_string containing the names of the registers that need to be read. It need not contain the full
 *           path of the register i.e. the datapoint name and hardware chip (EgressFpga, GBE) must be neglected. If the requested
 *           register does not exist in the recipe an empty string will be returned in the data, but no error. If it does not exist
 *           for hardware reads then it will fail and exceptionInfo will contain a PVSS error for that register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register. The details of this string
 *           can be found in fwUkl1_parseEgressFpgaConfigRegistersForGetEgressFpga() documentation. Will return `Fail' in the event the read is not successful.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_getEgressFpgaConfigRegisters(const string& ukl1Name, bool readRecipe, const string& recipeName, dyn_string registerNames, dyn_string& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getEgressFpgaConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".EgressFpga.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data from the reads.
  dyn_dyn_char ddcData;
  //Holds the status of each hardware read. Required by the parsing function,
  //but will be empty and hence ignored in the case of the recipe access.
  dyn_int callStatusList;
  dynClear(callStatusList);
  if (readRecipe) {
    //When we get the recipe data it will give us all registers in the recipe and also data.
    //This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    //Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if ( -1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData) ) {
      //Note that the length of the registerNames array is calculated for each loop.
      //This is important if we remove elements from the array, the total length does need to be updated.
      for (int element = 1; element <= dynlen(registerNames); ++element) {
	//This take the specific register name and returns the index of its occurance in the full list.
	//Relies on the fact that the list is unique.
	//It is this index that we can use to over write the appropriate data element.
	int index = dynContains(tmpRegisterNames, registerNames[element]);
        //Now we know where the appropriate recipe data is copy it into the ddcData array.
        //It is possible that this data does not occur in this recipe type, hence we must protect against this.
        if ( 0 != index ) {
	  ddcData[element] = tmpDdcData[index];
        }//if(0!=index)
        else {
          //If the element doesn't exist then pad the ddcData array with an empty array.
          //This allows the parse data function to pick up on it and put a null value
          //into the into the return data.
          ddcData[element] = makeDynChar();
        }//else(0!=index)
      }
    }//if(-1!=fwhw_getRecipe())
    else {
      //It is most likely that the recipe just didn't exist.
      //Populate the ddcData with nothing such that it can return nothing, but it will contain the appropriate number of elements of nothing.
      for (int element = 1; element <= dynlen(registerNames); ++element) {
         ddcData[element] = makeDynChar();
      }//for element
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getEgressFpgaConfigRegisters(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "1");
    }//else(-1!=fwHw_getRecipe())
    
  }//if(readRecipe)
  else {
    //The read status will need to be checked to ensure that the function succeeded and there is a valid call list.
    fwCcpc_read(registerNames, ddcData, callStatusList);
  }//else(readRecipe)

  //Now parse the read data to see if it is any good. Let the parsing function sort out any errors.
  fwUkl1_parseEgressFpgaConfigRegistersForGet(ukl1Name, registerNames, ddcData, data, callStatusList, exceptionInfo);
	
  //Done.
  return;
}//fwUkl1_getEgressFpgaConfigRegisters()

/*!
 * This writes the Ingress FPGA channel settings. All registers must be set at the same time and the function
 * will return without writing to the hardware if not all the registers are set.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on the Ingress FPGA that is to be written to. Range: 0 to 8.
 * \param  inFpga Number of the Ingress FPGA that is to be written. Range:0 to 3.
 * \param  writeRecipe If TRUE then the function will write the given register settings to the configurationDB recipe given by
 *           recipe name. FALSE it will write the register settings to the UKL1 board.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 *          This array is only required if writeRecipe=TRUE, otherwise it can be empty.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The accepted values are described in
 *           fwUkl1_parseIngressFpgaChannelConfigForSet();
 * \param  data By default a 16-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described in fwUkl1_parseIngressFpgaChannelWriteConfiguration() documentation.
 * \param  verifyWrite If this is set to TRUE write will be verified to be successful. Not used if writeDb=TRUE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * If the recipe name does not exist it will be created. If the recipe type does not exist then the recipe may lose its type and
 * all data associated with it. No checks are performed on the recipe type at present.
 */
void fwUkl1_setIngressFpgaChannelConfigRegisters(string ukl1Name, unsigned channel, unsigned inFpga, bool writeRecipe, const dyn_string& recipeNameAndType, dyn_string registerNames, const dyn_string& data, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Makes a note of the exception info size.
  int exInfoSize = dynlen(exceptionInfo);
  //Check that there are enough names in the register array to write to all the settings.
  //Save the number of registers from the excepted number of registers that are required.
  const unsigned numRegs = 7;
  if ( numRegs != dynlen(registerNames) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Number of registers is not sufficient to write all the channel data. Require " + numRegs + ", given " + dynlen(registerNames) + ".", "1");
    return;
  }//if not enough registers
  //Check that the register names and data arrays are the same size.
  if ( numRegs != dynlen(data) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Number of registers to write does not match amount of data to be written.", "1");
    return;
  }//if not enough data for all registers

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_char dcData;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  const int exInfoSize = dynlen(exceptionInfo);
  fwUkl1_parseIngressFpgaChannelConfigRegisterForSet(ukl1Name, channel, inFpga, registerNames, data, dcData, exceptionInfo);
  //Check that we were able to parse the data.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Failed to parse input data, no data written.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info present.
  else {
    if (writeRecipe) {
      //We will save only the FIFO data in the recipe and will require that the the application of the recipe
      //handles the necessary settings in the mailbox config register.
      //It will have no masks.
      dyn_string mailboxRegs = makeDynString(ukl1Name + ".IngressFpga" + inFpga + ".Channel" + channel + ".ConfigurationFifo");
      dyn_dyn_char mailboxData;
      mailboxData[1] = dcData;

      //Now save the recipe.
      _fwUkl1_saveRecipe(ukl1Name, recipeNameAndType, mailboxRegs, mailboxData, "FWHW_LBUS", exceptionInfo);
      //and check for errors.
      if ( (exInfoSize < dynlen(exceptionInfo)) || (dynlen(exceptionInfo) == -1) ) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_writeIngressFpgaChannel(): Failed to save recipe for " + ukl1Name + ".", "1");
        exInfoSize = dynlen(exceptionInfo);
      }//if _fwUkl1_saveRecipe() failed.
    }//if(writeRecipe)
    else {
      //Contains all the necessary stages for writing the configuration data.
      _fwUkl1_setIngressFpgaChannelConfigRegisters(ukl1Name, channel, inFpga, dcData, verifyWrite, exceptionInfo);
      if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
      }//if setting of data failed.
    }//else(writeRecipe)
  }//else parsing succeeded.
  //Done.
  return;
}//fwUkl1_setIngressFpgaChannelConfigRegisters()

/*!
 * This reads the configuration settings for an Ingress FPGA configuration register.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on the Ingress FPGA that is to be written to. Range: 0 to 8.
 * \param  inFpga Number of the Ingress FPGA that is to be written. Range:0 to 3.
 * \param  readRecipe If TRUE then the function will read the settings from the given recipe name and return the settings
 *           contained within. FALSE and the returned values will come directly from the hardware registers.
 * \param  recipeName Name of the recipe to save the configuration information to if writeDb=TRUE, not used if writeDb=FALSE.
 *           The recipe must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE is type of run that the recipe is to be loaded
 *           e.g PHYSICS, CALIBRATION, can take any value, but should be descriptive of the type of run the recipe settings represent;
 *           FsmAction is the action in the FSM when these settings should be loaded e.g. Configure if the recipe is to be used to moved to
 *           the READY state, typically only the first letter is upper case.
 * \param  registerNames dyn_string containing the names of the registers that need to be read. It need not contain the full
 *           path of the register i.e. the datapoint name and hardware chip (EgressFpga, GBE) must be neglected. If the requested
 *           register does not exist in the recipe an empty string will be returned in the data, but no error. If it does not exist
 *           for hardware reads then it will fail and exceptionInfo will contain a PVSS error for that register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register. The details of this string
 *           can be found in fwUkl1_parseIngressFpgaChannelConfigRegistersForGet() documentation. Will return `Fail' in the event the read is not successful.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_getIngressFpgaChannelConfigRegisters(const string& ukl1Name, unsigned channel, unsigned fpga, bool readRecipe, const string& recipeName, const dyn_string& registerNames, dyn_string& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaChannelConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Keep track of the size of the exceptionInfo array.
  int exInfoSize = dynlen(exceptionInfo);
  //Make sure we have an empty data array.
  dynClear(data);
  
  //Do the reads...
  //Holds the data from the read.
  dyn_char dcData;
  if (readRecipe) {
    //If we are reading from the recipe then we can get the channel data from the same register that
    //it is written to.
    const string actualRegName = ukl1Name + ".IngressFpga" + fpga + ".Channel" + channel + ".ConfigurationFifo";
    //When we get the recipe data it will give us all registers in the recipe and also data.
    //This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    //Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if ( -1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData) ) {
      //This take the register name and returns the index of its occurance in the full list.
      //Relies on the fact that the list is unique.
      //It is this index that we can use to over write the appropriate data element.
      int index = dynContains(tmpRegisterNames, actualRegName);
      //Now we know where the appropriate recipe data is copy it into the ddcData array.
      //It is possible that this data does not occur in this recipe type, hence we must protect against this.
      if ( 0 != index ) {
        dcData = tmpDdcData[index];
      }//if(0!=index)
      else {
        //If the element doesn't exist then pad the ddcData array with an empty array.
        //This allows the parse data function to pick up on it and put a null value
        //into the into the return data.
        dcData = makeDynChar();
      }//else(0!=index)
    }//if(-1!=fwhw_getRecipe())
    else {
      //It is most likely that the recipe just didn't exist.
      //Populate the ddcData with nothing such that it can return nothing, but it will contain the appropriate number of elements of nothing.
      dcData = makeDynChar();
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaChannelConfigRegisters(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "1");
      exInfoSize = dynlen(exceptionInfo);
    }//else(-1!=fwHw_getRecipe())
  }//if(readRecipe)
  else {
    //Call get status for the appropriate Ingress FPGA.
    dyn_dyn_char tmpDdcData;
    fwUkl1_getIngressFpgaStatusRegister(ukl1Name, makeDynUInt(fpga), tmpDdcData, exceptionInfo);
    if ( (dynlen(exceptionInfo) == exInfoSize) && (-1 != exInfoSize) ) {
      //Now we have the data we need extract four bytes from the status word whose location depends on channel.
      for (int byte = 1; byte <= 4; ++byte) {
        //The first two bytes are junk that the parsing function consider to be a channel number in the recipe,
        //so must be here to avoid seg faults. The last two bytes are the real status data.
        //Offset 8 to get past the Ingress status, each channel contains 16 bytes, then offset into the channel
        //status by 4, before offseting by byte to get the relevant bytes.
        dcData[byte] = tmpDdcData[1][8+16*channel+4+byte];
      }//for byte
    }//if no error
    else {
      //Mark the error dcData will remain empty and the parsing function will handle it.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaChannelConfigRegisters(): Failed to read the status information for Ingress FPGA " + fpga + ". Cannot determine the settings for channel " + channel + " on that FPGA.", "1");
      exInfoSize = dynlen(exceptionInfo);
    }//else error
    
  }//else(readRecipe)

  //Now parse the read data to see if it is any good. Let the parsing function sort out any errors.
  fwUkl1_parseIngressFpgaChannelConfigRegisterForGet(ukl1Name, registerNames, dcData, data, exceptionInfo);
	
  //Done.
  return;
}//fwUkl1_getIngressFpgaChannelConfigRegisters()

/*!
 * This reads the Ingress FPGA status FIFO from the Ingress FPGAs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  inFpga Number of the Ingress FPGAs that are to be read, each element should correspond contain
 *           the number of the Ingress FPGA to read. Range:0 to 3.
 * \param  data dyn_dyn_char returned by reference. Each element contains a dyn_char which is the read status
 *           data for the corresponding inFpga element.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_getIngressFpgaStatusRegister(string ukl1Name, dyn_uint inFpga, dyn_dyn_char& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Get the number of Ingress FPGAs we want to read.
  const int numFpgas = dynlen(inFpga);
  if ( -1 == numFpgas ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): PVSS failed to determine the number of Ingress FPGAs to read from (failure of dynlen on inFgpa array). Cannot continue with status read.", "1");
    return;
  }//if(-1==numFpgas)
  
  //Make a note of how large the exception array currently is, we can determine if function calls change its size this way.
  int exInfoSize = dynlen(exceptionInfo);
  
  //Create an array to hold the data. Have to do this despite the fact that it is actually the same type
  //as the input argument data. It literally crashes the PVSS 3.6SP1 ui executable if we try and use the
  //argument passed by reference in the function name!
  dyn_dyn_char ddcData;

  //In order to read the status information we must first assert and then deassert the read pointer
  //reset to ensure that we always start from the beginning of the FIFO. Always reset all the pointers.
  fwUkl1_resetIngressFpgaStatusBufPointers(ukl1Name, exceptionInfo);
  if ( (exInfoSize == dynlen(exceptionInfo)) || (-1 != exInfoSize) ) {
    //Holds the list of status registers we should read from.
    dyn_string registerNames;
    //Determine the Ingress FPGAs we should read from.
    for (unsigned fpga = 1; fpga <= numFpgas; ++fpga) {
      registerNames[fpga] = ukl1Name + ".IngressFpga" + inFpga[fpga] + ".IngressFpgaStatus.StatusFifo";
    }//for
  
    //Do the reads...
    //Holds the status of each hardware read.
    dyn_int callStatusList;
    dynClear(callStatusList);
    //Check the overall status of the read.
    if ( fwCcpc_read(registerNames, ddcData, callStatusList) ) {
      //Check to see if we have actually been given any data.
      if ( numFpgas != dynlen(ddcData) ) {
        //Mark the error.
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): Requested to read " + numFpgas + " registers, only received data for " + dynlen(ddcData) + " (-1 indicates a PVSS error).", "2");
      }//if(0==dynlen(ddcData)
      //Check to see if there is any information about the individual writes.
      else if ( numFpgas != dynlen(callStatusList) ) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): Requested to read " + numFpgas + ", status information returned for only " + dynlen(callStatusList) + " (-1 indicates a PVSS error).", "2");
      }//elseif(0>=dynlen(callStatusList))
      //Now we have some data and also some information about the specific reads, so check them.
      else {
        //Loop over all the registers.
        for (int element = 1; element <= numFpgas; ++element) {
          //Check the status for the elements.
          if ( 0 != callStatusList[element] ) {
            //Construct the error message.
            string errMsg = "fwUkl1_getIngressFpgaStatusRegister(): " + registerNames[element] + " encountered ";
            if (callStatusList[element] > 0) {
              errMsg += "a problem writing to the CCPC server, return code: " + callStatusList[element] + ".";
            }//ccserv errpr
            else if (callStatusList[element] < 0) {
              errMsg += "a problem with PVSS.";
            }//PVSS error
            else {
              errMsg += "an unknown error.";
            }//Should only be possible if 0 and we don't enter this statement if it is 0!
            //Now raise the exception.
            fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
          }//if(0!=callStatusList[element])
          //Check to see if the specific element has any data.
          else if ( 0 == dynlen(ddcData[element]) ) {
            fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_ingressFpgaStautsRead(): No data read for register `" + registerNames[element] + "'.", "2");
          }//elseif(0>=dynlen(ddcData[element])
          //else everything is OK so don't do anything!
        }//for reg
      }//else check the retrieved data is valid.
    }//if(fwCcpc_read())
    else {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): Read function failed, no data returned.", "1");
    }//else(fwCcpc_read())
  }//if exception information not generated.
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getIngressFpgaStatusRegister(): Failed to reset status buffer pointers, cannot read sensible data from status registers. Aborting read.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//else exception information generated.
  
  //Now copy ddcData to data, which is returned by reference. This works but using it as a function argument
  //doesn't. Something far too complex is going on here.
  data = ddcData;

  //Done.
  return;
}//fwUkl1_getIngressFpgaStatusRegister()

/*!
 * Asserts and then deasserts a reset of the status buffer pointers. This must be done in order
 * to read any information from the Ingress status FIFOs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * This will reset all the status buffer pointers simulataneously.
 */
void fwUkl1_resetIngressFpgaStatusBufPointers(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetIngressFpgaStatusBufPointers(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  //The reset must first be asserted and then deasserted for the pointers to reset.
  //Assert the reset of FIFO pointer.
  const string fifoResetReg = ukl1Name + ".Resets.IngressFpgaStatusBufferPointers";
  //data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("0000000f");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetIngressFpgaStatusBufPointers(): Failed to assert status buffer pointer reset, status information will likely be invalid.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info

  //Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  //dataAndMask now just contains the mask as we want to write something different for the data.
  //The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetIngressFpgaStatusBufPointers(): Failed to de-assert status buffer pointer reset, status information will likely be invalid.","1");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info
  //Done.
  return;
}//fwUkl1_resetIngressFpgaStatusBufPointers()

/*!
 * Perform a partial reset of the UKL1 boards.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * Resets everything except the configuration registers e.g. logic, readout buffers, status registers etc..
 */
void fwUkl1_resetUkl1Partial(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetUkl1Partial(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  //The reset must first be asserted and then deasserted for the pointers to reset.
  //Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.PartialL1";
  //data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000020");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetPartialUkl1(): Failed to assert reset.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info

  //Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  //dataAndMask now just contains the mask as we want to write something different for the data.
  //The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetPartialUkl1(): Failed to de-assert status reset.","1");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info
  //Done.
  return;
}//fwUkl1_resetPartialUkl1()

/*!
 * Asserts and then deasserts a reset of the SPI3 RX read pointer.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * This is for a special test feature and will almost never need to be used.
 */
void fwUkl1_resetSPI3RXReadPointer(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetSPI3RXReadPointer(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  //The reset must first be asserted and then deasserted for the pointers to reset.
  //Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.SPI3RxReadPointer";
  //data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000010");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetSPI3RXReadPointer(): Failed to assert reset.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info

  //Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  //dataAndMask now just contains the mask as we want to write something different for the data.
  //The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetSPI3RXReadPointer(): Failed to de-assert status reset.","1");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info
  //Done.
  return;
}//fwUkl1_resetSPI3RXReadPointer()

/*!
 * Perform a reset of the DCMs.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * Resets the Egress clock manager and generally will never need to be used except in special test cases.
 */
void fwUkl1_resetDCMs(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_resetDCMs(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Used to track changes in the exception info array.
  int exInfoSize = dynlen(exceptionInfo);
  //The reset must first be asserted and then deasserted for the pointers to reset.
  //Assert the reset.
  const string fifoResetReg = ukl1Name + ".Resets.DCMs";
  //data and mask are the same in this case.
  dyn_dyn_char dataAndMask;
  dataAndMask[1] = fwCcpc_convertHexToByte("00000030");
  _fwUkl1_write(fifoResetReg, dataAndMask, dataAndMask, TRUE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetDCMs(): Failed to assert reset.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info

  //Deassert the reset of the FIFO pointer.
  dyn_dyn_char clearData;
  clearData[1] = fwCcpc_convertHexToByte("00000000");
  //dataAndMask now just contains the mask as we want to write something different for the data.
  //The mask is the same as before however.
  _fwUkl1_write(fifoResetReg, clearData, dataAndMask, FALSE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (exInfoSize == -1) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_resetDCMs(): Failed to de-assert status reset.","1");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception info
  //Done.
  return;
}//fwUkl1_resetDCMs()

// ==============
//    INTERNAL
// ==============

/*!
 * It is this code that takes the raw dyn_char data that is to be written to the FIFO and writes it to the
 * FIFO, performing all the necessary configuration of the FIFO to prepare it to be written to and also
 * triggers the transmission of the FIFO data once done.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  channel Input channel number on the Ingress FPGA that is to be written to. Range: 0 to 8.
 * \param  inFpga Number of the Ingress FPGA that is to be written. Range:0 to 3.
 * \param  dcData This array contains the data that is to be written to the status register in the form of a byte
 *           array.
 * \param  verifyWrite This will ensure that the configuration procedures of the write are successful. It cannot
 *           check the data written to the FIFO itself however.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_setIngressFpgaChannelConfigRegisters(const string& ukl1Name, unsigned channel, unsigned inFpga, dyn_char& dcData, bool verifyWrite, dyn_string& exceptionInfo) {
  //Keep track of the amount of exception information generated.
  int exInfoSize = dynlen(exceptionInfo);
  //Prepare the FIFO for writing.
  _fwUkl1_setIngressChannelConfigFifoReadyForData(ukl1Name, inFpga, verifyWrite, exceptionInfo);
  //Check that the register was prepared correctly.
  if ( (exInfoSize == dynlen(exceptionInfo)) && (-1 != exInfoSize) ) {
    //Write the data.
    dyn_string mailboxRegs = makeDynString(ukl1Name + ".IngressFpga" + inFpga + ".Channel" + channel + ".ConfigurationFifo");
    dyn_dyn_char mailboxData;
    dyn_dyn_char mailboxMasks;
    // - Commented until FIFO writing can be figured out.
    //This is the code for use with the real FIFO register when we get it working.
    mailboxData[1] = dcData;
    //mailboxMasks[1] = "ffffffff";
    dynClear(mailboxMasks);
    //Can't verify the FIFO write.
    _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, FALSE, exceptionInfo);
    if ( (exInfoSize == dynlen(exceptionInfo)) && (-1 != exInfoSize) ) {
      //Now we have written the data we can trigger the sending of it.
      _fwUkl1_sendIngressChannelConfigFifo(ukl1Name, verifyWrite, exceptionInfo);
      if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Trigger the transmission of the configuration data for Ingress FPGA " + inFpga + ", channel " + channel + ". Data not written.", "1");
        exInfoSize = dynlen(exceptionInfo);
      }//if _fwUkl1_sentIngressChannelConfigFifo() failed.
    }//if writing of FIFO data successful.
    else {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Failed to write configuration data for Ingress FPGA " + inFpga + ", channel " + channel + ". Configuration not written.", "1");
      exInfoSize = dynlen(exceptionInfo);
   }//else writing of FIFO data unsuccessful.
  }//if _fwUkl1_setIngressChannelConfigFifoReadForData() successful.
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setIngressFpgaChannelConfigRegisters(): Failed to prepare the configuration FIFO for Ingress FPGA " + inFpga + ", channel " + channel + " for writting. Data not written.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//else _fwUkl1_setIngressChannelConfigFifoReadForData() not successful.
  //Done.
  return;
}//_fwUkl1_setIngressFpgaChannelConfigRegisters()
      
/*!
 * Prepares the configuration FIFO for writting by setting the appropriate Ingerss FPGA number and asserting and
 * deasserting the pointer resets. It ensures that the transmit trigger has been deasserted.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  inFpga Number of the Ingress FPGAs that are to be read, each element should correspond contain
 *           the number of the Ingress FPGA to read. Range:0 to 3.
 * \param  verifyWrite If this is set to TRUE write will be verified to be successful.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_setIngressChannelConfigFifoReadyForData(const string& ukl1Name, int inFpga, bool verifyWrite, dyn_string& exceptionInfo) {
  //Holds the various bits of data that will need to be written.
  //This will only ever write to one register.
  dyn_string mailboxRegs = makeDynString(ukl1Name + ".EgressFpga.IngressConfigMailboxControl");
  dyn_dyn_char mailboxData;
  dyn_dyn_char mailboxMasks;
    
  //Deassert both read and write pointer resets and the transmit trigger (zeros to bits [6..4]),
  //while writing the Ingress FPGA to be configured (bits [9..8]).
  string tmpData = fwCcpc_convertDecToHex( inFpga << 8 );
  fwUkl1_padString(tmpData, 4);
  mailboxData[1] = fwCcpc_convertHexToByte(tmpData);
  //Want to clear bits [6..4] and set bit [9..8].
  mailboxMasks[1] = fwCcpc_convertHexToByte("00000370");
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  //Assert the read and write pointers' reset.
  mailboxData[1] = fwCcpc_convertHexToByte("00000030");
  //The mask is the same as the data to be written in this case.
  mailboxMasks[1] = mailboxData[1];
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  //Deassert the read and write pointers' reset.
  mailboxData[1] = fwCcpc_convertHexToByte("00000000");
  //The mask is the same as before so we don't need to update it.
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
    
  //Done.
  return;
}//_fwUkl1_setIngressChannelConfigFifoReadyForData()

/*!
 * Triggers the tranmission of the Ingress configuration FIFO data.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  verifyWrite If this is set to TRUE write will be verified to be successful.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_sendIngressChannelConfigFifo(const string& ukl1Name, bool verifyWrite, dyn_string& exceptionInfo) {
  //Trigger the FIFO tranmission to Ingress FPGA from Egress FPGA.
  dyn_string mailboxRegs;
  dyn_dyn_char mailboxData;
  dyn_dyn_char mailboxMasks;
  mailboxRegs[1]  = ukl1Name + ".EgressFpga.IngressConfigMailboxControl";
  mailboxData[1]  = fwCcpc_convertHexToByte("00000040");
  mailboxMasks[1] = fwCcpc_convertHexToByte("00000040");
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);

  //Deassert the FIFO tranmission trigger.
  mailboxData[1] = fwCcpc_convertHexToByte("00000000");
  //The mask is the same as before.
  _fwUkl1_write(mailboxRegs, mailboxData, mailboxMasks, verifyWrite, exceptionInfo);
  
  //Done.
  return;
}//_fwUkl1_sendIngressChannelConfigFifo()


// =========
//    GBE
// =========

/*!
 * This writes to a given set of registers on the giga-bit ethernet (GBE) card. It can be used to write to either the
 * registers on the UKL1 board or to a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  writeRecipe If TRUE then the function will write the given register settings to the configurationDB recipe given by
 *           recipe name. FALSE it will write the register settings to the UKL1 board.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 *          This array is only required if writeRecipe=TRUE, otherwise it can be empty.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The accepted values are described in
 *           fwUkl1_parseGbeWriteRegisters();
 * \param  data By default a 32-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described in fwUkl1_parseGbeWriteRegisters() documentation.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_setGbeConfigRegisters(string ukl1Name, bool writeRecipe, const dyn_string& recipeNameAndType, dyn_string registerNames, const dyn_string& data, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbeConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbeConfigRegisters(): Number of registers to write does not match amount of data to be written.", "1");
    return;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  const int exInfoSize = dynlen(exceptionInfo);
  fwUkl1_parseGbeConfigRegistersForSet(ukl1Name, registerNames, data, ddcData, masks, exceptionInfo);
  //Check that we were able to parse the data.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_setGbeConfigRegisters(): Failed to parse input data, not all data was written.", "1");
  }//if exception info present.
  
  if (writeRecipe) {
    //Our save recipe function requires the masks to have already been appended to the data.
    const int numElements = dynlen(ddcData);
    for (int element = 1; element <= numElements; ++element) {
      dynAppend(ddcData[element], masks[element]);
    }//for element
    //Now save the recipe
    _fwUkl1_saveRecipe(ukl1Name, recipeNameAndType, registerNames, ddcData, "FWHW_GBE", exceptionInfo);
    //and check for errors.
    //We know exInfoSize hasn't changed since its size was first found and that it isn't -1 to get to this stage.
    if ( (exInfoSize < dynlen(exceptionInfo)) || (dynlen(exceptionInfo) == -1) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbeConfigRegisters(): Failed to save recipe for " + ukl1Name + ".", "1");
  }//if(writeRecipe)
  else {
    //The data and registers have been parsed, so let us write to the hardware.
    _fwUkl1_write(registerNames, ddcData, masks, verifyWrite, exceptionInfo);
    if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == dynlen(exceptionInfo)) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbeConfigRegisters(): Failed to write to registers for " + ukl1Name + ".", "1");
  }//else(writeRecipe)

  //Done.
  return;
}//fwUkl1_setGbeConfigRegisters()

/*!
 * This reads from a given set of general GBE registers. The reads can either be from the hardware or they can be
 * done from a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  readRecipe If TRUE then the function will read the settings from the given recipe name and return the settings
 *           contained within. FALSE and the returned values will come directly from the hardware registers.
 * \param  recipeName Name of the recipe to save the configuration information to if writeDb=TRUE, not used if writeDb=FALSE.
 *           The recipe must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE is type of run that the recipe is to be loaded
 *           e.g PHYSICS, CALIBRATION, can take any value, but should be descriptive of the type of run the recipe settings represent;
 *           FsmAction is the action in the FSM when these settings should be loaded e.g. Configure if the recipe is to be used to moved to
 *           the READY state, typically only the first letter is upper case.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 32-bit word.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_getGbeConfigRegisters(string ukl1Name, bool readRecipe, const string& recipeName, dyn_string registerNames, dyn_string& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getGbeConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".GBE.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data from the reads.
  dyn_dyn_char ddcData;
  //Holds the status of each hardware read. Required by the parsing function,
  //but will be empty and hence ignored in the case of the recipe access.
  dyn_int callStatusList;
  dynClear(callStatusList);

  if (readRecipe) {
    //When we get the recipe data it will give us all registers in the recipe and also data.
    //This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    //Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if ( -1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData) ) {
      for (int element = 1; element <= numRegs; ++element) {
	//This take the specific register name and returns the index of its occurance in the full list.
	//Relies on the fact that the list is unique.
	//It is this index that we can use to over write the appropriate data element.
	int index = dynContains(tmpRegisterNames, registerNames[element]);
        //Now we know where the appropriate recipe data is copy it into the ddcData array.
        //It is possible that this data does not occur in this recipe type, hence we must protect against this.
        if ( 0 != index )
	  ddcData[element] = tmpDdcData[index];
        //else
          //fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_saveRecipe(): " + registerNames[element] + " was not found in recipe type. This data has not been saved as it is not stored in this recipe type.", "1");
        //Don't bother warning about this it generates an unnecessary error, just let it happen silently.
      }
    }//if(fwhw_getRecipe())
    else
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getGbeConfigRegisters(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "1");
    
  }//if(readRecipe)
  else {
    fwCcpc_read(registerNames, ddcData, callStatusList);
  }

  //Parse the data we just read and let it deal with any errors that occurred.
  fwUkl1_parseGbeConfigRegistersForGet(ukl1Name, registerNames, ddcData, data, callStatusList, exceptionInfo);

  //Done.
  return;
}//fwUkl1_getGbeConfigRegisters()

/*!
 * This writes to a given set of registers on a GBE port. It can be used to write to either the
 * registers on the UKL1 board or to a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  writeRecipe If TRUE then the function will write the given register settings to the configurationDB recipe given by
 *           recipe name. FALSE it will write the register settings to the UKL1 board.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 *          This array is only required if writeRecipe=TRUE, otherwise it can be empty.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The accepted values are described in
 *           fwUkl1_parseGbeWriteRegisters();
 * \param  data By default a 32-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_setGbePortConfigRegisters(string ukl1Name, unsigned port, bool writeRecipe, const dyn_string& recipeNameAndType, dyn_string registerNames, dyn_string data, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbePortConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbePortConfigRegisters(): Number of registers to write does not match amount of data to be written.", "1");
    return;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  const int exInfoSize = dynlen(exceptionInfo);
  fwUkl1_parseGbePortConfigRegistersForSet(ukl1Name, port, registerNames, data, ddcData, masks, exceptionInfo);
  //Check that we were able to parse the data.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_setGbePortConfigRegisters(): Failed to parse input data, not all data was written.", "2");
  }//if exception info present.
  
  if (writeRecipe) {
    //Our save recipe function requires the masks to have already been appended to the data.
    const int numElements = dynlen(ddcData);
    for (int element = 1; element <= numElements; ++element) {
      dynAppend(ddcData[element], masks[element]);
    }//for element
    //Now save the recipe
    _fwUkl1_saveRecipe(ukl1Name, recipeNameAndType, registerNames, ddcData, "FWHW_GBE", exceptionInfo);
    //and check for errors.
    //We know exInfoSize hasn't changed since its size was first found and that it isn't -1 to get to this stage.
    if ( (exInfoSize < dynlen(exceptionInfo)) || (dynlen(exceptionInfo) == -1) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbePortConfigRegisters(): Failed to save recipe for " + ukl1Name + ".", "1");
  }//if(writeRecipe)
  else {
    //The data and registers have been parsed, so let us write to the hardware.
    _fwUkl1_write(registerNames, ddcData, masks, verifyWrite, exceptionInfo);
    if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == dynlen(exceptionInfo)) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setGbePortConfigRegisters(): Failed to write to registers for " + ukl1Name + ".", "1");
  }//else(writeRecipe)

  //Done.
  return;
}//fwUkl1_setGbePortConfigRegisters()

/*!
 * This reads from a given set of GBE port registers. The reads can either be from the hardware or they can be
 * done from a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  port Output port on the GBE card that is to be set.
 * \param  readRecipe If TRUE then the function will read the settings from the given recipe name and return the settings
 *           contained within. FALSE and the returned values will come directly from the hardware registers.
 * \param  recipeName Name of the recipe to save the configuration information to if writeDb=TRUE, not used if writeDb=FALSE.
 *           The recipe must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE is type of run that the recipe is to be loaded
 *           e.g PHYSICS, CALIBRATION, can take any value, but should be descriptive of the type of run the recipe settings represent;
 *           FsmAction is the action in the FSM when these settings should be loaded e.g. Configure if the recipe is to be used to moved to
 *           the READY state, typically only the first letter is upper case.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 32-bit word.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_getGbePortConfigRegisters(string ukl1Name, unsigned port, bool readRecipe, const string& recipeName, dyn_string registerNames, dyn_string& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getGbePortConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".GBE.PORT" + port + ".";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data from the reads.
  dyn_dyn_char ddcData;
  //Holds the status of each hardware read. Required by the parsing function,
  //but will be empty and hence ignored in the case of the recipe access.
  dyn_int callStatusList;
  dynClear(callStatusList);

  if (readRecipe) {
    //When we get the recipe data it will give us all registers in the recipe and also data.
    //This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    //Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if ( -1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData) ) {
      for (int element = 1; element <= numRegs; ++element) {
	//This take the specific register name and returns the index of its occurance in the full list.
	//Relies on the fact that the list is unique.
	//It is this index that we can use to over write the appropriate data element.
	int index = dynContains(tmpRegisterNames, registerNames[element]);
        //Now we know where the appropriate recipe data is copy it into the ddcData array.
        //It is possible that this data does not occur in this recipe type, hence we must protect against this.
        if ( 0 != index )
	  ddcData[element] = tmpDdcData[index];
        //else
          //fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_saveRecipe(): " + registerNames[element] + " was not found in recipe type. This data has not been saved as it is not stored in this recipe type.", "1");
        //Don't bother warning about this it generates an unnecessary error, just let it happen silently.
      }
    }//if(fwhw_getRecipe())
    else
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getGbePortConfigRegisters(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "1");

  }//if(readRecipe)
  else {
    fwCcpc_read(registerNames, ddcData, callStatusList);
  }

  //Parse the data we just read and let it deal with any errors that occurred.
  fwUkl1_parseGbePortConfigRegistersForGet(ukl1Name, port, registerNames, ddcData, data, callStatusList, exceptionInfo);

  //Done.
  return;
}//fwUkl1_getGbePortConfigRegisters()

// ========
//    I2C
// ========

/*!
 * This writes to a given set of registers on TTCrx. It can be used to write to either the
 * registers on the UKL1 board or to a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  writeRecipe If TRUE then the function will write the given register settings to the configurationDB recipe given by
 *           recipe name. FALSE it will write the register settings to the UKL1 board.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 *          This array is only required if writeRecipe=TRUE, otherwise it can be empty.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register. The accepted values are described in
 *           fwUkl1_parseGbeWriteRegisters();
 * \param  data By default an 8-bit value that is to be written to the register represented as a hex string, it need not be zero padded.
 *           Certain settings accept or require an alternative format as described for each register above.
 * \param  verifyWrite If this is set to TRUE then the register will be read back from to ensure that
 *           the data written corresponds to that read. Default: FALSE.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_setTtcrxConfigRegisters(string ukl1Name, bool writeRecipe, const dyn_string& recipeNameAndType, dyn_string registerNames, dyn_string data, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setTtcrxConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Check that the register names and data arrays are the same size.
  //Save only the lenght of one of the arrays as the other must be the same size, in this case it is the registerNames.
  const unsigned numRegs = dynlen(registerNames);
  if ( numRegs != dynlen(data) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setTtcrxConfigRegisters(): Number of registers to write does not match amount of data to be written.", "1");
    return;
  }

  //Holds the data in the form that the fwCcpc_write function expects.
  dyn_dyn_char ddcData;
  //Holds the mask for each element.
  dyn_dyn_char masks;

  //Converts the data given as strings in the calling arguments to byte arguments required by the fwCcpc_write.
  //Also it creates the masks required for each write and pads the register names, such that they refer to the appropriate
  //data point elements.
  const int exInfoSize = dynlen(exceptionInfo);
  fwUkl1_parseTtcrxConfigRegistersForSet(ukl1Name, registerNames, data, ddcData, masks, exceptionInfo);
  //Check that we were able to parse the data.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_setTtcrxConfigRegisters(): Failed to parse input data, not all data was written.", "2");
  }//if exception info present.
  
  if (writeRecipe) {
    //Our save recipe function requires the masks to have already been appended to the data.
    const int numElements = dynlen(ddcData);
    for (int element = 1; element <= numElements; ++element) {
      dynAppend(ddcData[element], masks[element]);
    }//for element
    //Now save the recipe
    _fwUkl1_saveRecipe(ukl1Name, recipeNameAndType, registerNames, ddcData, "FWHW_I2C", exceptionInfo);
    //and check for errors.
    //We know exInfoSize hasn't changed since its size was first found and that it isn't -1 to get to this stage.
    if ( (exInfoSize < dynlen(exceptionInfo)) || (dynlen(exceptionInfo) == -1) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setTtcrxConfigRegisters(): Failed to save recipe for " + ukl1Name + ".", "1");
  }//if(writeRecipe)
  else {
    //The data and registers have been parsed, so let us write to the hardware.
    _fwUkl1_write(registerNames, ddcData, masks, verifyWrite, exceptionInfo);
    if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == dynlen(exceptionInfo)) )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_setTtcrxConfigRegisters(): Failed to write to registers for " + ukl1Name + ".", "1");
  }//else(writeRecipe)

  //Done.
  return;
}//fwUkl1_setTtcrxConfigRegisters()

/*!
 * This reads from a given set of TTCrx registers. The reads can either be from the hardware or they can be
 * done from a configurationDB recipe.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  readRecipe If TRUE then the function will read the settings from the given recipe name and return the settings
 *           contained within. FALSE and the returned values will come directly from the hardware registers.
 * \param  recipeName Name of the recipe to save the configuration information to if writeDb=TRUE, not used if writeDb=FALSE.
 *           The recipe must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE is type of run that the recipe is to be loaded
 *           e.g PHYSICS, CALIBRATION, can take any value, but should be descriptive of the type of run the recipe settings represent;
 *           FsmAction is the action in the FSM when these settings should be loaded e.g. Configure if the recipe is to be used to moved to
 *           the READY state, typically only the first letter is upper case.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           read from, which corresponds to the appropriate hardware register.
 * \param  data dyn_string returned by reference that contains a string representing the value in the register.
 *           It will be zero padded to the nearest 32-bit word.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void fwUkl1_getTtcrxConfigRegisters(string ukl1Name, bool readRecipe, const string& recipeName, dyn_string registerNames, dyn_string& data, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_TtcrxConfigRegisters(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Make sure we have an empty data array.
  dynClear(data);
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".TTCRX.";
  //Useful to know how many elements we may need to loop over.
  const unsigned numRegs = dynlen(registerNames);
  //Add prefix to each name and convert the data into a byte array.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Add the prefix.
    registerNames[element] = prefix + registerNames[element];
  }

  //Do the reads...
  //Holds the data from the reads.
  dyn_dyn_char ddcData;
  //Holds the status of each hardware read. Required by the parsing function,
  //but will be empty and hence ignored in the case of the recipe access.
  dyn_int callStatusList;
  dynClear(callStatusList);

  if (readRecipe) {
    //When we get the recipe data it will give us all registers in the recipe and also data.
    //This will hold the registers we are given and then we can check them against the list of requested registers to see what data to parse.
    dyn_string tmpRegisterNames;
    //Holds the data from the recipe, but it will be all the data stored in the recipe and will later be extracted.
    dyn_dyn_char tmpDdcData;
    if ( -1 != fwHw_getRecipe(ukl1Name, recipeName, tmpRegisterNames, tmpDdcData) ) {
      for (int element = 1; element <= numRegs; ++element) {
	//This take the specific register name and returns the index of its occurance in the full list.
	//Relies on the fact that the list is unique.
	//It is this index that we can use to over write the appropriate data element.
	int index = dynContains(tmpRegisterNames, registerNames[element]);
        //Now we know where the appropriate recipe data is copy it into the ddcData array.
        //It is possible that this data does not occur in this recipe type, hence we must protect against this.
        if ( 0 != index )
	  ddcData[element] = tmpDdcData[index];
        //else
          //fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_saveRecipe(): " + registerNames[element] + " was not found in recipe type. This data has not been saved as it is not stored in this recipe type.", "1");
        //Don't bother warning about this it generates an unnecessary error, just let it happen silently.
      }
    }//if(fwhw_getRecipe())
    else
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getTtcrxConfigRegisters(): Failed to retrieve the recipe " + recipeName + " for " + ukl1Name + ". Recipe may not exist.", "1");
    
  }//if(readRecipe)
  else {
    fwCcpc_read(registerNames, ddcData, callStatusList);
  }

  //Parse the data we just read and let it deal with any errors that occurred.
  fwUkl1_parseTtcrxConfigRegistersForGet(ukl1Name, registerNames, ddcData, data, callStatusList, exceptionInfo);

  //Done.
  return;
}//fwUkl1_getTtcrxConfigRegisters()

/*!
 * A function to read the serial number from a UKl1 board.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string Contain the 32-bit serial number as a single hex string. The string will return `Fail' if it does not succeed.
 *           The read string is byte swapped and it will be converted to the appropriate byte order. If this cannot be done the
 *           original byte ordered string is returned if possible and this is noted in exceptionInfo.
 */
string fwUkl1_getUkl1SerialNumber(const string& ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1SerialNumber(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
  const string prefix = ukl1Name + ".PROM.";

  //Name of the register to read the ID from.
  dyn_string registerNames;
  registerNames[1] = prefix + "Id";

  //This will hold the serial number once read.
  string serialNumber = "";
  //Do the reads...
  //Holds the data in the form that the fwCcpc_read function returns.
  dyn_dyn_char ddcData;
  //Holds the status of the write.
  dyn_int callStatusList;
  const bool status = fwCcpc_read(registerNames, ddcData, callStatusList);

  //Check the arrays are well formed.
  if ( 0 != dynlen(callStatusList) ) {
    //Put the data into a single string for return.
    if ( 0 != callStatusList[1] ) {
      //Construct the error message.
      string errMsg = "fwUkl1_getUkl1SerialNumber(): Encounterd ";
      if (callStatusList[1] > 0) {
	errMsg += "a problem writing to the CCPC server " + callStatusList[1] + ".";
      } else if (callStatusList[1] < 0) {
	errMsg += "a problem with PVSS.";
      } else {
	errMsg += "an unknown error.";
      }
      //Mark the error.
      serialNumber = "Fail";
      fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
    }//if(0!=callStatusList[1])
    else {
      //The ID number can just be converted straight to a hex string.
      const int exInfoSize = dynlen(exceptionInfo);
      serialNumber = fwUkl1_convertByteToHexLittleEndian(ddcData[1], exceptionInfo);
      if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
        //Attempt to return the big endian version of the serial number.
        serialNumber = fwCcpc_convertByteToHex(ddcData[1]);
        fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_getUkl1SerialNumber(): Byte swap of serial number failed. Displayed serial number will be byte swapped.", "2");
      }//if an exception was present.
    }//else(0!=callStatusList[1]
  }//if(0!=dynlen(callStatusList))
  else {
    //The function didn't execute properly as the callStatusList size is zero.
    serialNumber = "Fail";
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1SerialNumber(): Failed to read from UKL1.", "1");
  }

  //Return the ID number.
  return serialNumber;
}//fwUkl1_getUkl1SerialNumber()

/*!
 * This function allows the temperature probes to be read.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return dyn_int Temperatures read from the temperature sensors around the UKL1 board. The return elements are as follows:
 *   \li 0 Top of the board near Ingress FPGA 0, sensor on lowest register address.
 *   \li 1 Middle of the board, sensor on middle register address.
 *   \li 2 Bottom of the board near Ingress FPGA 3, sensor on highest address.
 *           -1 will be returned if there is an error, which will be described in exceptionInfo.
 */
dyn_int fwUkl1_getUkl1Temperatures(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1Temperatures(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Holds the temperature values.
  dyn_int temperatureReadings;

  //Loop over all the temperature sensors.
  for (int tempSens = 1; tempSens <= FWUKL1_TEMPERATURE_SENSORS; ++tempSens) {
    //Define the string to be added to each of the register names such that they are written to the appropriate datapoint element.
    const string prefix = ukl1Name + ".Temp " + tempSens + ".";

    //Name of the register to read the ID from.
    dyn_string registerNames;
    registerNames[1] = prefix + "TempRead";

    //Do the reads...
    //Holds the data in the form that the fwCcpc_read function returns.
    dyn_dyn_char ddcData;
    //Holds the status of the write.
    dyn_int callStatusList;
    fwCcpc_read(registerNames, ddcData, callStatusList);

    //Check the arrays are well formed.
    if ( 0 != dynlen(callStatusList) ) {
      //Put the data into a single string for return.
      if ( 0 != callStatusList[1] ) {
	//Construct the error message.
	string errMsg = "fwUkl1_getUkl1Temperatures(): Failed to read temperature sensor number " + tempSens + ", encountered ";
	if (callStatusList[1] > 0) {
	  errMsg += "a problem writing to the CCPC server, return code " + callStatusList[1] + ".";
	} else if (callStatusList[1] < 0) {
	  errMsg += "a problem with PVSS.";
	} else {
	  errMsg += "an unknown error.";
	}
	//Now raise an exception.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
	//Mark the error.
	temperatureReadings[tempSens] = -1;
      }
      else {
	//The ID number can just be converted straight to a hex string.
	temperatureReadins[tempSens] = fwCcpc_convertByteToDec(ddcData[1]);
      }
    }//if ( 0 != dynlen(callStatusList) )
    else {
      //The function didn't execute properly as the callStatusList size is zero.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1Temperatures(): No data returned from temperature sensor number " + tempSens + ".", "1");
    }

  }//for tempSens

  //Return the temperature readings.
  return temperatureReadings;
}//fwUkl1_getUkl1Temperatures()


// ===========
//   SUPPORT
// ===========

/*!
 * In order to write to specific registers on the FPGAs we must first convert the address
 * to include the address of the FPGAs from the glue card perspective.
 *
 * \param  addr Address of the register to be written to.
 * \return unsigned Full address of the FPGA register.
 */
unsigned _fwUkl1_fpgaizeAddress(unsigned addr) {
  // Based on code written by Steve Wotton, Cambridge.
  // The address sent to lb_{write|read}_hword should be above 0x20000
  // to be outside the GBE & gluecard ranges
  // The FPGA is sensitive only to bits 2-13 of the address
  //  (=> the FPGA is not sensitive to the 0x2???? that indicates
  //      that an address is outside the GBE/gluecard ranges)
  // So whenever we want to address the FPGA, set the 13th bit:
  //   the FPGA will only respond to addresses for which the 13th bit has been set
  //
  // So: << 2    translates a (32-bit) int address into (8-bit) byte address
  //     0x20000 gets us outside the address ranges of the GBE and gluecard
  //               (but will be ignored by the FPGA)
  //     0x2000  should stop the gluecard from reacting to the operation
  //               (but will be ignored by the FPGA)
  //
  // => (int) register addresses 0x0 to 0x7ff should be usable by the FPGA
  //
  return (0x08000000 + ( 0x800 | ( addr<<2 ) ) );
}//_fwUkl1_fpgaizeAddress()

/*!
 * Writes the given data to the given UKL1 board. The masks are applied when writing the data to ensure that only specific
 * bits of a register are written to.
 *
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * \param  ddcData Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write.
 * \param  masks Each element contains a bit mask that is applied when writing the corresponding element in the ddcData array
 *           to the corresponding register in the registerNames array. It ensure that only the desired bits are written to in
 *           the register.
 * \param  verifyWrite If this is set to TRUE write will be verified to be successful.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_write(const dyn_string& registerNames, const dyn_dyn_char& ddcData, const dyn_dyn_char& masks, bool verifyWrite, dyn_string& exceptionInfo) {
  //Check we are being asked to write something.
  if ( 0 >= dynlen(registerNames) ) {
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_write(): List of registers to write to empty, no data written to the hardware.", "2");
    return;
  }//if(0>=dynlen(registerNames))
  //We will read the data back from the registers once it is written. The read data is stored here.
  dyn_dyn_char readData;
  //Holds the status of the writes.
  dyn_int callStatusList;
//  if ( fwCcpc_writeRead(registerNames, ddcData, masks, readData, callStatusList) ) {
  readData = ddcData;
  if ( fwCcpc_write(registerNames, ddcData, masks, callStatusList) ) {
    //See if we want to verify that the write was successfull.
    if (verifyWrite) {
      //Call the verification function.
      int exInfoSize = dynlen(exceptionInfo);
      _fwUkl1_verifyWrite(registerNames, ddcData, masks, readData, exceptionInfo);
      if ( (exInfoSize < dynlen(exceptionInfo)) && (exInfoSize != -1) )
        fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_write(): Failed to verify data written written to some/all of the registers.", "1");
    }//if(verifyWrite)
  }//if(fwCcpc_write)
  else {
    //Check where the errors occured.
    const unsigned statusLen = dynlen(callStatusList);
    if ( 0 != statusLen ) {
      for (unsigned element = 1; element <= statusLen; ++element) {
        if ( 0 != callStatusList[element] ) {
          //Construct the error message.
	  string errMsg = "_fwUkl1_write(): " + registerNames[element] + " encountered ";
	  if (callStatusList[element] > 0)
            errMsg += "a problem writing to the CCPC server, return code: " + callStatusList[element] + ".";
	  else if (callStatusList[element] < 0)
            errMsg += "a problem with PVSS.";
	  else
            errMsg += "an unknown error.";
	  
          fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "2");
        }//if(0!=callStatusList[element]
      }//for element
    }//if(0!=statusLen)
    else
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_write(): All writes failed.", "1");
  }//else(fwCcpc_write)
  
  //Done.
  return;
}//_fwUkl1_write()

/*!
 * Save a list of recipes to the given recipe, which is of the given type. The data should contain the actual value that would be
 * written to the hardware and include the mask that should be used when writing the data.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  recipeNameAndType Two element array containing:
 *   \li Element 1 - The name of the recipe to save the data to. It must be of the form "RUN_TYPE/FsmAction", where RUN_TYPE
 *         is descriptive of the type of run the settings are saved for e.g. PHYSICS, CALIBRATION; FsmAction is the action of
 *         FSM state transition where the recipe should be loaded e.g. Configure, the first letter only should be capitalised.
 *   \li Element 2 - The second element contains the recipe type that it is to be saved to, this can be anything provided it already exists.
 * \param  registerNames dyn_string that contains the name of the HwTypeCCPCUKL1 datapoint element to be
 *           written to, which corresponds to the appropriate hardware register.
 * \param  ddcDataAndMask Each element of this array contains an array of char that make up the words that are to be written to the
 *           register in the form required by a fwCcpc_write and this includes the mask that is to be used for the data.
 * \param  regType Hardware register type that the data is being saved for. It accepts the constants defined by the fwHw.ctl lib.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_saveRecipe(const string& ukl1Name, const dyn_string& recipeNameAndType, const dyn_string& registerNames, const dyn_dyn_char& ddcDataAndMask, int regType, dyn_string& exceptionInfo) {
  //Check that the recipeNameAndType is populated with two values as required.
  if ( 2 == dynlen(recipeNameAndType) ) {
    //We must write to all the registers in the recipe otherwise it will populate them with the last values written to the board.
    //We must therefore get the recipe first and then update the relevant values and write it back to the recipe.
    //Holds a list of all the registers in the recipe to be written.
    dyn_string fullRegisterNames;
    //Holds a list of all the data, including masks, to be written into the recipe.
    dyn_dyn_char fullDdcData;
    //Before we try and get the recipe we should get that it exists, as if it doesn't trying to retrieve it will fail.
    dyn_string recipes = fwHw_getRecipes(ukl1Name);
    //Also note whether or not the recipe already exists.
    bool recipeAlreadyExists;
    //Check to see if the register we are going to write to exists.
    //dynContains returns either the index of the element that the string is contained within, 0 if it does not exists or -1 in the event of an error.
    if ( 0 < dynContains(recipes, recipeNameAndType[1]) ) {
      //The recipe does exist, thus we should get it and overwrite the relevant elements.
      recipeAlreadyExists = TRUE;
      if ( -1 != fwHw_getRecipe(ukl1Name, recipeNameAndType[1], fullRegisterNames, fullDdcData) ) {
        const int numElements = dynlen(registerNames);
        for (int element = 1; element <= numElements; ++element) {
  	  //This take the specific register name and returns the index of its occurance in the full list.
  	  //It is this index that we can use to over write the appropriate data element.
          //It assumes there is only one occurrance of a register in the recipe. This should always be the case!
  	  int index = dynContains(fullRegisterNames, registerNames[element]);
  	  //Now we know where the appropriate recipe data is copy it into the ddcData array converting from its string format.
          //It is possible that this data does not occur in this recipe type, hence we must protect against this.
          if ( 0 != index )
            fullDdcData[index] = ddcDataAndMask[element];
          //else
            //fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_saveRecipe(): " + registerNames[element] + " was not found in recipe type. This data has not been saved as it is not stored in this recipe type.", "1");
          //Don't bother warning about this it generates an unnecessary error, just let it happen silently.
        }//for element
      }//if(-1!=fwHw_getRecipe)
      else
       fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_saveRecipe(): Failed to find recipe `" + recipeNameAndType[1] + "' for " + ukl1Name + ". Cannot update recipe settings.", "1");
    }//if(dynContains)
    else {
      //As the recipe doesn't exist we are creating it. Therefore we just give the recipe all the data we have.
      //The save recipe function will place the last values written to the hardware in the other recipe registers.
      //These can be updated by other function calls.
      fullRegisterNames = registerNames;
      fullDdcData = ddcDataAndMask;
      //The recipe doesn't already exist.
      recipeAlreadyExists = FALSE;
    }//else(dynContains)

    //The recipe needs to know the recipe type that we are using. Has to be for the full list.
    dyn_int regTypes;
    const int fullLen = dynlen(fullRegisterNames);
    for (int i = 1; i <= fullLen; ++i) {
      //All the registers are of the same type, LBUS, so just populate array with that value. Where is memset when you need it?
      regTypes[i] = regType;
    }//for i

    //Now we can save the ddcData to the recipe.
    //If this function fails and the recipe already exists then there has been an error, however this function could still apparently fail if the recipe
    //doesn't already exist, but it does write the appropriate values to the recipe. Require it fails and the recipe already exists for us to indicate an error.
    if ( (-1 == fwHw_saveRecipe(ukl1Name, recipeNameAndType[2], recipeNameAndType[1], fullRegisterNames, regTypes, fullDdcData)) && recipeAlreadyExists )
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_saveRecipe(): Failed to save the recipe " + recipeNameAndType[1] + " for " + ukl1Name + ".", "1");

  }//if(2 == dynlen(recipeNameAndType))
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_saveRecipe(): Recipe name and/or type not given for " + ukl1Name + " . Cannot save recipe.", "1");
  }//else(2 == dynlen(recipeNameAndType))
  
  //Done.
  return;
}//_fwUkl1_saveRecipe()

/*!
 * Reads from the registers on the L1 board to ensure that the data just written has been written successfully.
 *
 * \param  registerNames Names of the registers that were written to.
 * \param  writtenData The data that was written to the registers that the read back values should be compared against.
 * \param  masks Any masks that should be applied to the read data, such that only specific bits in a register are compared
 *           against the write bits.
 * \param  readData The data that was read back from the registers to compare against the written data.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * This function is designed to check the data written and read using the fwCcpc_writeRead function.
 */
void _fwUkl1_verifyWrite(dyn_string registerNames, dyn_dyn_char writtenData, dyn_dyn_char masks, dyn_dyn_char readData, dyn_string& exceptionInfo) {
  //Number of elements of data to loop through.
  const unsigned numRegs = dynlen(registerNames);
  //Check to see if any masks are supplied and if so flag them for use.
  bool applyMasks = ( numRegs==dynlen(masks) ? TRUE:FALSE );
  //Check the data.
  for (unsigned element = 1; element <= numRegs; ++element) {
    //Number of bytes written.
    const int dataLen  = dynlen(writtenData[element]);
    //Number of bytes read back.
    const int checkLen = dynlen(readData[element]);
    //Check they are the same size.
    if (checkLen != dataLen) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_verifyWrite(): " + registerNames[element] + " written and read data lengths differ when verifying write, " + dataLen + " and " + checkLen + " bytes big respectively. Read data 0x" + fwCcpc_convertByteToHex(readData[element]) + ", and written data 0x" + fwCcpc_convertByteToHex(writtenData[element]) + " (premask).", "2");
    }//if(checkLen!=data)
    else {
      //Only check the data if they are the same size.
      //Use to note if an element in the read array differ.
      bool diff = FALSE;
      //This will provide us with our loop index. The two arrays have been found to be the same size, so just set to either.
      for (unsigned index = 1; index <= dataLen; ++index) {
        //Apply the masks that are given to the read data, if masks are supplied.
        if (applyMasks)
          readData[element][index] &= masks[element][index];
        //Now check the data against the read values.
        if ( readData[element][index] != writtenData[element][index] ) {
          diff = TRUE; 
          //We can't break here as otherwise the readData will not be fully masked and that value included in the error will make no sense.
          //break;
        }
      }//for index
      //Will only raise the exception displaying the whole value and not for each differing element.
      if (diff)
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_verifyWrite(): " + registerNames[element] + " read data, " + fwCcpc_convertByteToHex(readData[element]) + ", and written data, " + fwCcpc_convertByteToHex(writtenData[element]) + ", differ.", "2");
    }//else(checkLen!=dataLen)
  }//for element

  //Done.
  return;
}//_fwUkl1_verifyWrite()

//@}


// =========================
//  STATE MACHINE FUNCTIONS
// =========================

/** @defgroup SectionState FSM interface
 *    A group of functions that can be used to perform state machine actions
 *    for a UKL1 board. The following actions are defined:
 *    \li Configure - Loads of the configuration registers on the UKL1 board with the settings defined in the database.
 *          It will leave the board in the READY state. It is slow transition.
 *    \li Start - Moves the board into the RUNNING state. Enables data to flow through the UKL1 board. Fast transition.
 *    \li Stop  - Moves the board into the READY state. Prevents data flow through the UKL1 board. It will ignore TFC
 *          commands and cannot output over the GBE card.
 *    \li Reset  - Performs a reset of the UKL1. Currently the actions of this are ill defined.
 *    \li Reload - Reloads the FPGA (Ingress/FE and Egress/BE) firmware from the onboard memory.
 *    \li Recover - Currently does nothing.
 *
 *    None of the defined functions change the FSM state, this is left to the User to ensure the FSM is in the appropriate
 *    state after the function has been called.
 *  @{
 */

/*!
 * Gets the state of the current UKL1.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \return string State that the UKL1 board is currently in, empty string if the board is not present
 *           and hence there is no associated state.
 */
string fwUkl1_getUkl1State(const string& ukl1Name) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getUkl1State(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Holds the state to be returned.
  string sState = "";
  //Check the DPE exists.
  const string dpeName = ukl1Name + ".status";
  if ( dpExists(dpeName) )
    dpGet(ukl1Name + ".status", sState);
  //No else as we have initialised ourselves to the appropriate case for DPE not existing.
  return sState;
}//fwUkl1_getUkl1State()

/*!
 * Loads a specific recipe to a UKL1 board from the configuration database.
 *
 * \param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * \param  recipe Name of the recipe that is to be applied to the UKL1 board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * This will move the UKL1 board to a configured state, it is typically a long operation and
 * it is recommended that a fwUkl1_stop() is performed once completed. It will not update the
 * state of the UKL1, this must be done by the User.
 */
void fwUkl1_configure(string ukl1Name, string recipe, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Holds the registers that can be set with this recipe.
  dyn_string regs;
  //Holds the recipe data.
  dyn_dyn_anytype data;
  //Keeps track of the number of errors that occured.
  int exInfoSize = dynlen(exceptionInfo);

  //When performing the configure we will always configure as much as possible and indicate any errors via
  //exception info. Boards may still be mostly usable.
    
  //For the moment we will still configure the hardcoded settings of the GBE from value defined settings and not DB.
  //Probably will never use the DB.
  //This performs some configuration routines for the GBE, setting the GBE into its running mode.
  _fwUkl1_configureHardcodedSettings(ukl1Name, exceptionInfo);
  //Check for errors.
  if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to initialise GBE or TTCrx.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//if _fwUkl1_configureHardcodedSettings() failed
  
  //The source IP and MAC addresses must be configured dynamically from the serial number.
  _fwUkl1_configureSourceAddresses(ukl1Name, exceptionInfo);
  if ( (dynlen(exceptionInfo) > exInfoSize) && (-1 != exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to initialise some/all of the source IP and MAC addresses, and L1 ID.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//if _fwUkl1_configureSourceAddresses()
  
  //Holds the registers and data retrieved from this recipe.
  dyn_string regs;
  dyn_dyn_char data;
  //Get the recipe that we wish to apply.
  if ( -1 != fwHw_getRecipe(ukl1Name, recipe, regs, data) ) {
    //Now we have the recipe we must remove the channel configuration from it for later application.
    //If successful this will modify the regs and data such that fwHw_applyRecipe() will only apply the Egress
    //and GBE settings. If it is not successful everything will be applied, but the channel configuration data
    //will not be set.
    _fwUkl1_configureIngressChannelSettings(ukl1Name, regs, data, exceptionInfo);
    if ( (dynlen(exceptionInfo) > exInfoSize) && (-1 != exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "fwUkl1_configure(): Failed to configure the UKL1 input channel specific settings from the recipe `" + recipe + "'.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if _fwUkl1_configureSourceAddresses()
    
    //Apply the recipe with the Egress and GBE data.\
    if ( -1 == fwHw_applyRecipe(ukl1Name, regs, data) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to apply the recipe `" + recipe + "' for the Egress and GBE settings.", "1");
      exInfoSize = dynlen(exceptionInfo);
    }//if(-1==fwHw_applyRecipe())
  }//if(-1!=fwHw_getRecipe())
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_configure(): Failed to get the recipe `" + recipe + "'. Cannot configure the recipe specific settings.", "1");
    exInfoSize = dynlen(exceptionInfo);
  }//else(-1!=fwHw_getRecipe())

  //Apply the pixel masks recipe by here when we have some.
  //Don't apply them in the channel configuration as we have to get them from a recipe.

  //Done.
  return;
}//fwUkl1_configure()

/*!
 * Start does only two things and that is to enable the TFC decoding on the UKL1 and enable the link from
 * the BE FPGA to the GBE card for data output.
 *
 * \param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * \param  recipe Name of the recipe that is to be applied to the UKL1 board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * This will move the UKL1 to the start state where it is ready to take data. It will not update the state
 * once complete this is upto the User.
 */
void fwUkl1_start(string ukl1Name, string recipe, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_start(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Holds the registers that can be set with this recipe.
  dyn_string regs;
  //Holds the recipe data.
  dyn_dyn_anytype data;
  
  //Get and apply the recipe, for the moment ignore the domain argument.
  if ( -1 == fwHw_getApplyRecipe(ukl1Name, recipe, "") ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_start(): Failed to get and apply the recipe `" + recipe + "'.", "1");
  }//if(-1==fw_getApplyRecipe())

  //Done.
  return;
}//fwUkl1_start()

/*!
 * Stop does only two things and that is to disable the TFC decoding on the UKL1 and disable the link from
 * the BE FPGA to the GBE card for data output.
 *
 * \param  ukl1Name Name of the UKL1 board as defined in the DIM server.
 * \param  recipe Name of the recipe that is to be applied to the UKL1 board.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * This will move the UKL1 to the start state where it will not be able to see triggers sent from the TFC and also
 * is not capable of outputting data. It will not update the state once complete this is upto the User.
 */
void fwUkl1_stop(string ukl1Name, string recipe, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_stop(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //Holds the registers that can be set with this recipe.
  dyn_string regs;
  //Holds the recipe data.
  dyn_dyn_anytype data;
  
  //Get and apply the recipe, for the moment ignore the domain argument.
  if ( -1 == fwHw_getApplyRecipe(ukl1Name, recipe, "") ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_stop(): Failed to get and apply the recipe `" + recipe + "'.", "1");
  }//if(-1==fw_getApplyRecipe())

  //Done.
  return;
}//fwUkl1_stop()

/*!
 * Performs a reset of the FPGAs on the L1 board. Effects still under discussion.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_reset(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_reset(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //The reset is performed by setting the GPIO line 5 low for a few 10s of milliseconds and then setting it back to high.
  //First set it low.
  int callStatus = fwCcpc_GPIOSet(ukl1Name, FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_LOW);
  //Check for errors.
  if ( 0 == callStatus ) {
    //We have driven it low, wait a few milli-seconds for the reset to take affect.
    delay(0,20);
    //Only worth trying to do this if it succeeded.
    int callStatus = fwCcpc_GPIOSet(ukl1Name, FWCCPC_GPIO_LINE_5, FWCCPC_GPIO_HIGH);
    //Check for errors.
    if ( 0 != callStatus ) {
	//Construct the error message.
	string errMsg = "fwUkl1_reset(): Failed to set the GPIO line 5 high during a reset, unable to desassert the reset. Encountered";
	if (callStatus > 0) {
	  errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
	} else if (callStatus < 0) {
	  errMsg += " a problem with PVSS.";
	} else {
	  errMsg += " an unknown error.";
	}
	//Now raise the exception.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
    }
  } else {
    //Construct the error message.
    string errMsg = "fwUkl1_reset(): Failed to set the GPIO line 5 low during a reset, unable to reset. Encountered";
    if (callStatus > 0) {
	errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
    } else if (callStatus < 0) {
	errMsg += " a problem with PVSS.";
    } else {
	errMsg += " an unknown error.";
    }
    //Now raise the exception.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
  }

  //Done.
  return;
}//fwUkl1_reset()

/*!
 * Performs a reload of the FPGA firmware on the L1 board. It will reset all values back to their defaults.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 */
void fwUkl1_reload(string ukl1Name, dyn_string& exceptionInfo) {
  //Check we have a UKL1 name, this really upsets things if we don't.
  if ("" == ukl1Name) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_reload(): UKL1 name was empty, cannot proceed.", "1");
    return;
  }//if(""==ukl1Name)
  //The reload is performed by setting the GPIO line 6 low for a few 10s of milliseconds and then setting it back to high.
  //First set it low.
  int callStatus = fwCcpc_GPIOSet(ukl1Name, FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_LOW);
  //Check for errors.
  if ( 0 == callStatus ) {
    //We have driven it low, wait a few milli-seconds for the reload to take affect.
    delay(0,20);
    //Only worth trying to do this if it succeeded.
    int callStatus = fwCcpc_GPIOSet(ukl1Name, FWCCPC_GPIO_LINE_6, FWCCPC_GPIO_HIGH);
    //Check for errors.
    if ( 0 != callStatus ) {
	//Construct the error message.
	string errMsg = "fwUkl1_reload(): Failed to set the GPIO line 6 high during a reload, unable to desassert the reload. Encountered";
	if (callStatus > 0) {
	  errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
	} else if (callStatus < 0) {
	  errMsg += " a problem with PVSS.";
	} else {
	  errMsg += " an unknown error.";
	}
	//Now raise the exception.
	fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
    }
  } else {
    //Construct the error message.
    string errMsg = "fwUkl1_reload(): Failed to set the GPIO line 6 low during a reload, unable to reload. Encountered";
    if (callStatus > 0) {
	errMsg += " a problem writing to the CCPC server, return code " + callStatus + ".";
    } else if (callStatus < 0) {
	errMsg += " a problem with PVSS.";
    } else {
	errMsg += " an unknown error.";
    }
    //Now raise the exception.
    fwUkl1Exception_raise(exceptionInfo, "ERROR", errMsg, "1");
  }

  //Done.
  return;
}//fwUkl1_reload()

/*!
 * Recovers the UKL1 board from an error state. This will return the UKL1 to an unconfigured state and the values in
 * the registers cannot be guarantied to be the same as before recovering. The User should update the FSM state appropriately.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void
 *
 * The interface to this function is unlikely to change, but it might.
 */
void fwUkl1_recover(string ukl1Name, dyn_string& exceptionInfo) {
  //Do something to recover from an error...
  //this is actually quite useful as it is, it allows us to return to NOT_READY without doing anything.
  return;
}//fwUkl1_recover()

/*!
 * Configures the last byte of the source IP address and the fifth byte of the source MAC address from the ID number in the
 * EPROM. It also sets the L1 ID to the least significant byte of the serial number.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 */
void _fwUkl1_configureSourceAddresses(const string& ukl1Name, dyn_string& exceptionInfo) {
  //Get the serial number from the board.
  string serialNumber = fwUkl1_getUkl1SerialNumber(ukl1Name, exceptionInfo);
  if ( "Fail" == serialNumber ) {
    //This is going likely to be the prototype board, if the board genuinely doesn't exist then we will encounter some errors.
    //The prototype board can have a serial number of 40, it is high to indicate what happened.
    serialNumber = "60110040";
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureSourceAddresses(): Configuring board without a serial number, using 0x60110040.", "1");
  }

  //This prefixes the register names.
  const string prefix = ukl1Name + ".EgressFpga.";
  //These are required when writing to the board registers.
  dyn_string regNames;
  dyn_dyn_char data;
  dyn_dyn_char masks;
  
  //Now get the last two characters (byte) from this number.
  const string boardNum = substr(serialNumber, strlen(serialNumber)-2, 2);
  //Construct the MAC address. Pad the first 4 bytes with 0s and the last byte also.
  //It must have extra 16-bit words added in order to convert the data from 16- to 32-bit word for writing to the registers.
  regNames[1] = prefix + "MacSourceAddress";
  data[1] = fwCcpc_convertHexToByte("0000" + boardNum + "000000000000000000");
  //Need a mask for a 96-bit word write, 2 32-bit words.
  //The last two 32-bit words (actually last two bytes but 16- to 32-bit funniness comes into play)
  //are reserved. The first of these bytes we must configure, the other must be set to zero and the UKL1 firmware will deal with it.
  masks[1] = makeDynChar(0x00, 0x00, 0xff, 0xff, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00);
  //Now construct the IP address. Need to pad the first 3 bytes with 0s, plus the extra padding to convert from 16- to 32-bit
  //words for the register writes.
  regNames[2] = prefix + "IpSourceAddress";
  data[2] = fwCcpc_convertHexToByte("000000" + boardNum + "00000000");
  //Need a 48-bit mask for this.
  //The last 8-bits are reserved and not User configurable. These are written what we now write.
  masks[2] = makeDynChar(0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00, 0x00);
  //Now set the serial number.
  regNames[3] = prefix + "L1Id";
  string l1Id = boardNum;
  fwUkl1_padString(l1Id, 4);
  data[3] = fwCcpc_convertHexToByte(l1Id);
  //Need an 8-bit mask for this.
  masks[3] = makeDynChar(0x00, 0x00, 0x00, 0xff);
  
  //Now write this data. Have to bypass the BE FPGA write as this can't set the bytes we want to.
  //Use the _fwUkl1_write instead as this takes the raw byte data and can optionally verify the writes.
  //Don't verify writes until the update to the CCPC library fixes the problem where writeRead returns the previous written value.
  const int exInfoSize = dynlen(exceptionInfo);
  _fwUkl1_write(regNames, data, masks, FALSE, exceptionInfo);
  if ( (exInfoSize < dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureSourceAddresses(): Failed to configure the source IP and MAC addresses, and the L1 ID. Events may not be routed correctly.", "1");
  }//if exceptionInfo generated.
  
  //Done.
  return;
}//_fwUkl1_configureSourceAddresses()

/*!
 * Configures all the settings on the GBE that are never changed by the User and all values are hardcoded into this function.
 * It also setups the single TTCrx register that is hardcoded to a UKL1 specific setting.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 *
 * The values set here are defined by the LHCb technical note Quad Gigabit Ethernet plug-in card and are correct as of
 * issue 2.2 revision 0 of that document. They must be performed in a specific order.
 */
void _fwUkl1_configureHardcodedSettings(const string& ukl1Name, dyn_string& exceptionInfo) {
  //Used to track the size of the exception info array to see if new exception have been added.
  int exInfoSize = dynlen(exceptionInfo);

  //This holds the list of registers that to be written
  dyn_string registerList;
  //and their corresponding data.
  dyn_string sData;
  //both MUST be cleared after each write.

  //Set up the TTCrx control register as desired.
  registerList[1] = "Control";
  sData[1] = "a1";
  //Don't write to a recipe and don't verify the write.
  //As we are not writing to a recipe we don't need to give it a recipe name and type.
  fwUkl1_setTtcrxConfigRegisters(ukl1Name, FALSE, makeDynString(), registerList, sData, FALSE, exceptionInfo);
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): TTCrx not properly initialised.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);

  //Place MAC, SPI3 TX and RX, RX and TX FIFOs in reset.
  registerList[1] = "MACSoftReset";
  sData[1] = "0000000f";
  registerList[2] = "SPI3ConfigTrnmtGlobal";
  sData[2] = "000c000f";
  registerList[3] = "RxFifoPrtReset";
  sData[3] = "0000000f";
  registerList[4] = "TxFifoPrtReset";
  sData[4] = "0000000f";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Disable all ports.
  registerList[1] = "PortEnable";
  sData[1] = "Disable,Disable,Disable,Disable";
  registerList[2] = "Clock";
  sData[2] = "00000000";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Select copper mode
  registerList[1] = "Mode";
  sData[1] = "0000000f";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);
  
  //Check for errors before we call a new function, ensures the tree forms properly.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): GBE not properly initialised.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception
  
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    // Select copper mode for each port.
    registerList[1] = "MAC";
    sData[1] = "00000002";
    //Perform the write.
    fwUkl1_setGbePortConfigRegisters(ukl1Name, port, registerList, sData, registerList, sData, FALSE, exceptionInfo);
    //Clear the arrays for next use.
    dynClear(registerList);
    dynClear(sData);
    //Delaying for the shortest time possible.
    delay(0,1);

    //Set full duplex mode for each port.
    registerList[1] = "Duplex";
    sData[1] = "00000001";
    //Perform the write.
    fwUkl1_setGbePortConfigRegisters(ukl1Name, port, registerList, sData, registerList, sData, FALSE, exceptionInfo);
    //Clear the arrays for next use.
    dynClear(registerList);
    dynClear(sData);
    //Delaying for the shortest time possible.
    delay(0,1);
    
    //Check for errors before we call a new function, ensures the tree forms properly.
    if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): GBE port " + port + " not properly initialised.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception
  }//for port

  //Enable clocks for all active channels
  registerList[1] = "Clock";
  sData[1] = "0000000f";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Remove resets
  registerList[1] = "MACSoftReset";
  sData[1] = "00000000";
  registerList[2] = "SPI3ConfigTrnmtGlobal";
  sData[2] = "0000000f";
  registerList[3] = "RxFifoPrtReset";
  sData[3] = "00000000";
  registerList[4] = "TxFifoPrtReset";
  sData[4] = "00000000";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Check for errors before we call a new function, ensures the tree forms properly.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): GBE not properly initialised.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception
  
  // Enable automatic padding and CRC generation for all active channels
  registerList[1] = "Config";
  // Port setting
  sData[1] = "000011cd";
  for (unsigned port = 0; port < FWUKL1_GBE_PORTS; ++port) {
    //Perform the write.
    fwUkl1_setGbePortConfigRegisters(ukl1Name, port, registerList, sData, registerList, sData, FALSE, exceptionInfo);
    //Check for errors before we call a new function, ensures the tree forms properly.
    if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): GBE port " + port + " not properly initialised.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if exception
  }//for port
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure SPI3 width 32 bit, RVAL pause and active channel
  registerList[1] = "SPI3ConfigRcv";
  sData[1] = "00ffff80";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  // Setup Rx interface: configure RX CRC stripping and check
  registerList[1] = "RxFifoEnable";
  sData[1] = "000000f0";
  //Perform the write.
  fwUkl1_setGbeConfigRegisters(ukl1Name, registerList, sData, registerList, sData, FALSE, exceptionInfo);
  
  //Check for errors before we call a new function, ensures the tree forms properly.
  if ( (exInfoSize != dynlen(exceptionInfo)) || (-1 == exInfoSize) ) {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "_fwUkl1_configureHardcodedSettings(): GBE not properly initialised.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//if exception
  
  //Clear the arrays for next use.
  dynClear(registerList);
  dynClear(sData);
  //Delaying for the shortest time possible.
  delay(0,1);

  //Done.
  return;
}//_fwUkl1_configureHardcodedSettings()

/*!
 * This writes the Ingress input channel configuration settings to the UKL1s.
 *
 * \param  ukl1Name Name of the UKL1 board as it appears in the DIM server.
 * \param  regs This should contain all the registers that are found in a UKL1_Configure type recipe.
 *           The function will separate out the Channel settings and the other settings returning, by reference,
 *           the other settings that this function does not write. These can then be applied by fwHw_applyRecipe().
 * \param  data This should contain all the data that is retrieved from a recipe and the function will, as for
 *         regs, separate out the data it wants, apply it, and return the remaining data by reference for use with
 *         fwHw_applyRecipe().
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return void.
 * 
 */
void _fwUkl1_configureIngressChannelSettings(const string& ukl1Name, dyn_string& regs, dyn_dyn_char& data, dyn_string& exceptionInfo) {
  //Keep track of the number of exceptions that have been generated.
  int exInfoSize = dynlen(exceptionInfo);
  //Determine where in the full register list are the registers that are set for channels.
  dyn_bool chanRegPos = patternMatch("*.Channel?.*", regs);
  if ( 0 < dynlen(chanRegPos) ) {
    //Holds the names of the registers that we wish to apply and the corresponding data.
    dyn_string channelRegs;
    dyn_dyn_char channelData;
    //Note where we put the register and data into the arrays.
    int index = 0;
    //Now loop through the returned bool array and determine where the channel data is.
    for (int reg = 1; reg <= dynlen(chanRegPos); ++reg) {
      if (chanRegPos[reg]) {
        ++index;
        //Copy the desired register to the channel register names array and then remove the register.
        channelRegs[index] = regs[reg];
        dynRemove(regs, reg);
        dynRemove(chanRegPos, reg);
        //also the data
        channelData[index]  = data[reg];
        dynRemove(data, reg);
        //Now decrement the reg number.
        --reg;
      }//if(chanRegPos[reg])
      //No need for an else as we want to leave the other register untouched.
    }//for reg
    //We can just apply the settings to the appropriate channel.
    const int numChanRegs = dynlen(channelRegs);
    //Check to see if we matched against anything.
    if ( 0 == numChanRegs ) {
      fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Failed to find any configuration settings for the Ingress FPGA channels. These have not been configured.", "2");
      exInfoSize = dynlen(exceptionInfo);
    }//if(0==numChanRegs)
    
    //so we know who to configure.
    unsigned channel = 0;
    unsigned inFpga  = 0;
    for (int chanReg = 1; chanReg <= numChanRegs; ++chanReg) {
      //Get the channel and Ingress number.
      //This is the pattern that sscanf must look for.
      const string format = ukl1Name + ".IngressFpga%u.Channel%u.ConfigurationFifo";
      if ( 2 == sscanf(channelRegs[chanReg], format, inFpga, channel) ) {
        //The FALSE is because we are not going to verify the writes. Want recipe application to be as fast as possible.
        _fwUkl1_setIngressFpgaChannelConfigRegisters(ukl1Name, channel, inFpga, channelData[chanReg], false, exceptionInfo);
        if ( (dynlen(exceptionInfo) > exInfoSize) || (-1 == exInfoSize) ) {
          fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Failed to configure Ingress FPGA " + inFpga + " channel " + channel + ". This channel may not work correctly.");
          exInfoSize = dynlen(exceptionInfo);
        }//channel configuration error.
      }//if(-1!=sscanf())
      else {
        fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Failed to determine the channel and Ingress FPGA to write the configuration data to for register `" + channelRegs[chanReg] + ". sscanf() failed to identify the channel and Ingress FPGA number in that string.", "2");
        exInfoSize = dynlen(exceptionInfo);
      }//else(-1!=sscanf())              
    }//for chanReg
  }//if(0<dynlen(chanRegPos)
  else {
    //In this case we couldn't find any channel configuration registers!
    //Warn people and then apply the full recipe anyway.
    fwUkl1Exception_raise(exceptionInfo, "WARNING", "_fwUkl1_configureIngressChannelSettings(): Could not find any channel configuration settings in this recipe. No input channel configuration was done.", "2");
    exInfoSize = dynlen(exceptionInfo);
  }//else(0<dynlen(chanRegPos))
  //Done.
  return;
}//_fwUkl1_configureIngressChannelSettings()

//@}


// ===================
//  SUPPORT FUNCTIONS
// ===================

/** @defgroup SectionSupport Support funtions
 *     These are miscellaneous functions that provide functionality that does not fit into any of the other categories.
 *  @{
 */

/*!
 * Strips the system name from the given string. Will remove only the system name. If no system name is present
 * then the effects are undefined, unless there are no semi-colons present in the string.
 *
 * \param  dpName String containing the DP name that the system name is to be removed from.
 * \return string dpName minus the beginning system name.
 */
string fwUkl1_removeSystemName(string dpName) {
  //Will hold the stripped DP name, will rely on it being initialised to NULL.
  string strippedDpName = "";

  //First break up the string into its semi-colon parts.
  const dyn_string semiColonDelimited = strsplit(dpName, ':');
  //Determine how many parts we have.
  const int numSec = dynlen(semiColonDelimited);
  if ( 1 == numSec ) {
    //No semi-colons found just return the original string.
    strippedDpName = dpName;
  } else if ( 2 <= numSec ) {
    //Start at the second element, bypassing the system name and then append the other elements.
    for (int sec = 2; sec <= numSec; ++sec) {
      strippedDpName += semiColonDelimited[sec];
    }
  } else {
    //Must be no elements, just return the original string which is probably NULL :S
    strippedDpName = dpName;
  }

  //Done.
  return strippedDpName; 
}//fwUkl1_removeSystemName()

/*!
 * Takes the recipe name in the form RUN_MODE/Action and returns the recipe type the recipe should be save as.
 *
 * \param  recipeName The name of the recipe as it is save in the configuration data base.
 * \param  exceptionInfo Error information. If dynlen is zero no errors occurred. The standard error codes (defined in library
 *           documentation) are used.
 * \return string The recipe type that the recipe should be saved under. An empty string is returned
 *           if the recipe type cannot be determined, along with exception information detailing the problem.
 *
 * The recipe type required is determine from the Action part of the recipe name, if this is not present it
 * cannot be determined. The return value can be used directly in any recipe saving function.
 * \n
 * This function works entire from the recipe name and no gauranties are given that it is the actual type.
 * If the recipe is not named correctly or is currently saved under the wrong type this function cannot
 * know this.
 */
string fwUkl1_getRecipeType(const string& recipeName, dyn_string& exceptionInfo) {
  //Holds the recipe type.
  string recipeType = "";
  //Check to see if we have a name.
  if ( "" != recipeName ) {
    //Try and split the recipe name using a `/' as a delimiter.
    dyn_string tmpRunModeAndType = strsplit(recipeName, "/");
    //Check to see we have a name and type.
    const int numEle = dynlen(tmpRunModeAndType);
    if ( 1 < numEle ) {
      //It must have at least two elements for it to be of the appropriate form
      //(taking into account the fwFSMConfDB form with many /).
      //The type is always in the last element regardless of form.
      if ( "Configure" == tmpRunModeAndType[numEle] )
        recipeType = FWUKL1_CONFIGURE_TYPE;
      else if ( ("Start" == tmpRunModeAndType[numEle]) || ("Stop" == tmpRunModeAndType[numEle]) )
        recipeType = FWUKL1_START_TYPE;
      //Only need to mark the error with else as the element is already an empty string as promised.
      else
        fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not determine the recipe type from the recipe name `" + recipeName + "'. The action is believed to be `" + tmpRunModeAndType[2] + "'. No recognisable action present in the name (must be of the form RUN_MODE/Action).", "1");
    }//if(1<numEle)
    else {
      //Can't do anything with either, mark the error and move on. The return values are already initialised to their error states.
      fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not determine the recipe type from the recipe name `" + recipeName + "'. No forward slash, /, present in the name to delimit the run mode and action (must be of the form RUN_MODE/Action).", "1");
    }//else
  }//if(""!=recipeName)
  else {
    fwUkl1Exception_raise(exceptionInfo, "ERROR", "fwUkl1_getRecipeType(): Could not identify recipe type as the recipe name string was empty.", "1");
  }//else(""!=recipeName)
  
  //Done.
  return recipeType;
}//fwUkl1_parseRecipeNameAndIdentifyType()

//@}
