// ======================================
//  NAME    : fwTell1 Library
//
//  VERSION : 1
//
//  REVISION: 0
//
//  RELEASE : 2006/Oct/07
//
//  PLATFORM: PVSS II
//
//  AUTHOR  : Stefan Koestner
//
//  EMAIL   : stefan.koestner@cern.ch
// ======================================


#uses "fwCcpc.ctl"
#uses "fwHw.ctl"
#uses "fwTell1.ctl"



/*! \mainpage fwTell1 Library Documentation
 *
 *  Library of PVSS functions regularly used for the operation of the TELL1-board
 *
 *  Modifications: none
 *
 *  Usage: JCOP framework internal, public
 *
 *  PVSS manager usage: VISION, CTRL
 *
 *  \n
 *  \n
 *  \li \ref SectionConstants
 *  \n
 *  \n
 *  \li \ref SectionArbitrary
 *  \n
 *  \n
 *  \li \ref SectionHardware
*/






// ======================================
//  CONSTANTs & GLOBAL VARs
// ======================================

/** @defgroup SectionConstants Constants
 *  List of constants that can be used by the user.
 *  @{
*/



/** This constant represents nothing.
 *  \n
*/
const int					FWTELL1_NOTHING = 0;



/** This constant represents nothing.
 *  \n
*/
global string				FWTELL1_BOARDTYPE = "NOT_DEF";





/** @} */


//const string			FWCCPC_CONFIG_NAME = "CcpcConfig";
//const string			FWCCPC_PREFIX = "ccpc/";
//const int         FWCCPC_SERVER_VERSION = 220;
//const int					FWCCPC_TIME_OUT = 10;
//global dyn_string fwCcpcList;	
//global int				fwCcpcStateManager;
//global bool				fwCcpcChangeScriptsFlag;







// ======================================
//  ARBITRARY FUNCTIONs
// ======================================

/** @defgroup SectionArbitrary Arbitrary Functions
 *  List of regulary used functions.
 *  @{
*/



/** This function cuts out a bit-array from a hexadecimal string.
 *
 *  \param [in] pHex hexadecimal number (represented as a string) from which an array of bits is cut out
 *  \param [in] pUpper the most significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \param [in] pLower the least significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \return \e the bit array represented as a hexadecimal number (string not integer!)
 *
 *  \b Example: - under construction (returns 0 by default)
 *
 *  \code
 *  (...)
 * // read a register and represent a few adjacent bits out of the full register as a number on the panel
 * dyn_dyn_char data;  
 * dyn_int status;
 * dyn_string registersDP;
 * dynClear(registersDP);
 * dynAppend(registersDP, "dist_1:myTell1_OT.SL.CMNCTRL.SLCtrlReg1.readings"); 
 * fwCcpc_read(registersDP, data, status)
 *  SLCtrlReg1.text = fwTell1_BitsFromString(fwCcpc_convertByteToHex(data[2]),31,24);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string fwTell1_BitsFromString(string pHex, int pUpper, int pLower){
   string casted;
   int number;
   int dummy;
   int mask = 0;
   int i;
 
   number = fwCcpc_convertHexToDec(pHex);
   number = number >> pLower;   
   
   for(i = 0; i <= (pUpper - pLower); i++){
      dummy = 1 << i;
      mask = dummy + mask;
   }
   dummy = number & mask;
   casted = fwCcpc_convertDecToHex(dummy);
   return casted;
}



/** This function cuts out a bit-array from an integer.
 *
 *  \param [in] pNumber an integer number from which an array of bits is cut out
 *  \param [in] pUpper the most significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \param [in] pLower the least significant bit (counted from zero - conform with Guido's ECS docu!)
 *  \return \e the bit array represented as an integer number 
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 * // read a register and represent a few adjacent bits out of the full register as a number on the panel
 * dyn_dyn_char data;  
 * dyn_int status;
 * dyn_string registersDP;
 * dynClear(registersDP);
 * dynAppend(registersDP, "dist_1:myTell1_OT.SL.CMNCTRL.SLCtrlReg1.readings"); 
 * fwCcpc_read(registersDP, data, status)
 *  SLCtrlReg1.text = fwTell1_BitsFromString(fwCcpc_convertByteToDec(data[2]),31,24);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_BitsFromInt(int pNumber, int pUpper, int pLower){
   int dummy;
   int mask = 0;
   int i;
   pNumber = pNumber >> pLower;   
   
   for(i = 0; i <= (pUpper - pLower); i++){
      dummy = 1 << i;
      mask = dummy + mask;
   }
   dummy = pNumber & mask;
   return dummy;  
}






/** This function is used currently by Cedric and will be implemented propperly soon.
 *
 *  \param [in] imput decimal number to be converted.
 *  \param [in] total decimal number to be converted.
 *  \return \e string hexadecimal number.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string result;
 *
 *  result = fwCcpc_convertDecToHex(1127);   // convert decimal number '1127' to the corresponding hexadecimal number
 *  DebugTN("Hexadecimal: " + result);   // output the result of the conversion
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string complStr(string imput, int total)
{	
	 string zeros ;
		if(strlen(imput) != total){
	    for(int h = 1 ; h <= total - strlen(imput); h++){
		       
		    zeros = zeros + "0";
		 }
    }
		
	  string output = zeros + imput;
		
		return output;
}



/** @} */






// ======================================
//  COMMON HARDWARE FUNCTIONs
// ======================================

/** @defgroup SectionHardware Hardware Functions
 *  List of common functions operating with the Tell1 board.
 *  @{
*/



/** TELL1 boards of different subdetectors can be recognised by reading various bits 
 *  of a special register. This function allows to identify the board type without previously
 *  subscribing to this register. The global variable FWTELL1_BOARDTYPE can be used to store the
 *  returned string which is a short acronym for the board type. By default this variable is "NOT_DEF".
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \param [out] status represents the return value of the read function used inside the code.
 *  \return \e string returns the abriations of the board type
 *
 *  \b Example:
 *  
 *  \code
 *  (...)
 *  int status;
 *  if (FWTELL1_BOARDTYPE == "NOT_DEF")
 *     FWTELL1_BOARDTYPE = fwTell1_TypeOfBoard("pctell07",status);
 *  if (status == 0)
 *     DebugN("Reading of register successfull!");
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

string fwTell1_TypeOfBoard(string CCPC, int &status){
   dyn_char data;
   string dummy;
   int address;
   string TypeOfBoard;

   address = fwCcpc_convertHexToDec("4001000");
   status = fwCcpc_LBUSRead(CCPC, address, FWCCPC_LBUS_BITS_32,1, data);
   if(status == 0){
      dummy = fwTell1_BitsFromString(fwCcpc_convertByteToHex(data),7,4);
      //DebugTN("***********dummy:");
      //DebugN(dummy);      
      if(dummy == "1"){
         TypeOfBoard = "VELO";
      }   
      else if(dummy == "2"){
         TypeOfBoard = "ST";
      }
      else if(dummy == "3"){
         TypeOfBoard = "OT";
      }   
      else if(dummy == "4"){
         TypeOfBoard = "CALO";
      }
      else if(dummy == "5"){
         TypeOfBoard = "MUON";
      }
      else if(dummy == "6"){
         TypeOfBoard = "L0MUON";
      }
      else if(dummy == "7"){
         TypeOfBoard = "L0DU";
      } 
      else if(dummy == "8"){
         TypeOfBoard = "L0PUS";
      }
      else if(dummy == "9"){
         TypeOfBoard = "RICH";
      }
      else {
         TypeOfBoard = "NOT_DEF";
      }     
   } else{
        DebugTN("Warning: LBUSRead failed!");
        TypeOfBoard = "ERR";
   }
   //DebugTN(TypeOfBoard);
   return TypeOfBoard;
}








/** This function sets bit Nr.1 in the MDIO-Control register (0x683) of the MAC chip to enable/disable the austoscan feature. 
 *  The flag 'true' (=default) enables the autoscan, 'false' will disable the autoscan.
 *  It checks if the WriteRead function terminated successfully and compares the written bit with the value that was read back.
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \param [in] pFlag to enable or disable autoscan feature of MAC chip.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *  \code
 *  (...)
 *  string CCPC;
 *  flag = true;
 *  //get the ccpc of the Tell1 and enable the autoscan feature
 *  dummy = DeviceUnitDP + ".settings";
 *  dpGet(dummy,CCPC);
 *  fwTell1_enableMDIOCtrl(CCPC,true);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_enableMDIOCtrl(string CCPC, bool pFlag = true){
   int error = 0;
   dyn_char data;
   dyn_char mask;
dyn_char dataRead;
int dummy;
int bit;

   int address = fwCcpc_convertHexToDec("683");
   if(pFlag){
      data = fwCcpc_convertHexToByte("00000002");
      mask = fwCcpc_convertHexToByte("00000002");
   }
   else{
      dyn_char data = fwCcpc_convertHexToByte("00000000");
      dyn_char mask = fwCcpc_convertHexToByte("00000002");
   }  
   
   if(fwCcpc_GBEWriteRead(CCPC, address, 1, data, mask, dataRead )){
      DebugTN("Warning: GBEWrite failed!");
      error |= 1;
   }     
   else{
      error = 0;
   }


 dummy = fwCcpc_convertByteToDec(dataRead);
 bit = fwTell1_BitsFromInt(dummy,1,1);


 DebugTN("MDIO Autoscan - Bit value: ");
 DebugN(bit); 



 if(pFlag){
    if(bit != 1){
        error |= 1;
        DebugTN("Error when reading back register - MDIO Autoscan may be not set!");
    }
 }
 else{ 
   if(bit != 0){
        error |= 1;
        DebugTN("Error when reading back register - MDIO Autoscan may be not disabled!");
   }
 }

   return error;
}





/** This function loads the FPGA from EPC16 (just for TELL1).
 *
 *  \param [in] pCCPC name of the CCPC.
 *  \return \e true no error found.
 *  \return \e false error found.
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  if (fwTell1_EPC16ToFPGA("pclbcecs03") == 0)   // load the FPGA from EPC16
 *     (...)
 *  else
 *     DebugTN("Error found!");
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/
bool fwTell1_EPC16ToFPGA(string pCCPC)
{

	if (_fwCcpc_commonOperation(pCCPC, "5", true) == "-1")
		return false;		// ~OK
	else
		return true;		// OK	
		
}







/** Checks if the .cfg file was written propperly by evaluating a part of a certain register .
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *                                               -1 LBUS access failed
 *                                                or positive value (5-bit number)
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkCFG(ccpc))
 *     DebugTN("FPGAs are not configured propperly!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkCFG(string CCPC){
   int status = 0;
   int error = 0;
   dyn_char data;
   int address;
   int myhex;

   address = fwCcpc_convertHexToDec("1000004");   
   if(fwCcpc_LBUSRead(CCPC, address, FWCCPC_LBUS_BITS_32, 1, data)){
      DebugTN("Warning: LBUSRead failed!");
      return -1;
   }      

   myhex = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 15, 11);
   DebugTN("Check CFG: Returned 5-bit number (hex):");
   DebugN(fwCcpc_convertDecToHex(myhex));
 
   if( myhex == 0x1F){
      //FPGA init OK
      return 0;
   }else{
      //FPGA init NOT OK
      return myhex;
   }
}




/** Checks if the CCPC is running by checking the invalid bit of the 'common.operationStatus' service.
 *
 *  \param [in] CCPC name of the credit card PC.
 *  \return \e int the errorcode of this library: 0 if no error occured 
 *                                                1 requested CCPC was not found
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkCCPC(ccpc))
 *     DebugTN("Checked CCPC is not running!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkCCPC(string CCPC){
 
   //dyn_string tmp0;
   //dyn_string tmp1;
   //dynClear(tmp0);
   //dynClear(tmp1);
   //int i;	
   //CcpcOn = 0
   bool invalidBit;

//a temporary hack to set the invalid bit - later service is published immediately on start up
   
   int a, b, c; 
   fwCcpc_getServerVersion("pctell07", a, b, c); 
   DebugTN(a, b, c); 


   DebugTN("Check if server is connected!");
   dpGet("CCPC/"+CCPC+".common.operationStatus:_original.._invalid",invalidBit);
   //DebugTN(invalidBit);
   if(invalidBit == true){
      DebugTN("Server "+CCPC+" is down!");
      return 1;
   } else{
      DebugTN("Server "+CCPC+" found!");
      return 0;
   }
}
   


/*
   DebugTN("chkCCPC - looking for command:"); 
   if (isFunctionDefined("fwDim_createConfig")){
         fwDim_findCommands(FWCCPC_CONFIG_NAME, FWCCPC_PREFIX  + CCPC + "/CmndCommonOperation", tmp0, tmp1);	
                  DebugN(FWCCPC_CONFIG_NAME, FWCCPC_PREFIX + CCPC + "/CmndCommonOperation");

      if(dynlen(tmp0) == 1){
         DebugTN("ccpc found!");
         return 0;}
      else{
	 DebugTN("ccpc not found!");
         return 1;}	
    }
*/




/** Checks if GBE is operational - currently under construction (returns 0 by default!)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkGBE(){
   int status = 0;
   return status;
}




/** Checks if TTC is operational (fibre is plugged). If fibre is unplugged the I2C interface is pertubated. This function is reading 3-times from the status register. In addition the last two bits from the status register are checked.
 *
 *  \param [in] CCPC name of the CCPC.
 *  \return \e int the errorcode of this library: 0 if no error occured
 *                                                1  clock or data recovery circuit not locked on incoming data stream
 *                                                2  high-resolution phase shifters not working propperly!
 *                                                3  both bits are set
 *                                               -1  I2C-read of status register failed (minimum once out of three trials)
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  string ccpc = "pctell07";
 *
 *  if(fwTell1_chkTTC(ccpc))
 *     DebugTN("WARNING: Fibre might not be plugged on TTC!"
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkTTC(string CCPC){
   int i;
   int error = 0;
   dyn_char data;
   int dllReady;
   //indicates that the high-resolution phase shifters are working propperly
   int pllReady;
   //indicates that the clock and data recovery circuit is locked on the incoming data stream
   
   //int address = fwCcpc_convertHexToDec("58");
   int address = 0x58;
   for(i=1; i <= 3; i++){
      if(fwCcpc_I2CReadSub(CCPC, 0, address, 22, 1, -1, FWCCPC_I2C_SEPARATED, 0, data)){
         DebugN("Warning: TTCRead failed!");
         error += 1;
      }     
   }
   if(error){
      DebugTN("Could not read I2C interface - check clock and fibre of TTCRx!");
      return -1;
   }
      
   error = 0;
   pllReady = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 7, 7);
   dllReady = fwTell1_BitsFromInt(fwCcpc_convertByteToDec(data), 6, 6);
   if((pllReady == 0)){
      DebugTN("TTC: Clock or data recovery circuit not locked on incoming data stream!");
      error += 1;
   }
   if(dllReady==0){
      DebugTN("TTC: High-resolution phase shifters not working propperly!");
      error += 2;
   }
   return error;
}





/** Checks if ORx is operational - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkORx(){
   int status = 0;
   return status;
}




/** Checks if ARx is operational - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/

int fwTell1_chkARx(){
   int status = 0;
   return status;
}









/** Checks if processing on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/


int fwTell1_chkProcessing(){
   int status = 0;
   return status;
}




/** Checks if data flow on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkFlow(){
   int status = 0;
   return status;
}



/** Checks if memory on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkMemory(){
   int status = 0;
   return status;
}



/** Checks if data on Tell1 board is ok  - under construction (returns 0 by default)
 *
 *  \return \e int the errorcode of this library: 0 if no error occured
 *
 *  \b Example:
 *
 *  \code
 *  (...)
 *  dyn_string myList;
 *  int        i;
 *
 *  fwCcpc_get(myList);   // get a list of CCPC servers running in the moment
 *  DebugTN("CCPC servers: " + dynlen(myList));   // display number of CCPC servers running in the moment
 *  for(i = 1; i <= dynlen(myList); i++)
 *     DebugTN(myList[i]);
 *  (...)
 *  \endcode
 *
 *  \n
 *  \n
*/



int fwTell1_chkData(){
   int status = 0;
   return status;
}






/** @}*/










// ======================================
//  INTERNAL FUNCTIONs (NOT TO BE USED)
// ======================================


/*
string _fwCcpc_getConfigName()
{

	return FWCCPC_CONFIG_NAME;
	
}




string _fwCcpc_getPrefix()
{

	return FWCCPC_PREFIX;
	
}
*/