//
// PC host application to transmit and receive USB frames to the C8051
// device SPItoUSB interface.  This application is based on the SiLabs
// USBExpress library
//

#include <stdio.h>
#include <windows.h>
#include "SiUSBXp.h"
#include "chimaera.h"

// Enumerate, locate and open device

SI_STATUS openDevice( char *deviceType, unsigned idx, HANDLE *deviceHandle )
{
SI_STATUS rc;
unsigned found = 0;

// Find how many devices are attached

DWORD NumDevices;

  rc = SI_GetNumDevices( &NumDevices );
  if ( rc != SI_SUCCESS )
  {
    printf( "Si_GetNumDevices(): rc = %d\n", rc );
    return rc;
  }

  // Identify the correct device

DWORD deviceIndex;
char deviceString[256];

  for (deviceIndex=0; deviceIndex<NumDevices; deviceIndex++ )
  {
    rc = SI_GetProductString( deviceIndex, deviceString, SI_RETURN_DESCRIPTION );
    if ( rc != SI_SUCCESS )
    {
      printf( "Si_GetProductString(): rc = %d\n", rc );
      return rc;
    }
    printf( "SI_GetProductString(): [%d] [%s] rc = %d\n", deviceIndex, deviceString, rc );
    if ( !strcmp( deviceString,deviceType ) )
    {
      if ( found == idx ) break;
      found++;
    }
  }

  if ( deviceIndex == NumDevices )
  {
    printf( "Device not found\n" );
    return SI_DEVICE_NOT_FOUND;
  }

// Open the device

  rc = SI_Open( deviceIndex, deviceHandle );
  if ( rc != SI_SUCCESS )
  {
    printf( "Si_Open(): rc = %d\n", rc );
    return rc;
  }

// Set the read and write timeout to 5s

  rc = SI_SetTimeouts(5000,5000);
  if ( rc != SI_SUCCESS )
  {
    printf( "Si_SetTimeouts(): rc = %d\n", rc );
    return rc;
  }

  return SI_SUCCESS;

}

// Close device

SI_STATUS closeDevice( HANDLE deviceHandle )
{
SI_STATUS rc;

  rc = SI_Close( deviceHandle );
  if ( rc != SI_SUCCESS )
  {
    printf( "Si_Close(): rc = %d\n", rc );
    return rc;
  }
  return rc;
}

// Send command to device

unsigned requestDevice( HANDLE deviceHandle, devicePacket_t &packet )
{

SI_STATUS rc;
DWORD len;

// Send request

  rc = SI_Write( deviceHandle, &packet, 8, &len );
  if ( rc != SI_SUCCESS )
  {
    printf( "requestDevice(): Si_Write(): rc = %d. cmd = %#x\n", rc, packet.cmd );
    return 0;
  }
  if ( len != 8 )
  {
    printf( "requestDevice(): Si_Write(): Data under-run error. len = %d. cmd = %#x\n", len, packet.cmd );
    return 0;
  }

  return 1;

}

// Read the block of data and send acknowledge to device

SI_STATUS readDeviceData( HANDLE deviceHandle, devicePacket_t &buf, unsigned nBytes )
{

SI_STATUS rc;
DWORD len;

// Read data

  rc = SI_Read( deviceHandle, (BYTE *)&buf, nBytes, &len );
  if ( rc != SI_SUCCESS )
  {
    printf( "readDeviceData(): Si_Read(): rc = %d\n", rc );
    return rc;
  }
  if ( len != nBytes )
  {
    printf( "readDeviceData(): Si_Read(): Data under-run error. len = %d\n", len );
    return 0;
  }

  //  printf( "readDeviceData(): Received %d data bytes\n", len );

  return SI_SUCCESS;

}

// Read the status block sent from device

void showLastStatus( devicePacket_t &packet )
{

  printf( "showLastStatus(): cmd=%#x, len=%#x\n", packet.cmd, packet.length );
  printf( "PIO[2,1,0] = [%#x,%#x,%#x]\n", packet.status.PIO[2],packet.status.PIO[1],packet.status.PIO[0] );
  printf( "Sequence # = %d\n", packet.status.seq );
  printf( "Parity errors[0]: %#x\n", packet.status.parityErrorCount[0] );
  printf( "Parity errors[1]: %#x\n", packet.status.parityErrorCount[1] );
  printf( "Parity errors[2]: %#x\n", packet.status.parityErrorCount[2] );
  printf( "Parity errors[3]: %#x\n", packet.status.parityErrorCount[3] );
  printf( "Event counter: %#x\n", packet.status.eventCount );
  printf( "Buffer counter: %#x\n", packet.status.bufferCount );
  printf( "Rx error flags: %#x\n", packet.status.rxerr );
  printf( "Rx inhibit flags: %#x\n", packet.status.rxinhibit );
  printf( "Extra: %#x\n", packet.status.extra );
  printf( "ADC[0]: %#x\n", packet.status.ADC[0] );
  printf( "ADC[1]: %#x\n", packet.status.ADC[1] );


  return;

}

void showLastData( devicePacket_t &packet )
{
  for (unsigned i=0; i<16 ; i++ )
  {
    if ( i%4 == 0 )  printf( "[%0#4x]", i );
    printf( " %0#10x", packet.data[i] );
    if ( i%4 == 3 ) printf( "\n" );
  }
  return;
}

unsigned requestDeviceConfiguration( HANDLE deviceHandle, devicePacket_t &msg )
{

SI_STATUS rc;
DWORD len;

 printf( "Configuration data = %#x/%#x\n", msg.config[0], msg.config[1] );

// Send header

  msg.cmd = deviceHeader_t::ConfigurationData;
  msg.length = 4;
  rc = SI_Write( deviceHandle, &msg, 8, &len );
  if ( rc != SI_SUCCESS )
  {
    printf( "requestDeviceConfiguration(): Si_Write(): rc = %d.\n", rc );
    return 0;
  }
  if ( len != 8 )
  {
    printf( "requestDeviceConfiguration(): Si_Write(): Data under-run error. len = %d.\n", len );
    return 0;
  }

  return 1;

}

// Request write to IO Port

unsigned requestPIOWrite( HANDLE deviceHandle, unsigned PIO )
{

SI_STATUS rc;
DWORD len;
devicePacket_t packet;


// Send request

  packet.cmd = deviceHeader_t::PIOWriteRequest;
  packet.length = 4;
  packet.data[0] = PIO;
  rc = SI_Write( deviceHandle, &packet, 8, &len );
  if ( rc != SI_SUCCESS )
  {
    printf( "requestPIOWrite(): Si_Write(): rc = %d\n", rc );
    return 0;
  }
  if ( len != 8 )
  {
    printf( "requestPIOWrite(): Si_Write(): Data under-run error. len = \n", len );
    return 0;
  }

  return 1;

}

// Flush the Chimaera buffer

unsigned flushBuffer(  HANDLE deviceHandle, unsigned maxPacketRequest, bool show )
{
unsigned packetsRequested;
devicePacket_t cmdPacket;
devicePacket_t lastStatus;
devicePacket_t lastData;

// Request status to find how much (if any) data are buffered

  cmdPacket.cmd = deviceHeader_t::StatusRequest;
  SI_STATUS rc = requestDevice( deviceHandle, cmdPacket );
  rc = readDeviceData( deviceHandle, lastStatus, 68 );

  if ( lastStatus.status.bufferCount == 0 )
  {
    return 0;
  }

  unsigned packets = (lastStatus.status.bufferCount >> 4) + 1;

// Request flush

  cmdPacket.cmd = deviceHeader_t::FlushRequest;
  rc = requestDevice( deviceHandle, cmdPacket );

// Extract all the data from the buffer

  for ( unsigned packet=0; packet<packets; packet+=packetsRequested )
  {

    unsigned packetsRemaining = packets - packet;
    if ( packetsRemaining >= maxPacketRequest )
    {
      packetsRequested = maxPacketRequest;
    }
    else
    {
      packetsRequested = packetsRemaining;
    }

    cmdPacket.cmd = deviceHeader_t::PacketRequest;
    cmdPacket.uc[0] = packetsRequested; // Number of packets
    requestDevice( deviceHandle, cmdPacket );
    for ( unsigned char nread=packetsRequested; nread; nread-- )
    {
      readDeviceData( deviceHandle, lastData, 68 );
      if ( show ) showLastData( lastData );
      // Write data to file here
    }
  }

  return lastStatus.status.bufferCount;
}


