//-----------------------------------------------------------------------------
// bounce.c
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <c8051f320.h>        
#include <stddef.h>       
#include "USB_API.h"
#include "chimaera.h"

//-----------------------------------------------------------------------------
// Global CONSTANTS
//-----------------------------------------------------------------------------

sbit P20 = P2^0;
sbit P21 = P2^1;
sbit Led1 = P2^2;                         // LED='1' means OFF
sbit Led2 = P2^3;

code const BYTE VendorString[] = {0x2a,0x03,'C',0,'a',0,'v',0,'e',0,'n',0,'d',0,'i',0,'s',0,'h',0,' ',0,'L',0,'a',0,'b',0,'.',0,' ',0,'(',0,'H',0,'E',0,'P',0,')',0};
code const BYTE ProductString[] = {0xe,0x03,'C',0,'B',0,' ',0,'U',0,'S',0,'B',0};
code const BYTE Serial[] =	{0x0A,0x03,'0',0,'0',0,'0',0,'1',0};
code const BYTE MaxPower = 15;            // Max current = 16 mA (8 * 2)
code const BYTE PwAttributes = 0x80;      // Self-powered, remote wakeup not supported
code const UINT bcdDevice = 0x100;        // Device release number 1.00
code const BYTE TEMP_ADD = 112;           // This constant is added to Temperature

unsigned short nByte;
devicePacket_t packet;
devicePacketISR_t packet_isr;
unsigned rxInterrupt = 0;
unsigned txInterrupt = 0;
unsigned waitUSB = 1;
unsigned short cnt = 0;

//-----------------------------------------------------------------------------
// Main Routine
//-----------------------------------------------------------------------------
void main(void) 
{
  unsigned rc;
  unsigned char packetsRemaining;
  unsigned char packetPending = 0;
  unsigned char seq = 0;
  unsigned char txSlots = 2;

   PCA0MD &= ~0x40;                       // Disable Watchdog timer


   USB_Init(0xCBCB,0x0001,VendorString,ProductString,Serial,MaxPower,PwAttributes,bcdDevice);
   OSCICN |= 0x3; // System clock divide. Set for fastest system clock speed.
   Initialize();
   USB_Int_Enable();

   P0MDIN   = 0xFF;    // P2.0-7 set digital input
   P0MDOUT  = 0x00;    // P2.2,3 set push-pull (LEDs)
   P0       = ~0x00;   // Open-drain outputs set high impedance
   P1MDIN   = 0xFF;    // P2.0-7 set digital input
   P1MDOUT  = 0x00;    // P2.2,3 set push-pull (LEDs)
   P1       = ~0x00;   // Open-drain outputs set high impedance
   P2MDIN   = 0xFF;    // P2.0-7 set digital input
   P2MDOUT  = 0x0F;    // P2.2,3 set push-pull (LEDs)
   P2       = ~0x0F;   // Open-drain outputs set high impedance

   XBR1     = 0x40;    // Enable Crossbar

// Wait for the USB to be ready

   Led1 = 1;
   Led2 = 1;

   while ( waitUSB == 1 ) ;

   Led1 = 0;

// Continuous transaction loop

  while ( 1 )
  {

    if ( txInterrupt == 1 )
    {
      txInterrupt = 0;
      txSlots++;
    }

    if ( (packetPending == 1) && (txSlots > 0) )
    {
      txSlots--;
      rc = Block_Write( (BYTE *)&packet, 68 );
      packetPending = 0;
    }

    if ( rxInterrupt == 1 )
    {
      rxInterrupt = 0;
      packetsRemaining = packet_isr.cdata[0];
    }


    if ( (packetPending == 0) && (packetsRemaining > 0) )
    {

      P20 = 1;

      for ( nByte=0; nByte < 64; nByte++ )
      {
        packet.cdata[nByte] = (nByte+seq)&0xff;
      }

      packet.cmd = Data;
      packet.length.uc[0] = 64;
      packet.length.uc[1] = 0;

      P20 = 0;

      packetsRemaining--;
      seq++;

      packetPending = 1;

    }

   }

}

//-----------------------------------------------------------------------------
// Initialization Subroutines
//-----------------------------------------------------------------------------

void Initialize(void)
{

  return;
}

//-------------------------
// Suspend_Device
//-------------------------
// Called when a DEV_SUSPEND interrupt is received.
// - Calls USB_Suspend()
//
void Suspend_Device(void)
{
// Don't disable peripherals. Device is self-powered.

   USB_Suspend();                       // Put the device in suspend state
}


// USB ISR

void USB_Device_ISR(void) interrupt 16
{
  BYTE INTVAL = Get_Interrupt_Source();

  if (INTVAL & RX_COMPLETE)
  {
    Block_Read( (BYTE *)&packet_isr, 8 );
    switch ( packet_isr.cmd )
    {
    case PacketRequest:       // Request next data packet
      rxInterrupt = 1;
      break;
    default:
      break;
    }
 
  }


  if (INTVAL & TX_COMPLETE)
  {
    txInterrupt = 1;
  }

  if (INTVAL & DEV_SUSPEND)
  {
    Suspend_Device();
  }

  if (INTVAL & DEV_CONFIGURED)
  {
    waitUSB = 0;
  }

}
