/*
 * @(#)TextureAnim.java	1.5 99/04/23
 *
 * Copyright (c) 1997-1999 by Sun Microsystems, Inc. All Rights Reserved.
 * 
 * Sun grants you ("Licensee") a non-exclusive, royalty free, license to use,
 * modify and redistribute this software in source and binary code form,
 * provided that i) this copyright notice and license appear on all copies of
 * the software; and ii) Licensee does not utilize the software in a manner
 * which is disparaging to Sun.
 * 
 * This software is provided "AS IS," without a warranty of any kind. ALL
 * EXPRESS OR IMPLIED CONDITIONS, REPRESENTATIONS AND WARRANTIES, INCLUDING ANY
 * IMPLIED WARRANTY OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE OR
 * NON-INFRINGEMENT, ARE HEREBY EXCLUDED. SUN AND ITS LICENSORS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING, MODIFYING
 * OR DISTRIBUTING THE SOFTWARE OR ITS DERIVATIVES. IN NO EVENT WILL SUN OR ITS
 * LICENSORS BE LIABLE FOR ANY LOST REVENUE, PROFIT OR DATA, OR FOR DIRECT,
 * INDIRECT, SPECIAL, CONSEQUENTIAL, INCIDENTAL OR PUNITIVE DAMAGES, HOWEVER
 * CAUSED AND REGARDLESS OF THE THEORY OF LIABILITY, ARISING OUT OF THE USE OF
 * OR INABILITY TO USE SOFTWARE, EVEN IF SUN HAS BEEN ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGES.
 * 
 * This software is not designed or intended for use in on-line control of
 * aircraft, air traffic, aircraft navigation or aircraft communications; or in
 * the design, construction, operation or maintenance of any nuclear
 * facility. Licensee represents and warrants that it will not use or
 * redistribute the Software for such purposes.
 */

package demos.Paint;
    
import java.awt.*;
import java.awt.event.*;
import java.awt.image.BufferedImage;
import javax.swing.*;
import AnimatingContext;
import DemoSurface;
import DemoPanel;
import CustomControls;


/**
 * TexturePaint animation with controls for transformations.
 */
public class TextureAnim extends DemoSurface implements AnimatingContext, CustomControls {

    public static final Color colorblend = new Color(0f, 0f, 1f, .5f);
    private BufferedImage bimg;
    private Rectangle tilerect;
    private TexturePaint texture;
    private boolean newtexture;
    private boolean bouncesize = false;
    private boolean bouncerect = true;
    private boolean rotate = false;
    private boolean shearx = false;
    private boolean sheary = false;
    private boolean showanchor = true;
    private boolean quality = false;
    private int tilesize;
    private AnimVal w, h, x, y, rot, shx, shy;
    private DemoControls controls;


    public TextureAnim() {
        makeImage(32);
        tilesize = bimg.getWidth();
        w = new AnimVal(0, 200, 3, 10, tilesize);
        h = new AnimVal(0, 200, 3, 10, tilesize);
        x = new AnimVal(0, 200, 3, 10, 0);
        y = new AnimVal(0, 200, 3, 10, 0);
        rot = new AnimVal(-360, 360, 5, 15, 0);
        shx = new AnimVal(-50, 50, 3, 10, 0);
        shy = new AnimVal(-50, 50, 3, 10, 0);
        tilerect = new Rectangle(x.getInt(), y.getInt(),
                                 w.getInt(), h.getInt());
        texture = new TexturePaint(bimg, tilerect);
        controls = new DemoControls(this);
    }


    public String[] getCustomControlsConstraints() {
        return new String[] { BorderLayout.NORTH };
    }


    public Component[] getCustomControls() {
        return new Component[] { (Component) controls };
    }


    public void customControlsThread(int state) {
        if (state == CustomControls.START) {
            controls.start();
        } else if (state == CustomControls.STOP) {
            controls.stop();
        }
    }



    public void makeImage(int size) {
        bimg = new BufferedImage(size, size, bimg.TYPE_INT_BGR);
        Graphics2D g2d = bimg.createGraphics();
        g2d.setColor(Color.white);
        g2d.fillRect(0, 0, size, size);
        for (int j = 0; j < size; j++) {
            float red = j / (float) size;
            for (int i = 0; i < size; i++) {
                float green = i / (float) size;
                g2d.setColor(new Color(1.0f - red, 1.0f - green, 0.0f, 1.0f));
                g2d.drawLine(i, j, i, j);
            }
        }
        newtexture = true;
    }


    public void reset(int width, int height) {
        x.newlimits(-width/4, width/4 - w.getInt());
        y.newlimits(-height/4, height/4 - h.getInt());
    }


    public void step(int width, int height) {
        if (tilesize != bimg.getWidth()) {
            tilesize = bimg.getWidth();
        }
        if (bouncesize) {
            w.anim();
            h.anim();
            x.newlimits(-width/4, width/4 - w.getInt());
            y.newlimits(-height/4, height/4 - h.getInt());
        } else {
            if (w.getInt() != tilesize) {
                w.set(tilesize);
                x.newlimits(-width/4, width/4 - w.getInt());
            }
            if (h.getInt() != tilesize) {
                h.set(tilesize);
                y.newlimits(-height/4, height/4 - h.getInt());
            }
        }
        if (bouncerect) {
            x.anim();
            y.anim();
        }
        if (newtexture ||
            x.getInt() != tilerect.x || y.getInt() != tilerect.y ||
            w.getInt() != tilerect.width || h.getInt() != tilerect.height)
        {
            newtexture = false;
            int X = x.getInt();
            int Y = y.getInt();
            int W = w.getInt();
            int H = h.getInt();
            tilerect = new Rectangle(X, Y, W, H);
            texture = new TexturePaint(bimg, tilerect);
        }
    }



    public void drawDemo(int width, int height, Graphics2D g2) {

        g2.translate(width/2, height/2);
        if (rotate) {
            rot.anim();
            g2.rotate(Math.toRadians(rot.getFlt()));
        } else {
            rot.set(0);
        }
        if (shearx) {
            shx.anim();
            g2.shear(shx.getFlt()/100, 0.0f);
        } else {
            shx.set(0);
        }
        if (sheary) {
            shy.anim();
            g2.shear(0.0f, shy.getFlt()/100);
        } else {
            shy.set(0);
        }
        g2.setPaint(texture);
        g2.fillRect(-1000, -1000, 2000, 2000);
        if (showanchor) {
            g2.setColor(Color.black);
            g2.setColor(colorblend);
            g2.fill(tilerect);
        }
    }


    public static void main(String argv[]) {
        final DemoPanel dp = new DemoPanel(new TextureAnim());
        JFrame f = new JFrame("Java2D Demo - TexureAnim");
        f.addWindowListener(new WindowAdapter() {
            public void windowClosing(WindowEvent e) {System.exit(0);}
            public void windowDeiconified(WindowEvent e) { 
                dp.surface.start(); 
            }
            public void windowIconified(WindowEvent e) { 
                dp.surface.stop(); 
            }
        });
        f.getContentPane().add("Center", dp);
        f.pack();
        f.setSize(new Dimension(400,300));
        f.show();
        dp.surface.start();
    }

    static class AnimVal {
        float curval;
        float lowval;
        float highval;
        float currate;
        float lowrate;
        float highrate;

        public AnimVal(int lowval, int highval,
                       int lowrate, int highrate) {
            this.lowval = lowval;
            this.highval = highval;
            this.lowrate = lowrate;
            this.highrate = highrate;
            this.curval = randval(lowval, highval);
            this.currate = randval(lowrate, highrate);
        }

        public AnimVal(int lowval, int highval,
                       int lowrate, int highrate,
                       int pos) {
            this(lowval, highval, lowrate, highrate);
            set(pos);
        }

        public float randval(float low, float high) {
            return (float) (low + Math.random() * (high - low));
        }

        public float getFlt() {
            return curval;
        }

        public int getInt() {
            return (int) curval;
        }

        public void anim() {
            curval += currate;
            clip();
        }

        public void set(float val) {
            curval = val;
            clip();
        }

        public void clip() {
            if (curval > highval) {
                curval = highval - (curval - highval);
                if (curval < lowval) {
                    curval = highval;
                }
                currate = - randval(lowrate, highrate);
            } else if (curval < lowval) {
                curval = lowval + (lowval - curval);
                if (curval > highval) {
                    curval = lowval;
                }
                currate = randval(lowrate, highrate);
            }
        }

        public void newlimits(int lowval, int highval) {
            this.lowval = lowval;
            this.highval = highval;
            clip();
        }
    }  // End AnimVal class


    static class DemoControls extends JPanel implements ActionListener, Runnable {

        TextureAnim demo;
        JToolBar toolbar;
        JComboBox combo;
        Thread thread;

        public DemoControls(TextureAnim demo) {
            this.demo = demo;
            setBackground(Color.gray);
            add(toolbar = new JToolBar());
            toolbar.setFloatable(false);
            addTool("BO", "bounce", true);
            addTool("SA", "show anchor", true);
            addTool("RS", "resize", false);
            addTool("RO", "rotate", false);
            addTool("SX", "shear x", false);
            addTool("SY", "shear y", false);
            add(combo = new JComboBox());
            combo.addActionListener(this);
            combo.addItem("8");
            combo.addItem("16");
            combo.addItem("32");
            combo.addItem("64");
            combo.setSelectedIndex(2);
            addMouseListener(new MouseAdapter() {
                public void mouseClicked(MouseEvent e) {
                    if (thread == null) start(); else stop();
                }
            });
        }


        public void addTool(String str, String toolTip, boolean state) {
            JButton b = (JButton) toolbar.add(new JButton(str));
            b.setBackground(state ? Color.green : Color.lightGray);
            b.setSelected(state);
            b.setToolTipText(toolTip);
            b.addActionListener(this);
        }


        public void actionPerformed(ActionEvent e) {
            if (e.getSource() instanceof JComboBox) {
                int size = Integer.parseInt((String) combo.getSelectedItem());
                demo.makeImage(size);
                return;
            }
            JButton b = (JButton) e.getSource();
            b.setSelected(!b.isSelected());
            b.setBackground(b.isSelected() ? Color.green : Color.lightGray);
            if (b.getText().equals("BO")) {
                demo.bouncerect = b.isSelected();
            } else if (b.getText().equals("SA")) {
                demo.showanchor = b.isSelected();
            } else if (b.getText().equals("RS")) {
                demo.bouncesize = b.isSelected();
            } else if (b.getText().equals("RO")) {
                demo.rotate = b.isSelected();
            } else if (b.getText().equals("SX")) {
                demo.shearx = b.isSelected();
            } else if (b.getText().equals("SY")) {
                demo.sheary = b.isSelected();
            }
        }

        public Dimension getPreferredSize() {
            return new Dimension(200,37);
        }


        public void start() {
            if (thread != null) {
                return;
            }
            thread = new Thread(this);
            thread.setPriority(Thread.MIN_PRIORITY);
            thread.setName("Paint.TextureAnim DemoControls Thread");
            thread.start();
        }


        public synchronized void stop() {
            if (thread != null) {
                thread.interrupt();
            }
            thread = null;
            notifyAll();
        }


        public void run() {
            Thread me = Thread.currentThread();
            while (thread == me) {
                for (int i = 2; i < toolbar.getComponentCount(); i++) {
                    try {
                        thread.sleep(4444);
                    } catch (InterruptedException e) { return; }
                    ((JButton) toolbar.getComponentAtIndex(i)).doClick();
                }
            }
            thread = null;
        }
    } // End DemoControls class
} // End TextureAnim class
