/*

A display class that implements persistence

Steve Wotton, 2000
Cavendish Lab (HEP)

*/
package cbsw.pbw;

import java.awt.*;
import java.awt.event.*;
import java.awt.geom.*;
import java.awt.Graphics2D.*;
import java.awt.image.*;
import java.awt.print.*;
import javax.print.attribute.*;
import javax.print.attribute.standard.*;
import java.net.*;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.text.html.*;
import javax.swing.border.*;
import java.util.*;
import java.util.Date;
import java.text.DateFormat;
import cbsw.pbw.*;

public class PixieBoogieWoogie extends JPanel implements MouseMotionListener, MouseListener, ActionListener, ChangeListener, Printable
{

JPopupMenu popupMenu;
ButtonGroup rotateGroup = new ButtonGroup();
JColorChooser colorChooser = new JColorChooser();
BufferedImage bi;
Graphics2D big2d;
Color fgColor = Color.yellow;
Color bgColor = Color.blue;
Color gridColor = Color.black;
BasicStroke gridStroke = new BasicStroke(2.0f);
int pixelHeight, pixelWidth, nxpixels, nypixels;
float persistence;
float saturationCount;
int quadrantRotation = 0;
boolean adaptiveDisplay;
boolean printMono = false;

// Intensity of each pixel. This increments by one for each new
// hit on the pixel and decays with time.

float[][] intensity;
// Coordinates of pixel having highest intensity
int xPixelMax = 0, yPixelMax = 0;

static final int defaultPersistence = 100;
static final int defaultSaturation = 1;
// If adaptiveDisplay is set true, the pixel brightness will scale automatically so the the brightest pixel just saturates.
static final boolean defaultAdaptiveDisplay = false;

AffineTransform atg2d;

public PixieBoogieWoogie( int nx, int ny, int width, int height )
{

   nxpixels = nx;
   nypixels = ny;
   pixelWidth = width;
   pixelHeight = height;

   persistence = ((float)defaultPersistence)/100.0f;
   saturationCount = (float)defaultSaturation;

   intensity = new float[nx][ny];

   adaptiveDisplay = defaultAdaptiveDisplay;

   setOpaque( true );
   //   setBackground( new Color(75,200,150) );

   bi = new BufferedImage(nx*pixelWidth,ny*pixelHeight,BufferedImage.TYPE_INT_RGB);
   big2d = bi.createGraphics();
   big2d.clearRect( 0, 0, nxpixels*pixelWidth, nypixels*pixelHeight );
   atg2d = big2d.getTransform();

   big2d.setStroke(gridStroke);

   AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   big2d.setComposite(ac);

   drawBackground( big2d, bgColor );
   drawGrid( big2d, gridColor );
   //   draw();

   setPreferredSize( new Dimension(nx*width,ny*height) );
   //   setBorder( BorderFactory.createTitledBorder("Image area") );

   popupMenu = new JPopupMenu();
   makePopupMenu( popupMenu, this );

   addMouseListener( this );
   addMouseMotionListener( this );


}

public void makePopupMenu( JPopupMenu m, ActionListener l )
{

// Colour buttons

// Make "Print" menuitem

        JMenuItem menuItem = new JMenuItem( "Print..." );
        menuItem.addActionListener( this );
        menuItem.setActionCommand( "print" );

        m.add( menuItem );

        m.addSeparator();

        menuItem = m.add( "Foreground colour..." );
        menuItem.setActionCommand("fgcolor");
        menuItem.addActionListener( l );

        menuItem = m.add( "Background colour..." );
        menuItem.setActionCommand("bgcolor");
        menuItem.addActionListener( l );

        menuItem = m.add( "Grid colour..." );
        menuItem.setActionCommand("gridcolor");
        menuItem.addActionListener( l );

        m.addSeparator();

        JMenu rotateMenu = new JMenu( "Rotation" );

        JRadioButtonMenuItem rotateButton = new JRadioButtonMenuItem( "0 deg" );
        rotateButton.addActionListener(this);
        rotateButton.setActionCommand("rotate0");
        rotateButton.setSelected( true );
        rotateMenu.add( rotateButton );
        rotateGroup.add( rotateButton );
        rotateButton = new JRadioButtonMenuItem( "90 deg" );
        rotateButton.addActionListener(this);
        rotateButton.setActionCommand("rotate90");
        rotateMenu.add( rotateButton );
        rotateGroup.add( rotateButton );
        rotateButton = new JRadioButtonMenuItem( "180 deg" );
        rotateButton.addActionListener(this);
        rotateButton.setActionCommand("rotate180");
        rotateMenu.add( rotateButton );
        rotateGroup.add( rotateButton );
        rotateButton = new JRadioButtonMenuItem( "270 deg" );
        rotateButton.addActionListener(this);
        rotateButton.setActionCommand("rotate270");
        rotateMenu.add( rotateButton );
        rotateGroup.add( rotateButton );

        m.add( rotateMenu );

   return;
}

// MouseMotionListener implementation

public void mouseDragged( MouseEvent e ){ return; }


public void mouseMoved( MouseEvent e )
{
   int x = e.getX();
   int y = e.getY();
   int nx = x/pixelWidth;
   int ny = y/pixelHeight;

   if ( nx < 0 || nx >= nxpixels || ny < 0 || ny >= nypixels ) return;

   //   applet.showStatus( "(x,y)=(" + nx + "," + ny + ") intensity=" + intensity[nx][ny] );

   return;
}

// MouseListener implementation

    public void mousePressed( MouseEvent e )
    {
	//       applet.showStatus( "Pressed x:" + e.getX() + " y:" + e.getY() );
       if ( e.isPopupTrigger() )
       {
	   popupMenu.show( e.getComponent(), e.getX(), e.getY() );
       }
       return;
    }

    public void mouseReleased( MouseEvent e )
    {
	//       applet.showStatus( "Released x:" + e.getX() + " y:" + e.getY() );
       if ( e.isPopupTrigger() )
       {
	   popupMenu.show( e.getComponent(), e.getX(), e.getY() );
       }
       return;
    }

    public void mouseClicked( MouseEvent e )
    {
	//	applet.showStatus( "Clicked x:" + e.getX() + " y:" + e.getY() );
       return;
    }

    public void mouseEntered( MouseEvent e )
    {
	//	applet.showStatus( "Entered x:" + e.getX() + " y:" + e.getY() );
       return;
    }

    public void mouseExited( MouseEvent e )
    {
	//	applet.showStatus( "Exited x:" + e.getX() + " y:" + e.getY() );
       return;
    }


// ActionListener implementation

public void actionPerformed( ActionEvent e )
{
   String cmd = e.getActionCommand();

  if ( cmd.equals( "rotate0" ) )
  {
     setRotation( 0 );
  }
  else if ( cmd.equals( "rotate90" ) )
  {
     setRotation( 1 );
  }
  else if ( cmd.equals( "rotate180" ) )
  {
     setRotation( 2 );
  }
  else if ( cmd.equals( "rotate270" ) )
  {
     setRotation( 3 );
  }
  else if ( cmd.equals( "clear" ) )
  {
     clear();
  }
  else if ( cmd.equals( "fgcolor" ) || cmd.equals( "bgcolor" ) || cmd.equals( "gridcolor" ) )
  {
     setColor( cmd );
  }
   else if ( cmd.equals( "print" ) )
   {   
      PrinterJob printJob = PrinterJob.getPrinterJob();
      PrintRequestAttributeSet printAttributes = new HashPrintRequestAttributeSet();
      printAttributes.add(MediaSizeName.ISO_A4);
      printAttributes.add(OrientationRequested.PORTRAIT);
      //      printAttributes.add(Chromaticity.MONOCHROME);
      printAttributes.add(new JobName("PBW",null));
      //      printJob.pageDialog( printAttributes );
      if (printJob.printDialog(printAttributes))
      {
	  try
	  {
            Chromaticity chrom = (Chromaticity)printAttributes.get(Chromaticity.class);
            if ( chrom != null )
            {
	      if ( chrom.equals( Chromaticity.COLOR ) )
	      {
	        printMono = false;
	      }
              else
	      {
	        printMono = true;
	      }
            }
	    printJob.setPrintable(this);
	    printJob.print(printAttributes);  
	  }
	  catch (PrinterException pe)
	  {
            System.err.println(pe);
	  }
      }
   }
  return;
}

// Change listener implementation

public void stateChanged( ChangeEvent e )
{
   JSlider source = (JSlider)e.getSource();
   String name = source.getName();
   int value = source.getValue();
   if ( name.equals("persistence") )
   {
      setPersistence( value );
   }
   else if ( name.equals("saturation") )
   {
      setSaturationCount( value );
   }
   source.setToolTipText( Integer.toString(value) );
}

public void drawGrid( Graphics2D g, Color gc )
{
   AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   g.setComposite(ac);

   g.setColor(gc);
   for ( int x=0; x<nxpixels; x++ )
   {
      for ( int y=0; y<nypixels; y++ )
      {
	 g.drawRect(x*pixelWidth, y*pixelHeight, pixelWidth, pixelHeight);
      }
   }
   return;
}

public void drawBackground( Graphics2D g, Color bc )
{
   AlphaComposite ac = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   g.setComposite(ac);

   g.setColor(bc);
   g.fillRect( 0, 0, nxpixels*pixelWidth, nypixels*pixelHeight);

   return;
}

public void refresh()
{

   if ( !adaptiveDisplay )
   {
     for ( int x=0; x<nxpixels; x++ )
     {
       for ( int y=0; y<nypixels; y++ )
       {
	 intensity[x][y] = persistence * intensity[x][y];
       }
     }
   }

   draw();
   repaint();

   return;
}

public void clear()
{

   for ( int x=0; x<nxpixels; x++ )
   {
      for ( int y=0; y<nypixels; y++ )
      {
	  intensity[x][y] = 0.0f;
      }
   }

   drawBackground( big2d, bgColor );
   drawGrid( big2d, gridColor );

   return;
}

public void setPersistence( int val )
{
  persistence = ((float)val)/100.0f; 
  drawBackground( big2d, bgColor );
  drawGrid( big2d, gridColor );

  return;
}
public void setSaturationCount( int val )
{
  saturationCount = (float)val;
  drawBackground( big2d, bgColor );
  drawGrid( big2d, gridColor );

  return;
}

public void setAdaptiveDisplay( boolean val )
{
  adaptiveDisplay = val;
  drawBackground( big2d, bgColor );
  drawGrid( big2d, gridColor );

  return;
}

public void setForegroundColor( Color col ){ fgColor = col; return; }
public Color getForegroundColor(){ return fgColor; }
public void setBackgroundColor( Color col )
{
   bgColor = col;
   return;
}
public Color getBackgroundColor(){ return bgColor; }
public void setGridColor( Color col )
{
   gridColor = col;
   return;
}
public Color getGridColor(){ return gridColor; }

public void setColor( String cmd )
{
Color newColor = null;

   if ( cmd.equals( "fgcolor" ) )
     newColor = colorChooser.showDialog( this, "Foreground colour chooser", fgColor );
   else if ( cmd.equals( "bgcolor" ) )
     newColor = colorChooser.showDialog( this, "Background colour chooser", bgColor );
   else if ( cmd.equals( "gridcolor" ) )
     newColor = colorChooser.showDialog( this, "Grid colour chooser", gridColor );

   if ( newColor != null )
   {
	if ( cmd.equals( "fgcolor" ) )
	   setForegroundColor( newColor );
	else if ( cmd.equals( "bgcolor" ) )
	   setBackgroundColor( newColor );
	else if ( cmd.equals( "gridcolor" ) )
	   setGridColor( newColor );
   }

   return;
}

public void hitPixel( int x, int y )
{
   if ( (x>=nxpixels) || (y>=nypixels) || (x<0) || (y<0) )
   {
       //      applet.showStatus( "Array index out of bounds [x][y]=[" + x + "][" + y + "]" );
      return;
   }

   intensity[x][y] += 1.0;
   if ( intensity[x][y] > intensity[xPixelMax][yPixelMax] )
   {
      xPixelMax = x;
      yPixelMax = y;
   }

   //   drawPixel(x,y);
   
   return;
}

public void hitPixel( int x, int y, double w )
{
   if ( (x>=nxpixels) || (y>=nypixels) || (x<0) || (y<0) )
   {
       //      applet.showStatus( "Array index out of bounds [x][y]=[" + x + "][" + y + "]" );
      return;
   }

   intensity[x][y] += w;
   if ( intensity[x][y] > intensity[xPixelMax][yPixelMax] )
   {
      xPixelMax = x;
      yPixelMax = y;
   }

   //   drawPixel(x,y);

   return;
}

public void setRotation( int quadrant )
{
   quadrantRotation = quadrant;

   AffineTransform qrot = AffineTransform.getQuadrantRotateInstance(quadrantRotation, nxpixels*pixelWidth/2.0, nypixels*pixelHeight/2.0);
   //   big2d.setTransform( atg2d );
   big2d.transform(qrot);

   drawBackground( big2d, bgColor );
   drawGrid( big2d, gridColor );

   return;
}

public void paintNonBuffered( Graphics2D g, Paper pa, boolean printMono )
{

// Save the current transform

   AffineTransform tr = g.getTransform();

// Choose scaling to fit page

   double xscale;
   double yscale;
   if (quadrantRotation==0 || quadrantRotation == 2)
   {
     xscale = pa.getImageableWidth()/(nxpixels*pixelWidth);
     yscale = pa.getImageableHeight()/(nypixels*pixelHeight);
   }
   else
   {
     xscale = pa.getImageableWidth()/(nypixels*pixelHeight);
     yscale = pa.getImageableHeight()/(nxpixels*pixelWidth);
   }

   double scale = xscale<yscale ? xscale : yscale;   

// Translate, rotate and scale to fit page

   AffineTransform qrot = AffineTransform.getQuadrantRotateInstance(quadrantRotation );

   g.translate( (pa.getImageableX()+pa.getImageableWidth()/2.0), (pa.getImageableY()+pa.getImageableHeight()/2.0) );
   g.scale( scale, scale );
   g.transform(qrot);
   g.translate( -nxpixels*pixelWidth/2.0, -nypixels*pixelHeight/2.0 );

// Draw background

   AlphaComposite ac3 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   g.setComposite(ac3);

// Paint hit pixels

   if ( printMono )
   {
     drawBackground( g, Color.white );
     drawGrid( g, Color.white );
     g.setColor( Color.black );
   }
   else
   {
     drawBackground( g, bgColor );
     drawGrid( g, gridColor );
     g.setColor(fgColor);
   }
   for ( int x=0; x<nxpixels; x++ )
   {
      for ( int y=0; y<nypixels; y++ )
      {
         if ( intensity[x][y] >= 0.01 )
	 {
            float alpha;
            if ( adaptiveDisplay == true )
	    {
              alpha = intensity[x][y]/intensity[xPixelMax][yPixelMax];
	    }
            else
	    {
              alpha = intensity[x][y]/saturationCount;
	    }
            if ( alpha > 1.0f ) alpha = 1.0f;
	    AlphaComposite ac2 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,alpha );
	    g.setComposite(ac2);
	    g.fillRect(x*pixelWidth+1, y*pixelHeight+1, pixelWidth-2, pixelHeight-2);
	 }
      }
   }

// Draw a frame around the image

   AlphaComposite ac2 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   g.setComposite(ac2);
   g.setColor( Color.black );
   g.drawRect( 0, 0, nxpixels*pixelWidth, nypixels*pixelHeight );

// Restore the original transform

   g.setTransform( tr );

   return;
}

public void paintComponent(Graphics g)
{
    //   super.paintComponent(g); // paints the parent

   Graphics2D g2 = (Graphics2D)g.create();

   g2.drawImage( bi, 0, 0, this );

   g2.dispose();

   return;
}

// Draw full pixel array into buffered image

public void draw()
{

   AlphaComposite ac1 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
// Draw array

   for ( int x=0; x<nxpixels; x++ )
   {
     for ( int y=0; y<nypixels; y++ )
     {
       if ( intensity[x][y] >= 0.01 )
       {
         float alpha;
         if ( adaptiveDisplay == true )
         {
           alpha = intensity[x][y]/intensity[xPixelMax][yPixelMax];
         }
         else
         {
           alpha = intensity[x][y]/saturationCount;
         }
         if ( alpha > 1.0f ) alpha = 1.0f;
         if ( alpha > 0.1 )
	 {
           big2d.setComposite(ac1);
           big2d.setColor(bgColor);
           big2d.fillRect(x*pixelWidth+1, y*pixelHeight+1, pixelWidth-2, pixelHeight-2);
           AlphaComposite ac2 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,alpha );
           big2d.setComposite(ac2);
           big2d.setColor(fgColor);
           big2d.fillRect(x*pixelWidth+1, y*pixelHeight+1, pixelWidth-2, pixelHeight-2);
	 }
       }
     }
   }

   return;
}

// Draw single pixel into buffered image

public void drawPixel(int x, int y)
{

// Draw pixel background

   AlphaComposite ac3 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,1.0f );
   big2d.setComposite(ac3);

   big2d.setColor(bgColor);
   big2d.fillRect( x*pixelWidth, y*pixelHeight, pixelWidth, pixelHeight);

   big2d.setColor(gridColor);
   big2d.drawRect(x*pixelWidth, y*pixelHeight, pixelWidth, pixelHeight);

// Draw hit pixel

   big2d.setColor(fgColor);
   if ( intensity[x][y] >= 0.01 )
   {
     float alpha;
     if ( adaptiveDisplay == true )
     {
       alpha = intensity[x][y]/intensity[xPixelMax][yPixelMax];
     }
     else
     {
       alpha = intensity[x][y]/saturationCount;
     }
     if ( alpha > 1.0f ) alpha = 1.0f;
     AlphaComposite ac2 = AlphaComposite.getInstance(AlphaComposite.SRC_OVER,alpha );
     big2d.setComposite(ac2);
     big2d.fillRect(x*pixelWidth+1, y*pixelHeight+1, pixelWidth-2, pixelHeight-2);
   }

   return;
}
//
public int print(Graphics g, PageFormat pf, int pi) throws PrinterException
{
   if (pi >= 1)
   {
            return Printable.NO_SUCH_PAGE;
   }

   Paper pa = pf.getPaper();

   //   System.out.println( "Paper (pidx,x,y,width,height)=(" + pi + "," + pa.getImageableX() + "," + pa.getImageableY() + "," + pa.getImageableWidth() + "," + pa.getImageableHeight() + ")" );

   Graphics2D g2D = (Graphics2D)g;
   Date now = new Date();
   String dateString = DateFormat.getDateTimeInstance().format(now);
   String titleString = new String( dateString );

   g2D.drawString( titleString, (int)pa.getImageableX(), (int)pa.getImageableY() + (int)pa.getImageableHeight() );

   paintNonBuffered(g2D, pa, printMono );

   return Printable.PAGE_EXISTS;
}

}
//














