// This program is a proxy readout task started from the MAPMTFEB Java GUI
// Communicates with GUI through stdin and stdout.
#include <stdlib.h>
#include <unistd.h>
#include "arpa/inet.h"
#include "file.h"
#include "mdfrich.h"
#include <vector>
#include <string>
#include "mapmtfeb.h"
#include "semaphore.h"
#include <pthread.h>
// For the FIFO
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

using namespace std;

const unsigned int buflen=65536;
struct ip_header_t
{
  unsigned short header_length :4;
  unsigned short version :4;
  unsigned short ecn :2;
  unsigned short dscp :6;
  unsigned char total_length[2];
  unsigned char identification[2];
  unsigned short flags_offset;
  unsigned short ttl :8;
  unsigned short protocol :8;
  unsigned short checksum;
  unsigned char srcaddr[4];
  unsigned char destaddr[4];
};

struct mepHeader_t
{
  unsigned index; // ID of first event in MEP
  unsigned short count; // Number of events in MEP
  unsigned short length; // Number of bytes (including this header)
  unsigned tag; // Formerly partition ID
};

struct mepEventHeader_t
{
  unsigned short event; // Event ID
  unsigned short length; // Number of bytes (not including this header)
};

struct msgHeader_t
{
  unsigned short length; // Number of bytes following header
  unsigned short type;
  unsigned info;
  unsigned seq;
};

union msg_u { msgHeader_t hdr; unsigned char uc[sizeof(msgHeader_t)];} msg;

bool seekpFlag = false;
bool debugConfig = true;
bool debugIP = false;
bool debugMEP = false;
bool debugRawBank = false;
bool debugRICH = false;
unsigned runNumber = 1;
int eventsSeen = 0;
int mepsSeen = 0;
int maxEvents = 0;

vector<fstream*> fspointers;
fstream* fs = NULL;
int fifo = -1;
semaphore *sem = NULL;

static int openAllStreams(void);
static void closeAllStreams(void);
static void showBuffer(unsigned char *);
static void *readData(void *);

vector<string> ipAddresses;
vector<string> macAddresses;
vector<string> filenames;

main(int argc, char **argv)
{
int protocol = 0xf4;
char *ptr;

  for(int i=0; i<argc; i++){cerr<<argv[i]<<" ";}
  cerr<<endl;

  const char *keyFile = NULL;

  for (int arg=1;arg<argc;arg++)
  {
    if ( !strcmp("--protocol",argv[arg]))
    {
      protocol = strtol(argv[++arg],0,0); 
    }
    else if ( !strcmp("--keyfile",argv[arg]))
    {
      keyFile = argv[++arg]; 
    }
    else if ( !strcmp("--debug-ip",argv[arg]))
    {
      debugIP = true; 
    }
    else if ( !strcmp("--debug-mep",argv[arg]))
    {
      debugMEP = true; 
    }
    else if ( !strcmp("--debug-raw",argv[arg]))
    {
      debugRawBank = true; 
    }
    else if ( !strcmp("--debug-rich",argv[arg]))
    {
      debugRICH = true; 
    }
  }

// A semaphore to protect file operations
  if (keyFile == NULL)
  {
    cerr<<"main(): Semaphore keyfile undefined. Exiting..."<<endl;
    exit(0);
  }
  sem = new semaphore(keyFile,1,0666);
  delete sem;
  sem = new semaphore(keyFile,1,0666);
  cerr<<"main(): New semaphore with key="<<hex<<sem->getKey()<<dec<<endl;

  int osk = socket(PF_INET, SOCK_RAW, protocol);
  if ( osk == -1 )
  {
    cerr << "main(): Failed to create the socket." << endl; 
    cerr << errno << endl;
    exit(0);
  }

  // Create the thread to read the raw socket
  pthread_t tid;
  int rc = pthread_create(&tid,NULL,&readData,NULL);

  sockaddr_in dst;
  dst.sin_family = AF_INET;
  dst.sin_port = 0;
  //inet_pton(AF_INET,ip_address,&dst.sin_addr.s_addr);

  cerr << "main(): Start forwarding packets..." << endl;

unsigned char buf[buflen];

  while(1)
  {
    // Read message header

    for ( int bytesReceived=0;bytesReceived<sizeof(msgHeader_t); )
    {
      size_t len = sizeof(msgHeader_t)-bytesReceived;
      len = read(0,msg.uc+bytesReceived,len);

      if (len == 0) { cerr<<"main(): EOF detected while reading message header. Exiting."<<endl; exit(0); }
      if (len > 0) bytesReceived += len;
    }
    if( debugConfig ){cerr<<"Received header ===> msg.hdr.[type,length,info,seq]=["<<msg.hdr.type<<","<<msg.hdr.length<<","<<msg.hdr.info<<","<<msg.hdr.seq<<"]"<<endl;}

    // Get the payload (if any)

    if (msg.hdr.length > 0)
    {
      for ( int bytesReceived=0;bytesReceived<msg.hdr.length; )
      {
        size_t len = msg.hdr.length-bytesReceived;
        len = read(0,buf+2+bytesReceived,len);
        if (len == 0) {cerr<<"main(): EOF detected while reading message payload. Exiting."<<endl;exit(0);}
        if (len > 0) bytesReceived += len;
      }

      buf[0] = 0;
      buf[1] = 0;
    }

    int rc;

    // Check what action to take
    switch ( msg.hdr.type )
    {
    case 1: // Start or stop acquisition
      {
        sem->get();

        string ip_str = "";
        char* ip_chars = new char[16]; 
        unsigned char* ptr;
        bool matchflag = 0;
        febmap_t* feb = 0;
        char* dna_chars = new char[4];
        eventsSeen = 0;
        mepsSeen = 0;
        maxEvents = msg.hdr.info;

        ip_str = "";
        ptr = buf + 2; // first 2 bytes of buf are empty
        sprintf( dna_chars, "%02X%02X%02X%02X", *(ptr+3), *(ptr+2), *(ptr+1), *ptr );
        ptr += 4; // 4 bytes to hold dna
        feb = reinterpret_cast<febmap_t*>(ptr);
        sprintf(ip_chars,"%d.%d.%d.%d",int(feb->ipsrc.uc[3]),int(feb->ipsrc.uc[2]),int(feb->ipsrc.uc[1]),int(feb->ipsrc.uc[0]));
        if( debugConfig ) {cerr<<"[1] dna="<<dna_chars<<" ipsrc="<<ip_chars<<endl;}
        ip_str = string(ip_chars);
        matchflag = 0;
        for(int i=0; i<ipAddresses.size(); i++)
        {
          if( !ip_str.compare(ipAddresses[i]) )
          {
            matchflag = 1;
            if( debugConfig ){cerr<<"[1] found match for ip address of config buffer"<<endl;}
            if( debugConfig ){cerr<<"[1] "<< ip_str <<endl;}
            const char* parsed_ip_addr = ipAddresses[i].c_str();
            inet_pton(AF_INET,parsed_ip_addr,&dst.sin_addr.s_addr);
            int sent = sendto(osk,reinterpret_cast<void *>(buf),msg.hdr.length+2,0,reinterpret_cast<sockaddr *>(&dst),sizeof(sockaddr_in));
            if(sent == -1)
            {
              cerr<<"[1] sendto() errno="<<errno<<endl;
            }
            if( debugConfig ){cerr<<"[1] sent buffer to "<<ip_str<<endl;}
          }
        }
        if(matchflag == 0){cerr<<"[1] did not find match for ip address specified in buffer"<<endl;}

        // tidy up
        delete ip_chars;
        delete dna_chars;

	if (debugConfig) cerr<<"[1] Send response "<<msg.hdr.seq<<endl;
        rc = write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);

        sem->release();
        break;
      }

    case 2: // New run.
// Ensure all output streams are closed
        closeAllStreams();
// Get run number
        runNumber = msg.hdr.info;
        cerr << "[2] Run number=" << runNumber << endl;
// Open new output streams
        rc = openAllStreams();
        if(rc == -1){closeAllStreams(); cerr << "[2] openAllStreams() rc=-1. Error opening stream. Data recording disabled." << endl;}
        else if(rc == -2){closeAllStreams(); cerr << "[2] openAllStreams() rc=-2 (File already exists). Data recording disabled." << endl; }
        write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);
        break;

    case 3: // End run
        cerr<<"[3] Ending run"<<endl;
        closeAllStreams();
        write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);
        break;

    case 4: // Setup proxy
        if( debugConfig ){cerr<<"[4] Proxy setup."<<endl;}
        sem->get();

        macAddresses.clear();
        ipAddresses.clear();
        filenames.clear();
        ptr = reinterpret_cast<char *>(buf) + 2;
// Parse the (mac,ip,filename) triplets
        while( string(ptr).compare("--")!=0 )
        {
          macAddresses.push_back(string(ptr)); ptr+=(strlen(ptr)+1); 
          ipAddresses.push_back(string(ptr));  ptr+=(strlen(ptr)+1);
          filenames.push_back(string(ptr)); ptr+=(strlen(ptr)+1);
// Make a static ARP entry
          char syscmd[256];
          if( debugConfig ){cerr<<"[4] MAC="<<macAddresses.back()<<" IP="<<ipAddresses.back()<<" filename="<<filenames.back()<<endl;}
          sprintf(syscmd,"/sbin/arp -s %s %s",ipAddresses.back().c_str(),macAddresses.back().c_str());
// NB: system call generates a signal that interrupts semaphore.
	  system(syscmd);
        }
        ptr+=3;
// Parse the FIFO name
        while ( string(ptr).compare("--")!=0 )
        {
          if( debugConfig ){cerr<<"[4] FIFO="<<ptr<<endl;}
          if (fifo > 2)
          {
            if( debugConfig ){cerr<<"[4] close fifo fd="<<fifo<<endl;}
	    close(fifo);
            fifo = -1;
	  }
// Open read/write so that the open does not block while waiting for a reader
	  fifo = open(ptr,O_RDWR|O_TRUNC|O_NONBLOCK,0666);
          if (fifo < 0) { fifo=-1; cerr<<"[4] Failed to open FIFO "<<ptr<<endl;}
          ptr+=(strlen(ptr)+1);
        }
        write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);
        sem->release();
        if( debugConfig ){cerr<<"[4] Proxy setup done."<<endl;}
        break;

    default:
      cerr<<"[?] Unknown message type received"<<endl;
      write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);
      break;
    }
  }
} //main

static void closeAllStreams(void)
{
  sem->get();
  for(int i=0; i<fspointers.size(); i++)
  {
    fs = fspointers[i];
    if ( fs != NULL )
    {
      delete (fs);
      fs = NULL;
    }
  }
  fspointers.clear();
  sem->release();

  return;
}

static int openAllStreams(void)
{
  // for each filename
  sem->get();
  for(int i=0; i<filenames.size(); i++)
  {
    char fileName[128];
    const char* parsed_filename = filenames[i].c_str();
    sprintf(fileName,parsed_filename,runNumber);

  // Open for reading to make sure it doesn't already exist.
    fs = new fstream(fileName,ios::in | ios::binary);
    if ( fs->is_open())
    {
      cerr<<"openAllStreams(): "<<fileName<<" already exists. return(-2)"<<endl;
      sem->release();
      return (-2);
    }
  // Reopen (create) for writing
    cerr<<"openAllStreams(): Opening "<<fileName<<" for writing..."<<endl;
    fs->clear();
    fs->open(fileName,ios::out | ios::binary);
    if ( !fs->is_open())
    {
      cerr<<"openAllStreams() "<<fileName<<" was not opened. return(-1)"<<endl;
      sem->release();
      return(-1);
    }
    fspointers.push_back(fs);
  }
  sem->release();
  return (0);
}

// Read and store events

static void *readData( void *arg )
{
  char fifobuf[512]; // Enough for headers + binary data + 2*MAROC ADC
  cerr<<"readData(): Waiting for data..."<< endl;

  int sk = socket(PF_INET, SOCK_RAW, 0xf2);
  if ( sk == -1)
  {
    cerr << "readData(): Failed to create the socket. errno=" << errno << endl;
    exit(0);
  }

  xsockstream *xs = new xsockstream(sk);

unsigned char buf[buflen];
unsigned char *ptr;

  for(;;)
  {
    fs = NULL;

    int rc = xs->read(reinterpret_cast<void *>(buf),buflen);
    if (rc < 0) {cerr<<"readData(): Read failed. rc="<<rc<<", errno="<<errno<<endl;break;}
    mepsSeen++;

    showBuffer( buf );
    ptr = buf;
    ip_header_t *ip = reinterpret_cast<ip_header_t *>(ptr);
    ptr += sizeof(ip_header_t);

    mepHeader_t *mep = reinterpret_cast<mepHeader_t *>(ptr);
    ptr += sizeof(mepHeader_t);

// Special case. MSB of tag is used to signal end.
// Close all streams, not just the one identified for this MEP.

    if (mep->tag & 0x80000000)
    {
      cerr<<"readData(): MSB of tag signals end of run!!"<<endl;
      closeAllStreams();
      continue;
    }

    // Identify the source (by IP address) and assign the corresponding output stream
    char* SourceIp_chars = new char[16];
    string SourceIp = "";
    sprintf(SourceIp_chars,"%d.%d.%d.%d",ip->srcaddr[0],ip->srcaddr[1],ip->srcaddr[2],ip->srcaddr[3]);
    SourceIp = string(SourceIp_chars);
    delete SourceIp_chars;

    sem->get();
    fs = NULL;
    for(int i=0; i<ipAddresses.size(); i++)
    {
      if( !SourceIp.compare(ipAddresses[i]) )
      {
        if (fspointers.size()>i) fs = fspointers[i];
      }
    }
    sem->release();

    unsigned char *nextEventPtr = ptr;
    for (unsigned ievt=0; ievt<mep->count; ievt++)
    {
      ptr = nextEventPtr;
      mepEventHeader_t *evt = reinterpret_cast<mepEventHeader_t *>(ptr);
      ptr += sizeof(mepEventHeader_t);
      nextEventPtr = ptr + evt->length;

      rawHeader_t *raw = reinterpret_cast<rawHeader_t *>(ptr);
      ptr += sizeof(rawHeader_t);

      ptr += sizeof(ingressHeader_t);

      l1Header_xc0_t *l1 = reinterpret_cast<l1Header_xc0_t *>(ptr);

// Write the event

      eventsSeen++;

      mdfHeader_t mdf;
      mdf.length = sizeof(mdfHeader_t) + sizeof(eventHeader_t) + raw->length;
      mdf.length_1 = mdf.length;
      mdf.length_2 = mdf.length;
      mdf.checksum = 0;
      mdf.compression = 0;
      mdf.size = 7;
      mdf.version = 1;
      mdf.dataType = 3;

      eventHeader_t mdfEvt;
      memset(reinterpret_cast<void *>(&mdfEvt),0,sizeof(eventHeader_t));
      mdfEvt.run = runNumber;
      mdfEvt.orbit = l1->frame;
      mdfEvt.bunch = mep->tag;

// Write the MDF data. If recording is not enabled, fs will be NULL
      sem->get();
      if (fs != NULL)
      {
        fs->clear();  //make sure fs is not stuck in some error state
        if (seekpFlag) fs->seekp(0);
        fs->write(reinterpret_cast<char *>(&mdf),sizeof(mdfHeader_t));
        fs->write(reinterpret_cast<char *>(&mdfEvt),sizeof(eventHeader_t));
        fs->write(reinterpret_cast<char *>(raw),raw->length);
        if ( (fs->rdstate() & fstream::failbit ) != 0 ){cerr<<"Error writing to file"<<endl;}
        fs->flush();
      }
// Write also to FIFO
      int fifomsglen = sizeof(mdfHeader_t) + sizeof(eventHeader_t) + raw->length;
      if ((fifo > 2) && (fifomsglen <= 512))
      {
        int off = 0;
        memcpy(fifobuf+off,&fifomsglen,4); off+=4;
        memcpy(fifobuf+off,&mdf,sizeof(mdfHeader_t)); off+=sizeof(mdfHeader_t);
        memcpy(fifobuf+off,&mdfEvt,sizeof(eventHeader_t)); off+=sizeof(eventHeader_t);
        memcpy(fifobuf+off,raw,raw->length);
// Fifo is non-blocking. Doesn't matter if it fails. The amount of data is less than PIPE_BUF so either all data are written or none.
        write(fifo,fifobuf,fifomsglen+4);
      }
      sem->release();
      if (eventsSeen%1000 == 1) cerr<<"readData(): eventsSeen="<<eventsSeen<<endl;
      if ( maxEvents > 0 && eventsSeen==maxEvents)
      {
        write(1,reinterpret_cast<void *>(&msg.hdr.seq),4); fflush(stdout);
        maxEvents = 0;
      }
    }
    if (mepsSeen%1000 == 1) cerr<<"readData(): mepsSeen="<<mepsSeen<<endl;
  }

  xs->close();
  delete (xs);

  cerr<<"readData(): End."<<endl;

  return 0;
}


static void showBuffer( unsigned char *buf )
{
  unsigned char *ptr = buf;
  ip_header_t *ip = reinterpret_cast<ip_header_t *>(ptr);
  ptr += sizeof(ip_header_t);
  if (debugIP) fprintf(stderr,"IP\t Header length: %d",ip->header_length);
  if (debugIP) fprintf(stderr,"\t Version: %d",ip->version);
  if (debugIP) fprintf(stderr,"\t ECN: %d",ip->ecn);
  if (debugIP) fprintf(stderr,"\t DSCP: %d\n",ip->dscp);

  unsigned short len = ip->total_length[0]<<8 | ip->total_length[1];
  if (debugIP) fprintf(stderr,"IP\t Total length: %#06x",len);
  unsigned short id = ip->identification[0]<<8 | ip->identification[1];
  if (debugIP) fprintf(stderr,"\t Identification: %#06x\n",id);

  if (debugIP) fprintf(stderr,"IP\t Flags_offset: %#06x",ip->flags_offset);
  if (debugIP) fprintf(stderr,"\t TTL: %#04x",ip->ttl);
  if (debugIP) fprintf(stderr,"\t Protocol: %#04x\n",ip->protocol);
  if (debugIP) fprintf(stderr,"IP\t Header checksum: %d\n",ip->checksum);

  if (debugIP) fprintf(stderr,"IP\t Source: %d.%d.%d.%d",ip->srcaddr[0],ip->srcaddr[1],ip->srcaddr[2],ip->srcaddr[3]);
  if (debugIP) fprintf(stderr,"\t Destination: %d.%d.%d.%d\n",ip->destaddr[0],ip->destaddr[1],ip->destaddr[2],ip->destaddr[3]);

  mepHeader_t *mep = reinterpret_cast<mepHeader_t *>(ptr);
  ptr += sizeof(mepHeader_t);
  if (debugMEP) fprintf(stderr,"MEP\t Event index: %#010x",mep->index);
  if (debugMEP) fprintf(stderr,"\t Event count: %#06x\n",mep->count);
  if (debugMEP) fprintf(stderr,"MEP\t Word count: %#06x",mep->length);
  if (debugMEP) fprintf(stderr,"\t Tag: %#06x\n",mep->tag);

  unsigned char *nextEventPtr = ptr;
  for (unsigned ievt=0; ievt<mep->count; ievt++)
  {
    mepEventHeader_t *evt = reinterpret_cast<mepEventHeader_t *>(nextEventPtr);
    ptr += sizeof(mepEventHeader_t);
    nextEventPtr = ptr + evt->length;

    if (debugMEP) fprintf(stderr,"Event\t ID: %#06x",evt->event);
    if (debugMEP) fprintf(stderr,"\t Word count: %#06x\n",evt->length);

    rawHeader_t *raw = reinterpret_cast<rawHeader_t *>(ptr);
    ptr += sizeof(rawHeader_t);
    if (debugRawBank) fprintf(stderr,"Bank\t Magic: %#06x",raw->magic);
    if (debugRawBank) fprintf(stderr,"\t Length: %d",raw->length);
    if (debugRawBank) fprintf(stderr,"\t Type: %#04x",raw->type);
    if (debugRawBank) fprintf(stderr,"\t Version: %#04x\n",raw->version);
    if (debugRawBank) fprintf(stderr,"Bank\t Source ID: %#06x\n",raw->source);

    ingressHeader_t *ing = reinterpret_cast<ingressHeader_t *>(ptr);
    ptr += sizeof(ingressHeader_t);
    if (debugRICH) fprintf(stderr,"Ingress\t evtid: %#04x",ing->event);
    if (debugRICH) fprintf(stderr,"\t bxid: %#04x\n",ing->bxid);
    if (debugRICH) fprintf(stderr,"Ingress\t channelMask: %#06x",ing->channelMask);
    if (debugRICH) fprintf(stderr,"\t id: %d",ing->id);
    if (debugRICH) fprintf(stderr,"\t truncation flag: %d\n",ing->truncated);

    l1Header_xc0_t *l1 = reinterpret_cast<l1Header_xc0_t *>(ptr);
    ptr += sizeof(l1Header_xc0_t);
    if (debugRICH) fprintf(stderr,"L1[0]\t pdid: %#06x",l1->hpd);
    if (debugRICH) fprintf(stderr,"\t evtid: %#06x\n",l1->event);
    if (debugRICH) fprintf(stderr,"L1[0]\t nzcount: %#06x",l1->nzcount);
    if (debugRICH) fprintf(stderr,"\t flags: [GFMZ]=[%d%d%d%d]\n",l1->gt,l1->ext,l1->mode,l1->zs);
    if (debugRICH) fprintf(stderr,"L1[1,2]\t Frame/Event/BX = %#06x/%#06x/%#010x\n",l1->frame,l1->evid,l1->timestamp);

// 8 words contain (non-zs)
    unsigned *ui = reinterpret_cast<unsigned *>(ptr);
    ptr += 32;
    if (debugRICH)
    {  
      for (unsigned i=1; i<9; i++)
      {
        fprintf(stderr,"%#010x\n",ui[i]);
      }
    }
  }
  return;
}
