///////////////////////////////////////////////////////////////////////////
// This file contains definition of functions to produce 
// DAQ request for DCS data single read (this is the DDC package DdcRequest class modified for SCT DAQ needs)
//					
//  Created 21.05.2003
//  by A. Abdesselam
///////////////////////////////////////////////////////////////////////////

#include 	<string>
#include 	<vector>
#include 	<algorithm>

#include <ipc/partition.h>
//IS stuff
#include <is/isinfoany.h>
#include <is/isnamedinfo.h>

using namespace std;

#include "sctddc/SCTDdcRequest.hxx"

SCTDdcRequest::SCTDdcRequest(IPCPartition& partition, string& isServer)
	: m_requestReady(false), m_partition(partition), m_requestServer(isServer)
{ 
  m_infoReceiver = new ISInfoReceiver(partition);
}

// initilizations related to request vector of float
void SCTDdcRequest::requestReady_vec_init(unsigned int vecSize)
  {
    //reset the current index
    m_requestReady_vec_index = 0;
    //erase all previous if any (pehaps from previous call)
    m_requestReady_vec.erase(m_requestReady_vec.begin(), m_requestReady_vec.end());
    //initialize the vector to false 
    for(unsigned int i=0; i<vecSize; i++) 
      m_requestReady_vec.push_back(false);
    //initialise the local map (pehaps from previous call)
    m_floatVecRequestResponse.erase(m_floatVecRequestResponse.begin(), 
				    m_floatVecRequestResponse.end());
  }

//test if all the responses came
bool SCTDdcRequest::getRequestReady_vec()
  {
    bool allRequestReady = true;
    for(unsigned int i=0; i<m_requestReady_vec.size(); i++) 
      allRequestReady = allRequestReady && m_requestReady_vec[i];
    return  allRequestReady;
  }

//tell that one response came:  request =true
void SCTDdcRequest::setRequestReady_vec(bool request)
  {
    m_requestReady_vec[m_requestReady_vec_index] = request;
    m_requestReady_vec_index++;
  }
//---------------------------------------------------------------------------------------------------------
void requestCallback(ISCallbackInfo* isCallback)
//---------------------------------------------------------------------------------------------------------
{
  std::string value_str;
  OWLDate value_date;
  OWLTime value_time;
  //get the calling object (to modify its attributes)
  SCTDdcRequest* sctDdcRequest = (SCTDdcRequest*)(isCallback->parameter());

  //define ISInfoAny, we don't know appriori the type of the object in the IS
  ISInfoAny dcsData;
  if(isCallback->reason() != ISInfoDeleted) 
    {
      string dpeName = isCallback->name();
      isCallback->value(dcsData);
    }
  // usaly the number of attribute is 2: time and value
  int N = dcsData.countAttributes( );
  
  //cout << "=============== type = " << dcsData.getType() << std::endl;
  for (int i=0; i<N; i++)
    {
      switch(dcsData.getAttributeType())
	{
	case ISType::Boolean:
	  //bool
	  bool value_bool;
	  dcsData >> value_bool;
	  sctDdcRequest->setIntRequestResponse(int(value_bool));
	  sctDdcRequest->setFloatRequestResponse(float(value_bool));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::S8:
	case ISType::U8:
	case ISType::S16:
	  //short
	  short value_s;
	  dcsData >> value_s;
	  sctDdcRequest->setIntRequestResponse(int(value_s));
	  sctDdcRequest->setFloatRequestResponse(float(value_s));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::U16:
	  //unsigned short
	  unsigned short value_us;
	  dcsData >> value_us;
	  sctDdcRequest->setIntRequestResponse(int(value_us));
	  sctDdcRequest->setFloatRequestResponse(float(value_us));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::S32:
	  //long
	  long value_int;
	  dcsData >> value_int;
	  sctDdcRequest->setIntRequestResponse(value_int);
	  sctDdcRequest->setFloatRequestResponse(float(value_int));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::U32:
	  //unsigned lon
	  unsigned long value_uint;
	  dcsData >> value_uint;
	  sctDdcRequest->setIntRequestResponse(value_uint);
	  sctDdcRequest->setFloatRequestResponse(float(value_uint));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::Float:
	  //float
	  float value_float;
	  dcsData >> value_float;
	  sctDdcRequest->setFloatRequestResponse(value_float);
	  sctDdcRequest->setIntRequestResponse(int(value_float));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::Double:
	  //double
	  double value_d;
	  dcsData >> value_d;
	  sctDdcRequest->setFloatRequestResponse(float(value_d));
	  sctDdcRequest->setIntRequestResponse(int(value_d));//just for protection
	  sctDdcRequest->setRequestReady(true);
	  break;
	case ISType::String:
	  //std::string
	  dcsData >> value_str;
	  sctDdcRequest->setStringRequestResponse(value_str);
	  sctDdcRequest->setRequestReady(true);
	  break;

	case ISType::Date:
	  //OWLDate
	  dcsData >> value_date;
	  std::cerr<<"SCTDdcRequest::requestCallback() DDC_ERROR: variable from dcs is date, "
		   <<"not implemented inddcRequest class"<< std::endl;
	  break;
	case ISType::Time:
	  //OWLTine
	  dcsData >> value_time;
	  //cerr<<"DDC_ERROR: variable from dcs is time, "
	  //    <<"not implemented inddcRequest class"<< endl;
	  break;
	  
	  
	default:
	  cout << "SCTDdcRequest::requestCallback() DDC_ERROR: Invalid Type" 
	       << (long)dcsData.getAttributeType() << std::endl;
	}
    }
}

//---------------------------------------------------------------------------------------------------------
void floatArrayRequestCallback(ISCallbackInfo* isCallback)
//---------------------------------------------------------------------------------------------------------
{
  OWLTime value_time;
  //get the calling object (to modify its attributes)
  SCTDdcRequest* sctDdcRequest = (SCTDdcRequest*)(isCallback->parameter());

  //define ISInfoAny, we don't know appriori the type of the object in the IS
  ISInfoAny dcsData;
  if(isCallback->reason() != ISInfoDeleted) 
    {
      string dpeName = isCallback->name();
      isCallback->value(dcsData);
    }

  //read the first item that is the time
  dcsData >> value_time;

  //read then the array of float
  float *float_array;
  size_t  array_size;

  dcsData.get(&float_array, array_size);
  sctDdcRequest->setFloatArrayRequestResponse(float_array, int(array_size));
  sctDdcRequest->setRequestReady(true);
  
  return;
}

//---------------------------------------------------------------------------------------------------------
void requestFloatVecCallback(ISCallbackInfo* isCallback)
//---------------------------------------------------------------------------------------------------------
{
  std::string value_str;
  OWLDate value_date;
  OWLTime value_time;
  string dpeName;

  //get the calling object (to modify its attributes)
  SCTDdcRequest* sctDdcRequest = (SCTDdcRequest*)(isCallback->parameter());

  //define ISInfoAny, we don't know appriori the type of the object in the IS
  ISInfoAny dcsData;
  if(isCallback->reason() != ISInfoDeleted) 
    {
      dpeName = isCallback->name();
      //get the name of the prameter; the fromat of dpeName is: ISserve.path.parameter.Recv (.Send)
      unsigned int position = dpeName.find(".Recv");
      if(position != string::npos) dpeName = dpeName.substr(0,position);
      position = dpeName.find(".Send");
      if(position != string::npos) dpeName = dpeName.substr(0,position);
      dpeName = dpeName.substr(dpeName.find_last_of(".")+1, dpeName.size());
      isCallback->value(dcsData);
    }
  // usaly the number of attribute is 2: time and value
  int N = dcsData.countAttributes( );
  
  //cout << "    " << N << " attribute(s):" << std::endl;
  for (int i=0; i<N; i++)
    {
      switch(dcsData.getAttributeType())
	{
	case ISType::Boolean:
	  //bool
	  bool value_bool;
	  dcsData >> value_bool;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_bool));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::S8:
	case ISType::U8:
	case ISType::S16:
	  //short
	  short value_s;
	  dcsData >> value_s;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_s));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::U16:
	  //unsigned short
	  unsigned short value_us;
	  dcsData >> value_us;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_us));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::S32:
	  //long
	  long value_int;
	  dcsData >> value_int;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_int));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::U32:
	  //unsigned lon
	  unsigned long value_uint;
	  dcsData >> value_uint;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_uint));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::Float:
	  //float
	  float value_float;
	  dcsData >> value_float;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, value_float);
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::Double:
	  //double
	  double value_d;
	  dcsData >> value_d;
	  sctDdcRequest->setfloatVecRequestResponse(dpeName, float(value_d));
	  sctDdcRequest->setRequestReady_vec(true);
	  break;
	case ISType::String:
	  //std::string
	  dcsData >> value_str;
	  std::cerr
	    <<"SCTDdcRequest::requestFloatVecCallback(..) DDC_ERROR: variable ("<<dpeName
		   <<") is string, that is not supported in ddcRequest class"<< std::endl;
	  break;
	case ISType::Date:
	  //OWLDate
	  dcsData >> value_date;
	  std::cerr
	    <<"SCTDdcRequest::requestFloatVecCallback(..) DDC_ERROR: variable ("<<dpeName
		   <<") is date, that is not supported in ddcRequest class"<< std::endl;
	  break;
	case ISType::Time:
	  //OWLTine
	  dcsData >> value_time;
	  //cerr<<"DDC_ERROR: variable from dcs is time, "
	  //    <<"not implemented inddcRequest class"<< endl;
	  break;
	  
	default:
	  cout << "SCTDdcRequest::requestCallback() DDC_ERROR: Invalid Type" 
	       << (long)dcsData.getAttributeType() << std::endl;
	}
    }
  //dcsData.reset();
}

//this method is to read int from DCS
//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::ddcSendRequest(string& dpeName, int& value, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  // IS server and PVSS DataPoint address
  string server_dpeName = m_requestServer + "." + dpeName;

  // first subscribe for the request repply 
  rc = m_infoReceiver->subscribe(server_dpeName.c_str(), requestCallback, (void*)this);
  if(rc != ISInfo::Success) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << server_dpeName
	   << " subscription fault " << rc << ". No IS server to send request" << endl;
    }
  
  time_t startRequestTime = time(&startRequestTime);
  // then send request

  //================================code from Slava
  ISInfo::Status exportResult = ISInfo::Success;
  string isObName;
  time_t currTime;
  
  isObName = m_requestServer + '.';
  isObName += DDC_DAQ_REQUEST;
  
  DcsSingleData<string>* ddcRequestForData 
    = new DcsSingleData<string>(time(&currTime), m_partition, (char*)isObName.c_str());
  ddcRequestForData->setValue(server_dpeName);
  exportResult = ddcRequestForData->checkin();
  if(exportResult == ISInfo::CommFailure) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " 
	   << ctime((const time_t*)&currTime) << " IS server "
	   << m_requestServer << " for DAQ's data requests in partition "
	   << m_partition.name() << " does not run!" << endl;
    }
  //delete ddcRequestForData;
  //==================================end of code from Slava
  
  // wait for the request to be ready or the time to be out
  int deltaRequestTime = 0;
  while(!m_requestReady && deltaRequestTime< timeout)
    {
      usleep(10); // wait in micro seconds
      time_t timenow = time(&timenow);
      deltaRequestTime = timenow - startRequestTime;
    }
  // if time is out
  if(!m_requestReady)
    {
      cerr << "DDC_ERROR: request waiting time for "<<dpeName
	   <<" is out!!, stupid value is returned!!"<<endl;
      m_infoReceiver->unsubscribe(server_dpeName.c_str());
      value = 9999999;
      delete ddcRequestForData;
      return;
    }

  //retune requested value
  value = m_intRequestResponse;
 //reset request to not ready (enable fro another request)
  m_requestReady = false;

  // Remove the request
  ddcRequestForData->remove();
  delete ddcRequestForData;
  m_infoReceiver->unsubscribe(server_dpeName.c_str());


  return;
}

//this method is to read array of float (dyn_Float) from DCS
//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::ddcSendRequest(string& dpeName, float* float_array, int& array_size, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  // IS server and PVSS DataPoint address
  string server_dpeName = m_requestServer + "." + dpeName;

  // first subscribe for the request repply 
  rc = m_infoReceiver->subscribe(server_dpeName.c_str(), 
				 floatArrayRequestCallback, (void*)this);
  if(rc != ISInfo::Success) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << server_dpeName
	   << " subscription fault " << rc << ". No IS server to send request" << endl;
    }
  
  time_t startRequestTime = time(&startRequestTime);
  // then send request

  //================================code from Slava
  ISInfo::Status exportResult = ISInfo::Success;
  string isObName;
  time_t currTime;
  
  isObName = m_requestServer + '.';
  isObName += DDC_DAQ_REQUEST;
  
  DcsSingleData<string>* ddcRequestForData 
    = new DcsSingleData<string>(time(&currTime), m_partition, (char*)isObName.c_str());
  ddcRequestForData->setValue(server_dpeName);
  exportResult = ddcRequestForData->checkin();
  if(exportResult == ISInfo::CommFailure) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " 
	   << ctime((const time_t*)&currTime) << " IS server "
	   << m_requestServer << " for DAQ's data requests in partition "
	   << m_partition.name() << " does not run!" << endl;
    }
  //delete ddcRequestForData;
  //==================================end of code from Slava
  
  // wait for the request to be ready or the time to be out
  int deltaRequestTime = 0;
  while(!m_requestReady && deltaRequestTime< timeout)
    {
      usleep(10); // wait in micro seconds
      time_t timenow = time(&timenow);
      deltaRequestTime = timenow - startRequestTime;
    }
  // if time is out
  if(!m_requestReady)
    {
      cerr << "DDC_ERROR: request waiting time for "<<dpeName
	   <<" is out!!, stupid value is returned!!"<<endl;
      m_infoReceiver->unsubscribe(server_dpeName.c_str());
      float_array = NULL;
      delete ddcRequestForData;
      return;
    }

  //retune requested array
  array_size = m_floatArraySizeRequestResponse;
  for(int i=0; i<array_size; i++)
    {
      float_array[i] = m_floatArrayRequestResponse[i];
    }
  //reset request to not ready (enable fro another request)
  m_requestReady = false;

  // Remove the request
  ddcRequestForData->remove();
  delete ddcRequestForData;
  m_infoReceiver->unsubscribe(server_dpeName.c_str());

  return;
}

//this method is to read float from DCS
//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::ddcSendRequest(string& dpeName, float& value, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  // IS server and PVSS DataPoint address
  string server_dpeName = m_requestServer + "." + dpeName;

  // first subscribe for the request repply 
  rc = m_infoReceiver->subscribe(server_dpeName.c_str(), requestCallback, (void*)this);
  if(rc != ISInfo::Success) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest()  DDC_ERROR: " << server_dpeName
	   << " subscription fault " << rc << ". No IS server to send request" << endl;
    }
  
  time_t startRequestTime = time(&startRequestTime);
  // then send request

  //==========code from Slava
  ISInfo::Status exportResult = ISInfo::Success;
  string isObName;
  time_t currTime;
  
  isObName = m_requestServer + '.';
  isObName += DDC_DAQ_REQUEST;
  
  DcsSingleData<string>* ddcRequestForData 
    = new DcsSingleData<string>(time(&currTime), m_partition, (char*)isObName.c_str());
  ddcRequestForData->setValue(server_dpeName);
  exportResult = ddcRequestForData->checkin();
  if(exportResult == ISInfo::CommFailure) {
    cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << ctime((const time_t*)&currTime) 
	 << " IS server "
	 << m_requestServer << " for DAQ's data requests in partition "
	 << m_partition.name() << " does not run!" << endl;
  }
  //delete ddcRequestForData;
  //===========end of code from Slava
  
  // wait for the request to be ready or the time to be out
  int deltaRequestTime = 0;
  while(!m_requestReady && deltaRequestTime< timeout)
    {
      usleep(10); // wait in micro seconds
      time_t timenow = time(&timenow);
      deltaRequestTime = timenow - startRequestTime;
    }
  // if time is out
  if(!m_requestReady)
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: request waiting time for "
	   <<dpeName<<" is out!!"<<endl;
      m_infoReceiver->unsubscribe(server_dpeName.c_str());
      value = 9999999;
      delete ddcRequestForData;
      return;
    }

  //retune requested value
  value = m_floatRequestResponse;
 //reset request to not ready (enable fro another request)
  m_requestReady = false;

  // Remove the request
  ddcRequestForData->remove();
  delete ddcRequestForData;
  m_infoReceiver->unsubscribe(server_dpeName.c_str());

  return;
}

//this method is to read string from DCS
//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::ddcSendRequest(string& dpeName, string& value, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  // IS server and PVSS DataPoint address
  string server_dpeName = m_requestServer + "." + dpeName;

  // first subscribe for the request repply 
  rc = m_infoReceiver->subscribe(server_dpeName.c_str(), requestCallback, (void*)this);
  if(rc != ISInfo::Success) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << server_dpeName
	   << " subscription fault " << rc << ". No IS server to send request" << endl;
    }
  
  time_t startRequestTime = time(&startRequestTime);
  // then send request

  //==========code from Slava
  ISInfo::Status 	exportResult = ISInfo::Success;
  string 			isObName;
  time_t			currTime;
  
  isObName = m_requestServer + '.';
  isObName += DDC_DAQ_REQUEST;
  
  DcsSingleData<string>* ddcRequestForData 
    = new DcsSingleData<string>(time(&currTime), m_partition, (char*)isObName.c_str());
  ddcRequestForData->setValue(server_dpeName);
  exportResult = ddcRequestForData->checkin();
  if(exportResult == ISInfo::CommFailure) {
    cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " 
	 << ctime((const time_t*)&currTime) << " IS server "
	 << m_requestServer << " for DAQ's data requests in partition "
	 << m_partition.name() << " does not run!" << endl;
  }
  //delete ddcRequestForData;
  //==========end of code from Slava
  
  // wait for the request to be ready or the time to be out
  int deltaRequestTime = 0;
  while(!m_requestReady && deltaRequestTime< timeout)
    {
      usleep(10); // wait in micro seconds
      time_t timenow = time(&timenow);
      deltaRequestTime = timenow - startRequestTime;
    }
  // if time is out
  if(!m_requestReady)
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: request waiting time for "
	   <<dpeName<<" is out!!"<<endl;
      m_infoReceiver->unsubscribe(server_dpeName.c_str());
      value = 9999999;
      delete ddcRequestForData;
      return;
    }

  //retune requested value
  value = m_stringRequestResponse;
  //reset request to not ready (enable fro another request)
  m_requestReady = false;

  // Remove the request
  ddcRequestForData->remove();
  delete ddcRequestForData;
  m_infoReceiver->unsubscribe(server_dpeName.c_str());

  return;
}

//this method is to read several (vector) of float from DCS. Rq: intger are converted to float!
//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::ddcSendRequest(std::vector<string>& dpeName_vec, 
				   std::map<string, float>& values_map, int timeout)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  //number of parameters (request) in the vector
  unsigned int numberOfRequest = dpeName_vec.size();
  // IS server and PVSS DataPoint address
  std::vector<string> server_dpeName;
  // first subscribe for the request repply for all the parameters (datapoint elements)
  for(unsigned int i_requ=0; i_requ<numberOfRequest; i_requ++)
    {
      //name for IS server must be of the form is_server.parameter_path
      server_dpeName.push_back(m_requestServer + "." + dpeName_vec[i_requ]);
      //subscribe
      rc = m_infoReceiver->subscribe(server_dpeName[i_requ].c_str(), requestFloatVecCallback, 
				     (void*)this);
      if(rc != ISInfo::Success) 
	{
	  cerr << " SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << server_dpeName[i_requ]
	   << " subscription fault " << rc << ". No IS server to send request" << endl;
	}
    }

  // initilizations related to request vector of float
  requestReady_vec_init(numberOfRequest);
  //start sending request time
  time_t startRequestTime = time(&startRequestTime);
  // then send request

  //================================code from Slava

  ISInfo::Status exportResult = ISInfo::Success;
  string isObName;
  time_t currTime;
  
  isObName = m_requestServer + '.';
  isObName += DDC_DAQ_REQUEST;
  
  DcsDataArray<string>* ddcRequestForData 
    = new DcsDataArray<string>(time(&currTime), m_partition, (char*)isObName.c_str());
  for(unsigned int i_requ=0; i_requ<numberOfRequest; i_requ++)
    {
      ddcRequestForData->addValue(server_dpeName[i_requ]);
    } 
  
  exportResult = ddcRequestForData->checkin();
  
  if(exportResult == ISInfo::CommFailure) 
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: " << ctime((const time_t*)&currTime) 
	   << " IS server "
	   << m_requestServer << " for DAQ's data requests in partition "
	   << m_partition.name() << " does not run!" << endl;
    }
  //delete ddcRequestForData;
  //==================================end of code from Slava
  // wait for the request to be ready or the time to be out
  int deltaRequestTime = 0;
  while(!getRequestReady_vec() && deltaRequestTime< timeout)
    {
      usleep(10); // wait in micro seconds
      time_t timenow = time(&timenow);
      deltaRequestTime = timenow - startRequestTime;
    }

  // if time is out
  if(!getRequestReady_vec())
    {
      cerr << "SCTDdcRequest::ddcSendRequest() DDC_ERROR: request waiting time for vector"
	   <<" of float is out, or may be just some of the requests didn't come!!"<<endl;
    }

  //retune requested value
  values_map = m_floatVecRequestResponse;

  // Remove the request
  ddcRequestForData->remove();
  delete ddcRequestForData;

  //unsubsribe
  for(unsigned int i_requ=0; i_requ<numberOfRequest; i_requ++)
    {
      //name for IS server must be of the form is_server.parameter_path
      string server_dpeName = m_requestServer + "." + dpeName_vec[i_requ];
      //unsubscribe
      m_infoReceiver->unsubscribe(server_dpeName.c_str());
    }

  return;
}

//---------------------------------------------------------------------------------------------------------
void getCallback(ISCallbackInfo* isCallback)
//---------------------------------------------------------------------------------------------------------
{
  OWLTime value_time;
  //get the calling object (to modify its attributes)
  float *value = (float*)(isCallback->parameter());

  //define ISInfoAny, we don't know appriori the type of the object in the IS
  ISInfoAny dcsData;

  if(isCallback->reason() != ISInfoDeleted) 
    {
      isCallback->value(dcsData);
    }
 
  // usaly the number of attribute is 2: time and value
  int N = dcsData.countAttributes( );

  for ( int i = 0; i < N; i++ )
    {
      switch ( dcsData.getAttributeType() )
	{
	case ISType::Float:
	  //float
	  dcsData >> *value;
	  break;
	  
	case ISType::Time:
	  //OWLTine
	  dcsData >> value_time;
	  break;
	  
	default:
	  cout << " SCTDdcRequest::getCallback() ERROR: Invalid Type" 
	       << (long)dcsData.getAttributeType() << std::endl;
	}
    }
  return;
}

//---------------------------------------------------------------------------------------------------------
void SCTDdcRequest::readFromIS(string dpeName, float& value)
//---------------------------------------------------------------------------------------------------------
{
  
  ISInfo::Status rc;
  // IS server and PVSS DataPoint address
  string server_dpeName = "DDC_DT." + dpeName;

  // first subscribe for the request repply 
  rc = m_infoReceiver->subscribe(server_dpeName.c_str(), getCallback, (void*)(&value));
  if(rc != ISInfo::Success) 
    {
	  cerr << "SCTDdcRequest::readFromIS() DDC_ERROR: " << server_dpeName
	       << " subscription fault " << rc << ". No IS server to send request" << endl;
    }
  return;
}

