/**
ColorScaleManager - class to retrieve configuration information for display GUI
@author robinson
*/
package DisplayGUI;
import java.util.*;
import java.io.*;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import java.awt.Color;


public class ColorScaleManager implements TestDataInfo {
    private static ColorScaleManager instance = new ColorScaleManager();

    private Map defaultMap;  // map of titles to default color scales
    private Map nonDefaultMap; // map of titles to non-default color scales
    private Set persistedSet;  // list of persisted color scales
    private final String IS_MONITOR_TITLE="Number of published objects";
    private final String MODULE_GROUP_SELECTION_TITLE = "Selected module grouping";
    private final String SCTDB_COMPARISON_TITLE = "Comparison with SCTDAQ Reference";
    private final String IVSCAN_TITLE="Sensor Current (microAmps)";

    private ColorScaleManager() {
        Refresh();
        }

    public static ColorScaleManager getInstance() {
        return instance;
    }
    public void Refresh() {
       persistedSet = new HashSet();
       initialiseDefaults();
       readPersistedScales();
    }
    public ColorScale getColorScale(int display, int testIndex, int parameterIndex, int mouseOption, int displayOption) {


       String scaleTitle;
       if((scaleTitle=getTitleString(display,testIndex,parameterIndex,mouseOption,displayOption))==null) {
//           System.err.println("SctGUI::ColorScaleManager: No scale defined for : "+display+","+testIndex+","+parameterIndex+","+displayOption);
           return null;
           }
       if(nonDefaultMap.containsKey(scaleTitle)) return (ColorScale)nonDefaultMap.get(scaleTitle);
       if(defaultMap.containsKey(scaleTitle)) return (ColorScale)defaultMap.get(scaleTitle);
       System.err.println("SctGUI::ColorScaleManager: No scale defined for title '"+scaleTitle+"'");
       return null;
       }

    public void updateScale(ColorScale colorScale, boolean persist) {
       boolean isDefault=false;
       ColorScale defaultScale = new ColorScale((ColorScale)defaultMap.get(colorScale.getTitle()));
       if(colorScale.equals(defaultScale)) {
          nonDefaultMap.remove(colorScale.getTitle());
          isDefault=true;
          }
       else nonDefaultMap.put(colorScale.getTitle(),colorScale);

       if(persist) {
          if(!isDefault) persistedSet.add(colorScale.getTitle());
          persistScales();
          }
       }

    public ColorScale getDefaultColorScale(String scaleTitle) {
        ColorScale cs = (ColorScale)defaultMap.get(scaleTitle);
        return cs;
        }
    public void clearAll() {
        System.out.println("SctGUI: Clearing "+nonDefaultMap.size()+" color scales...");
        persistedSet.clear();
        nonDefaultMap.clear();
        persistScales();
        }


    public boolean isDefault(ColorScale colorScale) {
         return (!nonDefaultMap.containsKey(colorScale.getTitle()));
         }

    public void readPersistedScales() {
       ColorScale thisScale = null;
       Pattern scalePattern = Pattern.compile("%COLORSCALE TITLE=(.*)LOG=(\\d+)GRAD=(\\d+)");
       Pattern pointPattern = Pattern.compile("VALUE=([\\d\\.-]+)LABEL=(.*)RED=(\\d+)GREEN=(\\d+)BLUE=(\\d+)");
       nonDefaultMap = new java.util.HashMap();
       int scaleCount=0;
       try {
       File prefFile = Preferences.PreferencesInterface.getInstance().getPrefsFile();
       if(!prefFile.exists()) return;
       BufferedReader in = new BufferedReader(new FileReader(prefFile));
       String line;
       while((line=in.readLine())!=null) {
          Matcher matcher = scalePattern.matcher(line);
          if(matcher.matches()) {
              if(thisScale!=null) {
//                   thisScale.printData();
                   if(thisScale.getNoPoints()>=2) {
                       nonDefaultMap.put(thisScale.getTitle(),thisScale);
                       persistedSet.add(thisScale.getTitle());
                       scaleCount++;
                       }
                   else System.err.println("SctGUI::ColorScaleManager: <2 scale points!!!");
                   }
              String title = line.substring(matcher.start(1),matcher.end(1));
              boolean isLog = line.substring(matcher.start(2),matcher.end(2)).equals("1");
              boolean isGrad = line.substring(matcher.start(3),matcher.end(3)).equals("1");
              thisScale = new ColorScale(title,isLog,isGrad);
              continue;
              }
          matcher = pointPattern.matcher(line);
          if(matcher.matches()) {
              if(thisScale==null) {
                 System.err.println("SctGUI::ColorScaleManager: Null scale point for line "+line);
                 continue;
                 }
              String value = line.substring(matcher.start(1),matcher.end(1));
              String label = line.substring(matcher.start(2),matcher.end(2));
              String red = line.substring(matcher.start(3),matcher.end(3));
              String green = line.substring(matcher.start(4),matcher.end(4));
              String blue = line.substring(matcher.start(5),matcher.end(5));
              thisScale.add(Double.parseDouble(value),new Color(Integer.parseInt(red),Integer.parseInt(green),Integer.parseInt(blue)),label);
              }
 
           }
       if(thisScale!=null) {
//            thisScale.printData();
            if(thisScale.getNoPoints()>=2) {
                       nonDefaultMap.put(thisScale.getTitle(),thisScale);
                       persistedSet.add(thisScale.getTitle());
                       scaleCount++;
                       }
            else System.err.println("SctGUI::ColorScaleManager: <2 scale points!!!");
            }
       in.close();

        }catch(Exception io) {System.err.println("SctGUI::ColorScaleManager: Failed to read persisted color scales: "+io.toString());}
        if(scaleCount>0) System.out.println("SctGUI::ColorScaleManager: Imported "+scaleCount+" color scales");
       }

    public void persistScales() {
       List contents = new ArrayList();
       try {
       File prefFile = Preferences.PreferencesInterface.getInstance().getPrefsFile();
       if(prefFile.exists()) {
         BufferedReader in = new BufferedReader(new FileReader(prefFile));
         String line;
         while((line=in.readLine())!=null) {
           if(line.startsWith("%COLORSCALE")) break; // get content up to start of color scale definitions
           contents.add(line);
           }
         in.close();
         }
       BufferedWriter out = new BufferedWriter(new FileWriter(prefFile));
       for(int i=0;i<contents.size();i++) {
           out.write((String)contents.get(i));
           out.newLine();
           }
      for (Iterator i=nonDefaultMap.values().iterator(); i.hasNext(); ) {
              ColorScale thisColorScale = (ColorScale) i.next();
              String title = thisColorScale.getTitle();
              if(!persistedSet.contains(title)) continue;
              String isLog = thisColorScale.isLogScale() ? "1" : "0";
              String isGrad = thisColorScale.isGradientPaint() ? "1" : "0";
              out.write("%COLORSCALE TITLE="+title+"LOG="+isLog+"GRAD="+isGrad);
              out.newLine();
              List scalePoints = thisColorScale.getScalePoints();
              for(int y=0;y<scalePoints.size();y++) {
                  ColorScalePoint scalePoint = (ColorScalePoint)scalePoints.get(y);
                  Color color = scalePoint.getColor();
                  out.write("VALUE="+scalePoint.getValue()+"LABEL="+scalePoint.getLabel()+"RED="+color.getRed()+"GREEN="+color.getGreen()+"BLUE="+color.getBlue());
                  out.newLine();
                  }
              }
        out.close();
        }catch(Exception io) {System.err.println("SctGUI::ColorScaleManager: Failed to persist color scales: "+io.toString());}
    }
       

    public void initialiseDefaults() {

        defaultMap = new java.util.HashMap();
        ColorScale colorScale;

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0");
        colorScale.add(1.0, java.awt.Color.green, "1");
        defaultMap.put(colorScale.setTitle(IS_MONITOR_TITLE),colorScale);

        colorScale = new ColorScale(-1.0,java.awt.Color.white, "None");
        colorScale.add(0.0, java.awt.Color.orange, "0");
        colorScale.add(1.0, java.awt.Color.cyan,"1"); 
        colorScale.add(2.0, java.awt.Color.blue,"2"); 
        colorScale.add(3.0, java.awt.Color.pink,"3"); 
        colorScale.add(4.0, java.awt.Color.green,"4");
        colorScale.add(5.0, java.awt.Color.magenta,"5");  
        colorScale.add(6.0, java.awt.Color.yellow,"6"); 
        colorScale.add(7.0, java.awt.Color.lightGray,"7"); 
        defaultMap.put(colorScale.setTitle(MODULE_GROUP_SELECTION_TITLE),colorScale);

        colorScale = new ColorScale(-1.0,java.awt.Color.white, "No Data");
        colorScale.add(1.0, java.awt.Color.green,"Good");
        colorScale.add(2.0, java.awt.Color.red,"Poor"); 
        defaultMap.put(colorScale.setTitle(SCTDB_COMPARISON_TITLE),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.green, "Pass");
        colorScale.add(1.0,java.awt.Color.orange, "Problem");
        colorScale.add(2.0,java.awt.Color.red, "Fail");
        defaultMap.put(colorScale.setTitle("Result Status"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.green, "0",15.0,java.awt.Color.red,"15");
        defaultMap.put(colorScale.setTitle("Defect Count"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.orange, "0",5.0,java.awt.Color.green,"5.0");
        defaultMap.put(colorScale.setTitle("Token"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.orange, "0",5.0,java.awt.Color.green,"5.0");
        defaultMap.put(colorScale.setTitle("RToken"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.orange,"Not Tested");
        colorScale.add(1.0, java.awt.Color.green, "Minimal Tested");
        defaultMap.put(colorScale.setTitle("Token Comment"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.orange,"Not Tested");
        colorScale.add(1.0, java.awt.Color.green, "Minimal Tested");
        defaultMap.put(colorScale.setTitle("RToken Comment"),colorScale);

        colorScale = new ColorScale(0., java.awt.Color.green, "0");
        colorScale.add(1.0,java.awt.Color.red,">0");
        defaultMap.put(colorScale.setTitle("Defective pipelines"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0",20.0,java.awt.Color.red,"20");
        colorScale.add(10.0,java.awt.Color.green,"10");
        colorScale.add(15.0,java.awt.Color.green,"15");
        defaultMap.put(colorScale.setTitle("Strobe Delay"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0",300.0,java.awt.Color.red,"300.");
        colorScale.add(100.0,java.awt.Color.green,"100.");
        colorScale.add(200.0,java.awt.Color.green,"200.");
        defaultMap.put(colorScale.setTitle("VT50(mV)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",10.0,java.awt.Color.red,"10.");
        colorScale.add(1.0,java.awt.Color.green,"1.");
        colorScale.add(5.0,java.awt.Color.green,"5.");
        defaultMap.put(colorScale.setTitle("VT50(mV) RMS"),colorScale);

        colorScale = new ColorScale(30.0, java.awt.Color.red, "30",70.0,java.awt.Color.red,"70");
        colorScale.add(40.0,java.awt.Color.green,"40");
        colorScale.add(60.0,java.awt.Color.green,"60");
        defaultMap.put(colorScale.setTitle("Gain (mV/fC)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",10.0,java.awt.Color.red,"10.");
        colorScale.add(1.0,java.awt.Color.green,"1.");
        colorScale.add(5.0,java.awt.Color.green,"5.");
        defaultMap.put(colorScale.setTitle("Gain RMS"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0.",100.0,java.awt.Color.red,"100.");
        colorScale.add(10.0,java.awt.Color.green,"10");
        colorScale.add(90.0,java.awt.Color.green,"90");
        defaultMap.put(colorScale.setTitle("Offset (mV)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",10.0,java.awt.Color.red,"10.");
        colorScale.add(1.0,java.awt.Color.green,"1.");
        colorScale.add(5.0,java.awt.Color.green,"5.");
        defaultMap.put(colorScale.setTitle("Offset RMS"),colorScale);

        colorScale = new ColorScale(10.0, java.awt.Color.blue, "10.",16.0,java.awt.Color.red,"16.");
        colorScale.add(12.0,java.awt.Color.green,"12.");
        colorScale.add(14.0,java.awt.Color.green,"14.");
        defaultMap.put(colorScale.setTitle("Output Noise (mV)"),colorScale);

        colorScale = new ColorScale(1000.0, java.awt.Color.blue, "1000",2000.0,java.awt.Color.red,"2000");
        colorScale.add(1300.0,java.awt.Color.green,"1300");
        colorScale.add(1600.0,java.awt.Color.green,"1600");
        defaultMap.put(colorScale.setTitle("Input Noise (enc)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0.",100.0,java.awt.Color.red,"100");
        colorScale.add(30.0,java.awt.Color.green,"30.");
        colorScale.add(80.0,java.awt.Color.green,"80.");
        defaultMap.put(colorScale.setTitle("Noise RMS (enc)"),colorScale);

        colorScale = new ColorScale(0.0000001, java.awt.Color.blue, "1E-7",0.0001,java.awt.Color.red,"1E-4");
        colorScale.add(0.000001,java.awt.Color.green,"1E-6");
        colorScale.add(0.00001,java.awt.Color.green,"1E-5");
        colorScale.setLogScale(true);
        defaultMap.put(colorScale.setTitle("Occupancy"),colorScale); 

        colorScale = new ColorScale(0.0000001, java.awt.Color.blue, "1E-7",0.0001,java.awt.Color.red,"1E-4");
        colorScale.add(0.000001,java.awt.Color.green,"1E-6");
        colorScale.add(0.00001,java.awt.Color.green,"1E-5");
        colorScale.setLogScale(true);
        defaultMap.put(colorScale.setTitle("RMS Occupancy"),colorScale); 

        colorScale = new ColorScale(0, java.awt.Color.blue, "0");
        colorScale.add(1,java.awt.Color.green,"1");
        colorScale.add(2,java.awt.Color.red,"2");
        colorScale.add(3,java.awt.Color.yellow,"3");
        defaultMap.put(colorScale.setTitle("Trim Range"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0.",250.0,java.awt.Color.red,"250");
        colorScale.add(50.0,java.awt.Color.green,"50.");
        colorScale.add(200.0,java.awt.Color.green,"200.");
        defaultMap.put(colorScale.setTitle("Target"),colorScale); 

        colorScale = new ColorScale(0., java.awt.Color.green, "0");
        colorScale.add(1.0,java.awt.Color.red,">0");
        defaultMap.put(colorScale.setTitle("N(untrimmed)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",20.0,java.awt.Color.green,"20.");
        defaultMap.put(colorScale.setTitle("Step size (mV)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",1.0,java.awt.Color.green,"1.");
        defaultMap.put(colorScale.setTitle("Step Size RMS"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0",10.0,java.awt.Color.green,"10");
        colorScale.add(12.0,java.awt.Color.green,"12");
        colorScale.add(20.0,java.awt.Color.red,"20");
        defaultMap.put(colorScale.setTitle("TimeWalk(ns)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0",20.0,java.awt.Color.red,"20");
        colorScale.add(10.0,java.awt.Color.green,"10");
        colorScale.add(15.0,java.awt.Color.green,"15");
        defaultMap.put(colorScale.setTitle("TimeWalk(ns)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.red, "0",40.0,java.awt.Color.red,"40");
        colorScale.add(20.0,java.awt.Color.green,"20");
        colorScale.add(30.0,java.awt.Color.green,"30");
        defaultMap.put(colorScale.setTitle("TCal"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",500.0,java.awt.Color.red,"500");
        colorScale.add(200.0,java.awt.Color.green,"200");
        colorScale.add(400.0,java.awt.Color.green,"400");
        defaultMap.put(colorScale.setTitle("Sensor HV(Volts)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",2.0,java.awt.Color.red,"2.0");
        colorScale.add(0.5,java.awt.Color.green,"0.5");
        colorScale.add(1.0,java.awt.Color.green,"1.0");
        defaultMap.put(colorScale.setTitle("Sensor Current(uA)"),colorScale); 

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",35.0,java.awt.Color.red,"35.0");
        colorScale.add(20.0,java.awt.Color.green,"20.0");
        colorScale.add(30.0,java.awt.Color.green,"30.0");
        defaultMap.put(colorScale.setTitle("Hybrid T0(degC)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0",35.0,java.awt.Color.red,"35.0");
        colorScale.add(20.0,java.awt.Color.green,"20.0");
        colorScale.add(30.0,java.awt.Color.green,"30.0");
        defaultMap.put(colorScale.setTitle("Hybrid T1(degC)"),colorScale);

        colorScale = new ColorScale(3.0, java.awt.Color.blue, "3.0",4.0,java.awt.Color.red,"4.0");
        colorScale.add(3.5,java.awt.Color.green,"3.5");
        defaultMap.put(colorScale.setTitle("Chip VCC (Volts)"),colorScale);

        colorScale = new ColorScale(800.0, java.awt.Color.blue, "800",1200.0,java.awt.Color.red,"1200");
        colorScale.add(1000.0,java.awt.Color.green,"1000");
        defaultMap.put(colorScale.setTitle("Chip ICC (mA)"),colorScale);

        colorScale = new ColorScale(3.5, java.awt.Color.blue, "3.5",4.5,java.awt.Color.red,"4.5");
        colorScale.add(4.0,java.awt.Color.green,"4.0");
        defaultMap.put(colorScale.setTitle("Chip VDD (Volts)"),colorScale);

        colorScale = new ColorScale(450.0, java.awt.Color.blue, "450",550.0,java.awt.Color.red,"550");
        colorScale.add(500.0,java.awt.Color.green,"500");
        defaultMap.put(colorScale.setTitle("Chip IDD (mA)"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0.0",24.0,java.awt.Color.yellow,"24.0");
        defaultMap.put(colorScale.setTitle("Rx Delay"),colorScale);

        colorScale = new ColorScale(80.0, java.awt.Color.blue, "80.0",120.0,java.awt.Color.yellow,"120.0");
        defaultMap.put(colorScale.setTitle("Rx Threshold"),colorScale);

        colorScale = new ColorScale(0.0, java.awt.Color.blue, "0.0",2.0,java.awt.Color.red,"2.0");
        colorScale.add(0.5,java.awt.Color.green,"0.5");
        colorScale.add(1.5,java.awt.Color.green,"1.5");
        defaultMap.put(colorScale.setTitle(IVSCAN_TITLE),colorScale);

        } 

    private String getTitleString(int display, int testIndex, int parameterIndex, int mouseOption, int displayOption) {

        
        switch(display) {
        case DisplayStatus.IS_MONITOR:
               return IS_MONITOR_TITLE;
        case DisplayStatus.MODULE_GROUP_SELECTION:
               return MODULE_GROUP_SELECTION_TITLE;
        case DisplayStatus.IV_DATA:
               switch(parameterIndex) {
                  case 0:
                     return "Result Status";
                  case 1:
                     return IVSCAN_TITLE;
                  case 2:
                     if(mouseOption==-1) return null;
                     return dcsOptions[mouseOption];
                  default:
                  }
        case DisplayStatus.TEST_DATA:
           switch(parameterIndex) {
              case 0: // result status
              case 1: // defect count
                return testParameters[testIndex][parameterIndex];
              case 2: // DCS stuff
                if(mouseOption==-1) return null;
                return dcsOptions[mouseOption];
              default:
                switch(displayOption) {
                  case 0:
                  case 1:
                    if(testIndex==-1 || parameterIndex==-1) return null;
                    return testParameters[testIndex][parameterIndex];
                  default:
                   return SCTDB_COMPARISON_TITLE;    
                  } 
              }        
        case DisplayStatus.SCTDB_DATA:
        case DisplayStatus.SCTDB_RODDAQ_DATA:
           if(testIndex==-1 || parameterIndex==-1) return null;
           switch(parameterIndex) {
              case 2: // DCS stuff
                return dcsOptions[mouseOption];
              default:
                return testParameters[testIndex][parameterIndex];
                }
        case DisplayStatus.DCS_DATA:
              return dcsOptions[parameterIndex];
        default:
        }
        return null;
        } 



}
