/*
 * ColorScale.java
 *
 * Created on 26 June 2003, 14:34
 */

package DisplayGUI;
import java.awt.Color;
import java.util.*;
import java.awt.Graphics2D.*;
import java.awt.geom.*;
import java.awt.GradientPaint;
import java.awt.FontMetrics;
/**
 * Class to hold and draw color scale information
 * @author  robinson
 */
public class ColorScale implements displayParams {
//    Vector scalePoints;
    List scalePoints;
    boolean isGradientPaint;                               ///<Does scale contain gradient colors?
    boolean logScale;                                      ///<Display log scale
    String title=null;
    SortedMap sortedMap;

    /** Creates a new instance of colorScale with gradient colors */
    public ColorScale(double xlow, Color loColor, String loLabel, double xHigh, Color hiColor, String hiLabel) {
//        scalePoints = new Vector();
//        scalePoints.addElement(new ColorScalePoint(new Double(xlow),loColor,loLabel));
//        scalePoints.addElement(new ColorScalePoint(new Double(xHigh),hiColor,hiLabel));
        scalePoints = new ArrayList();
        scalePoints.add(new ColorScalePoint(new Double(xlow),loColor,loLabel));
        scalePoints.add(new ColorScalePoint(new Double(xHigh),hiColor,hiLabel));
        isGradientPaint=true;
        logScale=false;
    }
    public ColorScale(String title, boolean isLog, boolean isGrad) {
         this.title=title;
         logScale=isLog;
         isGradientPaint=isGrad;
         scalePoints = new ArrayList();
//         scalePoints=new Vector();
         }


    /**  Creates an instance of colorScale with solid colors */
    public ColorScale(double xlow, Color loColor, String loLabel) {
        scalePoints = new ArrayList();
        scalePoints.add(new ColorScalePoint(new Double(xlow),loColor,loLabel));
//        scalePoints = new Vector();
//        scalePoints.addElement(new ColorScalePoint(xlow,loColor,loLabel));
        isGradientPaint=false;
        logScale=false;
        }

    public ColorScale(String title, List scalePoints, boolean isGradientPaint, boolean isLogScale) {
        this.scalePoints = scalePoints;
        this.title=title;
        this.isGradientPaint=isGradientPaint;
        this.logScale=isLogScale;
        }
    public ColorScale(ColorScale newScale) {
        title=newScale.getTitle();
        isGradientPaint=newScale.isGradientPaint();
        logScale=newScale.isLogScale();
        scalePoints = new ArrayList(newScale.getScalePoints());
        }

    public String setTitle(String title) {
         this.title=title;
         return title;
         }
    public String getTitle() {
         if(title==null) return "No title";
         return title;
         }
    public void setLogScale(boolean log) {
        logScale=log;
        }
    public boolean isLogScale() {
        return logScale;
        }
    public void setGradientPaint(boolean isGradientPaint) {
        this.isGradientPaint=isGradientPaint;
        }
    public boolean isGradientPaint() {
        return isGradientPaint;
        }

    /**  Add a new point on the scale with specified value, color, and label */
    public void add(double value, Color color, String label) {
        scalePoints.add(new ColorScalePoint(new Double(value),color,label));
        orderList();
        }
//************************************************* reorder list by value
    private void orderList() {

        sortedMap = new TreeMap();              // sort the points by value
        for(int i=0;i<scalePoints.size();i++) {
           ColorScalePoint csp = (ColorScalePoint)scalePoints.get(i);
           sortedMap.put(csp.getValue(),csp);
           }
        scalePoints = new ArrayList();
        for (Iterator i=sortedMap.values().iterator(); i.hasNext(); ) {       
          scalePoints.add((ColorScalePoint)i.next());
          }        
    }

    /**  Draw the color scale at x,y */    
    public void drawColorScale(java.awt.Graphics2D g2, int x, int y) {

            double firstDataPoint = ((ColorScalePoint)scalePoints.get(0)).getValue().doubleValue();
            double lastDataPoint = ((ColorScalePoint)scalePoints.get(scalePoints.size()-1)).getValue().doubleValue();
            if(!isGradientPaint) {
                 drawSolidColorScale(g2,x,y);
                 return;
                 }
            if(logScale) {
                lastDataPoint = Math.log(lastDataPoint)/Math.log(10.0);
                firstDataPoint = Math.log(firstDataPoint)/Math.log(10.0);
                }

            Rectangle2D.Double drawRect;
            ColorScalePoint firstPoint = (ColorScalePoint)scalePoints.get(0);
            double lastValue = firstPoint.getValue().doubleValue();
            if(logScale) lastValue = Math.log(lastValue)/Math.log(10.0); // get log to base 10
            Color lastColor = firstPoint.getColor();
            String thisLabel = firstPoint.getLabel();
            
            drawAxisLabel(g2,x,y, thisLabel);
            
            int xBox = x;
                      
            for(int i=0;i<scalePoints.size();i++) {
                ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
                double nextValue = thisPoint.getValue().doubleValue();
                if(logScale) nextValue=Math.log(nextValue)/Math.log(10.0);
                Color nextColor = thisPoint.getColor();
                thisLabel = thisPoint.getLabel();
 
                double thisFraction = (nextValue-lastValue)/(lastDataPoint-firstDataPoint);
                int xValue = (int)(thisFraction*(double)colorBoxWidth);
//                System.out.println("Drawing for value "+Double.toString(nextValue));
//                System.out.println("Drawing for color "+nextColor.toString());
//                System.out.println("Drawing for label "+thisLabel);
//                System.out.println("Fraction="+Double.toString(thisFraction));
//                System.out.println("xValue "+Integer.toString(xValue));
    
                drawAxisLabel(g2,xBox+xValue,y, thisLabel); 
                
                Color fillColor= lastColor;


                if(!nextColor.equals(lastColor)) {
                    GradientPaint colorGradient = new GradientPaint(xBox, y, lastColor,xValue+xBox,y,nextColor);
                    g2.setPaint(colorGradient);
                }
                else {
                    g2.setPaint(lastColor);
                   }                    

                int xWidth = (i==scalePoints.size()-1) ? x+colorBoxWidth-xBox : xValue;
                drawRect = new Rectangle2D.Double(xBox, y, xWidth, colorBoxHeight);
                g2.fill(drawRect);
                // now the tickmarks and labels
                                
                lastColor=nextColor;
                lastValue=nextValue;
                xBox+=xValue;
            }
            
            // now draw the outline
             drawRect = new Rectangle2D.Double(x, y, colorBoxWidth, colorBoxHeight);
//             System.out.println("Bad2: "+Double.toString(drawRect.getX())+","+Double.toString(drawRect.getY())+","+Double.toString(drawRect.getWidth())+","+Double.toString(drawRect.getHeight()));
             g2.setPaint(Color.black);
             g2.draw(drawRect);
            
            
 
            
    }

    public void drawSolidColorScale(java.awt.Graphics2D g2, int x, int y) {

            Rectangle2D.Double drawRect;            
            int nboxes = scalePoints.size();
            int boxWidth = (int)((double)colorBoxWidth/(double)nboxes);        
            
            int xBox = 0;
            
            for(int i=0;i<scalePoints.size();i++) {
                ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
                double nextValue = thisPoint.getValue().doubleValue();
                if(logScale) nextValue=Math.log(nextValue)/Math.log(10.0);
                Color nextColor = thisPoint.getColor();
                String thisLabel = thisPoint.getLabel();
                
                drawAxisLabel(g2,x+xBox+boxWidth/2,y, thisLabel);                 
                g2.setPaint(nextColor);

                drawRect = new Rectangle2D.Double(x+xBox, y, boxWidth, colorBoxHeight);
                g2.fill(drawRect);
                // now the tickmarks and labels
                                
                xBox+=boxWidth;
            }
            
            // now draw the outline
             drawRect = new Rectangle2D.Double(x, y, nboxes*boxWidth, colorBoxHeight);
//             System.out.println("Bad2: "+Double.toString(drawRect.getX())+","+Double.toString(drawRect.getY())+","+Double.toString(drawRect.getWidth())+","+Double.toString(drawRect.getHeight()));
             g2.setPaint(Color.black);
             g2.draw(drawRect);
                        
    }

    /** Draw axis label at x,y */
    public void drawAxisLabel(java.awt.Graphics2D g2, int x, int y, String label) {
        g2.setPaint(Color.black);

        g2.draw(new Line2D.Double(x, y, x, y-tickHeight));
        g2.setFont(displayScaleFont);
        g2.setPaint(darkBlueColor);
        FontMetrics metrics = g2.getFontMetrics();
        int stringWidth = metrics.stringWidth(label);
        int stringHeight = metrics.getHeight();  
        g2.drawString(label,x-stringWidth/2,y-tickHeight-tickLabelOffSet);
 
    }

    /** get color for given value */
    public Color getColor(Double dValue) {
//      System.out.println("requested color for "+value);
//      printData();
      if(dValue==null) return Color.white;
      double value =dValue.doubleValue();

      if(!isGradientPaint) {
 
          for(int i=0;i<scalePoints.size();i++) {
               ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
                double thisValue = thisPoint.getValue().doubleValue();
                if(value<=thisValue) return thisPoint.getColor();
                }
           return ((ColorScalePoint)scalePoints.get(scalePoints.size()-1)).getColor();
         }
       else {
           if(logScale) value = Math.log(value)/Math.log(10.0);
           Color loColor = ((ColorScalePoint)scalePoints.get(0)).getColor();
           double loValue = ((ColorScalePoint)scalePoints.get(0)).getValue().doubleValue();
           if(logScale) loValue = Math.log(loValue)/Math.log(10.0);
           if(value<=loValue) return loColor;

           for(int i=1;i<scalePoints.size();i++) {
             ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
//             System.out.println("Point "+i+":"+thisPoint.getValue().toString()+","+thisPoint.getLabel()+","+thisPoint.getColor().toString());

             double hiValue = (logScale) ? Math.log(thisPoint.getValue().doubleValue())/Math.log(10.0) : thisPoint.getValue().doubleValue();
             Color hiColor = thisPoint.getColor();
//             System.out.println("Data between "+Double.toString(loValue)+" and "+Double.toString(hiValue));
             if(value<=hiValue) {
                    int loRed = loColor.getRed();
                    int hiRed = hiColor.getRed();
                    int loBlue = loColor.getBlue();
                    int hiBlue = hiColor.getBlue();
                    int loGreen = loColor.getGreen();
                    int hiGreen = hiColor.getGreen();
                    double fraction = (value-loValue)/(hiValue-loValue);
//                    System.out.println("fraction="+Double.toString(fraction));
//                    System.out.println("scalecolor: "+loRed+","+hiRed+":"+loGreen+","+hiGreen+":"+loBlue+","+hiBlue);
                    if(fraction<0.0) fraction=0.0;
                    if(fraction>1.0) fraction=1.0;
                    double redValue = (double)(hiRed-loRed)*fraction;
                    double blueValue = (double)(hiBlue-loBlue)*fraction;
                    double greenValue = (double)(hiGreen-loGreen)*fraction;
//                    int newRed=loRed+(int)redValue;
//                    int newGreen=loGreen+(int)greenValue;
//                    int newBlue=loBlue+(int)blueValue;
//                    System.out.println("newcolor: "+newRed+","+newGreen+","+newBlue);
                    return new Color(loRed+(int)redValue,loGreen+(int)greenValue,loBlue+(int)blueValue);
                    }
               loColor=hiColor;
               loValue=hiValue;
              }
            return loColor;  // ie, return the hiColor because value must be > greatesthiValue
           }

    }

   public String getLabel(Double value) {
         if(value==null) return new String("No Data");
         try {
           if(sortedMap.containsKey(value)) {
             String theLabel = ((ColorScalePoint)sortedMap.get(value)).getLabel(); 
             if(theLabel.indexOf("<")==-1 && theLabel.indexOf(">")==-1) return theLabel;
             }
          }catch(Exception e){};  // in case value cannot be represented as a number
          return value.toString();
         }

  public List getScalePoints() {
      return scalePoints;
     }
  public int getNoPoints() {
      return scalePoints.size();
     }
  public ColorScalePoint getColorScalePoint(int index) {
      if(index<0) return null;
      return (ColorScalePoint) scalePoints.get(index);
      }
  public void setColorScalePoint(int index, ColorScalePoint csp) {
      scalePoints.set(index,csp);
      }
  public void setLastPoint(double lastValue) {
      ColorScalePoint scalePoint = (ColorScalePoint)scalePoints.get(scalePoints.size()-1);
      scalePoint.setValue(new Double(lastValue));
      scalePoint.setLabel(Double.toString(lastValue));
      }
  public void setLastPoint(int lastValue) {
      ColorScalePoint scalePoint = (ColorScalePoint)scalePoints.get(scalePoints.size()-1);
      scalePoint.setValue(new Double(lastValue));
      scalePoint.setLabel(Integer.toString(lastValue));
      }
  public void removePoint(int index) {
      if(index<0) return;
      scalePoints.remove(index);
      }
  public void add(ColorScalePoint newPoint) {
      scalePoints.add(newPoint);
      orderList();
      }

  public void printData() {
      System.out.println("ColorScale");
      System.out.println("----------");
      System.out.println(title);
      System.out.println("log="+logScale+" gradient="+isGradientPaint);
      System.out.println("Points:");
      for(int i=0;i<scalePoints.size();i++) {
       ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
       System.out.println("value,label,color="+thisPoint.getValue().doubleValue()+","+thisPoint.getLabel()+","+thisPoint.getColor().toString());
       }
      }

  public boolean equals(ColorScale otherScale) {
      if(logScale != otherScale.isLogScale()) return false;
      if(isGradientPaint != otherScale.isGradientPaint()) return false;
      if(!title.equals(otherScale.getTitle())) return false;
      if(scalePoints.size()!=otherScale.getNoPoints()) return false;
      List otherScalePoints = otherScale.getScalePoints();
      for(int i=0;i<scalePoints.size();i++) {
         ColorScalePoint thisPoint = (ColorScalePoint)scalePoints.get(i);
         ColorScalePoint otherPoint = (ColorScalePoint)otherScalePoints.get(i);
         if(!thisPoint.equals(otherPoint)) return false;
         }
       return true;
       }
    

    
}
