#include <stdlib.h>

#include <is/isinfo.h>
#include <rc/RCStateInfo.h>
#include <rc/RunParamsInfo.h>
#include <boost/lexical_cast.hpp>

#include <sys/types.h>
#include <sys/stat.h>

#include "configserver.h"
#include "Sct/ILUMarshalling.h"
#include "Sct/SctNames.h"
#include "Sct/IoExceptions.h"
#include "marshalling.h"

using namespace Sct;
using namespace SctConfiguration;
using namespace boost;

typedef sctConf_T_ilu__prefix_idlExceptionType_ConfigurationException ConfException;

static void myRaise(sctConfStatus *_status, ConfigurationException &c);

using namespace std;

void RCCallback(ISCallbackInfo * isc) {
    ConfigServer* ob = (ConfigServer*)isc->parameter();
    RCStateInfo info;
    isc->value(info);
    ob->RCEvent(info);
}

ConfigServer::ConfigServer(const char* oid, IPCServer* server) : IPCObject(oid, server), 
    infoReceiver(new ISInfoReceiver(SctNames::getPartition())), lastState(RCStateInfo::ACT_Initial)
{
    ///This fails if sctStructure has been updated without updating the definition of ABCDModule in configuration.idl
    assert(sizeof(sctConf_T_ABCDModule) == sizeof(ABCDModule));
    
    //Subscribe for RC updates
    infoReceiver->subscribe("RunCtrl", "RootCtrl", RCCallback, this);
}

ConfigServer::~ConfigServer() 
{
}

void ConfigServer::RCEvent(const RCStateInfo& info) {
    if (info.activity == RCStateInfo::ACT_Paused) return;
    
    //Get RunParams
    RunParamsInfo runInfo;
    ISInfo::Status status = SctNames::getISDictionary().findValue("RunParams.RunParams", runInfo);
    if (status != ISInfo::Success) {
	IsException ise(status, "Could not get RunParams", __FILE__, __LINE__);
	ise.sendToMrs(MRS_DIAGNOSTIC);
    }
    
    
    if (info.activity != lastState && info.activity == RCStateInfo::ACT_Running) {
	//Enter running
	string fileName = SctNames::getPersistentDir() + "/configArchive_" + lexical_cast<string>(runInfo.run_number) + "_start.xml.gz";
	config.saveConfiguration(fileName);
        chmod(fileName.c_str(), S_IRUSR | S_IRGRP);
    }
    
    if (info.activity != lastState && lastState == RCStateInfo::ACT_Running) {
	//Exit running
	string fileName = SctNames::getPersistentDir() + "/configArchive_" + lexical_cast<string>(runInfo.run_number) + "_end.xml.gz";
	config.saveConfiguration(fileName);	
        chmod(fileName.c_str(), S_IRUSR | S_IRGRP);
    }
    lastState = info.activity;
}

sctConf_T_PartitionList ConfigServer::listPartitions (sctConfStatus *_status) 
{
  sctConf_T_PartitionList result;

  result = _sctConf_T_PartitionList_sequence::Create(4, NULL);
  result->Clear(0);

  list<unsigned int> parts = config.listPartitions();

  for(list<unsigned int> :: const_iterator p = parts.begin();
      p!=parts.end();
      p++) {
    result->Append(*p);
  }

  return result;
}

sctConf_T_CrateList ConfigServer::listCratesInPartition (sctConfStatus *_status, ilu_Cardinal partition) 
{
  sctConf_T_CrateList result;

  result = _sctConf_T_CrateList_sequence::Create(4, NULL);
  result->Clear(0);

  list<unsigned int> crates = config.listCratesInPartition(partition);

  for(list<unsigned int> :: const_iterator c = crates.begin();
      c!=crates.end();
      c++) {
    result->Append(*c);
  }

  return result;
}

ilu_Boolean ConfigServer::isDummyCrate (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate)
{
  ilu_Boolean result = false;
  try {
    result = config.isDummyCrate(partition, crate);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_RodList ConfigServer::listRodsInCrate (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate) 
{
  sctConf_T_RodList result;

  result = _sctConf_T_RodList_sequence::Create(4, NULL);
  result->Clear(0);

  list<unsigned int> rods = config.listRodsInCrate(partition, crate);

  for(list<unsigned int> :: const_iterator r = rods.begin();
      r!=rods.end();
      r++) {
    result->Append(*r);
  }

  return result;
}

sctConf_T_MURList ConfigServer::listMURSInRod (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod) 
{
  sctConf_T_MURList result;

  result = _sctConf_T_MURList_sequence::Create(4, NULL);
  result->Clear(0);

  list<unsigned int> MURs = config.listMURSInRod(partition, crate, rod);

  for(list<unsigned int> :: const_iterator m = MURs.begin();
      m!=MURs.end();
      m++) {
    result->Append(*m);
  }

  return result;
}

sctConf_T_ModuleList ConfigServer::listModulesInMUR (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal MUR) 
{
  sctConf_T_ModuleList result;

  result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
  result->Clear(0);

  list<string> mods = config.listModulesInMUR(partition, MUR);

  for(list<string> :: const_iterator m = mods.begin();
      m!=mods.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

sctConf_T_ModuleList ConfigServer::listRModulesInMUR (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal MUR) 
{
  sctConf_T_ModuleList result;

  result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
  result->Clear(0);

  list<string> mods = config.listRModulesInMUR(partition, MUR);

  for(list<string> :: const_iterator m = mods.begin();
      m!=mods.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

sctConf_T_ModuleList ConfigServer::listAllModules (sctConfStatus *_status) 
{
  sctConf_T_ModuleList result = 0;

  try {
    result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
    result->Clear(0);

    list<string> mods = config.listAllModules();

    for(list<string> :: const_iterator m = mods.begin();
        m!=mods.end();
        m++) {
      char *copy = (char *)malloc((m->length()+1) * sizeof(char));
      strncpy(copy, m->c_str(), m->length());
      copy[m->length()] = 0;
      result->Append(copy);
    }
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_ModuleList ConfigServer::listUnusedModules (sctConfStatus *_status) 
{
  sctConf_T_ModuleList result = 0;
  try {
    result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
    result->Clear(0);

    list<string> mods = config.listUnusedModules();

    for(list<string> :: const_iterator m = mods.begin();
        m!=mods.end();
        m++) {
      char *copy = (char *)malloc((m->length()+1) * sizeof(char));
      strncpy(copy, m->c_str(), m->length());
      copy[m->length()] = 0;
      result->Append(copy);
    }
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_ABCDModule* ConfigServer::getModuleConfig (sctConfStatus *_status, ilu_T_CString mod) 
{
  try {
    ABCDModule realConfig = config.getModuleConfig(mod);
    return copyABCDModuleToILU(realConfig);

  } catch(ConfigurationException &c) {
    myRaise(_status, c);
    return 0;
  }
}

ilu_ShortCardinal ConfigServer::getModuleGroup (sctConfStatus *_status, ilu_T_CString mod)
{
  try {
    return config.getModuleGroup(mod);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::setModuleGroup (sctConfStatus *_status, ilu_T_CString mod, ilu_ShortInteger group)
{
  try {
    config.setModuleGroup(mod, group);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_RodConfig * ConfigServer::getRodConfig (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod) 
{
  sctConf_T_RodConfig *result = 0;

  try {
    result = copyRodConfigToILU(config.getRodConfig(partition, crate, rod));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_SlaveConfig * ConfigServer::getDefaultSlaveConfig (sctConfStatus *_status, ilu_Byte type)
{
  sctConf_T_SlaveConfig *result = 0;

  try {
    result = copySlaveConfigToILU(config.getDefaultSlaveConfig(type));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_TimConfig * ConfigServer::getTimConfig (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate) 
{
  TimConfig myTim;
  try {
    myTim = config.getTimConfig(partition, crate);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return copyTimConfigToILU(myTim);
}

sctConf_T_CharBlock ConfigServer::getFibreMappings (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  char * mappings = 0;
  try {
    mappings = config.getFibreMappings(partition, crate, rod);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
  return copyArrayToILU<_sctConf_T_CharBlock_sequence, char>(mappings, 48*3);
}

void ConfigServer::setFibreMapping (sctConfStatus *_status, 
                                    ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                    ilu_Cardinal channel, ilu_Cardinal tx, ilu_Cardinal rx0, ilu_Cardinal rx1)
{
  try {
    config.setFibreMapping(partition, crate, rod, channel, tx, rx0, rx1);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_MURType ConfigServer::getMURType (sctConfStatus *_status, ilu_Cardinal MUR) 
{
  MURType result = UNKNOWN;
  try {
    result = config.getMURType(MUR);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return (sctConf_T_MURType)result;
}

ilu_Cardinal ConfigServer::getFreeMurId (sctConfStatus *_status) 
{
  try {
    return config.getFreeMurId();
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
 }
}

void ConfigServer::translateToROD (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, 
                                   ilu_Cardinal * partition, ilu_Cardinal * crate, ilu_Cardinal * rod, ilu_Cardinal * channel) 
{
  try {
    unsigned int partitionOut, crateOut, rodOut, channelOut;

    config.translateToROD(MUR, mod, partitionOut, crateOut, rodOut, channelOut);

    *partition = partitionOut;
    *crate = crateOut;
    *rod = rodOut;
    *channel = channelOut;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToRROD (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, 
                                   ilu_Cardinal * partition, ilu_Cardinal * crate, ilu_Cardinal * rod, ilu_Cardinal * channel) 
{
  try {
    unsigned int partitionOut, crateOut, rodOut, channelOut;

    config.translateToRROD(MUR, mod, partitionOut, crateOut, rodOut, channelOut);

    *partition = partitionOut;
    *crate = crateOut;
    *rod = rodOut;
    *channel = channelOut;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToSN (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, 
                                  ilu_T_CString * sn) 
{
  try {
    std::string serialNum;

    config.translateToSN(MUR, mod, serialNum);

    *sn = new char[serialNum.length()+1];

    memcpy(*sn, serialNum.c_str(), serialNum.length());
    (*sn)[serialNum.length()] = 0;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_BOCChannelConfig * ConfigServer::getBOCConfig (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal channel) 
{
  sctConf_T_BOCChannelConfig *res = 0;
  try {
    res = new sctConf_T_BOCChannelConfig;

    BOCChannelConfig ret = config.getBOCConfig(partition, crate, rod, channel);

    res->current = ret.current;
    res->delay = ret.delay;
    res->markSpace = ret.markSpace;

    res->threshold0 = ret.threshold0;
    res->delay0 = ret.delay0;

    res->threshold1 = ret.threshold1;
    res->delay1 = ret.delay1;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return res;
}

sctConf_T_BOCGlobalConfig * ConfigServer::getBOCGlobalConfig (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod) 
{
  sctConf_T_BOCGlobalConfig *res = 0;
  try {
    res = copyBOCGlobalConfigToILU(config.getBOCGlobalConfig(partition, crate, rod));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return res;
}

void ConfigServer::printModuleConfig (sctConfStatus *_status, sctConf_T_ABCDModule* conf) 
{
  try {
    config.printModuleConfig(copyILUToABCDModule(*conf));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::configureBOCChannel (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal position, sctConf_T_BOCChannelConfig * conf) 
{
  try {
    BOCChannelConfig newConf;

    newConf.current = conf->current;
    newConf.delay = conf->delay;
    newConf.markSpace = conf->markSpace;

    newConf.threshold0 = conf->threshold0;
    newConf.delay0 = conf->delay0;

    newConf.threshold1 = conf->threshold1;
    newConf.delay1 = conf->delay1;

    config.configureBOCChannel(MUR, position, newConf); 
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToRMUR (sctConfStatus *_status, ilu_Cardinal RMUR, ilu_Cardinal rmod, ilu_Cardinal * MUR, ilu_Cardinal * mod)
{
  try {
    unsigned int m, module;
    config.translateToRMUR(RMUR, rmod, m, module);
    *MUR = m;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToBarrel (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, 
                                      ilu_Cardinal * barrel, ilu_Cardinal * row, ilu_Integer * number) 
{
  try {
    unsigned int b, r;
    int num;

    config.translateToBarrel(MUR, mod, b, r, num);

    *barrel = b;
    *row = r;
    *number = num;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToEndcap (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, 
                                      ilu_Integer * disk, ilu_Cardinal * ring, ilu_Cardinal * number) 
{
  try {
    int d;
    unsigned int r, n;
    config.translateToEndcap(MUR, mod, d, r, n);
    *disk = d;
    *ring = r;
    *number = n;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateToPowerSupply (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, ilu_Cardinal * partition, ilu_Cardinal * crate, ilu_Cardinal * channel) 
{
  try {
    unsigned int partitionOut, crateOut, channelOut;

    config.translateToPowerSupply(MUR, mod, partitionOut, crateOut, channelOut);

    *partition = partitionOut;
    *crate = crateOut;
    *channel = channelOut;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromROD (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal channel, 
                                     ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromROD(partition, crate, rod, channel, M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromRROD (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal channel, 
                                     ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromRROD(partition, crate, rod, channel, M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromSN (sctConfStatus *_status, ilu_T_CString sn, 
                                    ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromSN(std::string(sn), M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromRMUR (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal mod, ilu_Cardinal * RMUR, ilu_Cardinal * rmod)
{
  try {
    unsigned int rm, rmodule;
    config.translateFromRMUR(MUR, mod, rm, rmodule);
    *RMUR = rm;
    *rmod = rmodule;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromBarrel (sctConfStatus *_status, ilu_Cardinal barrel, ilu_Cardinal row, ilu_Integer number, 
                                        ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromBarrel(barrel, row, number, M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromEndcap (sctConfStatus *_status, ilu_Integer disk, ilu_Cardinal ring, ilu_Cardinal number, 
                                        ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromEndcap(disk, ring, number, M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::translateFromPowerSupply (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal channel, ilu_Cardinal * MUR, ilu_Cardinal * mod) 
{
  try {
    unsigned int M, module;
    config.translateFromPowerSupply(partition, crate, channel, M, module);
    *MUR = M;
    *mod = module;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::configureModuleFromStructure (sctConfStatus *_status, ilu_T_CString mod, sctConf_T_ABCDModule* conf)
{
  ABCDModule modConfig = copyILUToABCDModule(*conf);

  try {
    config.configureModuleFromStructure(mod, modConfig);
//     cout << "Can't update config\n";
//     throw ConfigurationException("Can't update config\n");
  } catch(ConfigurationException &e) {
    myRaise(_status, e);
  }
}

void ConfigServer::configureModuleFromFile (sctConfStatus *_status, ilu_T_CString filename)
{
  try {
    config.configureModuleFromFile(filename);
  } catch(ConfigurationException &e) {
    myRaise(_status, e);
  }
}

void ConfigServer::clearAll (sctConfStatus *_status)
{
  try {
    config.clearAll();
  } catch(ConfigurationException &e) {
    myRaise(_status, e);
  }
}

void ConfigServer::namePartition (sctConfStatus *_status, ilu_Cardinal partition, ilu_T_CString name)
{
  try {
    config.namePartition(partition, name);
  } catch(ConfigurationException &e) {
    myRaise(_status, e);
  }
}

// void ConfigServer::configureROD (sctConfStatus *_status, 
//                                  ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
//                                  sctConf_T_RodConfig * conf) 
// {
//   cout << "Can't update ROD config\n";
//   //  config.updateRodConfig(partition, crate, rod, conf);
// }

void ConfigServer::loadConfiguration (sctConfStatus *_status, ilu_T_CString filename) 
{
  try {
    config.loadConfiguration(filename);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::saveConfiguration (sctConfStatus *_status, ilu_T_CString filename) 
{
  try {
    config.saveConfiguration(filename);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::saveModuleConfiguration (sctConfStatus *_status, ilu_T_CString sn, ilu_T_CString filename) 
{
  try {
    config.saveModuleConfiguration(sn, filename);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

ilu_T_CString ConfigServer::getModuleConfigurationString (sctConfStatus *_status, ilu_T_CString modsn, sctConf_T_ABCDModule* mConf)
{
  ilu_T_CString result = 0;

  try {
    string buffer = config.getModuleConfigurationString(modsn, copyILUToABCDModule(*mConf));

    result = new char[buffer.length() + 1];

    strncpy(result, buffer.c_str(), buffer.length() + 1);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

void ConfigServer::writePowerSupplyConfiguration (sctConfStatus *_status, ilu_T_CString filename)
{
  try {
    config.writePowerSupplyConfiguration(filename);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::mapRODMUR (sctConfStatus *_status, 
                              ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                              ilu_Cardinal order, ilu_Cardinal number) 
{ 
  try {
    config.mapRODMUR(partition, crate, rod, order, number);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::unmapRODMUR (sctConfStatus *_status, 
                              ilu_Cardinal MUR) 
{
  try {
    config.unmapRODMUR(MUR);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::mapBarrelMUR (sctConfStatus *_status, 
                                 ilu_Cardinal MUR, 
                                 ilu_Cardinal barrel, ilu_Cardinal row, ilu_Integer position)
{
  try {
    config.mapBarrelMUR(MUR, barrel, row, position);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::unmapBarrelMUR (sctConfStatus *_status, ilu_Cardinal MUR)
{
  try {
    config.unmapBarrelMUR(MUR);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::mapEndcapMUR (sctConfStatus *_status, 
                                 ilu_Cardinal MUR, 
                                 ilu_Integer disk, ilu_Cardinal quadrant, ilu_Cardinal position)
{
  try {
    config.mapEndcapMUR(MUR, disk, quadrant, position);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::unmapEndcapMUR (sctConfStatus *_status, ilu_Cardinal MUR)
{
  try {
    config.unmapEndcapMUR(MUR);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::mapPowerChannel (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal number, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal channel)
{
  try {
    config.mapPowerChannel(MUR, number, partition, crate, channel);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::unmapPowerChannel (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal number)
{
  try {
    config.unmapPowerChannel(MUR, number);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::mapModuleMUR (sctConfStatus *_status, 
                                 ilu_Cardinal MUR, ilu_Cardinal order, ilu_Cardinal RMUR, ilu_Cardinal rorder, 
                                 ilu_T_CString number) 
{
  try {
    config.mapModuleMUR(MUR, order, RMUR, rorder, number);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::unmapModuleMUR (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal number)
{
  try {
    config.unmapModuleMUR(MUR, number);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::getMapMURROD (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal * partition, ilu_Cardinal * crate, ilu_Cardinal * rod, ilu_Cardinal * order)
{
  try {
    unsigned int partitionOut, crateOut, rodOut, orderOut;

    config.getMapMURROD(MUR, partitionOut, crateOut, rodOut, orderOut);
    *partition = partitionOut;
    *crate = crateOut;
    *rod = rodOut;
    *order = orderOut;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::getMapRODMUR (sctConfStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal order, ilu_Cardinal * MUR)
{
  try {
    unsigned int MURout;
    config.getMapRODMUR(partition, crate, rod, order, MURout);
    *MUR = MURout;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_MURType ConfigServer::getMapMURPhysical (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal * plane, ilu_Cardinal * section, ilu_Cardinal * position)
{ 
  MURType type = UNKNOWN;
  try {
    int p, s, pos;
    config.getMapMURPhysical(MUR, type, p, s, pos);
    *plane = p;
    *section = s;
    *position = pos;
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
  return (sctConf_T_MURType)type;
}

void ConfigServer::swapMURNames (sctConfStatus *_status, 
                                 ilu_Cardinal MUR1, ilu_Cardinal MUR2) 
{
  try {
    config.swapMURNames(MUR1, MUR2);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::modifyDefaultPowerParam (sctConfStatus *_status, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type, ilu_ShortReal value)
{
  try {
    config.modifyDefaultPowerParam(state, name, type, value);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::modifyHVCardParam (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal card, ilu_T_CString name, ilu_T_CString type, ilu_ShortReal value)
{
  try {
    config.modifyHVCardParam(crate, card, name, type, value);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::modifyLVCardParam (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal card, ilu_T_CString name, ilu_T_CString type, ilu_ShortReal value)
{
  try {
    config.modifyLVCardParam(crate, card, name, type, value);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::modifyPowerParam (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal number, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type, ilu_ShortReal value)
{
  try {
    config.modifyPowerParam(MUR, number, state, name, type, value);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::modifyPowerParamCC (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal channel, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type, ilu_ShortReal value)
{
  try {
    config.modifyPowerParamCC(crate, channel, state, name, type, value);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

ilu_ShortReal ConfigServer::getDefaultPowerParam (sctConfStatus *_status, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type)
{
  ilu_ShortReal result = 0.0;
  try {
    result = config.getDefaultPowerParam(state, name, type);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
  return result;
}

ilu_ShortReal ConfigServer::getHVCardParam (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal card, ilu_T_CString name, ilu_T_CString type)

{
  ilu_ShortReal result;

  try {
    result = config.getHVCardParam(crate, card, name, type);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

ilu_ShortReal ConfigServer::getLVCardParam (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal card, ilu_T_CString name, ilu_T_CString type)

{
  ilu_ShortReal result;

  try {
    result = config.getLVCardParam(crate, card, name, type);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

ilu_ShortReal ConfigServer::getPowerParam (sctConfStatus *_status, ilu_Cardinal MUR, ilu_Cardinal number, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type)

{
  ilu_ShortReal result;

  try {
    result = config.getPowerParam(MUR, number, state, name, type);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

ilu_ShortReal ConfigServer::getPowerParamCC (sctConfStatus *_status, ilu_Cardinal crate, ilu_Cardinal channel, ilu_T_CString state, ilu_T_CString name, ilu_T_CString type)

{
  ilu_ShortReal result;

  try {
    result = config.getPowerParamCC(crate, channel, state, name, type);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }

  return result;
}

sctConf_T_NameList ConfigServer::listDefaultPowerParams (sctConfStatus *_status)
{
  try {
    return copyStringListToILU<_sctConf_T_NameList_sequence>(config.listDefaultPowerParams());
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

ilu_ShortReal ConfigServer::getCratePowerParam (sctConfStatus *_status, ilu_Cardinal crate, ilu_T_CString name)
{
  try {
    return config.getCratePowerParam(crate, name);
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_NameList ConfigServer::listDCSChannelParams (sctConfStatus *_status)
{
  try {
    return copyStringListToILU<_sctConf_T_NameList_sequence>(config.listDCSChannelParams());
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_NameList ConfigServer::listDCSCardParams (sctConfStatus *_status)
{
  try {
    return copyStringListToILU<_sctConf_T_NameList_sequence>(config.listDCSCardParams());
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

sctConf_T_NameList ConfigServer::listDCSCrateParams (sctConfStatus *_status)
{
  try {
    return copyStringListToILU<_sctConf_T_NameList_sequence>(config.listDCSCrateParams());
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::configureROD (sctConfStatus *_status, 
                                 ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                 sctConf_T_RodConfig * conf) 
{
  try {
    config.configureROD(partition, crate, rod, copyILUToRodConfig(conf));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::configureTIM (sctConfStatus *_status, 
                                 ilu_Cardinal partition, ilu_Cardinal crate, 
                                 sctConf_T_TimConfig * conf) 
{
  try {
    config.configureTIM(partition, crate, copyILUToTimConfig(conf));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void ConfigServer::configureBOC (sctConfStatus *_status, 
                                 ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                 sctConf_T_BOCGlobalConfig * conf) 
{
  try {
    config.configureBOC(partition, crate, rod, copyILUToBOCGlobalConfig(conf));
  } catch(ConfigurationException &c) {
    myRaise(_status, c);
  }
}

void myRaise(sctConfStatus *_status, ConfigurationException &c) 
{
  ConfException *e = new ConfException;
  char *buffer = new char[strlen(c.what())+1];

  cout << "Raising exception " << c.what() << endl;

  strcpy(buffer, c.what());
  buffer[strlen(c.what())] = 0;
  e->detail = buffer;
  sctConf_G::RaiseException(_status, sctConf_E_ConfigurationException, e);
}
