#include <memory>

#include "configipc.h"
#include "marshalling.h"
#include "Sct/ILUMarshalling.h"

using namespace std;
using namespace Sct;
using namespace SctConfiguration;

void myThrow(sctConfStatus *_status);

ConfigIPC::ConfigIPC()
{
  IPCPartition p("SCT");
  handle = p.lookup(sctConf_C_Configuration_serverName,     // "Test_Server", 
                    sctConf_C_Configuration_instanceName,       // "SCTConfigurationServer", 
                    sctConf_T_Configuration::ILUClassRecord);
  if (handle == 0 ) {
    std::cout << "Error" << std::endl;
    throw ConfigurationException("Can't initialise! (Can't find IPC config)");
  }
}

ConfigIPC::~ConfigIPC()
{
}

std::list<unsigned int> ConfigIPC::listPartitions()
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_PartitionList myList = handle->listPartitions(&status);
  myThrow(&status);

  return copyILUToList<_sctConf_T_PartitionList_sequence, unsigned int>(myList);
}

std::list<unsigned int> ConfigIPC::listCratesInPartition(unsigned int partition)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_CrateList myList = handle->listCratesInPartition(&status, partition);
  myThrow(&status);

  return copyILUToList<_sctConf_T_CrateList_sequence, unsigned int>(myList);
}

std::list<unsigned int> ConfigIPC::listRodsInCrate(unsigned int partition, unsigned int crate)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_RodList myList = handle->listRodsInCrate(&status, partition, crate);
  myThrow(&status);

  return copyILUToList<_sctConf_T_RodList_sequence, unsigned int>(myList);
}

std::list<unsigned int> ConfigIPC::listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_MURList myList = handle->listMURSInRod(&status, partition, crate, rod);
  myThrow(&status);

  return copyILUToList<_sctConf_T_MURList_sequence, unsigned int>(myList);
}

std::list<std::string> ConfigIPC::listModulesInMUR(unsigned int partition, unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_ModuleList myList = handle->listModulesInMUR(&status, partition, MUR);
  myThrow(&status);

  return copyILUToList<_sctConf_T_ModuleList_sequence, string>(myList);
}

std::list<std::string> ConfigIPC::listRModulesInMUR(unsigned int partition, unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_ModuleList myList = handle->listRModulesInMUR(&status, partition, MUR);
  myThrow(&status);

  return copyILUToList<_sctConf_T_ModuleList_sequence, string>(myList);
}

std::list<std::string> ConfigIPC::listAllModules()
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_ModuleList myList = handle->listAllModules(&status);
  myThrow(&status);

  return copyILUToList<_sctConf_T_ModuleList_sequence, string>(myList);
}

std::list<std::string> ConfigIPC::listUnusedModules()
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_ModuleList myList = handle->listUnusedModules(&status);
  myThrow(&status);

  return copyILUToList<_sctConf_T_ModuleList_sequence, string>(myList);
}

bool ConfigIPC::isDummyCrate(unsigned int partition, unsigned int crate)
{
  sctConfStatus status;	
  status.returnCode = 0;
  bool ret = handle->isDummyCrate(&status, partition, crate);
  myThrow(&status);

  return ret;
}


ABCDModule ConfigIPC::getModuleConfig(const std::string module)
{
  sctConfStatus status;	
  status.returnCode = 0;
  auto_ptr<sctConf_T_ABCDModule> iluModule (handle->getModuleConfig(&status, const_cast<char *>(module.c_str())));
  myThrow(&status);

  return copyILUToABCDModule(*iluModule);
}

short ConfigIPC::getModuleGroup(const std::string module)
{
  sctConfStatus status;
  status.returnCode = 0;
  short result = handle->getModuleGroup(&status, const_cast<char *>(module.c_str()));
  myThrow(&status);
  return result;
}

void ConfigIPC::setModuleGroup(const std::string module, short group) 
{
  sctConfStatus status;
  status.returnCode = 0;
  handle->setModuleGroup(&status, const_cast<char *>(module.c_str()), group);
  myThrow(&status);
}

RodConfig ConfigIPC::getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_RodConfig *config = handle->getRodConfig(&status, partition, crate, rod);
  myThrow(&status);

  return copyILUToRodConfig(config);
}

SlaveConfig ConfigIPC::getDefaultSlaveConfig(char type)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_SlaveConfig *config = handle->getDefaultSlaveConfig(&status, type);
  myThrow(&status);

  return copyILUToSlaveConfig(config);
}

BOCChannelConfig ConfigIPC::getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                                         unsigned int channel)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_BOCChannelConfig *config = handle->getBOCConfig(&status, partition, crate, rod, channel);
  myThrow(&status);

  return copyILUToBOCChannelConfig(config);
}

BOCGlobalConfig ConfigIPC::getBOCGlobalConfig(unsigned int partition, unsigned int crate, unsigned int rod)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_BOCGlobalConfig *config = handle->getBOCGlobalConfig(&status, partition, crate, rod);
  myThrow(&status);

  return copyILUToBOCGlobalConfig(config);
}

char *ConfigIPC::getFibreMappings(unsigned int partition, unsigned int crate, unsigned int rod)
{
  sctConfStatus status;	
  status.returnCode = 0;
  char *array = new char[48*3];
  copyILUToArray<sctConf_T_CharBlock, char>(handle->getFibreMappings(&status, partition, crate, rod), array, 48*3);

  myThrow(&status);

  return array;
}

TimConfig ConfigIPC::getTimConfig(unsigned int partition, unsigned int crate)
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_TimConfig *config = handle->getTimConfig(&status, partition, crate);
  myThrow(&status);

  return copyILUToTimConfig(config);
}

void ConfigIPC::setFibreMapping(unsigned int partition, unsigned int crate, unsigned int rod, 
                                unsigned int channel, unsigned int tx, unsigned int rx0, unsigned int rx1)
{
  sctConfStatus status;	
  status.returnCode = 0;
  handle->setFibreMapping(&status, partition, crate, rod, channel, tx, rx0, rx1);
  myThrow(&status);
}

void ConfigIPC::printModuleConfig(const ABCDModule &conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  auto_ptr<sctConf_T_ABCDModule> config (copyABCDModuleToILU(conf));

  handle->printModuleConfig(&status, config.get());
  myThrow(&status);
}

MURType ConfigIPC::getMURType(unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;

  MURType result;

  result = (MURType)handle->getMURType(&status, MUR);
  myThrow(&status);

  return result;
}

unsigned int ConfigIPC::getFreeMurId() 
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned int mur = handle->getFreeMurId(&status);
  myThrow(&status);

  return mur;
}

void ConfigIPC::translateToROD(unsigned int MUR, unsigned int module,
                               unsigned int &partition, unsigned int &crate, 
                               unsigned int &rod, unsigned int &channel)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long p, c, r, ch;
  handle->translateToROD(&status, MUR, module, &p, &c, &r, &ch);
  partition = p; crate = c; rod = r; channel = ch;
  myThrow(&status);
}

void ConfigIPC::translateToRROD(unsigned int MUR, unsigned int module,
                                unsigned int &partition, unsigned int &crate, 
                                unsigned int &rod, unsigned int &channel)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long p, c, r, ch;
  handle->translateToRROD(&status, MUR, module, &p, &c, &r, &ch);
  partition = p; crate = c; rod = r; channel = ch;
  myThrow(&status);
}

void ConfigIPC::translateToSN(unsigned int MUR, unsigned int module,
                              std::string &sn)
{
  sctConfStatus status;	
  status.returnCode = 0;

  char *inputString = 0;

  handle->translateToSN(&status, MUR, module, &inputString);
  myThrow(&status);

  sn = inputString;
}

void ConfigIPC::translateToRMUR(unsigned int MUR, unsigned int module,
                                unsigned int &RMUR, unsigned int &rmodule)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long rm, rmod;
  handle->translateToRMUR(&status, MUR, module, &rm, &rmod);
  RMUR = rm; rmod = rmod;
  myThrow(&status);
}

void ConfigIPC::translateToBarrel(unsigned int MUR, unsigned int module,
                                  unsigned int &barrel, unsigned int &row, int &number)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long b, r;
  long n;
  handle->translateToBarrel(&status, MUR, module, &b, &r, &n);
  barrel = b; row = r; number = n;
  myThrow(&status);
}

void ConfigIPC::translateToEndcap(unsigned int MUR, unsigned int module,
                                  int &disk, unsigned int &ring, unsigned int &number)
{
  sctConfStatus status;	
  status.returnCode = 0;

  long d;
  unsigned long r, n;
  handle->translateToEndcap(&status, MUR, module, &d, &r, &n);
  disk = d; ring = r; number = n;
  myThrow(&status);
}

void ConfigIPC::translateToPowerSupply(unsigned int MUR, unsigned int module,
                                       unsigned int &partition, 
                                       unsigned int &crate, unsigned int &channel)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long p, c, ch;
  handle->translateToPowerSupply(&status, MUR, module, &p, &c, &ch);
  partition = p; crate = c; channel = ch;
  myThrow(&status);
}

void ConfigIPC::translateFromROD(unsigned int partition, unsigned int crate, 
                                 unsigned int rod, unsigned int channel,
                                 unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromROD(&status, partition, crate, rod, channel, &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::translateFromRROD(unsigned int partition, unsigned int crate,
                                  unsigned int rod, unsigned int channel,
                                  unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromRROD(&status, partition, crate, rod, channel, &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::translateFromSN(const std::string sn,
                                unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromSN(&status, const_cast<char *>(sn.c_str()), &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::translateFromRMUR (unsigned int RMUR, unsigned int rmodule, unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long mur, mod;
  handle->translateFromRMUR(&status, RMUR, rmodule, &mur, &mod);
  MUR = mur; module = mod;
  myThrow(&status);
}

void ConfigIPC::translateFromBarrel(unsigned int barrel, unsigned int row, int number,
                                    unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromBarrel(&status, barrel, row, number, &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::translateFromEndcap(int disk, unsigned int ring, unsigned int number,
                                    unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromEndcap(&status, disk, ring, number, &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::translateFromPowerSupply(unsigned int partition, 
                                         unsigned int crate, unsigned int channel, 
                                         unsigned int &MUR, unsigned int &module)
{
  sctConfStatus status;
  status.returnCode = 0;

  unsigned long m, mo;
  handle->translateFromPowerSupply(&status, partition, crate, channel, &m, &mo);
  MUR = m; module = mo;
  myThrow(&status);
}

void ConfigIPC::configureModuleFromStructure(const std::string module, const ABCDModule conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  auto_ptr<sctConf_T_ABCDModule> config (copyABCDModuleToILU(conf));

  handle->configureModuleFromStructure(&status, const_cast<char *>(module.c_str()), config.get());
  myThrow(&status);
}

void ConfigIPC::configureModuleFromFile(const std::string filename)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->configureModuleFromFile(&status, const_cast<char *>(filename.c_str()));
  myThrow(&status);
}

void ConfigIPC::configureROD(unsigned int partition, unsigned int crate, unsigned int rod, RodConfig conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  sctConf_T_RodConfig *config = copyRodConfigToILU(conf);

  handle->configureROD(&status, partition, crate, rod, config);
  myThrow(&status);
}

void ConfigIPC::configureTIM(unsigned int partition, unsigned int crate, TimConfig conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  sctConf_T_TimConfig *config = copyTimConfigToILU(conf);

  handle->configureTIM(&status, partition, crate, config);
  myThrow(&status);
}

void ConfigIPC::configureBOC(unsigned int partition, unsigned int crate, unsigned int rod, const BOCGlobalConfig &conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  sctConf_T_BOCGlobalConfig *config = copyBOCGlobalConfigToILU(conf);

  handle->configureBOC(&status, partition, crate, rod, config);
  myThrow(&status);
}

void ConfigIPC::clearAll()
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->clearAll(&status);
  myThrow(&status);
}

void ConfigIPC::namePartition(unsigned int partition, const std::string name)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->namePartition(&status, partition, const_cast<char *>(name.c_str()));
  myThrow(&status);
}

void ConfigIPC::configureBOCChannel(unsigned int MUR, unsigned int position, const BOCChannelConfig &conf)
{
  sctConfStatus status;	
  status.returnCode = 0;

  sctConf_T_BOCChannelConfig *config = copyBOCChannelConfigToILU(conf);

  handle->configureBOCChannel(&status, MUR, position, config);
  myThrow(&status);
}

void ConfigIPC::mapModuleMUR(unsigned int MUR, unsigned int order, unsigned int RMUR, unsigned int rorder, std::string number)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->mapModuleMUR(&status, MUR, order, RMUR, rorder, const_cast<char *>(number.c_str()));
  myThrow(&status);
}

void ConfigIPC::unmapModuleMUR(unsigned int MUR, unsigned int order)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->unmapModuleMUR(&status, MUR, order);
  myThrow(&status);
}

void ConfigIPC::mapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int number)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->mapRODMUR(&status, partition, crate, rod, order, number);
  myThrow(&status);
}

void ConfigIPC::unmapRODMUR(unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->unmapRODMUR(&status, MUR);
  myThrow(&status);
}

void ConfigIPC::mapBarrelMUR(unsigned int MUR, unsigned int barrel, unsigned int row, int position)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->mapBarrelMUR(&status, MUR, barrel, row, position);
  myThrow(&status);
}

void ConfigIPC::unmapBarrelMUR(unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->unmapBarrelMUR(&status, MUR);
  myThrow(&status);
}

void ConfigIPC::mapEndcapMUR(unsigned int MUR, int disk, unsigned int quadrant, unsigned int position)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->mapEndcapMUR(&status, MUR, disk, quadrant, position);
  myThrow(&status);
}

void ConfigIPC::unmapEndcapMUR(unsigned int MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->unmapEndcapMUR(&status, MUR);
  myThrow(&status);
}

void ConfigIPC::getMapMURROD(unsigned int number, unsigned int &partition, unsigned int &crate, unsigned int &rod, unsigned int &order)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long p, c, r, o;
  handle->getMapMURROD(&status, number, &p, &c, &r, &o);
  partition = p; crate = c; rod = r; order = o;
  myThrow(&status);
}

void ConfigIPC::getMapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, int unsigned order, unsigned int &MUR)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long m;
  handle->getMapRODMUR(&status, partition, crate, rod, order, &m);
  MUR = m;
  myThrow(&status);
}

void ConfigIPC::getMapMURPhysical(unsigned int MUR, MURType &type, int &plane, int &section, int &position)
{
  sctConfStatus status;	
  status.returnCode = 0;

  unsigned long pl, se, ps;
  handle->getMapMURPhysical(&status, type, &pl, &se, &ps);
  plane = pl; section = se; position = ps;
  myThrow(&status);
}

void ConfigIPC::swapMURNames(unsigned int MUR1, unsigned int MUR2)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->swapMURNames(&status, MUR1, MUR2);
  myThrow(&status);
}

void ConfigIPC::mapPowerChannel(unsigned int MUR, unsigned int number, unsigned int partition, unsigned int crate, unsigned int channel)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->mapPowerChannel(&status, MUR, number, partition, crate, channel);
  myThrow(&status);
}

void ConfigIPC::unmapPowerChannel(unsigned int MUR, unsigned int number)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->unmapPowerChannel(&status, MUR, number);
  myThrow(&status);
}

void ConfigIPC::modifyDefaultPowerParam(string state, string name, string type, float value)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->modifyDefaultPowerParam(&status, const_cast<char *>(state.c_str()),
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()), value);
  myThrow(&status);
}
 
void ConfigIPC::modifyHVCardParam(unsigned int crate, unsigned int card, string name, string type, float value)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->modifyHVCardParam(&status, crate, card, 
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()), value);
  myThrow(&status);
}

void ConfigIPC::modifyLVCardParam(unsigned int crate, unsigned int card, string name, string type, float value)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->modifyLVCardParam(&status, crate, card, 
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()), value);
  myThrow(&status);
}

void ConfigIPC::modifyPowerParam(unsigned int MUR, unsigned int number, string state, string name, string type, float value)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->modifyPowerParam(&status, MUR, number, const_cast<char*>(state.c_str()), 
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()), value);
  myThrow(&status);
}

void ConfigIPC::modifyPowerParamCC(unsigned int crate, unsigned int channel, string state, string name, string type, float value)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->modifyPowerParam(&status, crate, channel, const_cast<char*>(state.c_str()), 
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()), value);
  myThrow(&status);
}

float ConfigIPC::getDefaultPowerParam(std::string state, std::string name, std::string type) 
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getDefaultPowerParam(&status, const_cast<char *>(state.c_str()),
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()));
  myThrow(&status);

  return result;
}

float ConfigIPC::getHVCardParam(unsigned int crate, unsigned int card, string name, string type)
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getHVCardParam(&status, crate, card,
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()));
  myThrow(&status);

  return result;
}

float ConfigIPC::getLVCardParam(unsigned int crate, unsigned int card, string name, string type)
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getLVCardParam(&status, crate, card,
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()));
  myThrow(&status);

  return result;
}

float ConfigIPC::getPowerParam(unsigned int MUR, unsigned int number, string state, string name, string type)
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getPowerParam(&status, MUR, number, const_cast<char *>(state.c_str()),
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()));
  myThrow(&status);

  return result;
}

float ConfigIPC::getPowerParamCC(unsigned int crate, unsigned int channel, string state, string name, string type)
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getPowerParam(&status, crate, channel, const_cast<char *>(state.c_str()),
                           const_cast<char*>(name.c_str()), const_cast<char*>(type.c_str()));
  myThrow(&status);

  return result;
}

std::list<std::string> ConfigIPC::listDefaultPowerParams() 
{
  sctConfStatus status;	
  status.returnCode = 0;
  sctConf_T_NameList myList = handle->listDefaultPowerParams(&status);
  myThrow(&status);

  return copyILUToList<_sctConf_T_NameList_sequence, string>(myList);
}

float ConfigIPC::getCratePowerParam(unsigned int crate, string name)
{
  sctConfStatus status;	
  status.returnCode = 0;

  float result = handle->getCratePowerParam(&status, crate, const_cast<char *>(name.c_str()));
  myThrow(&status);

  return result;
}

std::list<std::string> ConfigIPC::listDCSChannelParams() 
{
#warning "Fix these!"
  std::list<std::string> result;
  return result;
}

std::list<std::string> ConfigIPC::listDCSCardParams()
{
  std::list<std::string> result;
  return result;
}

std::list<std::string> ConfigIPC::listDCSCrateParams()
{
  std::list<std::string> result;
  return result;
}

void ConfigIPC::loadConfiguration(const std::string &filename)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->loadConfiguration(&status, const_cast<char *>(filename.c_str()));
  myThrow(&status);
}

void ConfigIPC::saveConfiguration(const std::string filename)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->saveConfiguration(&status, const_cast<char *>(filename.c_str()));
  myThrow(&status);
}

void ConfigIPC::saveModuleConfiguration(const std::string modsn, const std::string filename)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->saveModuleConfiguration(&status, const_cast<char *>(modsn.c_str()), const_cast<char *>(filename.c_str()));
  myThrow(&status);
}

std::string ConfigIPC::getModuleConfigurationString(const std::string modsn, const ABCDModule config) 
{
  sctConfStatus status;	
  status.returnCode = 0;

  std::string result = handle->getModuleConfigurationString(&status, const_cast<char *>(modsn.c_str()), copyABCDModuleToILU(config));
  myThrow(&status);

  return result;
}

void ConfigIPC::writePowerSupplyConfiguration(std::string filename)
{
  sctConfStatus status;	
  status.returnCode = 0;

  handle->writePowerSupplyConfiguration(&status, const_cast<char *>(filename.c_str()));
  myThrow(&status);
}

typedef sctConf_T_ilu__prefix_idlExceptionType_ConfigurationException ConfException;

void myThrow(sctConfStatus *status) 
{
  if(status->returnCode != 0) {
    if(status->returnCode == ilu::ProtocolError) {
      throw ConfigurationException("ILU Protocol Error!"); // + status->values.anyvalue
    }
    ConfException *e = status->values.sctConf_E_ConfigurationException_Value;
 
    throw ConfigurationException(e->detail);
  }
}
