#include <limits>
#include <map>
#include <string>

/// A generic timing thing but local to sctConf at the moment

/// Get intel processor instruction counter
extern __inline__ unsigned long long int rdtsc()
{
  unsigned long long int x;
  __asm__ volatile (".byte 0x0f, 0x31" : "=A" (x));
  return x;
}

class Timings;

/// A timer produces a time to be stored in Timings
class Timer {
  Timings &store;
  std::string name;

  unsigned long long tStart;

 public:
  /// Give a Name and say which Timings to store the time in
  /**
     Automatically starts the Timer
   */
  Timer(Timings &st, std::string n) : store(st), name(n) { start(); } 

  /**
     If end hasn't already been called, then call it
  */
  ~Timer();

  /**
     Start timer
   */
  void start();

  /**
     Stop the timer and update Timings
   */
  void end();
};

/**
   Store of timings which collects some simple statistics
 */
class Timings {
  /**
     Running statistics for one type
   */
  struct Timing {
    int count;
    double sum;
    double min;
    double max;

    Timing() : count(0), sum(0), min(std::numeric_limits<double>::max()), max(0) {}
  };

  std::map<std::string, Timing > timings;

 public:
  Timings() {}

  /**
     Print a dump of all the statistics
   */
  void dumpTimings();

  /** 
     Insert a timing into the "database"
   */
  void addTiming(std::string name, unsigned long long int diff);
};

inline void Timer::start() 
{
  tStart = rdtsc();
}

inline void Timer::end() 
{
  unsigned long long tEnd = rdtsc();

  store.addTiming(name, tEnd - tStart);

  tStart = 0;
}

inline Timer::~Timer() 
{
  if(tStart) {
    end();
  }
}

inline void Timings::addTiming(std::string name, unsigned long long int diff)
{
  Timing &last = timings[name];
  last.count ++;
  last.sum += diff/1e9;
  last.max = std::max(last.max, diff/1e9);
  last.min = std::min(last.min, diff/1e9);
}

inline void Timings::dumpTimings()
{
  std::cout << "Timings dump\n";
  std::cout << "\"Name\",\"Count\",\"Total (GCycles)\",\"Mean\",\"Min\",\"Max\"\n";

  for(std::map<std::string, Timing >::const_iterator iter = timings.begin();
      iter != timings.end();
      iter++) {
    std::cout << '"' << iter->first 
              << "\",\"" << iter->second.count
              << "\",\"" << iter->second.sum 
              << "\",\"" << iter->second.sum/iter->second.count
              << "\",\"" << iter->second.min
              << "\",\"" << iter->second.max << "\"\n";
  }
}
