//#include <libxml/parser.h>
#include <libxml/tree.h>
#include <libxml/xpath.h>
#include <libxml/xinclude.h>

#include <stdexcept>

#include <time.h>

#include "configuration.h"

/**
    Parse the xml configuration and do XInclude processing
    Configuration file loaded from environment variable SCTDAQ_ROD_CONFIGURATION_PATH
*/
Configuration::Configuration() {
  const char *envname = "SCTDAQ_ROD_CONFIGURATION_PATH";

  char *filename = getenv(envname);

  if(filename) {
    document = xmlParseFile(filename);
  } else {
    cerr << "Configuration Environment: " << envname << " not set, don't know where to load config from\n";
    throw std::range_error("No configuration");
  }

  if(document) {
    //    processFileLinks(document);
    xmlXIncludeProcess(document);
    //    cout << "XIncludes : " << includeCount << endl; //processFileLinks(document);

    //saveConfiguration();
  } else {
   cerr << "Configuration Document not parsed!\n";
  }
  
  xmlXPathInit();
  xpathContext = xmlXPathNewContext(document);
}

/**
    Free the configuration xml
*/
Configuration::~Configuration() {
  //saveConfiguration();
  xmlFreeDoc(document);
}

/**
    Lookup the partitions defined in the configuration and return a list of integer ID's
*/
list<unsigned int> Configuration::listPartitions() {
  list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition");

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        if(xmlGetProp(curr, (xmlChar *)"id")) {
          result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
        } else {
          cerr << "This partition has no id attribute\n";
        }
      }
    } else {
      cerr << "Bad partition data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No result\n";
  }

  return result;
}

/**
    Lookup the crates defined in the specified partition and return a list of integer ID's
*/
list<unsigned int> Configuration::listCratesInPartition(unsigned int partition) {
  list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate", partition);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];
        if(xmlGetProp(curr, (xmlChar *)"id")) {
          result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
        } else {
          cerr << "This crate has no id attribute\n";
        }
      }
    } else {
      cerr << "Bad crate data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No result\n";
  }

  return result;
}

/**
    Lookup the RODs defined in the specified crate and return a list of integer ID's
*/
list<unsigned int> Configuration::listRodsInCrate(unsigned int partition, unsigned int crate) {
  list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod", partition, crate);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        if(xmlGetProp(curr, (xmlChar *)"id")) {
          result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
        } else {
          cerr << "This rod has no id attribute\n";
        }
      }
    } else {
      cerr << "Bad ROD data\n";
    }
    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No result rod\n";
  }

  return result;
}

/**
   Return integer id's of the MURs in the specified rod.
 */
list<unsigned int> Configuration::listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod) {
  list<unsigned int> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/MUR", partition, crate, rod);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        if(xmlGetProp(curr, (xmlChar *)"id")) {
          result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
        } else {
          cerr << "This MUR has no id attribute\n";
        }
      }
    } else {
      cerr << "Bad MUR data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No MUR result\n";
  }

  return result;
}

/**
   Return a list of serial numbers associated with the specified MUR
 */
list<string> Configuration::listModulesInMUR(unsigned int partition, unsigned int MUR) {
  list<string> result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate/rod/MUR[@id=%d]/module/sn/text()", partition, MUR);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      for(int i=0; i<queryResult->nodesetval->nodeNr; i++) {
        xmlNodePtr curr = queryResult->nodesetval->nodeTab[i];

        result.push_back((char *)curr->content);
      }
    } else {
      cerr << "Bad module data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No module/MUR result\n";
  }

  return result;
}

/**
    Find the configuration associated with the specified ROD.
    Return a RodConfig structure.
 */
RodConfig Configuration::getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod) {
  RodConfig result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr != 1) {
        cerr << "Wrong number of Rods found, id " << rod << " not unique? " << queryResult->nodesetval->nodeNr << endl;
      } else {
        xmlNodePtr c = queryResult->nodesetval->nodeTab[0];

        result = parseRodConfig(c);
      }
    } else {
      cerr << "Bad ROD config data " << partition << " " << crate << " " << rod << "\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No ROD config result\n";
  }

//    cout << "Found config for Rod " << rod << endl;
//    hex(cout);
//    cout << "baseAddress " << result.baseAddress << endl;
//    cout << "mapSize " << result.mapSize << endl;
//    dec(cout);
//    cout << "numSlaves " << result.numSlaves << endl;
//    for(unsigned int i=0; i<4; i++) {
//      cout << "emifFile " << result.slaves[i].emifFile << endl;
//      cout << "ipramFile " << result.slaves[i].ipramFile << endl;
//      cout << "idramFile " << result.slaves[i].idramFile << endl;
//      cout << "extFile " << result.slaves[i].extFile << endl;
//    }

  return result;
}

/**
    Convert string serial number indexing to partition/crate/rod/channel indexing.
    position returned in partition, crate, rod and channel arguments
 */
void Configuration::getModuleRodPosition(std::string module,
                                         unsigned int &partition, unsigned int &crate, unsigned int &rod, unsigned int &channel)
{
  char expression[1000];

  //  sprintf(expression, "/configuration/partition/crate/rod/MUR/module[sn=\"%s\"]", module.c_str());
  sprintf(expression, "/configuration/partition/crate/rod/MUR[module=\"%s\"]", module.c_str());

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  xmlNodePtr moduleNode = 0;

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr > 1) {
        cerr << "Too many matches (" << queryResult->nodesetval->nodeNr << endl;
      } else {
        moduleNode = queryResult->nodesetval->nodeTab[0];
      }
    } else {
      cerr << "Bad Module position data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No module position result\n";
  }

  if(moduleNode) {
    int moduleNumber, MUR;

    if(strcmp((char *)moduleNode->name, "module") == 0) {
      if(xmlGetProp(moduleNode, (xmlChar *)"id")) {
        moduleNumber = atoi((char *)xmlGetProp(moduleNode, (xmlChar *)"id"));
      }
    }

    xmlNodePtr parents = moduleNode->parent;
    if(strcmp((char *)parents->name, "MUR") == 0) {
      if(xmlGetProp(parents, (xmlChar *)"order")) {
        MUR = atoi((char *)xmlGetProp(parents, (xmlChar *)"order"));
      }
    }

    parents = parents->parent;
    if(strcmp((char *)parents->name, "rod") == 0) {
      if(xmlGetProp(parents, (xmlChar *)"id")) {
        rod = atoi((char *)xmlGetProp(parents, (xmlChar *)"id"));
      } else {
        cerr << "This rod has no id attribute\n";
      }
    }

    parents = parents->parent;
    if(strcmp((char *)parents->name, "crate") == 0) {
      if(xmlGetProp(parents, (xmlChar *)"id")) {
        crate = atoi((char *)xmlGetProp(parents, (xmlChar *)"id"));
      } else {
        cerr << "This crate has no id attribute\n";
      }
    }

    parents = parents->parent;
    if(strcmp((char *)parents->name, "partition") == 0) {
      if(xmlGetProp(parents, (xmlChar *)"id")) {
        partition = atoi((char *)xmlGetProp(parents, (xmlChar *)"id"));
      } else {
        cerr << "This partition has no id attribute\n";
      }
    }

    channel = MUR*8 + moduleNumber - 1;
  }
}

/**
   Return the configuration for the given module
 */
ABCDModule Configuration::getModuleConfig(string module) {
//    cout << "Asked to find configuration for module " << module << endl;

  char expression[1000];

  sprintf(expression, "/configuration/partition/crate/rod/MUR/module[sn=\"%s\"]", module.c_str());

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  xmlNodePtr moduleNode = 0;

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr > 1) {
        cerr << "Too many matches (" << queryResult->nodesetval->nodeNr << endl;
      } else {
        moduleNode = queryResult->nodesetval->nodeTab[0];
      }
    } else {
      cerr << "Bad module config data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No module config result\n";
  }

  return parseModuleConfig(moduleNode);
}

/**
    Parse the xml tree provided and return the module configuration
 */
const ABCDModule Configuration::parseModuleConfig(xmlNodePtr moduleNode) {
  ABCDModule result;

  for(int i=0; i<N_SCT_CHIPS; i++) {
    result.chip[i].active = 0;
  }

  if(moduleNode) {
    //    xmlElemDump(stdout, document, moduleNode);

    for(xmlNodePtr child = moduleNode->children;
        child;
        child = child->next) {
      if(child->type == XML_ELEMENT_NODE) {
        char * content = (char *)xmlNodeGetContent(child);

        //        cout << "Tag in module " << (char *)child->name << endl;;

        if(strcmp((char *)child->name, "active") == 0) {
//        if(xmlGetProp(child, (xmlChar *)"active") &&
//           xmlGetProp(child, (xmlChar *)"select")) {
//          result.active = atoi((char *)xmlGetProp(child, (xmlChar *)"active"));
//          result.select = atoi((char *)xmlGetProp(child, (xmlChar *)"select"));
//        } else {
//          cerr << "module config without active or select tags\n";
//        }
          result.active = atoi(content);
        } else if(strcmp((char *)child->name, "select") == 0) {
          result.select = atoi(content);
        } else if(strcmp((char *)child->name, "config") == 0) {
          /// Old config format
          if(xmlGetProp(child, (xmlChar *)"active") &&
             xmlGetProp(child, (xmlChar *)"select")) {
            result.active = atoi((char *)xmlGetProp(child, (xmlChar *)"active"));
            result.select = atoi((char *)xmlGetProp(child, (xmlChar *)"select"));
          } else {
            cerr << "module config without active or select tags\n";
          }
        } else if(strcmp((char *)child->name, "chip") == 0) {
          if(xmlGetProp(child, (xmlChar *)"id") &&
             xmlGetProp(child, (xmlChar *)"active") &&
             xmlGetProp(child, (xmlChar *)"address")) {
            int id = atoi((char *)xmlGetProp(child, (xmlChar *)"id"));
            int active = atoi((char *)xmlGetProp(child, (xmlChar *)"active"));
            int address;
            char *addressString = (char *)xmlGetProp(child, (xmlChar *)"address");
            address = strtol(addressString, NULL, 0);

            result.chip[id].active = active;
            result.chip[id].address = address;

            for(xmlNodePtr chipChild = child->children;
                chipChild;
                chipChild = chipChild->next) {
              char * chipContent = (char *)xmlNodeGetContent(chipChild);

              //            cout << "Tag in chip " << (char *)chipChild->name << endl;

              if(chipChild->type != XML_ELEMENT_NODE) {
              } else if(strcmp((char *)chipChild->name, "config") == 0) {
                char *eptr = chipContent;
                result.chip[id].basic.config.padding = 0;
                for(int i=0; i<11; i++) {
                  int value;
                  char *cptr = eptr;
                  value = strtol(cptr, &eptr, 0);
//                perror("Error converting");
                  switch(i) {
                  case 0: result.chip[id].basic.config.readoutMode = value; break;
                  case 1: result.chip[id].basic.config.calibMode = value; break;
                  case 2: result.chip[id].basic.config.trimRange = value; break;
                  case 3: result.chip[id].basic.config.edgeDetect = value; break;
                  case 4: result.chip[id].basic.config.mask = value; break;
                  case 5: result.chip[id].basic.config.accumulate = value; break;
                  case 6: result.chip[id].basic.config.inputBypass = value; break;
                  case 7: result.chip[id].basic.config.outputBypass = value; break;
                  case 8: result.chip[id].basic.config.master = value; break;
                  case 9: result.chip[id].basic.config.end = value; break;
                  case 10: result.chip[id].basic.config.feedThrough = value; break;
                  }
                }
              } else if(strcmp((char *)chipChild->name, "mask") == 0) {
                // Should be more robust?
                char *eptr = chipContent;
                for(int m=0; m<4; m++) {
                  char *cptr = eptr;
                  result.chip[id].basic.mask[m] = strtoul(cptr, &eptr, 0);
                }
              } else if(strcmp((char *)chipChild->name, "reg") == 0) {
                if(xmlGetProp(chipChild, (xmlChar *)"name")) {
                  char *name = (char *)xmlGetProp(chipChild, (xmlChar *)"name");

                  if(strcmp(name, "vthr") == 0) {
                    result.chip[id].basic.vthr = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name, "vcal") == 0) {
                    result.chip[id].basic.vcal = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name, "delay") == 0) {
                    result.chip[id].basic.delay = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name, "preamp") == 0) {
                    result.chip[id].basic.preamp = strtol(chipContent, NULL, 0);
                  } else if(strcmp(name, "shaper") == 0) {
                    result.chip[id].basic.shaper = strtol(chipContent, NULL, 0);
                  } else {
                    cerr << "Unknown tag for reg value " << name << endl;
                  }
                } else {
                  cerr << "chip reg with no name attribute\n";
                }
              } else if(strcmp((char *)chipChild->name, "rc_function") == 0) {
                if(xmlGetProp(chipChild, (xmlChar *)"type")) {
                  result.chip[id].caldata.rc_function = strtol((char *)xmlGetProp(chipChild, (xmlChar *)"type"), NULL, 0);

                  char *token = strtok(chipContent, " ");
                  int nextVal = -1;
                  do {
                    if(strcmp(token, "p0") == 0) {
                      nextVal = 0;
                    } else if(strcmp(token, "p1") == 0) {
                      nextVal = 1;
                    } else if(strcmp(token, "p2") == 0) {
                      nextVal = 2;
                    } else {
                      if(nextVal < 0) {
                        cerr << "Bad rc function parameters " << token << endl;
                      } else {
                        result.chip[id].caldata.rc_params[nextVal] = strtod(token, NULL);
                      }
                      nextVal = -1;
                    }
                  } while((token = strtok(NULL, " ")) != 0);
                } else {
                  cerr << "No type attribute on rc_function\n";
                }
              } else if(strcmp((char *)chipChild->name, "c_factor") == 0) {
                result.chip[id].caldata.c_factor = strtol(chipContent, NULL, 0);
              } else if(strcmp((char *)chipChild->name, "trim") == 0) {
                if(xmlGetProp(chipChild, (xmlChar *)"start") &&
                   xmlGetProp(chipChild, (xmlChar *)"end") ) {
                  
                  char *attr = (char *)xmlGetProp(chipChild, (xmlChar *)"start");
                  int start = strtol(attr, NULL, 0);
                  attr = (char *)xmlGetProp(chipChild, (xmlChar *)"end");
                  int end = strtol(attr, NULL, 0);

                  int length = end-start+1;

                  if((length%8) != 0) cerr << "Can't interpret trim values not in blocks of 8\n";
                  else {
                    char *endPtr = chipContent;
                    char *startPtr = chipContent;
                    int curr = start;
                    do {
                      if(curr > end) {
                        cerr << "Too many numbers in trim (range " << start << "-" << end << ")\n";
                        break;
                      }

                      startPtr = endPtr;
                      unsigned long trimValues = strtoul(startPtr, &endPtr, 0);

                      for(int i=0; i<8; i++) {
                        unsigned char val = (trimValues >> ((7-i)*4)) & 0x0000000f;

                        result.chip[id].trim[curr+i] = val;
                        if(val>9) val += 'a'-10;
                        else val += '0';
                      }
                      curr += 8;
                    } while(*endPtr!=0);
                  }
                } else {
                  cerr << "No start and end tags on trim\n";
                } // End trim attribute check
              } // End if over chip tags
            } // End loop over chip tags
          } else {
            cerr << "Bad attributes to chip\n";
          } // End chip attribute check
        } // End if over module tags
      } // Tag not an element
    } // End loop over module tags
  }

  return result;
}

/// Get list of register value pairs to set on the BOC (in the correct order)
list<pair<long, int> > Configuration::getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod) {
  list<pair<long, int> > result;

  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]/BOC", partition, crate, rod);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr != 1) {
        cerr << "Wrong number of BOCs found, id " << rod << " not unique? " << queryResult->nodesetval->nodeNr << endl;
      } else {
        xmlNodePtr c = queryResult->nodesetval->nodeTab[0];

        for(xmlNodePtr child = c->children;
            child;
            child = child->next) {
          if(child->type == XML_ELEMENT_NODE) {
            if(strcmp((char *)child->name, "reg") == 0) {
              long address, value;
              if(xmlGetProp(child, (xmlChar *)"address")) {
                address = strtol((char *)xmlGetProp(child, (xmlChar *)"address"), NULL, 0);
              } else if(xmlGetProp(child, (xmlChar *)"value")) {
                value = strtol((char *)xmlGetProp(child, (xmlChar *)"value"), NULL, 0);
              }

              result.push_back(make_pair(address, value));
            } else if(strcmp((char *)child->name, "regs") == 0) {
              long start, end;
              // Only mod 4 accesses access BOC registers
              int step=4, value;
              if(xmlGetProp(child, (xmlChar *)"start")) {
                start = strtol((char *)xmlGetProp(child, (xmlChar *)"start"), NULL, 0);
              } else if(xmlGetProp(child, (xmlChar *)"end")) {
                end = strtol((char *)xmlGetProp(child, (xmlChar *)"end"), NULL, 0);
              } else if(xmlGetProp(child, (xmlChar *)"step")) {
                step = strtol((char *)xmlGetProp(child, (xmlChar *)"step"), NULL, 0);
              } else if(xmlGetProp(child, (xmlChar *)"value")) {
                value = strtol((char *)xmlGetProp(child, (xmlChar *)"value"), NULL, 0);
              }

              // Start to end is inclusive of start
              for(long address=start; address<end; address+=step) {
                result.push_back(make_pair(address, value));
              }
            }
          }
        }
      }
    }
  }

  return result;
}

void Configuration::printModuleConfig(ABCDModule &conf) {
  for(int c=0; c<N_SCT_CHIPS; c++) {
    cout << "Chip: " << c << (conf.chip[c].active?" active ":" inactive ") << (int)conf.chip[c].address << endl;
    if(conf.chip[c].active) {
      cout << "Config " << hex << *(UINT16*)(&conf.chip[c].basic.config) << dec << endl;

      cout << conf.chip[c].basic.config.readoutMode << " ";
      cout << conf.chip[c].basic.config.calibMode << " ";
      cout << conf.chip[c].basic.config.trimRange << " ";
      cout << conf.chip[c].basic.config.edgeDetect << " ";
      cout << conf.chip[c].basic.config.mask << " ";
      cout << conf.chip[c].basic.config.accumulate << " ";
      cout << conf.chip[c].basic.config.inputBypass << " ";
      cout << conf.chip[c].basic.config.outputBypass << " ";
      cout << conf.chip[c].basic.config.master << " ";
      cout << conf.chip[c].basic.config.end << " ";
      cout << conf.chip[c].basic.config.feedThrough << " ";

      cout << endl;

      cout << "Mask " << conf.chip[c].basic.mask[0] << " " << conf.chip[c].basic.mask[1] << " " <<
        conf.chip[c].basic.mask[2] << " " << conf.chip[c].basic.mask[3] << endl;

      cout << "Rc function type " << (int)conf.chip[c].caldata.rc_function << " :";
      for(int r=0; r<3; r++) {
        cout << " " << conf.chip[c].caldata.rc_params[r];
      }
      cout << endl;
      cout << "Regs ";
      cout << " vthr " << (int)conf.chip[c].basic.vthr;
      cout << " vcal " << (int)conf.chip[c].basic.vcal;
      cout << " delay " << (int)conf.chip[c].basic.delay;
      cout << " preamp " << (int)conf.chip[c].basic.preamp;
      cout << " shaper " << (int)conf.chip[c].basic.shaper;
        //      cout << endl;
      for(int i=0; i<128; i++) {
        if(!(i%32)) cout << endl;
        hex(cout);
        cout << (int)conf.chip[c].trim[i];
        dec(cout);
      }
      cout << endl;
    }
  }
  cout << endl;
}


/** Replace the configuration for the given module with the data
    stored in the given configuration
 */
void Configuration::updateModuleConfig(std::string module, ABCDModule conf) {
  cout << "Asked to find configuration for module " << module << endl;

  char expression[1000];

  sprintf(expression, "/configuration/partition/crate/rod/MUR/module[sn=\"%s\"]", module.c_str());

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  xmlNodePtr moduleNode = 0;

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr > 1) {
        cerr << "Too many matches (" << queryResult->nodesetval->nodeNr << endl;
      } else {
        moduleNode = queryResult->nodesetval->nodeTab[0];
      }
    } else {
      cerr << "Bad update module data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No module update result\n";
  }

  return replaceModuleConfig(moduleNode, conf, module);
}

/** Replace the configuration in the given node with the data
    stored in the given configuration
*/
void Configuration::replaceModuleConfig(xmlNodePtr node, ABCDModule conf, string name) {
  xmlNodePtr newNode = xmlNewNode(NULL, (xmlChar *)"module");

  const int BUF_SIZE = 200;

  char buffer[BUF_SIZE];

  xmlNewChild(newNode, NULL, (xmlChar *)"sn", (xmlChar *)name.c_str());
  xmlNewChild(newNode, NULL, (xmlChar *)"source", (xmlChar *)"Module configuration saved during running");
  xmlNewChild(newNode, NULL, (xmlChar *)"location", (xmlChar *)"RAL");
  snprintf(buffer, BUF_SIZE, "%d", conf.active);
  xmlNewChild(newNode, NULL, (xmlChar *)"active", (xmlChar *)buffer);
  snprintf(buffer, BUF_SIZE, "%d", conf.select);
  xmlNewChild(newNode, NULL, (xmlChar *)"select", (xmlChar *)buffer);

  for(int chipNum=0; chipNum<12; chipNum++) {
    ABCDChip &currChip = conf.chip[chipNum]; 

    xmlNodePtr chip = xmlNewChild(newNode, NULL, (xmlChar *)"chip", (xmlChar *)"\n");
    snprintf(buffer, BUF_SIZE, "%02d", chipNum);
    xmlSetProp(chip, (xmlChar *)"id", (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "0x%x", currChip.address);
    xmlSetProp(chip, (xmlChar *)"address", (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "%d", currChip.active);
    xmlSetProp(chip, (xmlChar *)"active", (xmlChar *)buffer);

    ABCDConfig &chipConf = currChip.basic.config;

    snprintf(buffer, BUF_SIZE, "%d %d %d %d %d %d %d %d %d %d %d",
             chipConf.readoutMode, chipConf.calibMode, chipConf.trimRange,
             chipConf.edgeDetect, chipConf.mask, chipConf.accumulate,
             chipConf.inputBypass, chipConf.outputBypass, chipConf.master,
             chipConf.end, chipConf.feedThrough);
    xmlNewChild(chip, NULL, (xmlChar *)"config", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "0x%08x 0x%08x 0x%08x 0x%08x",
             currChip.basic.mask[0], currChip.basic.mask[1],
             currChip.basic.mask[2], currChip.basic.mask[3]);
    xmlNewChild(chip, NULL, (xmlChar *)"mask", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.vthr);
    xmlNodePtr reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"vthr");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.vcal);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"vcal");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.delay);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"delay");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.preamp);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"preamp");
    snprintf(buffer, BUF_SIZE, "0x%02x", currChip.basic.shaper);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"reg", (xmlChar *)buffer);
    xmlSetProp(reg, (xmlChar *)"name", (xmlChar *)"shaper");

    snprintf(buffer, BUF_SIZE, "p0 %e p1 %e p2 %e",
             currChip.caldata.rc_params[0], currChip.caldata.rc_params[1], currChip.caldata.rc_params[2]);
    reg = xmlNewChild(chip, NULL, (xmlChar *)"rc_function",
                      (xmlChar *)buffer);
    snprintf(buffer, BUF_SIZE, "%d", currChip.caldata.rc_function); 
    xmlSetProp(reg, (xmlChar *)"type", (xmlChar *)buffer);

    snprintf(buffer, BUF_SIZE, "%f", currChip.caldata.c_factor); 
    xmlNewChild(chip, NULL, (xmlChar *)"c_factor", (xmlChar *)buffer);

    for(int u=0; u<4; u++) {
      buffer[0] = '\0';
      for(int t=u*32; t<(u+1)*32; t+=8) {
        snprintf(buffer + strlen(buffer), BUF_SIZE, "0x%x%x%x%x%x%x%x%x ",
                 currChip.trim[t+0], currChip.trim[t+1], currChip.trim[t+2], currChip.trim[t+3],
                 currChip.trim[t+4], currChip.trim[t+5], currChip.trim[t+6], currChip.trim[t+7]);
      }
      reg = xmlNewChild(chip, NULL, (xmlChar *)"trim",
                        (xmlChar *)buffer);
      snprintf(buffer, BUF_SIZE, "0x%02x", u*32);
      xmlSetProp(reg, (xmlChar *)"start", (xmlChar *)buffer);
      snprintf(buffer, BUF_SIZE, "0x%02x", u*32+31);
      xmlSetProp(reg, (xmlChar *)"end", (xmlChar *)buffer);
    }
  }

  xmlReplaceNode(node, newNode);
  xmlFreeNode(node);
}

/** Replace the in memory configuration for the given ROD
    with the configuration provided.
 */
void Configuration::updateRodConfig(unsigned int partition, unsigned int crate, unsigned int rod, RodConfig conf) {
  char expression[1000];

  sprintf(expression, "/configuration/partition[@id=%d]/crate[@id=%d]/rod[@id=%d]", partition, crate, rod);

  xmlXPathObjectPtr queryResult = xmlXPathEval((xmlChar *)expression, xpathContext);

  if(queryResult) {
    if(queryResult->type == XPATH_NODESET && queryResult->nodesetval) {
      if(queryResult->nodesetval->nodeNr != 1) {
        cerr << "Wrong number of Rods found, id " << rod << " not unique? " << queryResult->nodesetval->nodeNr << endl;
      } else {
        xmlNodePtr oldConf = queryResult->nodesetval->nodeTab[0];

        xmlNodePtr firstNode = 0;

        char buffer[200];

        xmlNodePtr child = oldConf->children;
        while(child) {
          xmlNodePtr newChild = 0;
          char *name = (char *)child->name;

          xmlNodePtr next = child->next;
          if((strcmp("baseAddress", name) == 0)) {
            sprintf(buffer, "0x%lx", conf.baseAddress);
            newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"baseAddress", (xmlChar *)buffer);

            xmlReplaceNode(child, newChild);
            xmlFreeNode(child);
          } else if(strcmp("mapSize", name) == 0) {
            sprintf(buffer, "0x%lx", conf.mapSize);
            newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"mapSize", (xmlChar *)buffer);

            xmlReplaceNode(child, newChild);
            xmlFreeNode(child);
          } else if(strcmp("numSlaves", name) == 0) {
            sprintf(buffer, "%ld", conf.numSlaves);
            newChild = xmlNewChild(oldConf, NULL, (xmlChar *)"numSlaves", (xmlChar *)buffer);

            xmlReplaceNode(child, newChild);
            xmlFreeNode(child);
          } else if(strcmp("slave", name) == 0) {
            xmlUnlinkNode(child);
            xmlFreeNode(child);
          } else if(strcmp("MUR", name) == 0) {
            if(!firstNode) firstNode = child;
          }

          child = next;
        }

        for(int s=3; s>=0; s--) {
          xmlNodePtr slave = xmlNewChild(oldConf, NULL, (xmlChar *)"slave", (xmlChar *)"\n");
          
          sprintf(buffer, "%d", s);
          xmlSetProp(slave, (xmlChar *)"id", (xmlChar *)buffer);

          xmlNewChild(slave, NULL, (xmlChar *)"emifFile", (xmlChar *)conf.slaves[s].emifFile.c_str());
          xmlNewChild(slave, NULL, (xmlChar *)"ipramFile", (xmlChar *)conf.slaves[s].ipramFile.c_str());
          xmlNewChild(slave, NULL, (xmlChar *)"idramFile", (xmlChar *)conf.slaves[s].idramFile.c_str());
          xmlNewChild(slave, NULL, (xmlChar *)"extFile", (xmlChar *)conf.slaves[s].extFile.c_str());

          xmlAddPrevSibling(firstNode, slave);
          firstNode = slave;
        }

//      xmlReplaceNode(oldConf, newRod);

        //      xmlFreeNode(oldConf);
      }
    } else {
      cerr << "Bad rod update data\n";
    }

    xmlXPathFreeObject(queryResult);
  } else {
    cerr << "No rod update result\n";
  }
}

/** Write configuration to disc.
    This creates a unique filename based on the date and time
    and saves the current configuration to it.
 */
void Configuration::saveConfiguration() {
  const int BUFF_SIZE = 50;
  char buffer[BUFF_SIZE];

  time_t secs;

  secs = time(NULL);

  struct tm broke;
  broke = *(gmtime(&secs));
  // 20021112143712
  strftime(buffer, BUFF_SIZE, "Conf_dump_%Y%m%d%H%M%S.xml", &broke);

  xmlSaveFormatFile(buffer, document, 0);
  cout << "Saved the configuration as " << buffer << endl;
}


const RodConfig Configuration::parseRodConfig(xmlNodePtr c) {
  RodConfig result;

  if(c) {
    for(xmlNodePtr child = c->children;
        child;
        child = child->next) {
      if(child->type == XML_ELEMENT_NODE) {
        char * content = (char *)xmlNodeGetContent(child);

        if(strcmp((char *)child->name, "baseAddress") == 0) {
          result.baseAddress = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "mapSize") == 0) {
          result.mapSize = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "numSlaves") == 0) {
          result.numSlaves = strtol(content, NULL, 0);
        } else if(strcmp((char *)child->name, "slave") == 0) {
          if(xmlGetProp(child, (xmlChar *)"id")) {
            unsigned int slaveId = (atoi((char *)xmlGetProp(child, (xmlChar *)"id")));
            //                cout << "Found data for slave " << slaveId << endl;
            for(xmlNodePtr slaveChild = child->children;
                slaveChild;
                slaveChild = slaveChild->next) {
              if(strcmp((char *)slaveChild->name, "emifFile") == 0) {
                result.slaves[slaveId].emifFile = (char*)xmlNodeGetContent(slaveChild);
              } else if(strcmp((char *)slaveChild->name, "ipramFile") == 0) {
                result.slaves[slaveId].ipramFile = (char*)xmlNodeGetContent(slaveChild);
              } else if(strcmp((char *)slaveChild->name, "idramFile") == 0) {
                result.slaves[slaveId].idramFile = (char*)xmlNodeGetContent(slaveChild);
              } else if(strcmp((char *)slaveChild->name, "extFile") == 0) {
                result.slaves[slaveId].extFile = (char*)xmlNodeGetContent(slaveChild);
              }
            }
          } else {
            cerr << "This slave has no id attribute\n";
          }
        } else {
          //              cerr << "Extra tag in rod conf " << child->name << " \n";
        }
      } else {
        //      cout << "Node of a different type\n";
      }
    } // End while child
  //      result.push_back(atoi((char *)xmlGetProp(curr, (xmlChar *)"id")));
  }
  return result;
}
