#ifndef SCTAPI_CONFIGURATION
#define SCTAPI_CONFIGURATION

#include <list>
#include <string>
#include <stdexcept>

// For ABCDModule
#include <CommonWithDsp/sctStructure.h>

namespace SctConfiguration {

/** 
    The configuration files needed to initialise a ROD slave DSP
 */
struct SlaveConfig {
  std::string ipramFile;
  std::string idramFile;
  std::string extFile;
};

/**
   Configuration needed for ROD initialisation
 */
struct RodConfig {
/*   /// Is there a BOC? (Shouldn't be necessary) */
/*   bool bocPresent; */
  /// VME base address
  unsigned long baseAddress;
/*   /// VME map size */
/*   unsigned long mapSize; */
  /// How much resetting to do
  unsigned long resetLevel;
  /// Slave file configuration
  SlaveConfig slaves[4];
};

/**
   Configuration needed for TIM initialisation
 */
struct TimConfig {
  /// VME base address
  unsigned long baseAddress;
/*   /// VME map size */
/*   unsigned long mapSize; */
  /// L1A frequency for TIM generated triggers
  float trigFrequency;
  /// Reset frequency for TIM generated triggers
  float resetFrequency;
};

/**
   Configuration of a "channel" on a BOC

   This is the information associated with one module
   ie. one output channel for primary data and
   two input channels for streams 0 and 1
 */
struct BOCChannelConfig {
  /// Laser current
  int current;
  /// BPM delay 
  int delay;
  /// BPM markspace
  int markSpace;
  /// Input threshold (stream 0)
  int threshold0;
  /// Strobe delay (stream 0)
  int delay0;
  /// Input threshold (stream 1)
  int threshold1;
  /// Strobe delay (stream 1)
  int delay1;
};

struct BOCGlobalConfig {
  /**
     This is a bit mask in the same order as below.
     If a bit is set then the value is valid.
     If the value is not valid then the BOC default is assumed to be correct.
  */
  int validMask;

  int clockControl;
  int rxDataMode;
  int rxDacClear;
  int txDacClear;
  int vernierFinePhase;
  int vernierClockPhase0;
  int vernierClockPhase1;
  int bpmClockPhase;
  int bregClockPhase;
};

enum MURType {BARREL, ENDCAP, UNMAPPED, UNKNOWN};

/**
   Exception thrown by public configuration methods
 */
class ConfigurationException : public std::exception {
  std::string desc;
 public:
  /// Provide a description on construction
  ConfigurationException(const std::string &str) throw() : desc(str) {}

  ~ConfigurationException() throw() {}

  /// Override std::exception method
  virtual const char* what () const throw() { return desc.c_str(); }
};

/**
   Interface to XML configuration file.
   This will be replaced by an interface to the DAQ confDB system.

   The primary function of this class is the readout of configuration
   data from the database.

   Translation functions:

   Barrel
   barrel, row,     number
   3 to 6  0 to 55  -1 to -6 and 1 to 6

   Internals:
   position on row given as -1 for -ve numbers and 1 for +ve numbers

   Endcap
   disc     quadrant number 
   0 to 17  0 to 3   0 to 32

   Internals:
   position of MUR in quadrant
   number is left to right top to bottom 
      0000001222222
        4444455555
        1111133333
*/
class Configuration {
public:
  /**
     Free resources
  */
  virtual ~Configuration() {};

  /************* Get physical configuration ****************/

  /**
     List partition ids
     Lookup the partitions defined in the configuration and return a list of integer ID's
  */
  virtual std::list<unsigned int> listPartitions() = 0;

  /**
     List crate ids with reference to the given partition
     Lookup the crates defined in the specified partition and return a list of integer ID's
  */
  virtual std::list<unsigned int> listCratesInPartition(unsigned int partition) = 0;

  /**
     Check if this should be a dummy crate 
  */
  virtual bool isDummyCrate(unsigned int partition, unsigned int crate) = 0;

  /**
     List the rods present in a crate
     Lookup the RODs defined in the specified crate and return a list of integer ID's
  */
  virtual std::list<unsigned int> listRodsInCrate(unsigned int partition, unsigned int crate) = 0;

  /**
     List the MURs associated with a ROD
     Return integer id's of the MURs in the specified rod.

     FIXME:  Should be in numbered order
  */
  virtual std::list<unsigned int> listMURSInRod(unsigned int partition, unsigned int crate, unsigned int rod) = 0;

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the specified MUR

     FIXME:  Should be in numbered order
  */
  virtual std::list<std::string> listModulesInMUR(unsigned int partition, unsigned int MUR) = 0;

  /**
     List the modules in the given MUR
     Return a list of serial numbers associated with the redundancy links of the specified MUR

     FIXME:  Should be in numbered order (array?)
  */
  virtual std::list<std::string> listRModulesInMUR(unsigned int partition, unsigned int MUR) = 0;

  /**
     Return a list of serial numbers of all modules that have configuration

     This includes those that are not attached to a ROD or MUR
  */
  virtual std::list<std::string> listAllModules() = 0;

  /**
     Return a list of serial numbers of modules that have configuration but are not attached to an MUR.
  */
  virtual std::list<std::string> listUnusedModules() = 0;

  /************** Get configurations *****************/

  /**
     Return the configuration for the module

     Deprecated: Should take MUR argument
  */
  virtual ABCDModule getModuleConfig(const std::string module) = 0;

  /** 
     Return the group for this module
  */
  virtual short getModuleGroup(const std::string module) = 0;

  /**
     Set the group for this module
  */
  virtual void setModuleGroup(const std::string module, short group) = 0;

  /**
     Return the Rod configuration
     Find the configuration associated with the specified ROD.
     Return a RodConfig structure.
  */
  virtual RodConfig getRodConfig(unsigned int partition, unsigned int crate, unsigned int rod) = 0;

  /**
     Return the default slave configuration
     Find the configuration associated with the specified slave type.
     Return a SlaveConfig structure.
  */
  virtual SlaveConfig getDefaultSlaveConfig(char type) = 0;

  /**
     Get the BOC configuration of one of 48 channels 

     @param channel Module number (0-47)

     Input channels are paired as one.
  */
  virtual BOCChannelConfig getBOCConfig(unsigned int partition, unsigned int crate, unsigned int rod, 
                                        unsigned int channel) = 0;

  /**
     Get the BOCs global configuration
  */
  virtual BOCGlobalConfig getBOCGlobalConfig(unsigned int partition, unsigned int crate, unsigned int rod) = 0;

  /**
     Get the BOC fibre mappings

     An array of 48*3 chars (out, in0, in1)
  */
  virtual char *getFibreMappings(unsigned int partition, unsigned int crate, unsigned int rod) = 0;

  /**
     Return the TIM configuration
     Find the configuration associated with the specified TIM.
     Return a TimConfig structure.
  */
  virtual TimConfig getTimConfig(unsigned int partition, unsigned int crate) = 0;

  /**
     Set fibre mappings for one channel

     Any value over 48/96 leaves current value unchanged
   */
  virtual void setFibreMapping(unsigned int partition, unsigned int crate, unsigned int rod, 
                               unsigned int channel, unsigned int tx, unsigned int rx0, unsigned int rx1) = 0;

  /**
     Print module configuration to cout
  */
  virtual void printModuleConfig(const ABCDModule &conf) = 0;

  virtual MURType getMURType(unsigned int MUR) = 0;

  /**
     Return an unused MUR id
   */
  virtual unsigned int getFreeMurId() = 0;

  /************** Change indexing schemes ****************/

  /**
     Translate from MUR / module(1-6) number to p/k/r/c

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateToROD(unsigned int MUR, unsigned int module,
                              unsigned int &partition, unsigned int &crate, 
                              unsigned int &rod, unsigned int &channel) = 0;
 
  /**
     Translate from MUR / redundant module(1-6) number to p/k/r/c

     This returns the channel to which the module referred to by MUR/module 
     is connected in its redundant configuration

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateToRROD(unsigned int MUR, unsigned int module,
                               unsigned int &partition, unsigned int &crate, 
                               unsigned int &rod, unsigned int &channel) = 0;
 
  /**
     Translate from MUR / module(1-6) number to serial number

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateToSN(unsigned int MUR, unsigned int module,
                             std::string &sn) = 0;

  /**
     Translate from MUR / module(1-6) number to MUR/module of redundant connection

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateToRMUR(unsigned int MUR, unsigned int module,
                               unsigned int &RMUR, unsigned int &rmodule) = 0;

  /**
     Translate from MUR / module(1-6) number to geographical barrel system

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateToBarrel(unsigned int MUR, unsigned int module,
                                 unsigned int &barrel, unsigned int &row, int &number) = 0;

  /**
     Translate from MUR / module(1-6) number to geographical endcap system

     @param MUR MUR number
     @param module Module in MUR (1-6)
   */
  virtual void translateToEndcap(unsigned int MUR, unsigned int module,
                                 int &disk, unsigned int &ring, unsigned int &number) = 0;

  /**
     Translate from MUR / module(1-6) number to power supply channel address

     @param MUR MUR number
     @param module Module in MUR (1-6)
   */
  virtual void translateToPowerSupply(unsigned int MUR, unsigned int module,
                                      unsigned int &partition, 
                                      unsigned int &crate, unsigned int &channel) = 0;

  /**
     Translate to MUR / module(1-6) number from p/k/r/c

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateFromROD(unsigned int partition, unsigned int crate, 
                                unsigned int rod, unsigned int channel,
                                unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate to MUR / module(1-6) number from address of redundant module

     This returns the MUR/module of the module connected using its
     redundant link to the p/k/r/c address

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateFromRROD(unsigned int partition, unsigned int crate,
                                 unsigned int rod, unsigned int channel,
                                 unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate to MUR / module(1-6) number from serial number

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateFromSN(const std::string sn,
                               unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate from redundant MUR / module(1-6) number to MUR/module via redundant connection

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateFromRMUR(unsigned int RMUR, unsigned int rmodule,
                                 unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate to MUR / module(1-6) number from geographical barrel system

     @param MUR MUR number
     @param module Module in MUR (1-6)
  */
  virtual void translateFromBarrel(unsigned int barrel, unsigned int row, int number,
                                   unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate to MUR / module(1-6) number from geographical endcap system

     @param MUR MUR number
     @param module Module in MUR (1-6)
   */
  virtual void translateFromEndcap(int disk, unsigned int ring, unsigned int number,
                                   unsigned int &MUR, unsigned int &module) = 0;

  /**
     Translate from power supply channel address to MUR / module(1-6) number 

     @param MUR MUR number
     @param module Module in MUR (1-6)
   */
  virtual void translateFromPowerSupply(unsigned int partition, 
                                        unsigned int crate, unsigned int channel, 
                                        unsigned int &MUR, unsigned int &module) = 0;


  /************** Change configuration ******************/

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given configuration
  */
  virtual void configureModuleFromStructure(const std::string module, const ABCDModule conf) = 0;

  /**
     Update or add the configuration of a module stored in memory
     Replace the configuration for the given module with the data
     stored in the given file (the serial number is read from the file)
  */
  virtual void configureModuleFromFile(const std::string filename) = 0;

  /**
     Configure a ROD

     Replace the in memory configuration for the given ROD
     with the configuration provided.

     If no configuration then construct it
   */
  virtual void configureROD(unsigned int partition, unsigned int crate, unsigned int rod, RodConfig conf) = 0;

  /**
     Configure a TIM

     Replace the in memory configuration for the given TIM
     with the configuration provided.

     If no configuration then construct it
   */
  virtual void configureTIM(unsigned int partition, unsigned int crate, TimConfig conf) = 0;

  /**
     Set a BOC configuration
  */
  virtual void configureBOC(unsigned int partition, unsigned int crate, unsigned int rod, const BOCGlobalConfig &conf) = 0;

  /**
     Remove all configuration.
     Very dangerous!!
   */
  virtual void clearAll() = 0;

  /**
     Configuration creation:

     Create module (write configuration)

     Change translations

     Change MUR translations
   */

  /** 
     Name a partition (for information, any use?)
   */
  virtual void namePartition(unsigned int partition, const std::string name) = 0;

  /**
     Set a BOC configuration 
   */
/*virtual    void setBOCConfig(int partition, int crate, int rod, int channel, const BOCChannelConfig &conf) = 0; */
  
  /**
     Set a BOC channel configuration

     @param MUR MUR number
     @param position Module in MUR (1-6)
  */
  virtual void configureBOCChannel(unsigned int MUR, unsigned int position, const BOCChannelConfig &conf) = 0;

  /**
     Map a module (by serial number) onto a position in an MUR

     @param MUR MUR number
     @param order Module in MUR (1-6)
  */
  virtual void mapModuleMUR(unsigned int MUR, unsigned int order, unsigned int RMUR, unsigned int rorder, std::string number) = 0;

  /**
     Unmap a module from a position in an MUR

     @param MUR MUR number
     @param order Module in MUR (1-6)
  */
  virtual void unmapModuleMUR(unsigned int MUR, unsigned int order) = 0;
 
  /** 
     Map an MUR to a ROD
  */
  virtual void mapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int MUR) = 0;

  /** 
     Unmap an MUR from a ROD
  */
  virtual void unmapRODMUR(unsigned int MUR) = 0;

  /**
     Map Barrel MUR

     @param position -1 or 1
  */
  virtual void mapBarrelMUR(unsigned int MUR, unsigned int barrel, unsigned int row, int position) = 0;

  /**
     Unmap MUR from Barrel 
  */
  virtual void unmapBarrelMUR(unsigned int MUR) = 0;

  /** 
     Get mapping of MUR to ROD
  */
  virtual void getMapMURROD(unsigned int MUR, unsigned int &partition, unsigned int &crate, unsigned int &rod, unsigned int &order) = 0;

  /** 
     Get mapping of ROD to MUR

     @param MUR MUR number
     @param number Module in MUR (1-6)
  */
  virtual void getMapRODMUR(unsigned int partition, unsigned int crate, unsigned int rod, unsigned int order, unsigned int &number) = 0;

  /*
    Get mapping of MUR to physical structure
  */
  virtual void getMapMURPhysical(unsigned int MUR, MURType &type, int &plane, int &section, int &position) = 0;

  /**
     Swap two MUR names around

     This means in regards to everything about one MUR you can use the other name
     If either doesn't exist then just one MUR is renamed
  */
  virtual void swapMURNames(unsigned int MUR1, unsigned int MUR2) = 0;

/* #warning "Need to think more about endcap mappings" */

  /**
     Map Endcap MUR

     @param disk -1 to -9 and 1 to 9
  */
  virtual void mapEndcapMUR(unsigned int MUR, int disk, unsigned int quadrant, unsigned int position) = 0;

  /**
     Unmap Endcap MUR
  */
  virtual void unmapEndcapMUR(unsigned int MUR) = 0;

  /**
     Map Power channel
  */
  virtual void mapPowerChannel(unsigned int MUR, unsigned int number, unsigned int partition, unsigned int crate, unsigned int channel) = 0;

  /**
     Unmap Power channel
  */
  virtual void unmapPowerChannel(unsigned int MUR, unsigned int number) = 0;

  /**
     Set default value of any MoPS parameter
  */
  virtual void modifyDefaultPowerParam(std::string state, std::string name, std::string type, float value) = 0;

  /**
     Set MoPS HV card parameter
  */
  virtual void modifyHVCardParam(unsigned int crate, unsigned int card,
                                 std::string name, std::string type, float value) = 0;

  /**
     Set MoPS LV card parameter
  */
  virtual void modifyLVCardParam(unsigned int crate, unsigned int card,
                                 std::string name, std::string type, float value) = 0;

  /**
     Set MoPS channel parameter referenced by MUR/number
  */
  virtual void modifyPowerParam(unsigned int MUR, unsigned int number,
                                std::string state, std::string name, std::string type, float value) = 0;

  /**
     Set MoPS channel parameter referenced by MoPS crate/channel
  */
  virtual void modifyPowerParamCC(unsigned int crate, unsigned int channel,
                                  std::string state, std::string name, std::string type, float value) = 0;

  /**
     Get default power of any MoPS parameter
   */
  virtual float getDefaultPowerParam(std::string state, std::string name, std::string type) = 0;

  /**
     Get MoPS HV card parameter
  */
  virtual float getHVCardParam(unsigned int crate, unsigned int card,
                              std::string name, std::string type) = 0;

  /**
     Get MoPS LV card parameter
  */
  virtual float getLVCardParam(unsigned int crate, unsigned int card,
                              std::string name, std::string type) = 0;

  /**
     Get MoPS channel parameter referenced by MUR/number
  */
  virtual float getPowerParam(unsigned int MUR, unsigned int number,
                              std::string state, std::string name, std::string type) = 0;

  /**
     Get MoPS channel parameter referenced by MoPS crate/channel
   */
  virtual float getPowerParamCC(unsigned int crate, unsigned int channel,
                                std::string state, std::string name, std::string type) = 0;

  /**
     List the defined parameters
   */
  virtual std::list<std::string> listDefaultPowerParams() = 0;

  /**
     Get Crate level power supply parameter
  */
  virtual float getCratePowerParam(unsigned int crate, std::string name) = 0;

  /**
     List the channel parameters for DCS
   */
  virtual std::list<std::string> listDCSChannelParams() = 0;

  /**
     List the card parameters for DCS
   */
  virtual std::list<std::string> listDCSCardParams() = 0;

  /**
     List the crate parameters for DCS
   */
  virtual std::list<std::string> listDCSCrateParams() = 0;

  /** 
     Load configuration from named file
   */
  virtual void loadConfiguration(const std::string &filename) = 0;

  /************** Save configuration *********************/

  /**
     Save the current configuration

     Write configuration to disc.
     If no filename is provided a unique filename based on
     the date and time is created.

     Filename eg Conf_dump_20021210142311.xml.gz (in current directory)
  */
  virtual void saveConfiguration(const std::string filename = "") = 0;

  /**
     Save the current configuration for a module

     Write the configuration for the module modsn to disc.
  */
  virtual void saveModuleConfiguration(const std::string modsn, const std::string filename) = 0;

  /**
     Construct the XML for a module configuration and return as a string
  */
  virtual std::string getModuleConfigurationString(const std::string modsn, const ABCDModule config) = 0;

  /**
     Dump power supply configuration to a file
  */
  virtual void writePowerSupplyConfiguration(std::string filename) = 0;
};

}

#endif

