#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifndef WIN32
#include <unistd.h>
#endif
#include <sys/stat.h>

#include "primNames.h"
#include "commandListDefinitions.h"
#include "primListTableDefinitions.h"
#include "primParams.h"

char *prefix = "d:rod";

#if WIN32
bool swapSlash = false;
#else
bool swapSlash = true;
#endif

void loadClist(const char *fname, int depth);
void loadPlist(const char *fname, int depth);
char *spaces(int num);
char *translateName(const char *oldFname);
char *getName(int id);
char *getTaskName(int id);
void printDataChunk(unsigned int *ptr, unsigned int length, int depth);
void printFile(char *fname, int depth);

int main(int argc, char **argv) {
  if(argc == 2) {
    int len = strlen(argv[1]);
    // In windows, cygwin sees the files as .cli
    if(argv[1][len-3] == 'c' || argv[1][len-3] == 'C') {
      loadClist(argv[1], 0);
    } else if(argv[1][len-5] == 'c' || argv[1][len-5] == 'C') {
      loadClist(argv[1], 0);
    } else {
      loadPlist(argv[1], 0);
    }
  } else {
    printf("Usage: loadClist clist\n");
  }
  getc(stdin); 

  return 0;
}

void loadClist(const char *fname, int depth) {
  printf("%sLoading clist %s\n", spaces(depth), fname);

  struct stat fileInfo;

  stat(fname, &fileInfo);

  int fileSize;

#if WIN32
  fileSize = 1000000;
#else
  if(!S_ISREG(fileInfo.st_mode)) {
    printf("Not a regular file!\n");
    return;
  }

  fileSize = fileInfo.st_size;
#endif

  char *buffer;
  buffer = (char *)malloc(fileSize);

  FILE *fin = fopen(fname, "rb");

  if(!fin) {
	printf("Bad file %s\n", fname);
	return;
  }

  int ret = fread(buffer, 1, fileSize, fin);

  if(ret == fileSize) 
    printf ("%sRead file of size %d\n", spaces(depth), (int)fileSize);
  else {
    printf("Read failed\n");
    return;
  }

  char *endOfFile = buffer + ret;

  COMMAND_LIST *clist = (COMMAND_LIST *)buffer;

  printf("%sThere are %d commands\n", spaces(depth), clist->commandCount);

  int noCommands = clist->commandCount;

  for(int command = 0; command < noCommands; command++) {
    printf("%s%d:\n", spaces(depth), command);

    COMMAND *commStruct = &clist->command[command];
    // Command size if 624
    if((char *)commStruct > endOfFile) {
      printf("Oops %p %p\n", commStruct, endOfFile);
      break;
    }

    int type = commStruct->id;

    printf(" %sRep: %d, Cont?: %d\n", spaces(depth), commStruct->numRepetitions, commStruct->continueIfError);

    //  #define RW_VME_COMMAND_ID              0  
    //  #define RW_MASTER_COMMAND_ID           1
    //  #define RW_SLAVE_COMMAND_ID            2
    //  #define SEND_PRIM_LIST_COMMAND_ID      3
    //  #define COMPARE_FILES_COMMAND_ID       4
    //  #define BUS_TEST_ID                    5
    //  #define RUN_CHILD_COMMAND_LIST_ID      6
    //  #define RW_FPGA_FLASH_ID               7
    //  #define WRITE_MDSP_FLASH_ID            8
    //  #define BOC_SETUP_ID                   9
    //  #define RUN_EXTERNAL_PROGRAM_ID       10
    //  #define DELAY_COMMAND_ID              11

    char *params = (char *)&commStruct->params; // buffer+index+12;

    switch(type) {
    case RW_VME_COMMAND_ID:
      {
        RW_VME_PARAMS *data = (RW_VME_PARAMS*)params;
        printf(" %sRW VME %s 0x%x bytes, block? %d start 0x%x length 0x%x\n", spaces(depth), 
               data->readNotWrite?"READ":"WRITE", data->elementSize, data->noBlockAccess, data->startAddress, data->length);
        break;
      }
    case RW_MASTER_COMMAND_ID:
      {
        RW_MASTER_PARAMS *data = (RW_MASTER_PARAMS*)params;
        printf(" %sRW MASTER %s start 0x%x length 0x%x\n", spaces(depth),
               data->readNotWrite?"Read":"Write", data->startAddress, data->length);
        if(data->readNotWrite) {
          printf(" %s comp? %d, filename %s\n", spaces(depth),
                 data->read.compareWithData, data->read.fileName);
        } else {
          if(data->write.dataNotFile) {
            printf(" %s Immediate 0x%x\n", spaces(depth),
                   data->write.data);
          } else {
            printf(" %s File %s\n", spaces(depth),
                   data->write.fileName);
          }
        }
        break;
      }
    case RW_SLAVE_COMMAND_ID:
      {
        RW_SLAVE_PARAMS *data = (RW_SLAVE_PARAMS*)params;
        printf(" %sRW SLAVE %s slave %d, start %p, length 0x%x\n", spaces(depth),
               data->readNotWrite?"READ":"WRITE", data->slaveNumber, (void *)data->startAddress, data->length);
        if(data->readNotWrite) {
          printf(" %s compare %d, 0x%08x\n", spaces(depth), data->read.compareWithData, data->read.compData);
          printf(" %s outFile %s\n", spaces(depth), data->read.fileName);
        } else {
          if(data->write.dataNotFile) {
            printf(" %s data 0x%08x\n", spaces(depth), data->write.data);
          } else {
            printf(" %s inFile %s\n", spaces(depth), data->write.fileName);
          }
        }
        break;
      }
    case SEND_PRIM_LIST_COMMAND_ID:
      printf(" %sRun prim list (no rebuild? %d) %s\n", spaces(depth), 
             *(int*)(params), params+4);

      char plName[300];
      strcpy(plName, translateName(params+4));
      //      printf("Converted name %s\n", plName);
      loadPlist(plName, depth + 1);

      break;
    case COMPARE_FILES_COMMAND_ID:
      {
        COMPARE_FILES_PARAMS *data = (COMPARE_FILES_PARAMS*)params;
        printf(" %sCOMPARE FILES %s %s\n", spaces(depth),
               data->commonFile, data->rodFile);
        break;
      }
    case BUS_TEST_ID:
      printf(" %sBUS TEST\n", spaces(depth));
      break;
    case RUN_CHILD_COMMAND_LIST_ID:
      printf(" %sSub command list %s\n", spaces(depth), params);

      char clName[300];
      strcpy(clName, translateName(params));
      //printf("Converted name %s\n", clName);
      loadClist(clName, depth + 1);

      break;
    case RW_FPGA_FLASH_ID:
      printf(" %sRW_FPGA_FLASH\n", spaces(depth));
      break;
    case WRITE_MDSP_FLASH_ID:
      printf(" %sWRITE_MDSP_FLASH\n", spaces(depth));
      break;
    case BOC_SETUP_ID:
      printf(" %sBOC_SETUP\n", spaces(depth));
      break;
    case RUN_EXTERNAL_PROGRAM_ID:
      {
        RUN_EXTERNAL_PROGRAM_PARAMS *data = (RUN_EXTERNAL_PROGRAM_PARAMS *)params;
        printf(" %sRUN EXTERNAL %s on (%s)\n", spaces(depth),
               data->externalProgram, data->rodDataFile);
      }
      break;
    case DELAY_COMMAND_ID:
      printf(" %sDELAY %f seconds\n", spaces(depth), *(float *)params);
      break;
    default:
      printf(" %sUnknown Command type is %d\n", spaces(depth), type);
    }
  }

  free(buffer);
}

void loadPlist(const char *fname, int depth) {
  printf("%sLoading plist %s\n", spaces(depth), fname);

  struct stat fileInfo;

  stat(fname, &fileInfo);

  int fileSize;

#if WIN32
  fileSize = 1000000;
#else
  if(!S_ISREG(fileInfo.st_mode)) {
    printf("Not a regular file!\n");
    return;
  }

  fileSize = fileInfo.st_size;
#endif

  char *buffer;
  buffer = (char *)malloc(fileSize);

  FILE *fin = fopen(fname, "rb");

  if(!fin) {
	printf("Bad file %s\n", fname);
	return;
  }

  int ret = fread(buffer, 1, fileSize, fin);

  if(ret == fileInfo.st_size) 
    printf ("%sRead file of size %d\n", spaces(depth), (int)fileSize);
  else {
    printf("Read failed\n");
    return;
  }

  char *endOfFile = buffer + ret;

  LIST_TABLE* primitiveList = (LIST_TABLE*)buffer;

  //  int noCommands = *(int*)(buffer);

  printf("%sThere are %d primitives\n", spaces(depth), primitiveList->allPrimCount);

  //  char *basePrim = buffer+4*100+16+16;

  for(unsigned int prim = 0; prim < primitiveList->allPrimCount; prim++) {
    printf("%s%d:\n", spaces(depth), prim);

    PRIM_TABLE *primStruct = &primitiveList->primTable[prim];

    // Primitive size is 168 !
    int index = prim*168 + 4;
    if(index > fileInfo.st_size) {
      printf("File size Oops %d\n", (int)fileInfo.st_size);
      break;
    }

    // MSG_HEAD inPrimHeader
    MSG_HEAD *msg = &primStruct->inPrimHeader;
    printf(" %sIn  -- Len: %d, Index: %d, Id: %d, Rev: %d, (%s)\n", spaces(depth), 
           msg->length, msg->index, msg->id, msg->primRevision,
           getName(msg->id));
    //     int type = *(int*)(basePrim+index);

    // MSG_HEAD outPrim
    msg = &primStruct->outPrimHeader;
    printf(" %sOut -- Len: %d, Index: %d, Id: %d, Rev: %d, (%s)\n", spaces(depth), 
           msg->length, msg->index, msg->id, msg->primRevision,
           getName(msg->id));

    unsigned int *baseParams = (unsigned int *)&primStruct->params; // basePrim+index+32;

    switch(primStruct->outPrimHeader.id) { // *(int*)(basePrim+index + 24)) {
    case ECHO:
      {
        int type = (*(int*)baseParams+0);
        printf("%s  ECHO primitive type (%d) %s\n", spaces(depth), type, (type==0)?"Test pattern":"From File");
        int length = (*(int*)(baseParams + 104));
        if(type == 0) {
          printf("%s  Pattern: 0x%08x length %d\n", spaces(depth), (*(int*)(baseParams + 4)), length);
        } else {
          printf("%s  File: %s length %d\n", spaces(depth), (char*)(baseParams + 4), length);
        }
      }
      break;
    case RW_REG_FIELD:
      {
        RW_REG_FIELD_IN *data = (RW_REG_FIELD_IN *)baseParams;
        printf("%s  %s register 0x%x, offset 0x%x, width %d, value 0x%x\n", spaces(depth), 
               data->readNotWrite?"Read":"Write",
               data->registerID, data->offset, data->width, 
               data->dataIn);
      }
      break;
    case START_SLAVE_EXECUTING:
      {
        START_SLAVE_EXECUTING_IN *data = (START_SLAVE_EXECUTING_IN *)baseParams;
        printf("%s  Start executing slave %d, on/off %d, type %d, timeout %dus\n", spaces(depth), 
               data->slaveNumber, data->commOnOff, data->slaveType, data->timeoutInUsec);
      }
      break;
    case BUILD_STREAM:
      {
        BUILD_STREAM_IN *data = (BUILD_STREAM_IN *)baseParams;
        printf("%s  Cmd Buffer %d, reset %d, ChipAddress %d, data %d %d %d %d, fibre %d\n", spaces(depth),
               data->cmdBuff, data->reset, data->chipAddress, data->data[0], data->data[1], data->data[2], data->data[3], data->fibre);
        printf("%s   ", spaces(depth));
        for(int i=0; i<N_CMD_LIST_CMDS; i++) {
          printf("(%d %d), ", data->cmdList.cmd[i], data->cmdList.data[i]);
        }
        printf("\b\b\n");
      }
      break;
    case SEND_STREAM:
      printf("%s  Cmd Buffer %d, capture serial? %d\n", spaces(depth), 
             baseParams[0], baseParams[1]);
      break;
    case SEND_SLAVE_LIST:
      {
        PRIM_SEND_SLAVE_LIST_PARAMS *params = (PRIM_SEND_SLAVE_LIST_PARAMS *)baseParams;
        SEND_SLAVE_LIST_IN *data = &params->primParams;
        printf("%s  Send to slave %d, %d, (%p, %p)\n", spaces(depth), 
               data->slaveNumber, data->listLength, data->slavePrimList, data->slaveRepData);

        printf("%s  ToROD length %d, index %d, prims %d, rev %d\n", spaces(depth+1),
               params->inSlaveListHeader.length, 
               params->inSlaveListHeader.index, 
               params->inSlaveListHeader.numMsgs, 
               params->inSlaveListHeader.primListRevision);
        printf("%s  FromROD length %d, index %d, prims %d, rev %d\n", spaces(depth+1),
               params->outSlaveListHeader.length, 
               params->outSlaveListHeader.index, 
               params->outSlaveListHeader.numMsgs, 
               params->outSlaveListHeader.primListRevision);

        printf("%s   See following primitives for contents\n", spaces(depth);
      }
      break;
    case START_SLAVE_LIST:
      {
        START_SLAVE_LIST_IN *data = (START_SLAVE_LIST_IN *)baseParams;
        printf("%s  Start on slave %d, pause %d, reply %d\n", spaces(depth), 
               data->slaveNumber, data->pauseMasterList, data->getSlaveReply);
      }
      break;
    case EVENT_TRAP_SETUP:
      {
        //        EVENT_TRAP_SETUP_IN *data = (EVENT_TRAP_SETUP_IN *)baseParams;
        printf("%s  Event trap setup (lots of parameters)\n", spaces(depth));
        printDataChunk(baseParams, sizeof(EVENT_TRAP_SETUP_IN)/4, depth);
      }
      break;
    case RW_MODULE_DATA:
      { 
        RW_MODULE_DATA_PARAMS *params = (RW_MODULE_DATA_PARAMS *)baseParams;
        RW_MODULE_DATA_IN *data = &(params->inputParams);
        printf("%s  %s module data Struct %d moduleNum %d from: \n%s   %s\n", spaces(depth), 
               data->readNotWrite?("READ"):("WRITE"), data->structId, data->moduleNum, spaces(depth), 
               params->inFileName);
     }
      break;
    case START_TASK:
      {
        START_TASK_PARAMS *params = (START_TASK_PARAMS *) baseParams;
        START_TASK_IN_NOHIST *data = &(params->inputParams);
        printf("%s  Task %d (%s), Rev %d, Pri %d, Comp %d\n", spaces(depth),
               data->taskType, getTaskName(data->taskType), data->taskRevision, data->priority, data->completionFlag);

        if(data->taskType == HISTOGRAM_CTRL_TASK) {
          printf("%s  Input file name %s\n", spaces(depth), params->inDataFileName);
          printFile(translateName(params->inDataFileName), depth);
        } else {
          printDataChunk((unsigned int*)&(data->taskStruct), primStruct->inPrimHeader.length - 8, depth);
        }
      }
      break;
      // These have params so can't just print from file
    case RW_SLAVE_MEMORY:
      {
        PRIM_RW_SLAVE_MEMORY_PARAMS *params = (PRIM_RW_SLAVE_MEMORY_PARAMS *) baseParams;
        RW_SLAVE_MEMORY_IN *data = &(params->inputParams);
        printf("%s  Slave %d, %s, sAdd %p, mAdd %p, words %d\n", spaces(depth),
               data->slaveNumber, data->readNotWrite?"READ":"WRITE", data->slaveAddress, data->masterAddress, data->numWords);
        printf("%s  Filename %s\n", spaces(depth), params->inDataFileName); // outDataFileName is same address
      }
      break;
    case SEND_DATA:
      {
        char *types[] = {"Mirror", "Router", "Stream"};
        SEND_DATA_PARAMS *params = (SEND_DATA_PARAMS*) baseParams;
        SEND_DATA_IN *data = &(params->inputParams);
        printf("%s  Type %s (%d) \"AuxVal\" %d, repBuffer %d, timeout %d\n", spaces(depth),
               types[data->dataType], data->dataType, data->auxVal, data->repBufferFlag, data->timeout);
        if(data->repBufferFlag) {
          printf("%s   Write to: %s\n", spaces(depth),
                 params->outDataFileName);
        }
      }
      break;
    case POLL_REG_FIELD:
      {
        PRIM_POLL_REG_FIELD_PARAMS *params = (PRIM_POLL_REG_FIELD_PARAMS*) baseParams;
        POLL_REG_FIELD_IN *data = &(params->inputParams);
        printf("%s  Reg 0x%x, offset %d, width %d, desiredVal %d, timeout %dus\n", spaces(depth),
               data->registerID, data->offset, data->width, data->desiredValue, data->timeoutInUsec);
      }
      break;
    case RW_FIFO:
      {
        char *fifoNames[] = {"INPUT", "DEBUG", "EVENT", "TIM"};
        char *bankNames[] = {"BANK_A", "BANK_B", "BANK_C"};
        PRIM_RWFIFO_PARAMS *params = (PRIM_RWFIFO_PARAMS*) baseParams;
        RW_FIFO_IN *data = &(params->inputParams);
        printf("%s  id %s (%d), bank %s (%d), %s, elements %d, addr %p\n", spaces(depth),
               fifoNames[data->fifoId], data->fifoId, bankNames[data->bank], data->bank, 
               data->readNotWrite?"READ":"WRITE", data->numElements, data->dataBaseAdr);
        if(data->readNotWrite) {
          printf("%s   Read %d from: %s\n", spaces(depth),
                 params->inDataFile.size, params->inDataFile.name);
        } else {
          printf("%s   Write to: %s\n", spaces(depth),
                 params->outDataFileName);
        }
      }
      break;
    case TEST:
    case TRANS_SERIAL_DATA:
      break;
    default:
      if(primStruct->inPrimHeader.length > 4) {
        printDataChunk(baseParams, primStruct->inPrimHeader.length - 4, depth);
      }
      break;
    }
    
  }

  free(buffer);
}

char *spaces(int num) {
  static char buf[100];

  for(int i=0; i<num*2; i++) {
    buf[i] = ' ';
  }

  buf[num*2] = '\0';

  return buf;
}

// Swap slashes and remove :
char *translateName(const char *oldFname) {
  static char clName[300];
  char *nptr = clName;
  const char *optr = oldFname;

  for(; *optr!=0; optr++) {
    if(*optr == '\\') {
      if(swapSlash) {
        *nptr++ = '/';
      } else {
        *nptr++ = '\\';
      }
    } else if(*optr == ':') {
      for(int i=0; i<strlen(prefix); i++) {
        nptr[i-1] = prefix[i];
      }
      nptr += strlen(prefix) - 1;
    } else {
      *nptr++ = *optr;
    }
  }

  *nptr = 0;

  return clName;
}

char *getName(int id) {
  for(int i=0; primNames[i].id != -1; i++) {
    if(primNames[i].id == id) {
      return primNames[i].name;
    }
  }

  return 0;
}

char *getTaskName(int id) {
  for(int i=0; taskNames[i].id != -1; i++) {
    if(taskNames[i].id == id) {
      return taskNames[i].name;
    }
  }

  return 0;
}

void printDataChunk(unsigned int *ptr, unsigned int length, int depth) {
  for(unsigned int i=0; i<length; i++) {
    if(i%4==0) printf("%s%s   ", i?"\n":"", spaces(depth));
    printf("0x%08x ", ptr[i]);
  }
  printf("\n");
}


void printFile(char *fname, int depth) {
  printf("%sLoading File %s\n", spaces(depth), fname);

  struct stat fileInfo;

  stat(fname, &fileInfo);

  int fileSize;

#if WIN32
  fileSize = 1000000;
#else
  if(!S_ISREG(fileInfo.st_mode)) {
    printf("Not a regular file!\n");
    return;
  }

  fileSize = fileInfo.st_size;
#endif

  long *buffer;
  buffer = (long *)malloc(fileSize);

  FILE *fin = fopen(fname, "rb");

  if(!fin) {
	printf("Bad file %s\n", fname);
	return;
  }

  int ret = fread((char *)buffer, 1, fileSize, fin);

  if(ret == fileSize) 
    printf ("%sRead file of size %d\n", spaces(depth), (int)fileSize);
  else {
    printf("Read failed\n");
    return;
  }

  printf("%s", spaces(depth));
  for(int i=0; i<fileInfo.st_size/4; i++) {
    if(i && i%8 == 0) printf("\n%s", spaces(depth));
    printf("%08lx ", buffer[i]);
  }
  printf("\n");
}
