#include "SctData/FitScanResult.h"
#include "SctData/FitObject.h"
#include "SctDaqRootFileExt.h"

#include "../../AnalysisTests/AnalysisTestFramework.h"
#include "../../AnalysisTests/CutUtils.h"

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int index;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Index:Serial/C";

struct FitData {
    double Params[5];
    double ChiSq;
    
    FitData() {Params[0]=Params[1]=Params[2]=Params[3]=Params[4]=ChiSq=0;}
};
string DataStr = "P0/D:P1:P2:P3:P4:ChiSq";


//Globals variables:
FitData RodDaq;
FitData SctDaq;
Info info;

//Output files
string SCurveOutput = "${SCT_DAQ_ROOT}/SystemTests/logs/SCurveFit.root";
string StrobeOutput = "${SCT_DAQ_ROOT}/SystemTests/logs/StrobeDelayFit.root";
string TWOutput = "${SCT_DAQ_ROOT}/SystemTests/logs/TimeWalkFit.root";


enum Mode {
    SCURVE,
    STROBE,
    TIMEWALK
};


class FitCompare : public AnalysisTestFramework<FitScanResult> {
public:
    FitCompare();
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const FitScanResult& t);   
    virtual void setup();
    virtual void printAdditionalArgs();
    virtual int handleArg(int argc, int i, char** argv);
    virtual void summaryOutput();
    void FillTree(FitObject& ob, unsigned int index);
    
    Mode mode;
};



string getFileName(const TestInfo& info, unsigned int scanNumber) {
    ostringstream oss;
    oss << info.path << "/strun" << info.runNumber << "_" << scanNumber << ".root";
    return oss.str();
}

double histMean(TH1& hist, unsigned int start, unsigned int nBins) {
    double total = 0;
    for (unsigned int i=start; i<start+nBins; ++i) {
	total += hist.GetBinContent(i);
    }
    return total/nBins;
}

void FitCompare::compare(const FitScanResult& fit) {
    info.runNumber = fit.getHeader().getRunNumber();
    info.scanNumber = fit.getHeader().getScanNumber();
    strncpy(info.serial, fit.getHeader().getModuleName().c_str(), 14);
			      
    TestInfo testInfo;
    switch (mode) {
    case SCURVE:
	testInfo = moduleData.getResponseCurveInfo(fit.getHeader().getModuleName());
	break;
    case STROBE:
	testInfo = moduleData.getStrobeDelayInfo(fit.getHeader().getModuleName());
	break;
    case TIMEWALK:
	testInfo = moduleData.getTimeWalkInfo(fit.getHeader().getModuleName());
	break;
    default:
	cerr << "Unsupported mode" << endl;
    }
    
    SctDaqRootFileExt file(getFileName(testInfo, fit.getHeader().getScanNumber()));
    unsigned int cycle = file.getCycleNum(fit.getHeader().getModuleName());
    
    
    for (unsigned int link = 0; link<nLinkModule; ++link) {	
	if (mode == SCURVE) {
	    auto_ptr<TH1> gMean = file.getMeanGraph(link, cycle);
	    auto_ptr<TH1> gSigma = file.getSigmaGraph(link, cycle);	
	    auto_ptr<TH1> gChiSq = file.getChiSqGraph(link, cycle);
	    
	    for (unsigned int i=0; i<nChannelLink; ++i) {
		SctDaq.Params[1] = gMean->GetBinContent(i);
		SctDaq.Params[2] = gSigma->GetBinContent(i);	 
		SctDaq.ChiSq = gChiSq->GetBinContent(i);
		
		FitObject& fitOb = fit.getChannelFit(link, i);
		FillTree(fitOb, i + link*nChannelLink);	    
	    }		
	} else {
	    for (unsigned int i=0; i<nChipLink; ++i) {	
		FitObject& fitOb = fit.getChipFit(link, i);
		FillTree(fitOb, i + link*nChipLink);	    
	    }
	}
    }
}

void FitCompare::FillTree(FitObject& fitOb, unsigned int index) {
    //Note SCTDAQ stores in bin 0 - the underflow bin!	
    for (unsigned int i=0; i<fitOb.getNPar(); ++i) {
	RodDaq.Params[i] = fitOb.getParameter(i);
    }
    RodDaq.ChiSq = fitOb.getChiSquared() / (fitOb.getNDF() + fitOb.getNPar());
	    
    info.index = index;
    tree->Fill();
}

/**
  Create the TFile and TTree
  */
void FitCompare::setup() {
    string name;
    switch (mode) {
    case SCURVE:
	name = SCurveOutput;
	break;
    case STROBE:
	name = StrobeOutput;
	break;
    case TIMEWALK:
	name = TWOutput;
	break;
    default:
	cerr << "Ooops, unsupported mode" << endl;
    }    
    name = Env::substituteVariables(name);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("FitData", "Fit Scan Comparison Data");
    tree->Branch("Info", &info, InfoStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    info.serial[14] = '\0';
}



/**
  Downloads the response curve data
  */
void FitCompare::downloadData(string serialNumber) {
}

void FitCompare::publishData(SctTestApiStatus* status) {
    switch (mode) {
    case SCURVE:
	highLevelApi->responseCurve(status);
	break;
    case STROBE:
	highLevelApi->strobeDelay(status);
	break;
    case TIMEWALK:
	highLevelApi->timeWalk(status);
	break;
    default:
	cerr << "Ooops, unsupported mode" << endl;
    }
}

/**
  Generates some summary output from the TTree
  */
void FitCompare::summaryOutput() {
    if (mode != SCURVE) return;
    if (cut(*tree, "mean", "(RodDaq.P1-SctDaq.P1)/SctDaq.P1*100", 0.1, 0.5, 5) > 0) {
	++exitCode;
	cout << "Failed mean tail check" << endl;
    }
    if (cut(*tree, "sigma", "(RodDaq.P2-SctDaq.P2)/SctDaq.P2*100", 1, 2, 20) > 0) {
	++exitCode;
	cout << "Failed sigma tail check" << endl;
    }
    if (cut(*tree, "chi", "(RodDaq.ChiSq-SctDaq.ChiSq)", 0.3, 0.5, 10) > 0) {
	++exitCode;
	cout << "Failed ChiSq tail check" << endl;
    }
    exitCode += errorCode;
}

FitCompare::FitCompare() : AnalysisTestFramework<FitScanResult>(".*FitScanResult", "FittedData") {
    mode = SCURVE;
    nData = 10;
}

void FitCompare::printAdditionalArgs() {
    cout << "-strobe:           Do strobe delay fits instead" << endl;
    cout << "-timewalk:		Do strobe delay fits using the timewalk test instead" << endl;
    cout << "Default is to publish a response curve and do s-curve fitting" << endl;
}

int FitCompare::handleArg(int argc, int i, char** argv) {
    if (string(argv[i]) == "-strobe") {
	mode = STROBE;  
	nData = 1;
    } else if (string(argv[i]) == "-timewalk") {
	mode = TIMEWALK;
	nData = 10;
    } else {
	cout << "Invalid argument " << argv[i] << endl;
	printHelp();
    }
    return i;
}

int main(int argc, char** argv) {
    TH1::AddDirectory(true);
    FitCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}




