#include "SctData/StrobeDelayTestResult.h"
#include "../../AnalysisTestFramework.h"
#include "../../CutUtils.h"
#include "../../TestFileUtil.h"

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Chip:NScans:Serial/C";

struct SDData {
    double delay;
    int pass;
};
string DataStr = "Delay/D:Pass/I";

//Globals variables:
SDData RodDaq;                    //Our data
SDData SctDaq;                    //SCTDAQ data
Info info;                          //Some info

//Noise Occupancy stuff
string Ext = "_StrobeDelay.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/StrobeDelay.root";
string Arg = "-4";

class StrobeDelayCompare : public AnalysisTestFramework<StrobeDelayTestResult> {
public:
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const StrobeDelayTestResult& t);   	
    virtual void setup();
    virtual void summaryOutput();
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void StrobeDelayCompare::compare(const StrobeDelayTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    TestFileUtil::skipHeader(file);

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	string chipStr;
	info.chip = chip;
	file >> chipStr >> SctDaq.delay;
	//Ignore rest of line
	file.ignore(256, '\n');

	RodDaq.delay = sd.getOptimum(chip);
	RodDaq.pass = sd.getPassed();

        tree->Fill();
    }
}

void StrobeDelayCompare::summaryOutput() {
    if (cut(*tree, "delay", "(RodDaq.Delay-SctDaq.Delay)", 0.5, 0.5, 1.1, true) > 0) {
	++exitCode;
	cout << "Failed delay tail check" << endl;
    }   
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.01, 0.01, 0.01, true) > 0) {
	++exitCode;
	cout << "Failed pass tail check" << endl;
    }
    exitCode += errorCode;
}


/**
  Create the TFile and TTree
  */
void StrobeDelayCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("SDData", "StrobeDelay Comparison Data");
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("Info", &info, InfoStr.c_str());
    info.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void StrobeDelayCompare::downloadData(string serialNumber) {
    ostringstream oss;
    TestInfo info = moduleData.getStrobeDelayInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void StrobeDelayCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->strobeDelay(status);
}


int main(int argc, char** argv) {
    StrobeDelayCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}












