#include "SctData/NPtGainTestResult.h"
#include "SctData/ResponseCurve.h"
#include "../../AnalysisTestFramework.h"
#include "../../CutUtils.h"
#include "../../TestFileUtil.h"

struct ChannelInfo {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int channel;
    unsigned int nScans;
    float SCTDAQVersion;
    char serial[15];    
};
string ChannelInfoStr = "Run/i:Scan:Channel:NScans:SCTDAQVersion/F:Serial/C";

struct ChannelRCData {
    double gain;
    double noise;
    double offset;
    int code;
};
string ChannelDataStr = "Gain/D:Noise:Offset:Code/I";

struct ChipInfo {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    float SCTDAQVersion;
    char serial[15];    
};
string ChipInfoStr = "Run/i:Scan:Chip:NScans:SCTDAQVersion/F:Serial/C";

struct ChipRCData {
    double gain;
    double noise;
    double offset;
    double p0;
    double p1;
    double p2;
    int func;
    int pass;
};
string ChipDataStr = "Gain/D:Noise:Offset:P0:P1:P2:Func/I:Pass";

//Response curve stuff
string RCExt = "_rc.dat";
string RCExt2 = "_ResponseCurve.dat";
string RCOutput = "${SCT_DAQ_ROOT}/SystemTests/logs/ResponseCurve.root";
string RCArg = "-RC";
string RCArg2 = "-7";

//Three-point gain stuff
string GainExt = "_rc3.dat";
string GainExt2 = "_3PtGain.dat";
string GainOutput = "${SCT_DAQ_ROOT}/SystemTests/logs/ThreePointGain.root";
string GainArg = "-3ptgain";
string GainArg2 = "-5";

//Globals variables:
ChipRCData RodDaq;                    //Our data
ChipRCData SctDaq;                    //SCTDAQ data
ChipInfo info;                          //Some info

ChannelRCData chRodDaq;	
ChannelRCData chSctDaq;	
ChannelInfo chInfo;


class NPtGainCompare : public AnalysisTestFramework<NPtGainTestResult> {
public:
    NPtGainCompare();
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const NPtGainTestResult& t);   
    virtual float compareChip(const NPtGainTestResult& t);   	
    virtual void compareChannel(const NPtGainTestResult& t, float version);   	
    virtual void setup();
    virtual void printAdditionalArgs();
    virtual int handleArg(int argc, int i, char** argv);
    virtual void summaryOutput();
    string getFileName(string serial);
    string getRCFileName(string serial);
    
    bool rc;
    TTree* channelTree;
};

void NPtGainCompare::summaryOutput() {
    //Chip data tests
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.01, 0.01, 0.01, true) > 0) {
        ++exitCode;
        cout << "Failed pass tail check" << endl;
    }    
    if (cut(*tree, "gain", "(RodDaq.Gain-SctDaq.Gain)/SctDaq.Gain", 0.01, 0.01, 0.15, true) > 0) {
        ++exitCode;
        cout << "Failed gain tail check" << endl;
    }    
    if (cut(*tree, "noise", "(RodDaq.Noise-SctDaq.Noise)/SctDaq.Noise", 0.02, 0.02, 0.25, true) > 0) {
        ++exitCode;
        cout << "Failed noise tail check" << endl;
    }    
    if (cut(*tree, "offset", "(RodDaq.Offset-SctDaq.Offset)", 1, 10, 80, true) > 0) {
        ++exitCode;
        cout << "Failed offset tail check" << endl;
    }
    /*
      No checks on the output params - they are too correlated for it to mean anything
      Hopefully the aggregate/summary tests above and below will suffice
      
    if (cut(*tree, "p0", "(RodDaq.P0-SctDaq.P0)", 0.01, 0.01, 0.01, true) > 0) {
        ++exitCode;
        cout << "Failed p0 tail check" << endl;
    }    
    if (cut(*tree, "p1", "(RodDaq.P1-SctDaq.P1)", 0.01, 0.01, 0.01, true) > 0) {
        ++exitCode;
        cout << "Failed p1 tail check" << endl;
    }    
    if (cut(*tree, "p2", "(RodDaq.P2-SctDaq.P2)", 0.01, 0.01, 0.01, true) > 0) {
        ++exitCode;
        cout << "Failed p2 tail check" << endl;
    }    
    */
    
    //Channel data tests
    if (cut(*channelTree, "gain_chan", "(RodDaq.Gain-SctDaq.Gain)/SctDaq.Gain", 0.01, 0.01, 0.15, true, "RodDaq.Code==0 && SctDaq.Code==0") > 0) {
        ++exitCode;
        cout << "Failed gain tail check" << endl;
    }
    if (cut(*channelTree, "noise_chan", "(RodDaq.Noise-SctDaq.Noise)/SctDaq.Noise", 0.02, 0.02, 0.25, true, "RodDaq.Code==0 && SctDaq.Code==0") > 0) {
        ++exitCode;
        cout << "Failed noise tail check" << endl;
    }
    if (cut(*channelTree, "offset_chan", "(RodDaq.Offset-SctDaq.Offset)", 1, 10, 80, true, "RodDaq.Code==0 && SctDaq.Code==0") > 0) {
        ++exitCode;
        cout << "Failed offset tail check" << endl;
    }
    //Allow a very small %age of differences
    //Hopefully all these will be checked manually
    if (cut(*channelTree, "code", "(RodDaq.Code-SctDaq.Code)", 3, 800, 0.01, true) > 0.0015) {
        ++exitCode;
        cout << "Failed code tail check" << endl;
    }

    exitCode += errorCode;
}


void NPtGainCompare::compare(const NPtGainTestResult& sd) {
    float version = compareChip(sd);
    compareChannel(sd, version);
}

int getCode(const NPtGainTestResult& rc, unsigned int channel) {
    list<Defect> defects = rc.getDefects().getDefectsAffectingElement(ModuleElement::Channel(channel))->getAllDefects();
    
    int code = 0;
    if (defects.size() > 0) cout << "Channel: " << channel << endl;
    for (list<Defect>::const_iterator i=defects.begin(); i!=defects.end(); ++i) {
	cout << i->getPrototype().getName() << endl;
	if (i->getPrototype() == StandardDefects::DEAD) code += 0x1;
	else if (i->getPrototype() == StandardDefects::STUCKON) code += 0x2;
	else if (i->getPrototype() == StandardDefects::PARTBONDED) code += 0x8000;
	else if (i->getPrototype() == StandardDefects::UNBONDED) code += 0x1000;
	else if (i->getPrototype() == StandardDefects::NOISY) code += 0x4000;	
	else if (i->getPrototype() == StandardDefects::LO_GAIN) code += 0x100;	
	else if (i->getPrototype() == StandardDefects::HI_GAIN) code += 0x200;
	else if (i->getPrototype() == StandardDefects::LO_OFFSET) code += 0x400;
	else if (i->getPrototype() == StandardDefects::HI_OFFSET) code += 0x800;	
	//These are not SCTDAQ codes - should be ignored?
	else if (i->getPrototype() == StandardDefects::VLO_GAIN) code += 0x10000; 
	else if (i->getPrototype() == StandardDefects::UNDER) code += 0x20000; 
	else if (i->getPrototype() == StandardDefects::BADFIT) code += 0x30000; 
	else code += 100000;
    }
    return code;
}

string NPtGainCompare::getRCFileName(string serial) {
    ostringstream oss;
    oss << serial;
    if (rc) oss << RCExt;
    else oss << GainExt;
    
    return oss.str();
}

void NPtGainCompare::compareChannel(const NPtGainTestResult& rc, float version) {
    strncpy(chInfo.serial, rc.getModuleName().c_str(), 14);
    chInfo.runNumber = rc.getRunNumber();
    chInfo.scanNumber = rc.getScanNumberAt(0);
    chInfo.nScans = rc.getNScans();
    chInfo.SCTDAQVersion = version;
    
    string fileName = getRCFileName(rc.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    //Dump first line
    file.ignore(256, '\n');

    for (unsigned int channel = 0; channel<nChannelModule; ++channel) {
	string comment;
	file >> dec >> chInfo.channel >> hex >> chSctDaq.code >> chSctDaq.gain >> chSctDaq.offset >> chSctDaq.noise >> comment;
	//Ignore rest of line
	file.ignore(256, '\n');
	
	//Handle version differences and the fact that sometimes SCTDAQ records DEAD channels as
	//having lots of other defects too!  
	
	//Version >= 3.42 || Version <3.38
	if ((chInfo.SCTDAQVersion>3.415 || chInfo.SCTDAQVersion<3.3775) && (chSctDaq.code==4353 || chSctDaq.code==5377)) chSctDaq.code = 1;
	//3.38<=Version<=3.41
	if (chInfo.SCTDAQVersion>3.375 && chInfo.SCTDAQVersion<3.415 && chSctDaq.code==4481) chSctDaq.code = 1;
	
	if (channel != chInfo.channel) 
	    throw IllegalStateError("Read incorrect channel", __FILE__, __LINE__);
	
	const NPtGainTestResultData& data = rc.getChannelData(channel);

	chRodDaq.code = getCode(rc, channel);
	chRodDaq.gain = data.gain;
	chRodDaq.noise = data.noise;
	chRodDaq.offset = data.offset;

        channelTree->Fill();
    }
}

string NPtGainCompare::getFileName(string serial) {
    ostringstream oss;
    oss << serial << (rc ? RCExt2 : GainExt2);
    
    return oss.str();
}

//Returns the version
float NPtGainCompare::compareChip(const NPtGainTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    info.SCTDAQVersion = TestFileUtil::skipHeader(file);

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	info.chip = chip;
	double temp;
	string chipStr;
	file >> chipStr >> temp >> temp >> SctDaq.gain >> temp >> SctDaq.offset >> temp >> temp >> SctDaq.noise >> temp >> SctDaq.func
	     >> SctDaq.p0 >> SctDaq.p1 >> SctDaq.p2;
	//Ignore rest of line
	file.ignore(256, '\n');
	
	RodDaq.pass = sd.getPassed();
	const NPtGainTestResultData& data = sd.getChipData(chip);	
	RodDaq.gain = data.gain;
	RodDaq.noise = data.noise;
	RodDaq.offset = data.offset;
	RodDaq.func = data.rc->getIndex();
	TF1& f = *data.rc->getFunction();
	RodDaq.p0 = f.GetParameter(0);
	RodDaq.p1 = f.GetParameter(1);
	RodDaq.p2 = f.GetParameter(2);

        tree->Fill();
    }
    return info.SCTDAQVersion;
}


/**
  Create the TFile and TTree
  */
void NPtGainCompare::setup() {
    string name = Env::substituteVariables(rc ? RCOutput : GainOutput);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("NPtChip", "NPtGain Chip Comparison Data");
    channelTree = new TTree("NPtChannel", "NPtGain Channel Comparison Data");
    tree->Branch("RodDaq", &RodDaq, ChipDataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, ChipDataStr.c_str());
    tree->Branch("Info", &info, ChipInfoStr.c_str());
    
    channelTree->Branch("RodDaq", &chRodDaq, ChannelDataStr.c_str());
    channelTree->Branch("SctDaq", &chSctDaq, ChannelDataStr.c_str());
    channelTree->Branch("Info", &chInfo, ChannelInfoStr.c_str());    
    
    info.serial[14] = '\0';
    chInfo.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void NPtGainCompare::downloadData(string serialNumber) {
    ostringstream oss;
    ostringstream oss1;
    TestInfo info = rc ? moduleData.getResponseCurveInfo(serialNumber) : moduleData.get3PtGainInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << (rc ? RCArg : GainArg) << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    oss1 << "java ProdDatabase/getDBfiles " << (rc ? RCArg2 : GainArg2) << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
    system(oss1.str().c_str());
}

void NPtGainCompare::publishData(SctTestApiStatus* status) {
    if (rc) highLevelApi->responseCurve(status);
    else highLevelApi->threePointGain(status);
}

NPtGainCompare::NPtGainCompare() {
    channelTree = 0;
    rc = true;
}

void NPtGainCompare::printAdditionalArgs() {
    cout << "-3:           3pt-gain rather than response curve" << endl;
}

int NPtGainCompare::handleArg(int argc, int i, char** argv) {
    if (string(argv[i]) == "-3") {
	rc = false;    
    } else {
	printHelp();
    }
    return i;
}

int main(int argc, char** argv) {
    NPtGainCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}


