#include "SctData/NoiseOccupancyTestResult.h"
#include "SctData/RawScanResult.h"
#include "Sct/IS/IONameIS.h"
#include "../../AnalysisTestFramework.h"
#include "../../CutUtils.h"
#include "../../TestFileUtil.h"

using namespace SctData;
using namespace Sct::IS;

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    float SCTDAQVersion;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Chip:NScans:SCTDAQVersion/F:Serial/C";

struct NOData {
    double mean;
    double rms;
    double offset;
    double noise;
    int pass;
};
string DataStr = "Mean/D:Rms:Offset:Noise:Pass/I";

//Globals variables:
NOData RodDaq;                    //Our data
NOData SctDaq;                    //SCTDAQ data
Info info;                          //Some info

//Noise Occupancy stuff
string Ext = "_Noise.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/NoiseOccupancy.root";
string Arg = "-8";

class NoiseOccupancyCompare : public AnalysisTestFramework<NoiseOccupancyTestResult> {
public:
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const NoiseOccupancyTestResult& t);   
    virtual void summaryOutput();
    virtual void setup();
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void NoiseOccupancyCompare::summaryOutput() {
    if (cut(*tree, "mean", "(RodDaq.Mean-SctDaq.Mean)", 5e-5, 5e-5, 2e-4, true) > 0) {
        ++exitCode;
        cout << "Failed mean tail check" << endl;
    }
    //Allow 0.1% to fail this cut - due to SCTDAQ not excluding defective channels from calc
    if (cut(*tree, "rms", "(RodDaq.Rms-SctDaq.Rms)", 5e-5, 5e-5, 2e-4, true) > 0.001) {
        ++exitCode;
        cout << "Failed rms tail check" << endl;
    }
    if (cut(*tree, "Offset", "(RodDaq.Offset-SctDaq.Offset)/SctDaq.Offset", 0.01, 0.01, 0.03, true, "Info.SCTDAQVersion>3.401") > 0) {
        ++exitCode;
        cout << "Failed offset tail check" << endl;
    }
    if (cut(*tree, "Offset", "(RodDaq.Offset-SctDaq.Offset-9.5)", 1, 3, 10, true, "Info.SCTDAQVersion<3.401") > 0) {
        ++exitCode;
        cout << "Failed old version offset tail check" << endl;
    }
    if (cut(*tree, "noise", "(RodDaq.Noise-SctDaq.Noise)/SctDaq.Noise", 0.01, 0.02, 0.1, true) > 0) {
        ++exitCode;
        cout << "Failed noise tail check" << endl;
    }
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.01, 0.01, 0.01, true) > 0) {
        ++exitCode;
        cout << "Failed pass tail check" << endl;
    }
    exitCode += errorCode;
}

void NoiseOccupancyCompare::compare(const NoiseOccupancyTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    info.SCTDAQVersion = TestFileUtil::skipHeader(file);
    //Holds the RawScanResult if needed
    shared_ptr<RawScanResult> result;
    
    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	string chipStr;
	info.chip = chip;
	file >> chipStr >> SctDaq.offset >> SctDaq.mean >> SctDaq.rms >> SctDaq.noise;
	//Ignore rest of line
	file.ignore(256, '\n');
	
	//Correct offset if it is negative (SCTDAQ bug)
	if (SctDaq.offset < 0) {
	    if (!result) {
		try {
		    IONameIS name(RawScanResult::getUniqueID(sd.getHeader()), "SctData::RawScanResult", SctNames::getEventDataName());
		    shared_ptr<Serializable> ob = IOManagerIS::instance().read(name.getIOName());
		    result = dynamic_pointer_cast<RawScanResult>(ob);
		} catch (IoException& e) {}
	    }
	    if (result) {
		double target = 2.5*result->getConfiguration().getChipConfiguration(chip).getTrimTarget();
		SctDaq.offset+=target;
	    }
	}
	
	const ChipNOResult& data = sd.getChipResult(chip);

	RodDaq.mean = data.mean;
	RodDaq.offset = data.offset;
	RodDaq.rms = data.rms;
	RodDaq.noise = data.getNoise();
	RodDaq.pass = sd.getPassed();


        tree->Fill();
    }
}


/**
  Create the TFile and TTree
  */
void NoiseOccupancyCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("NOData", "NoiseOccupancy Comparison Data");
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("Info", &info, InfoStr.c_str());
    info.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void NoiseOccupancyCompare::downloadData(string serialNumber) {
    ostringstream oss;
    TestInfo info = moduleData.getNoiseOccupancyInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void NoiseOccupancyCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->noiseOccupancy(status);
}


int main(int argc, char** argv) {
    NoiseOccupancyCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}



















