#include "SctData/FullBypassTestResult.h"
#include "../../AnalysisTestFramework.h"
#include "../../TestFileUtil.h"
#include "../../CutUtils.h"

struct Info {
    unsigned int runNumber;
    unsigned int scanNumber;
    unsigned int chip;
    unsigned int nScans;
    float SCTDAQVersion;
    char serial[15];    
};
string InfoStr = "Run/i:Scan:Chip:NScans:SCTDAQVersion/F:Serial/C";

struct FBData {
    double vddDirect;
    double vddBypass;
    int statusDirect;
    int statusBypass;
    int pass;
};
string DataStr = "VddDirect/D:VddBypass:StatusDirect/I:StatusBypass:Pass";

//Globals variables:
FBData RodDaq;                    //Our data
FBData SctDaq;                    //SCTDAQ data
Info info;                          //Some info

//Noise Occupancy stuff
string Ext = "_FullByPass.dat";
string Output = "${SCT_DAQ_ROOT}/SystemTests/logs/FullBypass.root";
string Arg = "-2";

class FullBypassCompare : public AnalysisTestFramework<FullBypassTestResult> {
public:
    virtual void publishData(SctTestApiStatus* status);
    virtual void downloadData(string serialNumber);
    virtual void compare(const FullBypassTestResult& t);   
    virtual void summaryOutput();
    virtual void setup();
};


string getFileName(string serial) {
    ostringstream oss;
    oss << serial << Ext;
    
    return oss.str();
}

void FullBypassCompare::summaryOutput() {
    if (cut(*tree, "pass", "(RodDaq.Pass-SctDaq.Pass)", 0.1, 0.1, 0, true) > 0) {
        ++exitCode;
	cout << "Failed pass check" << endl;
    }
    if (cut(*tree, "VddDirect", "(RodDaq.VddDirect-SctDaq.VddDirect)", 0.1, 0.1, 0, true) > 0) {
        ++exitCode;
	cout << "Failed VddDirect check" << endl;
    }
    if (cut(*tree, "VddBypass", "(RodDaq.VddBypass-SctDaq.VddBypass)", 0.1, 0.1, 0, true) > 0) {
        ++exitCode;
	cout << "Failed VddBypass check" << endl;
    }
    if (cut(*tree, "StatusDirect", "(RodDaq.StatusDirect-SctDaq.StatusDirect)", 0.1, 0.1, 0, true) > 0) {
        ++exitCode;
	cout << "Failed StatusDirect check" << endl;
    }
    if (cut(*tree, "StatusBypass", "(RodDaq.StatusBypass-SctDaq.StatusBypass)", 0.1, 0.1, 0, true) > 0) {
        ++exitCode;
	cout << "Failed StatusBypass check" << endl;
    }
    exitCode += errorCode;
}

int getStatus(string comment) {
    if (comment == "Minimal tested") return 1;
    if (comment == "Not Tested") return -2;
    return -4;
}

void FullBypassCompare::compare(const FullBypassTestResult& sd) {
    strncpy(info.serial, sd.getModuleName().c_str(), 14);
    info.runNumber = sd.getRunNumber();
    info.scanNumber = sd.getScanNumberAt(0);
    info.nScans = sd.getNScans();
    
    string fileName = getFileName(sd.getModuleName());
    ifstream file (fileName.c_str());
    if (!file.good())
        throw IllegalStateError("Failed to open comparison file: " + fileName, __FILE__, __LINE__);
    
    SctDaq.pass = TestFileUtil::getPass(file);
    info.SCTDAQVersion = TestFileUtil::skipHeader(file);
    char buffer[1024];

    for (unsigned int chip = 0; chip<nChipModule; ++chip) {
	info.chip = chip;
	string chipStr;
	string directComment;
	string bypassComment;
		
	file >> chipStr >> SctDaq.vddDirect >> SctDaq.vddBypass;
	    
	//Skip an initial tab character
	file.ignore(1, '\t');
	file.getline(buffer, 1024, '\t');
	directComment = buffer;
	file.getline(buffer, 1024, '\n');
	bypassComment = buffer;
	
	SctDaq.statusBypass = getStatus(bypassComment);
	SctDaq.statusDirect = getStatus(directComment);
	
	RodDaq.pass = sd.getPassed();
	RodDaq.statusBypass = sd.getChipResult(chip).status_bypass;
	RodDaq.statusDirect = sd.getChipResult(chip).status_direct;
	RodDaq.vddBypass = sd.getChipResult(chip).vdd_bypass;
	RodDaq.vddDirect = sd.getChipResult(chip).vdd_direct;
	
        tree->Fill();
    }
}


/**
  Create the TFile and TTree
  */
void FullBypassCompare::setup() {
    string name = Env::substituteVariables(Output);
    file = new TFile(name.c_str(), "RECREATE");
    tree = new TTree("FBData", "FullBypass Comparison Data");
    tree->Branch("RodDaq", &RodDaq, DataStr.c_str());
    tree->Branch("SctDaq", &SctDaq, DataStr.c_str());
    tree->Branch("Info", &info, InfoStr.c_str());
    info.serial[14] = '\0';
}

/**
  Downloads the response curve data
  */
void FullBypassCompare::downloadData(string serialNumber) {
    ostringstream oss;
    TestInfo info = moduleData.getFullBypassInfo(serialNumber);
    oss << "java ProdDatabase/getDBfiles " << Arg << " -d -r " << info.runNumber << " -s " << info.scanNumber << "  " << serialNumber;
    system(oss.str().c_str());
}

/**
  Publishes all the data we need, looping over modules,
  then waits for it all to be fitted, analyzed
  */
void FullBypassCompare::publishData(SctTestApiStatus* status) {
    highLevelApi->fullBypass(status);
    //usleep(100000);
    sleep(1);
}


int main(int argc, char** argv) {
    FullBypassCompare sdc;
    sdc.analyzeAll(argc, argv);
    return sdc.getExitCode();
}












