#ifndef ANALYSISTESTFRAMEWORK_H
#define ANALYSISTESTFRAMEWORK_H

#include <vector>
#include <string>
#include <algorithm>
#include <is/isinfo.h>
#include <boost/shared_ptr.hpp>
#include <iostream>
#include <fstream>
#include <TTree.h>
#include <TApplication.h>
#include <TFile.h>
#include <TROOT.h>

#include "Sct/IS/IOManagerIS.h"
#include "Sct/Exception.h"
#include "Sct/Env.h"
#include "Sct/LogicErrors.h"
#include "Sct/SctNames.h"
#include "Sct/ILUMarshalling.h"
#include "Sct/SctParameters.h"

#include "SctData/DefectList.h"
#include "SctData/Defect.h"
#include "SctData/StandardDefects.h"
#include "SctData/ModuleElement.h"
#include "../SCTTestAPI/ModuleData.h"
#include "../SCTTestAPI/XmlSummaryFile.h"
#include "../SCTTestAPI/SctTestApi.hh"

using namespace Sct;
using namespace Sct::IS;
using namespace SctData;
using namespace SctTest;
using namespace boost;
using namespace std;

template<typename T>
class AnalysisTestFramework {
public:   
    /**
      Creates the framework.  The isFilter is a string which selects the object in IS.  It is 
      assumed that the serial number can be appended to it to create the correct filter.  The isServer
      is the server to get the objects from.
      */
    AnalysisTestFramework(string isFilter=".*TestResult", string isServer="TestData");
    virtual ~AnalysisTestFramework(){}
    
    void analyzeAll(int argc, char** argv);
    
    /**
      Call to print the help message and exit
      */
    void printHelp();
    
    /**
      Return the exit code
      */
    int getExitCode() {return exitCode;}
    
protected:
    /**
      May be overriden to handle additional args
      Should return the last used argument index
      If it doesn't understand the arg, it should call printHelp()
      */
    virtual int handleArg(int argc, int i, char** argv);
    
    /**
      Override to print addition help messages
      */
    virtual void printAdditionalArgs() {}
    
    /**
      Override to change print help header
      */
    virtual void printHelpHeader();
    
    virtual void summaryOutput() {}
    
    /**
      The main method for doing anything.  Sets the appropriate module serialNumber, then 
      calls publishData, then calls downloadData, waits for the data to be published, and calls
      compare.
      */
    virtual void doModule(string serialNumber);
    
    /**
      Waits for and returns the data corresponding to serialNumber
      Reads nData objects.
      */
    virtual vector<shared_ptr<T> > readData(string serialNumber);
    
    /**
      Should override this to publish the data
      */
    virtual void publishData(SctTestApiStatus* status) {}
    
    /**
      Should override this to download data from the database
      */
    virtual void downloadData(string serialNumber) {}
    
    virtual void compare(const T& t) {}
    
    virtual void setup() {}
        
    virtual void tearDown();
    
    int exitCode;	//The exit code
    bool readOnly;	//If true, only read and compare data, don't analyze
    bool debug;
    IPCObjectVar<SctTestApi_T_HighLevelApi> highLevelApi; //The high-level api
    unsigned int nModules;		//The number of modules in the list
    unsigned int nSkip;			//The number of modules to skip from the input file
    list<string> modules;             //The modules to do
    TFile* file;                    //Output file
    TTree* tree;                  //Output tree
    ModuleData moduleData;              //Here we hold all the info about the tests
    unsigned int maxModules;	    //Default Max number of modules to test
    unsigned int nData;			//The number of data objects to read
    string isFilter;			//To choose the wanted objects    
    string isServer;			//Holds the ISServer where we wait for object - default TestData
    list<string> ignoreModules;		//A list of modules to be ignored    
    string ignoreFile;			//The name of a file used for ignoring 'problem' modules
    
private:    
    void loadIgnoreFile();
    void handleArgs(int argc, char** argv);
    static void getRootApp();
};

template<typename T>
vector<shared_ptr<T> > AnalysisTestFramework<T>::readData(string serialNumber) {
    string filter = isFilter + ".*" + serialNumber;
    while (true) {
	ISInfoIterator it(SctNames::getPartition(), isServer.c_str(), filter.c_str() );
	ISInfoDictionary dict(SctNames::getPartition());
	//cout << it.entries() << " for isServer: " << isServer << " and filter: " << filter << " waiting for: " << nData << endl;
	if (it.entries() == nData) {
	    vector<shared_ptr<T> > data;
	    while (it()) {
		shared_ptr<Serializable> ob = IOManagerIS::instance().read(it);
		shared_ptr<T> result = dynamic_pointer_cast<T>(ob);
		if (!result)
		    throw IllegalStateError(string("Failed to read test: ") + it.name(), __FILE__, __LINE__);
		data.push_back(result);
	    }
	    return data;
	}
	sleep(1);
    }
}

/**
  Close the TFile
  */
template<typename T>
void AnalysisTestFramework<T>::tearDown() {
    file->Write();
    file->Close();
    delete file;
}

template<typename T>
void AnalysisTestFramework<T>::doModule(string serialNumber) {
    SctTestApiStatus status; 
    status.returnCode = 0;
    
    highLevelApi->setModuleSerial(&status, copyStringToILU(serialNumber));
    if (!readOnly) publishData(&status);
    if (!readOnly && status.returnCode != 0) {
	cerr << "Error publishing data: " << status.returnCode << endl;
	return;
    }
    try {
	downloadData(serialNumber);    
	vector<shared_ptr<T> > ob = readData(serialNumber);
	for (unsigned int i=0; i<ob.size(); ++i) {
	    compare(*ob[i]);
	}
    } catch(Throwable& e) {
	e.sendToMrs(MRS_ERROR);
    }
}


template<typename T>
void AnalysisTestFramework<T>::printHelpHeader() {
    cout << "RetrieveAndCompare [-n <num>] [-r] [-s <serials>] [-h/--help]" << endl;
    cout << "Compares SCTDAQ and SctRodDaq analyses" << endl;
    cout << "Arguments: " << endl;
}

template<typename T>
void AnalysisTestFramework<T>::printHelp() {
    printHelpHeader();
    cout << "-n <num>:     Do up to <num> modules only, otherwise do all" << endl;
    cout << "-r:           Read only - Useful for a re-run" << endl;
    cout << "-j <num>:     Skip the first <num> modules in the config files" << endl;
    cout << "-h/--help:    Print this help" << endl;
    cout << "-i <file>:    Ignore file - a file with serial numbers, 1 per line, that will be ignored - useful for 'problem' modules" << endl;
    cout << "-s <serials>: All arguments after -s are assumed to be module serial numbers" << endl;
    printAdditionalArgs();
    
    exit(0);
}

template<typename T>
int AnalysisTestFramework<T>::handleArg(int argc, int i, char** argv) {
    printHelp();
    return i;
}


template<typename T>
void AnalysisTestFramework<T>::handleArgs(int argc, char** argv) {
    for (int i=1; i<argc; ++i) {
	if (string(argv[i]) == "-s") {
	    for (int j=i+1; j<argc; ++j) {
		modules.push_back(argv[j]);
		++nModules;
	    }
	    return;
	}
	
	if (string(argv[i]) == "-n") {
	    if (i+1 == argc) printHelp();
	    istringstream iss(argv[++i]);
	    iss >> maxModules;
	} else if (string(argv[i]) == "-r") {
	    readOnly = true;
	} else if (string(argv[i]) == "-j") {
	    if (i+1 == argc) printHelp();
	    istringstream iss(argv[++i]);
	    iss >> nSkip;
	} else if (string(argv[i]) == "-i") {
	    if (i+1 == argc) printHelp();
	    ignoreFile = argv[++i];
	} else {
	    i = handleArg(argc, i, argv);
	}
	
    }
}


template<typename T>
void AnalysisTestFramework<T>::getRootApp() {
    //Create TApplication - necessary to prevent batch mode and no graphics on later ROOT versions
    //static TApplication myapp("myapp", 0, 0);
    gROOT->SetBatch(false);
    //return myapp;
}

template<typename T>
void AnalysisTestFramework<T>::loadIgnoreFile() {
    if (ignoreFile.length()==0) return;
    ifstream file(ignoreFile.c_str());
    if (!file.is_open() || !file.good()) {
	cerr << "Unable to open ignore file: " << ignoreFile << endl;
	return;
    }   
    string serial;
    while (file.good()) {
	file >> serial;
	if (serial[0] != '#' && serial.length() == 14 && count(ignoreModules.begin(), ignoreModules.end(), serial)==0) {
	    ignoreModules.push_back(serial);
	}
	file.ignore(256, '\n');
    }
}

template<typename T>
void AnalysisTestFramework<T>::analyzeAll(int argc, char** argv) {
    Sct::setExceptionHandlers(argv[0]);
    getRootApp();
    
    try {
        handleArgs(argc, argv);
	
	//Get HighLevelApi:
	highLevelApi = SctNames::getPartition().lookup(SctTestApi_C_HighLevelApi_serverName, SctTestApi_C_HighLevelApi_instanceName, SctTestApi_T_HighLevelApi::ILUClassRecord);
	if (highLevelApi == 0) //Can't use !highLevelApi!
	    throw IllegalStateError("Couldn't get HighLevelApi", __FILE__, __LINE__);
	
	SctTestApiStatus status;
	string dataFile = highLevelApi->getDataFile(&status);
	moduleData.load(Env::substituteVariables(dataFile));    
	
	loadIgnoreFile();
        setup();
	
	//Make sure we have a list of modules  
	if (modules.size() == 0) {	
	    string summaryFile = highLevelApi->getXmlSummaryFile(&status);
	    XmlSummaryFile file(summaryFile);
	    
	    while (file.hasMoreRecords() && nModules < maxModules) {
		XmlSummaryRecord xsr = file.getNextRecord();
		if (xsr.serialNumber.length() == 0)
		    continue;
		if (count(ignoreModules.begin(), ignoreModules.end(), xsr.serialNumber) != 0) {
		    cout << "Ignoring module: " << xsr.serialNumber << endl;
		    continue;
		}
		modules.push_back(xsr.serialNumber);
		++nModules;
	    }
	} 
	
	//Loop over all modules and do them 
	for (list<string>::iterator it=modules.begin(); it!=modules.end(); ++it) {
	    //Only do if not ignored
	    if (count(ignoreModules.begin(), ignoreModules.end(), *it) == 0)
		doModule(*it);
	    else 
		cout << "Ignoring module: " << *it << endl;
	}
	nModules = modules.size();
	
	file->cd();	//Make sure we are in the right directory
        summaryOutput();
    } catch (Throwable& e) {
        e.sendToMrs(MRS_FATAL);
        exitCode = -1;
    }
    tearDown();
}

template<typename T>
AnalysisTestFramework<T>::AnalysisTestFramework(string isFilter, string isServer) : isFilter(isFilter), isServer(isServer) {  
    debug = false;
    maxModules = 1000;
    file = 0;
    tree = 0;
    readOnly = false;
    exitCode = 0;
    nModules = 0;
    nSkip = 0;
    nData = 1;
}


#endif //ANALYSISTESTFRAMEWORK_H
