#ifndef SCTDATA_SUMMARYMANAGER_H
#define SCTDATA_SUMMARYMANAGER_H
#include <boost/utility.hpp>
#include <fstream>
#include <string>
#include <map>
#include <boost/shared_ptr.hpp>
#include "Sct/IoExceptions.h"
#include "Sct/LogicErrors.h"

using std::ofstream;
using std::map;
using std::ostream;
using std::string;

namespace SctData {
    class TestResult;
    /**
       Classes which convert SctData::TestResult data into SctDaq style summary files.
       The output is to a output stream, which is given to the SummaryManager. This output stream could be e.g. std::cout, or an output string stream, or perhaps the output file stream created by SummaryManager::getOutputFileStream(moduleName);
       Example code snippet:
       <PRE>
       ostringstream datastream;
       shared_ptr<const SctData::TestSummary::SummaryWriter> writer =
       SctData::TestSummary::SummaryManager::instance().getWriter(myTestResult.getClassName());
       writer->writeHeader(*m_result, datastream);
       writer->write(*m_result, datastream);
       myString = datastream.str();
       </PRE>
       
    */
    namespace TestSummary{
    class SummaryWriter;
	/**
	   Singleton manager provides the public interface to writing a summary for 
	   a particular TestResult
	*/

	class SummaryManager : boost::noncopyable {
	public:
	    /** get the instance of the singleton*/ 
	    static SummaryManager& instance() throw();
	    /** 
		For use by clients.
		Returns a string containing the Summary.
	     */
	    string write(const TestResult& t) throw (Sct::IoError, Sct::LogicError);
	    
	    /**
	      For use by clients
	      Writes to a given ostream the summary.
	      */
	    void write(const TestResult& t, ostream& stream) throw (Sct::IoError, Sct::LogicError);
	    
	    /** 
		concrete SummaryWriters should add themselves to the map using this mechanism 
		@param testname the name of the TestResult for which this writer will write
		@param writer a shared pointer to the concrete SummaryWriter 
	     */
	    bool addWriter(const string& testname, boost::shared_ptr<SummaryWriter> writer) throw(Sct::LogicError);
	    /** 
		get the appropriateSummaryWriter for the TestResult
		@param classname the name of the TestResult for which the summary is to be written.
	     */
	    boost::shared_ptr<const SummaryWriter> getWriter(const string& classname) const throw(Sct::LogicError);
	private:
      	    ~SummaryManager(); ///< do NOT delete the singleton! 
	    SummaryManager(); ///< Singleton.
	    SummaryManager(ostream& out) throw(); ///< not allowed to do this - singleton
	    
	    typedef map<string, boost::shared_ptr<const SummaryWriter> > WriterMap;
	    /** 
		maps TestResult names to SummaryWriters 
	    */
	    WriterMap theMap;
	};
    }
} // end of namespace SctDaqSummary

#endif // #ifndef SCTDATA_SUMMARYMANAGER
