#ifndef WORKERGROUP_H
#define WORKERGROUP_H
#include <queue>
#include <vector>
#include <algorithm>
#include <boost/thread.hpp>

#include "Sct/LogicErrors.h"

namespace SctService{

/**
  Control part of the observer/worker pattern.
  Contains the queue to which objects of type T are pushed.
  Also useful functions to control and query the worker threads.
  The template class ``T'' is the data/instructions which the workers at to work on.
  Derive this class to implement/modify the behaviour of the worker threads.  At the least,
  the work method needs an implementation.
  T must be DefaultConstructible
  For a simple example, see the test/MyWorkerGroup.h file
*/
template <class T>
class WorkerGroup : private boost::noncopyable {
public:
    typedef T DataType;
    
    /// Start this worker group. @param the number of worker threads to be started
    void go(unsigned nWorker=1);    

    /**
       get the number of objects in the queue
       @note for information only, not to be used in thread-control.
    */
    unsigned queueSize() const;
    /**
     * get the number of workers in the map
     */
    unsigned nWorkers() const;
    /**
     * get the number of threads which are busy.
     * @note for information only, not to be used in thread-control.
     */
    unsigned busy() const;
    /**
     * Raise flag to stop all worker threads.
     */
    void stop();
    /*
     * Raise flag to pause all worker threads.
     * @param true=pause, false=unpause/play/restart 
     */
    void setPaused(bool paused=true);
    /**
     * Call this to make sure that Worker threads continue to completion.
     */
    void join();
    /**
     * get the number of paused threads
     */
    unsigned paused() const;
    /**
     * Set sleep time for all workers, if no items are on the queue.
     * @param sleep time in msec
     */
    void setSleepTime(unsigned msec);
    /**
      Get the sleep time
      */
    unsigned getSleepTime() const {return m_sleeptime;}
    /*
     * Pop an object off the front of the queue. @return T to the object
     * if the queue isn't empty, otherwise, T()
     */
    DataType pop();
    /**
     * Push an object onto the back of the queue.
     */
    void push(DataType ob);    
    
    /// destructor
    virtual ~WorkerGroup() {;} 
    
    ///Constructor
    WorkerGroup() : m_run(true), m_pause(false), m_sleeptime(1) {}

protected:
    //Inner class to hold thread specific stuff
    class Worker {
    public:
	Worker(WorkerGroup<DataType>& group) : group(group), m_busy(false), m_paused(false) {}
	void operator() () {
	    group.threadMain(*this);
	}
	WorkerGroup<DataType>& group;
	volatile bool m_busy; ///< true when actively working
	volatile bool m_paused; ///<true if paused
    };
    
    /**
      The threadMain method.  Can be overriden if sub-classes want to completly change the
      behaviour!  Current implementation checks various flags for pausing, stopping etc,
      then calls popAndWork.  If popAndWork returns false, then it calls waitForData
      */
    virtual void threadMain(Worker& worker) throw();
    /**
      If the queue length != 0, removes an element from the queue and class work
      @returns false if the queue length is 0.
      */
    virtual bool popAndWork(Worker& worker);
    /**
      Called if there is no data on the queue.  Calls wait in default implementation.
      */
    virtual void waitForData();
    /**
      To be overriden!  Should actually do something with the data!
      */
    virtual void work(DataType data) throw() = 0;
    /**
      Causes this thread to sleep for msec milliseconds.
      */
    void wait(int msec);
    /**
      threadMain should call this with its parameter.
      */
    void addWorker(Worker& worker);
    
    typedef typename std::vector<Worker*> WorkerStore;
    typedef typename WorkerStore::iterator Iterator;
    typedef typename WorkerStore::const_iterator ConstIterator;
    
    /**
     * @name Worker thread information @{ 
     * This map allows each thread to access different particular 
     * information from its own Worker.
     */
     WorkerStore m_workers;

    /**
     * @name Queue @{ the queue itself contains pointers to objects to be
     * worked on.
     */
    std::queue<DataType> m_queue;
    

    /// boost thread group from which threads can be created
    boost::thread_group m_thread_group;
    
    volatile bool m_run;  ///< if false, threads are to stop.
    volatile bool m_pause; ///< does not execute work() if true
    volatile unsigned m_sleeptime; ///< time to sleep if queue is empty	
    
    boost::mutex m_queueAccess; ///< queueLock
    boost::mutex m_vectorAccess; ///< vectorLock    
    friend class Worker;
};


template <class T>
void WorkerGroup<T>::go(unsigned nWorker){    
    // create nWorker new threads
    for (unsigned i=0; i<nWorker; ++i) {
	m_thread_group.create_thread(Worker(*this));
    }
}

template <class T>
void WorkerGroup<T>::threadMain(Worker& worker) throw() { 
    //As the worker is owned by some boost thing, we just need to add it.
    addWorker(worker);
    
    while (m_run) {                        // if !m_run, thread will finish.
	while (m_pause && m_run) {     // while paused, and not stopped, wait in sleep loop.
	    worker.m_paused = true;
	    wait(m_sleeptime);
	}
	worker.m_paused = false;
	if (!m_run) return;                     // if no longer running, stop.
	
	if (!popAndWork(worker)) {
	    waitForData();    // if queue is empty, sleep.
	}
    }
    return;
}

template<class T>
void WorkerGroup<T>::addWorker(Worker& worker) {
    boost::mutex::scoped_lock lock(m_vectorAccess);
    m_workers.push_back(&worker);
}

template <class T>
void WorkerGroup<T>::push(DataType ob){
    boost::mutex::scoped_lock lock(m_queueAccess);
    if (lock.locked()) m_queue.push(ob);
    else throw Sct::IllegalStateError("Couldn't get queue lock", __FILE__, __LINE__);
}

template <class T>
T WorkerGroup<T>::pop(){
    boost::mutex::scoped_lock lock(m_queueAccess);
    if (lock.locked()) {
	if (m_queue.size()!=0) {
	    DataType a=m_queue.front();
	    m_queue.pop();
	    return a;
	}
	return DataType();
    } else throw Sct::IllegalStateError("Couldn't get queue lock", __FILE__, __LINE__);
}

template <class W>
bool WorkerGroup<W>::popAndWork(Worker& worker){
    boost::mutex::scoped_lock lock(m_queueAccess);
    if (lock.locked()) {	
	if (m_queue.size()!=0) {
	    DataType data=m_queue.front();
	    m_queue.pop();
	    lock.unlock();  		//Don't hold onto the lock for too long!
	    worker.m_busy = true;            // set the busy flag
	    work(data);                  // call the function to do work.
	    worker.m_busy = false;           // unset the busy flag 
	    return true;
	} else return false;
    } else throw Sct::IllegalStateError("Couldn't get queue lock", __FILE__, __LINE__);
}

template <class W>    
unsigned WorkerGroup<W>::queueSize() const{
    return m_queue.size();
}

template <class W>
unsigned WorkerGroup<W>::nWorkers() const{
    return m_workers.size();
}

template <class W>
void WorkerGroup<W>::stop(){
    m_run = false;
    join();
}

template <class W>
void WorkerGroup<W>::join(){
    m_thread_group.join_all();
}

template <class W>
void WorkerGroup<W>::setPaused(bool p){
    m_pause = p;
}

template <class W>
unsigned WorkerGroup<W>::paused() const{
    unsigned npaused=0;
    for (ConstIterator it=m_workers.begin(); it!=m_workers.end(); ++it) 
	if ( (*it)->m_paused ) ++npaused;

    return npaused;
}

template <class W>
unsigned WorkerGroup<W>::busy() const{
    unsigned nbusy=0;
    for (ConstIterator it=m_workers.begin(); it!=m_workers.end(); ++it) {
	if ( (*it)->m_busy ) ++nbusy;
    }
    return nbusy;
}

template <class W>
void WorkerGroup<W>::setSleepTime(unsigned t){
    m_sleeptime = t;
}

template <class W>
void WorkerGroup<W>::wait(int wait_time) {
    boost::xtime xt;
    boost::xtime_get(&xt, boost::TIME_UTC);
    xt.nsec += wait_time*1000000;
    boost::thread::sleep(xt);
}

template <class W>
void WorkerGroup<W>::waitForData(){
    wait(getSleepTime());
}


} // end of namepace SctService
#endif //#ifndef WORKERGROUP_H
