#include "FitDisplayer.h"
#include "DisplayManager.h"
#include "DisplayInfo.h"

#include "SctData/RawScanResult.h"
#include "SctData/FitScanResult.h"
#include "SctData/FitObject.h"
#include "Sct/IS/IOManagerIS.h"
#include "Sct/Archive/IONameArchiveFile.h"
#include "Sct/Archive/IOManagerArchiveFile.h"
#include "SctData/OccupancyProjector.h"
#include "SctData/ModuleElement.h"

#include <vector>
#include <sstream>
#include <algorithm>
#include <boost/shared_ptr.hpp>

#include <TH1.h>
#include <TF1.h>
#include <TCanvas.h>

using namespace std;
using namespace SctData;
using namespace Sct;
using namespace Sct::IS;
using namespace boost;

namespace SctDataDisplay {

class FitDisplayData : public DisplayData {
public:
    shared_ptr<const RawScanResult> raw;
    vector<shared_ptr<TCanvas> > canvas;
    vector<shared_ptr<TH1> > hists;
    vector<shared_ptr<TF1> > funcs;
};    
    
bool FitDisplayer::inMap = DisplayManager::addToMap("SctData::FitScanResult", shared_ptr<Displayer>(new FitDisplayer()));

shared_ptr<DisplayData> FitDisplayer::display(shared_ptr<const Sct::Serializable> serial, const DisplayInfo& info, std::ostream& os) {  
    shared_ptr<FitDisplayData> data (new FitDisplayData());
    const FitScanResult& fit = dynamic_cast<const FitScanResult&>(*serial);
    os << fit.getHeader().getScanNumber() << endl;
    
    try {
	string rawName = RawScanResult::getUniqueID(fit.getHeader());
	shared_ptr<Serializable> ob = IOManagerIS::instance().read("EventData.SctData::RawScanResult."+ rawName);
	data->raw = dynamic_pointer_cast<const RawScanResult>(ob);
    } catch (Sct::IoException& e) {
	e.sendToMrs(MRS_DIAGNOSTIC);
    }
    if (data->raw==0) try {
      Sct::Archive::IONameArchiveFile name(fit.getHeader().getUniqueID(),"SctData::RawScanResult");
      shared_ptr<Serializable> ob = Sct::Archive::IOManagerArchiveFile::instance().read(name.getIOName());
      data->raw = dynamic_pointer_cast<const RawScanResult>(ob);
    } catch (Sct::IoException& e) {
	e.sendToMrs(MRS_DIAGNOSTIC);
    }

    if (data->raw == 0)
        cerr << "Couldn't get RawScanResult associated with this FitScanResult" << endl;    
    
    if (info.displayChips.size() > 0)
	displayChips(fit, info, *data, os);
    
    if (info.displayChannels.size() > 0)
	displayChannels(fit, info, *data, os);
    
    //Now print out defects:
    os << endl << "Defects: " << endl;
    printDefectList(fit.getDefects(), os);   
    
    return data;
}

void FitDisplayer::displayChips(const FitScanResult& fit, const DisplayInfo& info, FitDisplayData& data, std::ostream& os) {
    shared_ptr<TCanvas> c = createCanvas("chipFits", "Chip Fits for module " + fit.getHeader().getModuleName());
    data.canvas.push_back(c);
    divideCanvas(info.displayChips.size(), *c);
 
    auto_ptr<OccupancyProjector> occ;
    if (data.raw) {
	occ.reset(new OccupancyProjector(*data.raw));
	occ->vetoSeriousDefects(fit.getDefects());
    }
	
    for (unsigned int i=0; i<info.displayChips.size(); ++i) {
        c->cd(i+1);
        ostringstream histName;
	histName << "chipHist" << info.displayChips[i];
	
        auto_ptr<TH1> h;
	if (data.raw) {
	    auto_ptr<TH1> htemp = occ->getOccupancy(histName.str().c_str(), ModuleElement::Chip(info.displayChips[i]));
            h = htemp;
	}
        FitObject& fo = fit.getChipFit(info.displayChips[i]);

	displayFit(fo, h, data, os);
	fo.print();
    }
    
}


void FitDisplayer::displayChannels(const FitScanResult& fit, const DisplayInfo& info, FitDisplayData& data, std::ostream& os) {
    shared_ptr<TCanvas> c;
    unsigned int nMax = min((unsigned int)25, info.displayChannels.size());
 
    auto_ptr<OccupancyProjector> occ;
    if (data.raw) 
	occ.reset(new OccupancyProjector(*data.raw));

    int j=0;
        
    for (unsigned int i=0; i<info.displayChannels.size(); ++i) {
        if (j%nMax == 0) {
	    c = createCanvas("channelFits", "Channel Fits for module " + fit.getHeader().getModuleName());
	    data.canvas.push_back(c);
	    divideCanvas(nMax, *c);
            j=0;
        }
        c->cd(j+1);

        ostringstream histName;
	histName << "channelHist" << info.displayChannels[i];
	
        auto_ptr<TH1> h;
	if (data.raw) {
	    auto_ptr<TH1> htemp = occ->getOccupancy(histName.str().c_str(), ModuleElement::Channel(info.displayChannels[i]));
	    h = htemp;
	}
	FitObject& fo = fit.getChannelFit(info.displayChannels[i]);
	fo.print();
	displayFit(fo, h, data, os);
	++j;
    }
}

void FitDisplayer::displayFit(FitObject& fo, auto_ptr<TH1> h, FitDisplayData& data, std::ostream& os) {
    if (h.get()) h->Draw();
    
    auto_ptr<TF1> f ( fo.makeRootTF1() );
    f->SetLineColor(2);
    f->SetLineWidth(1);
    f->SetFillStyle(0);
    if (h.get()) f->Draw("same");
    else f->Draw();
    
    for (unsigned int i=0; i<fo.getNPar(); ++i) {
	os << fo.getParameter(i) << "  " << f->GetParameter(i) << "  ";
    }
    os << endl;
	
    data.hists.push_back(shared_ptr<TH1>(h));
    data.funcs.push_back(shared_ptr<TF1>(f));

}

}
