#include <unistd.h>
#include <stdlib.h>
#include <iostream>

#include <mrs/message.h>
#include <is/isinfotmpl.h>

#include <algorithm>
#include <sstream>
#include <memory>

#include "Sct_SctApi.hh"
#include "SctApiServer.h"
#include "sctConf/configipc.h"
#include "SctApi/autoConfig.h"

#include "Sct/ILUMarshalling.h"
#include "sctConf/marshalling.h"
#include "marshalling.h"

#include "SctApi/SctApi.h"
#include "SctApi/SctApiImpl.h"

#include <boost/date_time/posix_time/posix_time.hpp>

using namespace Sct;
using namespace std;
using namespace boost::posix_time;

static void myRaise(Sct_SctApiStatus *_status, SctApi::SctApiException &s);

static inline SctApi::BankType myTranslateBank(Sct_SctApi_T_BankType bank);

/*
  ilu_Cardinal is long unsigned int
 */

SctApiServer *SctApiServer::singleton = 0;

SctApiServer::SctApiServer(SctApi::SctApi& api, const char* oid, IPCServer* server) : 
  IPCObject(oid, server), scans(0), 
  mrsStream(server->partition()), 
  isDict(server->partition()),
  ddcSingleTimeout(3), ddcMultipleTimeout(15), 
  api(api)
{
  publish();

  ISInfoInt isPresent(1);

  defaultScan = new ScanServer(ipcServer.get());
  api.setMrsStream(&mrsStream);
  api.setIsDictionary(&isDict);

  mrsStream << "SCTAPI_CONSTRUCT" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("Constructor called") << ENDM;

  if(isDict.contains("SCTAPIServer.present")) {
    std::cout << "*** Filename already in IS, will run anyway but there could be another service running?\n";
  } else {
    isDict.insert("SCTAPIServer.present", isPresent);
  }

  singleton = this;
}

SctApiServer::~SctApiServer() 
{
  mrsStream << "SCTAPI_DESTRUCT" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("Destructor called") << ENDM;

  isDict.remove("SCTAPIServer.present");

  singleton = 0;

  withdraw();

  std::cout << "Forcefully deleting " << scans.size() << " scans\n";

  for(std::list<Sct_SctApi_T_Scan *>::iterator i = scans.begin();
      i!= scans.end(); 
      i++) {
    delete *i;
  }

}

void SctApiServer::initialiseAll (Sct_SctApiStatus *_status, ilu_Cardinal runNumber)
{
  try {
    mrsStream << "SCTAPI_INIT" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("initialiseAll called") << ENDM;

    worker().initialiseAll(runNumber);

    ISInfoInt present(2);

    isDict.update("SCTAPIServer.present", present);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by initialiseAll: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::shutdownAll (Sct_SctApiStatus *_status)
{
  try {
    mrsStream << "SCTAPI_SHUTDOWN" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("shutdownAll called") << ENDM;

    worker().shutdownAll();

    ISInfoInt present(1);

    isDict.update("SCTAPIServer.present", present);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by shutdownAll: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::status (Sct_SctApiStatus *_status)
{
  worker().status();
}

void SctApiServer::awaitResponse (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  try {
    worker().awaitResponse((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, 5);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by awaitResponse: " << s.what() << endl;
    myRaise(_status, s);
  }
}

Sct_SctApi_T_DataBlock SctApiServer::getResponse (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  unsigned long length;

  unsigned long *body = worker().getResponse((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, length);

  return copyArrayToILU<_Sct_SctApi_T_DataBlock_sequence, unsigned long>(body, length);
}

void SctApiServer::configureBOC (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  try {
    worker().configureBOC((unsigned int) partition, (unsigned int)crate, (unsigned int)rod);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by configureBOC: " << s.what() << endl;
    myRaise(_status, s);
  }
}

Sct_SctApi_T_BOCMonitorSequence SctApiServer::getBOCMonitorArray (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  try {
    return copyVectorToILU<_Sct_SctApi_T_BOCMonitorSequence_sequence, double>(worker().getBOCMonitorArray((unsigned int) partition, crate, rod));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by getBOCMonitorArray: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::flashLED (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_ShortCardinal slaveNumber, ilu_Cardinal period, ilu_Cardinal flashes)
{
  try {
    worker().flashLED((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, slaveNumber, period, flashes);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by flashLED: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::echo (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, Sct_SctApi_T_DataBlock data)
{
  try {
    unsigned long ourLength = data->Length();

    unsigned long *array = new unsigned long[ourLength];
    unsigned long *body = copyILUToArray<Sct_SctApi_T_DataBlock, unsigned long>(data, array, ourLength);

    worker().echo((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, (unsigned int)ourLength, body);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by echo: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::createDebugPrimList (Sct_SctApiStatus *_status)
{
  try {
    worker().createDebugPrimList();
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by createDebugPrimList: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::addDebugPrimList (Sct_SctApiStatus *_status, ilu_Integer index, ilu_Integer id, ilu_Integer version, Sct_SctApi_T_DataBlock body)
{
  try {
    unsigned long ourLength = body->Length();
    unsigned long *array = new unsigned long[ourLength];
    copyILUToArray(body, array, ourLength);

    worker().addDebugPrimList(ourLength, index, id, version, array);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by addDebugPrimList: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::sendDebugPrimList (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  try {
    worker().sendDebugPrimList((unsigned int)partition, (unsigned int)crate, (unsigned int)rod);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by sendDebugPrimList: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::sendDebugSlavePrimList (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal slave, ilu_Boolean await, ilu_Boolean response)
{
  try {
    worker().sendDebugSlavePrimList((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, slave, await, response);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by sendDebugSlavePrimList: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::setRunNumber (Sct_SctApiStatus *_status, ilu_Cardinal newRun)
{
  worker().setRunNumber(newRun);
}

void SctApiServer::setScanNumber (Sct_SctApiStatus *_status, ilu_Cardinal newScan)
{
  worker().setScanNumber(newScan);
}

ilu_Cardinal SctApiServer::getRunNumber (Sct_SctApiStatus *_status)
{
  return worker().getRunNumber();
}

ilu_Cardinal SctApiServer::getScanNumber (Sct_SctApiStatus *_status)
{
  return worker().getScanNumber();
}

void SctApiServer::unsetDebugOption (Sct_SctApiStatus *_status, ilu_T_CString opt)
{
  worker().unsetDebugOption(opt);
}

void SctApiServer::setDebugOption (Sct_SctApiStatus *_status, ilu_T_CString opt)
{
  worker().setDebugOption(opt);
}

Sct_SctApi_T_DefectList SctApiServer::listEnabledDebugOptions (Sct_SctApiStatus *_status)
{
  Sct_SctApi_T_DefectList result;

  result = _Sct_SctApi_T_DefectList_sequence::Create(4, NULL);
  result->Clear(0);

  std::list<std::string> myList = worker().listEnabledDebugOptions();
  for(std::list<std::string> :: const_iterator m = myList.begin();
      m!=myList.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

Sct_SctApi_T_DefectList SctApiServer::listDebugOptions (Sct_SctApiStatus *_status)
{
  Sct_SctApi_T_DefectList result;

  result = _Sct_SctApi_T_DefectList_sequence::Create(4, NULL);
  result->Clear(0);

  std::vector<std::string> myList = worker().listDebugOptions();
  for(std::vector<std::string> :: const_iterator m = myList.begin();
      m!=myList.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

void SctApiServer::standardRegisterDump (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  worker().standardRegisterDump(partition, crate, rod);
}

void SctApiServer::standardRegisterDumpAll (Sct_SctApiStatus *_status)
{
  worker().standardRegisterDumpAll();
}

sctConf_T_ModuleList SctApiServer::getModuleList (Sct_SctApiStatus *_status) 
{
  sctConf_T_ModuleList result;

  result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
  result->Clear(0);

  std::list<std::string> myList = worker().getModuleList();
  for(std::list<std::string> :: const_iterator m = myList.begin();
      m!=myList.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

ilu_Boolean SctApiServer::isRODPresent (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  return worker().isRODPresent(partition, crate, rod);
}

void SctApiServer::loadConfiguration (Sct_SctApiStatus *_status, ilu_T_CString filename)
{
  try {
    worker().loadConfiguration(filename);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by loadConfiguration: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::storeModuleConfigurations (Sct_SctApiStatus *_status)
{
  try {
    worker().storeModuleConfigurations();
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by storeModuleConfigurations: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::configureAllModules (Sct_SctApiStatus *_status)
{
  try {
    worker().sendAllABCDModules(myTranslateBank(Sct_SctApi_T_BankType_PHYSICS_CONFIG));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by configureAllModules: " << s.what() << endl;
    myRaise(_status, s);
  }
}

sctConf_T_ABCDModule* SctApiServer::retrieveModule (Sct_SctApiStatus *_status, ilu_Cardinal mid)
{
  ABCDModule *body = worker().retrieveModule(mid);

  return copyABCDModuleToILU(*body);
}

ilu_ShortInteger SctApiServer::dspBlockDump (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                             ilu_Integer dspStart, ilu_Integer numWords, ilu_Integer dspNumber)
{
  return worker().dspBlockDump((unsigned int)partition, (unsigned int)crate, (unsigned int)rod,
                             dspStart, numWords, dspNumber);
}

Sct_SctApi_T_DataBlock SctApiServer::dspBlockRead (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Integer dspStart, ilu_Integer numWords, ilu_Integer dspNumber)
{
  unsigned long length;
  unsigned long *body = worker().dspBlockRead((unsigned int)partition, (unsigned int)crate, (unsigned int)rod,
                                            dspStart, numWords, dspNumber, length);

  return copyArrayToILU<_Sct_SctApi_T_DataBlock_sequence, unsigned long>(body, length);
}

ilu_ShortInteger SctApiServer::dspBlockWrite (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod,
                                              Sct_SctApi_T_DataBlock buffer, ilu_Cardinal dspAddress, 
                                              ilu_Integer dspNumber)
{
  unsigned long length = buffer->Length();
  unsigned long *array = new unsigned long[length];
  copyILUToArray<Sct_SctApi_T_DataBlock, unsigned long>(buffer, array, length);

  return worker().dspBlockWrite((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, 
                                array, dspAddress, length, dspNumber);
}

ilu_Cardinal SctApiServer::dspSingleRead (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                          ilu_Cardinal dspAddr, ilu_Integer dspNumber)
{
  return worker().dspSingleRead((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, 
                              dspAddr, dspNumber);
}

void SctApiServer::dspSingleWrite (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal dspAddr, ilu_Cardinal val, ilu_Integer dspNumber)
{
  worker().dspSingleWrite((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, dspAddr, val, dspNumber);
}

Sct_SctApi_T_DataBlock SctApiServer::readSlaveDsp (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_ShortInteger s, ilu_Cardinal add, ilu_Cardinal words)
{
  unsigned long *body = worker().primReadSlaveDsp((unsigned int)partition, (unsigned int)crate, (unsigned int)rod, s, add, words);

  return copyArrayToILU<_Sct_SctApi_T_DataBlock_sequence, unsigned long>(body, words);
}

ilu_Cardinal SctApiServer::findModuleSerial (Sct_SctApiStatus *_status, ilu_T_CString sn)
{
  return worker().findModule(sn);
}

ilu_T_CString SctApiServer::convertToString (Sct_SctApiStatus *_status, ilu_Cardinal mid)
{
  return copyStringToILU(worker().convertToString(mid));
}

ilu_Cardinal SctApiServer::findModule (Sct_SctApiStatus *_status, ilu_ShortCardinal mur, ilu_ShortCardinal number)
{
  return worker().findModule(mur, number);
}

ilu_Cardinal SctApiServer::findModuleRod (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal channel) 
{
  UINT32 mid = 0;
  mid |= (partition & 0xff) << 24;
  mid |= (crate & 0xff) << 16;
  mid |= (rod & 0xff) << 8;
  mid |= (channel & 0xff) << 0;
  return mid;
}

ilu_Cardinal SctApiServer::findBarrelModule (Sct_SctApiStatus *_status, 
                                             ilu_Cardinal barrel, ilu_Cardinal row, ilu_ShortInteger number)
{
  return worker().findBarrelModule(barrel, row, number);
}

void SctApiServer::convertToBarrelModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Cardinal * barrel, ilu_Cardinal * row, ilu_ShortInteger * number)
{
  int rnumber;
  worker().convertToBarrelModule((unsigned int)mid, *(unsigned int*)barrel, *(unsigned int*)row, rnumber);

  *number = rnumber;
}

ilu_Cardinal SctApiServer::findEndcapModule (Sct_SctApiStatus *_status, ilu_Integer disk, ilu_Integer ring, ilu_Integer number)
{
  return worker().findEndcapModule(disk, ring, number);
}

void SctApiServer::convertToEndcapModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Integer * disk, ilu_Cardinal * ring, ilu_Cardinal * number)
{
  worker().convertToEndcapModule((unsigned int)mid, *(int*)disk, *(unsigned int*)ring, *(unsigned int*)number);
}

void SctApiServer::getABCDModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_BankType bank)
{
  try {
    worker().getABCDModule(mid, myTranslateBank(bank));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by getABCDModule: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::getABCDModules (Sct_SctApiStatus *_status, Sct_SctApi_T_BankType bank)
{
  try {
    worker().getABCDModules(myTranslateBank(bank));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by getABCDModules: " << s.what() << endl;
    myRaise(_status, s);
  }
}

sctConf_T_ABCDModule * SctApiServer::getABCDModuleRaw (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                                       ilu_Cardinal slot, Sct_SctApi_T_BankType bank)
{
  try {
    boost::shared_ptr<ABCDModule> body = worker().getABCDModuleRaw(partition, crate, rod, slot, myTranslateBank(bank));
    return copyABCDModuleToILU(*(body.get()));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by getABCDModules: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::setABCDModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_BankType bank)
{
  try {
    worker().setABCDModule(mid, myTranslateBank(bank));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by setABCDModule: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::setABCDModules (Sct_SctApiStatus *_status, Sct_SctApi_T_BankType bank)
{
  try {
    worker().setABCDModules(myTranslateBank(bank));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by setABCDModules: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::sendABCDModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_BankType bank, ilu_Cardinal type)
{
  try {
    worker().sendABCDModule(mid, myTranslateBank(bank), SctApi::ConfigType(type));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by sendABCDModule: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::sendABCDModules (Sct_SctApiStatus *_status, Sct_SctApi_T_BankType bank)
{
  try {
    worker().sendAllABCDModules(myTranslateBank(bank));
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by sendABCDModules: " << s.what() << endl;
    myRaise(_status, s);
  }
}

void SctApiServer::modifyABCDMask (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_DataBlock mask)
{
  UINT32 array[48];
  worker().modifyABCDMask(mid, copyILUToArray<Sct_SctApi_T_DataBlock, UINT32> (mask, array, 48));
}

void SctApiServer::modifyABCDTrims (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_CharBlock trims){
  UINT8 array[12*128];
  worker().modifyABCDTrims (mid, copyILUToArray<Sct_SctApi_T_CharBlock, UINT8> (trims, array, 12*128) );
} 

void SctApiServer::modifyABCDRC (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Cardinal chip, ilu_ShortCardinal index, ilu_Real p0, ilu_Real p1, ilu_Real p2)
{
  cout << "SctApiServer::modifyABCDRC called" << endl;
  worker().modifyABCDRC(mid, chip, index, p0, p1, p2);
}

void SctApiServer::modifyABCDVar (Sct_SctApiStatus *_status, ilu_Cardinal typ, ilu_Real var)
{
  worker().modifyABCDVar(typ, var);
}

void SctApiServer::modifyABCDModuleVar (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Cardinal typ, ilu_Real var)
{
  worker().modifyABCDVar(mid, typ, var);
}

void SctApiServer::modifyABCDChipVar (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Cardinal chip, ilu_Cardinal typ, ilu_Real var)
{
  worker().modifyABCDVar(mid, chip, typ, var);
}

void SctApiServer::modifyABCDVarROD (Sct_SctApiStatus *_status, ilu_Cardinal typ, ilu_Real var, Sct_SctApi_T_BankType bank)
{
  worker().modifyABCDVarROD(typ, var, myTranslateBank(bank));
}

void SctApiServer::modifyABCDModuleVarROD (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Cardinal typ, ilu_Real var, Sct_SctApi_T_BankType bank)
{
  worker().modifyABCDVarROD(mid, typ, var, myTranslateBank(bank));
}

void SctApiServer::modifyBOCParam (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, 
                                   ilu_Cardinal channel, ilu_Cardinal type, ilu_Cardinal val)
{
  worker().modifyBOCParam(partition, crate, rod, channel, type, val);
}

void SctApiServer::lasersOff (Sct_SctApiStatus *_status)
{
  worker().lasersOff();
}

void SctApiServer::timSetFrequency (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, 
                                    ilu_Real trigFreq, ilu_Real rstFreq)
{
  worker().timSetFrequency(partition, crate, trigFreq, rstFreq);
}

void SctApiServer::freeTriggers (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate)
{
  worker().freeTriggers(partition, crate);
}

void SctApiServer::stopTriggers (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate)
{
  worker().stopTriggers(partition, crate);
}

void SctApiServer::timL1A (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate)
{
  worker().timL1A(partition, crate);
}

void SctApiServer::timCalL1A (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_ShortInteger delay)
{
  worker().timCalL1A(partition, crate, delay);
}

ilu_ShortInteger SctApiServer::timReadRegister (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_ShortInteger value) 
{
  return worker().timReadRegister(partition, crate, value);
}

void SctApiServer::timWriteRegister (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_ShortInteger reg, ilu_ShortInteger value)
{
  worker().timWriteRegister(partition, crate, reg, value);
}

void SctApiServer::sendTimTrigger (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, class Sct_SctApi_T_Trigger * trig) 
{
  TriggerServer *realTrigger = dynamic_cast<TriggerServer*>(trig);

  IPCObjectVar<TriggerServer>::duplicate(realTrigger);
  worker().sendTimTrigger(partition, crate, realTrigger->m_worker.get());
  realTrigger->decRef();
}

void SctApiServer::sendTrigger (Sct_SctApiStatus *_status, 
                                ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, class Sct_SctApi_T_Trigger * trig)
{
  TriggerServer *realTrigger = dynamic_cast<TriggerServer*>(trig);

  IPCObjectVar<TriggerServer>::duplicate(realTrigger);
  worker().sendTrigger(partition, crate, rod, realTrigger->m_worker.get());
  realTrigger->decRef();
}

void SctApiServer::doScan (Sct_SctApiStatus *_status, class Sct_SctApi_T_Scan * thisScan)
{
  mrsStream << "SCTAPI_SCAN" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("Starting scan") << ENDM;

  ISInfoInt scanning(1);
  isDict.insert("SCTAPIServer.scanning", scanning);

  ScanServer *serverScan = dynamic_cast<ScanServer*>(thisScan);

  IPCObjectVar<ScanServer>::duplicate(serverScan);

  try {
    worker().doScan(serverScan->m_worker);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by doScan: " << s.what() << endl;

//     serverScan->decRef();
    isDict.remove("SCTAPIServer.scanning");

    mrsStream << "SCAN_ABORT" << MRS_ERROR << MRS_QUALIF("SctApiServer") << MRS_TEXT((string("Scan aborted: ") + s.what()).c_str()) << ENDM;
    myRaise(_status, s);
  }

  // Don't decrease reference, so polling thread can keep it
//   realScan->decRef();
}

void SctApiServer::doRawScan (Sct_SctApiStatus *_status, class Sct_SctApi_T_Scan * thisScan, ilu_Integer delay, ilu_Integer width, ilu_Boolean configureModules, ilu_Boolean clockByTwo)
{
  mrsStream << "SCTAPI_RAWSCAN" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("Raw scan started") << ENDM;

  ISInfoInt scanning(1);
  isDict.insert("SCTAPIServer.scanning", scanning);

  ScanServer *realScan = dynamic_cast<ScanServer*>(thisScan);

  IPCObjectVar<ScanServer>::duplicate(realScan);
  try {
    worker().doRawScan(realScan->m_worker, delay, width, configureModules, clockByTwo);
  } catch(SctApi::SctApiException &s) {
    cout << "Exception thrown by doRawScan: " << s.what() << endl;

    isDict.remove("SCTAPIServer.scanning");

    mrsStream << "SCAN_RAW_ABORT" << MRS_ERROR << MRS_QUALIF("SctApiServer") << MRS_TEXT((string("Raw scan aborted: ") + s.what()).c_str()) << ENDM;
    myRaise(_status, s);
  }

  realScan->decRef();

  isDict.remove("SCTAPIServer.scanning");

  mrsStream << "SCTAPI_RAWSCAN" << MRS_INFORMATION << MRS_QUALIF("SctApiServer") << MRS_TEXT("Raw scan completed") << ENDM;
}

void SctApiServer::printBOCSetup (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  worker().printBOCSetup(partition, crate, rod);
}

Sct_SctApi_T_BOCChannelList SctApiServer::currentBOCSetup (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod) 
{
  return copyBOCChannelConfigVectorToILU(worker().currentBOCSetup(partition, crate, rod));
}

sctConf_T_BOCGlobalConfig * SctApiServer::currentBOCRegisters (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod) 
{
  return copyBOCGlobalConfigToILU(worker().currentBOCRegisters(partition, crate, rod));
}

void SctApiServer::printABCDModule (Sct_SctApiStatus *_status, ilu_Cardinal mid)
{
  worker().printABCDModule(mid);
}

void SctApiServer::printABCDRodModule (Sct_SctApiStatus *_status, ilu_Cardinal mid, Sct_SctApi_T_BankType bank)
{
  worker().printABCDRodModule(mid, myTranslateBank(bank));
}

void SctApiServer::decodeEvent (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_ShortInteger sl, ilu_ShortInteger index, ilu_Boolean extFlag, ilu_Boolean errorType)
{
  worker().decodeEvent(partition, crate, rod, sl, index, extFlag, errorType);
}

void SctApiServer::rawData (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Integer delay, ilu_Integer units, ilu_Boolean setMask)
{
  worker().rawData(partition, crate, rod, delay, units, setMask);
}

void SctApiServer::rawDataWithTrigger (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Integer delay, ilu_Integer units, ilu_Boolean setMask, class Sct_SctApi_T_Trigger * trig)
{
  TriggerServer *realTrigger = dynamic_cast<TriggerServer*>(trig);

  IPCObjectVar<TriggerServer>::duplicate(realTrigger);
  worker().rawData(partition, crate, rod, delay, units, setMask, realTrigger->m_worker.get());
  realTrigger->decRef();
}

void SctApiServer::sendL1A (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  worker().sendL1A(partition, crate, rod);
}

Sct_SctApi_T_CharBlock SctApiServer::probe (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  return copyVectorToILU<_Sct_SctApi_T_CharBlock_sequence, char>(worker().probe(partition, crate, rod));
}

Sct_SctApi_T_CharBlock SctApiServer::probeHarness (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal harness)
{
  return copyVectorToILU<_Sct_SctApi_T_CharBlock_sequence, char>(worker().probe(partition, crate, rod, harness));
}

Sct_SctApi_T_CharBlock SctApiServer::probeWithTrigger (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, class Sct_SctApi_T_Trigger * trig)
{
  TriggerServer *realTrigger = dynamic_cast<TriggerServer*>(trig);

  IPCObjectVar<TriggerServer>::duplicate(realTrigger);
  vector<char> ret = worker().probeWithTrigger(partition, crate, rod, realTrigger->m_worker.get());
  realTrigger->decRef();
  return copyVectorToILU<_Sct_SctApi_T_CharBlock_sequence, char>(ret);
}

Sct_SctApi_T_CharBlock SctApiServer::probeHarnessWithTrigger (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, class Sct_SctApi_T_Trigger * trig, ilu_Cardinal harness)
{
  TriggerServer *realTrigger = dynamic_cast<TriggerServer*>(trig);

  IPCObjectVar<TriggerServer>::duplicate(realTrigger);
  vector<char> ret = worker().probeWithTrigger(partition, crate, rod, realTrigger->m_worker.get(), harness);
  realTrigger->decRef();
  return copyVectorToILU<_Sct_SctApi_T_CharBlock_sequence, char>(ret);
}

Sct_SctApi_T_CharMatrix SctApiServer::probeScanHarnessWithTrigger (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, class Sct_SctApi_T_Scan * scan, ilu_Cardinal harness)
{
  ScanServer *realScan = dynamic_cast<ScanServer*>(scan);

  IPCObjectVar<ScanServer>::duplicate(realScan);
  vector<vector<char> > ret = worker().probeScan(partition, crate, rod, realScan->m_worker, harness);
  realScan->decRef();
  return copyVectorVectorToILU<_Sct_SctApi_T_CharMatrix_sequence, _Sct_SctApi_T_CharBlock_sequence, char>(ret);
}

ilu_Boolean SctApiServer::checkAllModulesProbe (Sct_SctApiStatus *_status, ilu_T_CString values)
{
  return worker().checkAllModulesProbe(values);
}

Sct_SctApi_T_AutoResultList SctApiServer::autoConfigureScan (Sct_SctApiStatus *_status)
{
  ::SctApi::AutoConf::AutoConfigurer configurer(worker());
  configurer.run();

  return copyAutoResultListToILU(configurer.getResults()); // worker().autoConfigureScan();
}

ilu_Cardinal SctApiServer::getRodSlot (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  return worker().getRodSlot(partition, crate, rod);
}

void SctApiServer::testLinkOutSelect (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod, ilu_Cardinal link)
{
  worker().testLinkOutSelect(partition, crate, rod, link);
}

ilu_Cardinal SctApiServer::testLinkOutRetrieve (Sct_SctApiStatus *_status, ilu_Cardinal partition, ilu_Cardinal crate, ilu_Cardinal rod)
{
  return worker().testLinkOutRetrieve(partition, crate, rod);
}

void SctApiServer::requestHardReset (Sct_SctApiStatus *_status, ilu_Cardinal mid)
{
  worker().requestHardReset(mid);
}

void SctApiServer::requestIVCurve (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_ShortReal start, ilu_ShortReal stop, ilu_ShortReal step, ilu_ShortCardinal delay, ilu_ShortReal currentLim) 
{
  worker().requestIVCurve(mid, start, stop, step, delay, currentLim);
}

void SctApiServer::setSelect (Sct_SctApiStatus *_status, ilu_Cardinal mid, ilu_Boolean value)
{
  worker().setSelect(mid, value);
}

void SctApiServer::setSelectAllInCrate (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Boolean value)
{
  worker().setSelectAllInCrate(crate, (bool)value, ddcMultipleTimeout);
}

void SctApiServer::changeRunState (Sct_SctApiStatus *_status, ilu_Integer state)
{
  worker().changeRunState(state);
}

void SctApiServer::ddcSetSingleTimeout (Sct_SctApiStatus *_status, ilu_ShortInteger timeout)
{
  ddcSingleTimeout = timeout;
}

ilu_ShortInteger SctApiServer::ddcGetSingleTimeout (Sct_SctApiStatus *_status) 
{
  return ddcSingleTimeout;
}

void SctApiServer::ddcSetMultipleTimeout (Sct_SctApiStatus *_status, ilu_ShortInteger timeout)
{
  ddcMultipleTimeout = timeout;
}

ilu_ShortInteger SctApiServer::ddcGetMultipleTimeout (Sct_SctApiStatus *_status)
{
  return ddcMultipleTimeout;
}

void SctApiServer::ddcChangeState (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer channel, ilu_ShortInteger state)
{
  worker().ddcChangeState(crate, channel, state, ddcMultipleTimeout);
}

void SctApiServer::ddcChangeStateAll (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_ShortInteger state)
{
  worker().ddcChangeStateAll(crate, state, ddcMultipleTimeout);
}

void SctApiServer::ddcLoadConfiguration (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_ShortInteger state)
{
  worker().ddcLoadConfiguration(crate, state, ddcMultipleTimeout);
}

void SctApiServer::ddcHardReset (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer channel)
{
  worker().ddcHardReset(crate, channel, ddcSingleTimeout);
}

void SctApiServer::ddcHardResetAll (Sct_SctApiStatus *_status, ilu_Integer crate)
{
  worker().ddcHardResetAll(crate, ddcMultipleTimeout);
}

ilu_Cardinal SctApiServer::ddcGetChannelParameterInt (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer channel, ilu_T_CString name)
{
  return worker().ddcGetChannelParameterInt(crate, channel, string(name), ddcSingleTimeout);
}

ilu_ShortReal SctApiServer::ddcGetChannelParameterFloat (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer channel, ilu_T_CString name)
{
  return worker().ddcGetChannelParameterFloat(crate, channel, string(name), ddcSingleTimeout);
}

ilu_T_CString SctApiServer::ddcGetChannelParameterString (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer channel, ilu_T_CString name)
{
  return copyStringToILU(worker().ddcGetChannelParameterString(crate, channel, string(name), ddcSingleTimeout));
}

ilu_Cardinal SctApiServer::ddcGetCardParameterInt (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer card, ilu_T_CString name)
{
  return worker().ddcGetCardParameterInt(crate, card, string(name), ddcSingleTimeout);
}

ilu_ShortReal SctApiServer::ddcGetCardParameterFloat (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer card, ilu_T_CString name)
{
  return worker().ddcGetCardParameterFloat(crate, card, string(name), ddcSingleTimeout);
}

ilu_T_CString SctApiServer::ddcGetCardParameterString (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_Integer card, ilu_T_CString name)
{
  return copyStringToILU(worker().ddcGetCardParameterString(crate, card, string(name), ddcSingleTimeout));
}

ilu_Cardinal SctApiServer::ddcGetCrateParameterInt (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_T_CString name)
{
  return worker().ddcGetCrateParameterInt(crate, string(name), ddcSingleTimeout);
}

ilu_ShortReal SctApiServer::ddcGetCrateParameterFloat (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_T_CString name)
{
  return worker().ddcGetCrateParameterFloat(crate, string(name), ddcSingleTimeout);
}

ilu_T_CString SctApiServer::ddcGetCrateParameterString (Sct_SctApiStatus *_status, ilu_Integer crate, ilu_T_CString name)
{
  return copyStringToILU(worker().ddcGetCrateParameterString(crate, string(name), ddcSingleTimeout));
}

void SctApiServer::ddcGetCrateParameters (Sct_SctApiStatus *_status, ilu_Integer crate, Sct_SctApi_T_NameList names, Sct_SctApi_T_NameValueList nameValues) 
{
  std::vector<std::string> myNames = copyILUToVector<Sct_SctApi_T_NameList, std::string> (names);

  std::map<std::string, float> myNameValues;

  worker().ddcGetCrateParameters(crate, myNames, myNameValues, ddcMultipleTimeout);

  nameValues = copyMapToNameValueList(myNameValues);
}

Sct_SctApi_T_Scan * SctApiServer::createScan (Sct_SctApiStatus *_status)
{
  std::cout << "Asked to create a scan\n";

  ScanServer *defScan = new ScanServer(ipcServer.get()); // , *defaultScan);

  scans.push_back(defScan);

  return IPCObjectVar<ScanServer>::duplicate(defScan);
}

Sct_SctApi_T_Scan * SctApiServer::getDefaultScan (Sct_SctApiStatus *_status)
{
  std::cout << "Asked for a reference to the default scan\n";

  return IPCObjectVar<ScanServer>::duplicate(defaultScan);
}

Sct_SctApi_T_Trigger * SctApiServer::createTrigger (Sct_SctApiStatus *_status)
{
  std::cout << "Asked to create a trigger\n";

  TriggerServer *defTrig = new TriggerServer(ipcServer.get());

  return IPCObjectVar<TriggerServer>::duplicate(defTrig);
}

Sct_SctApi_T_Trigger * SctApiServer::getDefaultTrigger (Sct_SctApiStatus *_status)
{
  std::cout << "Asked for the default trigger\n";

  return IPCObjectVar<TriggerServer>::duplicate(defaultTrigger);
}

Sct_SctApi_T_ScanList SctApiServer::knownScans (Sct_SctApiStatus *_status)
{
  std::cout << "In known scans\n";

  Sct_SctApi_T_ScanList myScans;

  myScans = new _Sct_SctApi_T_ScanList_sequence(10);

  std::cout << "Copying sequence of length " << scans.size() << std::endl;

  for(std::list<Sct_SctApi_T_Scan *>::iterator i=scans.begin();
      i != scans.end();
      i++) {
    ScanServer *thisScan = IPCObjectVar<ScanServer>::duplicate(dynamic_cast<ScanServer *>(*i));
    myScans->Append(thisScan);
  }

  return myScans;
}

SctApi::SctApi &SctApiServer::worker() 
{
  return api;
}

/**************** ScanServer ****************/

ScanServer::ScanServer(IPCServer* server) : IPCObject(getServerName().c_str(), server), m_worker(new SctApi::ScanDefImpl())
{
  std::cout << "Create a new ScanServer\n";

//    dynamic_cast<::Scan*>(this)->print();
//   std::cout << "trigsPerBurst = " << worker().getNTrigs() << std::endl; 

  TriggerServer *temp = new TriggerServer(ipcServer.get());
  std::cout << "TrigServer: " << temp;
  worker().setTrigger1(IPCObjectVar<TriggerServer>::duplicate(temp)->m_worker);
  std::cout << " getTrigger1() " << worker().getTrigger1() << std::endl;

  TriggerServer *temp2 = new TriggerServer(ipcServer.get());
  worker().setTrigger2(IPCObjectVar<TriggerServer>::duplicate(temp2)->m_worker);

//    if(!publish()) {
//      cerr << "Failed to publish new scan" << std::endl;
//    }

//    std::cout << "Published scan\n";
  std::cout << " ... ScanServer created\n";
}

ScanServer::ScanServer(IPCServer* server, const ScanServer &copy) 
  : IPCObject(getServerName().c_str(), server), m_worker(SctApi::ScanDefImpl::clone(copy.m_worker))
{
  std::cout << "Create a copy of a ScanServer\n";

//   TriggerServer *temp = new TriggerServer(ipcServer.get());
//   worker().setTrigger1(IPCObjectVar<TriggerServer>::duplicate(temp)->m_worker);

//   TriggerServer *temp2 = new TriggerServer(ipcServer.get());
//   worker().setTrigger2(IPCObjectVar<TriggerServer>::duplicate(temp2)->m_worker);

#warning "Need getTrigger to create new TriggerServer if none found?"

  std::cout << " ... ScanServer copied\n";
}

ScanServer::~ScanServer() 
{
//    return IPCObjectVar<ScanServer>::release(this);
  
//    withdraw();
}

void ScanServer::destroy(ipcStatus *_status) 
{
  std::list<Sct_SctApi_T_Scan *> &scans = SctApiServer::singleton->scans;

  std::list<Sct_SctApi_T_Scan *>::iterator item = find(scans.begin(), scans.end(), this);
  if(item == scans.end()) {
    std::cout << "*******Couldn't find destroyed scan in scans list**********!\n";
  } else {
    scans.erase(item);
  }
  decRef();
}


int ScanServer::nextNumber = 0;

std::string ScanServer::getServerName() {
  std::ostringstream serverName;
  serverName << "ScanServer" << nextNumber++;
  return serverName.str();
}

void ScanServer::print(Sct_SctApiStatus *_status) 
{
  std::cout << "Try to print a scan\n";
  worker().print();
  std::cout << "Gave up or sucess (printing scan)\n";
}

void ScanServer::configure (Sct_SctApiStatus *_status, ilu_ShortCardinal type, ilu_Real start, ilu_Real stop, ilu_Real step) 
{
  worker().configure(type, start, stop, step);
}

void ScanServer::configure2 (Sct_SctApiStatus *_status, ilu_ShortCardinal type, ilu_Real start, ilu_Real stop, ilu_Real step) 
{
  worker().configure2(type, start, stop, step);
}

class Sct_SctApi_T_Trigger * ScanServer::getTrigger1 (Sct_SctApiStatus *_status) 
{
  TriggerServer *defTrig = TriggerServer::getServer(worker().getTrigger1()); // new TriggerServer(ipcServer);

  return IPCObjectVar<TriggerServer>::duplicate(defTrig);
}

class Sct_SctApi_T_Trigger * ScanServer::getTrigger2 (Sct_SctApiStatus *_status) 
{
  TriggerServer *defTrig = TriggerServer::getServer(worker().getTrigger2()); // new TriggerServer(ipcServer);

  return IPCObjectVar<TriggerServer>::duplicate(defTrig);
}

void ScanServer::setTrigger1 (Sct_SctApiStatus *_status, class Sct_SctApi_T_Trigger * trig)
{
  TriggerServer *server = dynamic_cast<TriggerServer *>(trig);

  worker().setTrigger1(server->m_worker);
}

void ScanServer::setTrigger2 (Sct_SctApiStatus *_status, class Sct_SctApi_T_Trigger * trig) 
{
  TriggerServer *server = dynamic_cast<TriggerServer *>(trig);

  worker().setTrigger2(server->m_worker);
}

void ScanServer::setScanVariable1 (Sct_SctApiStatus *_status, ilu_ShortCardinal var) 
{
  worker().setScanVariable1(var);
}

void ScanServer::setScanVariable2 (Sct_SctApiStatus *_status, ilu_ShortCardinal var) 
{
  worker().setScanVariable2(var);
}

ilu_ShortCardinal ScanServer::getScanVariable1 (Sct_SctApiStatus *_status) 
{
  return worker().getScanVariable1();
}

ilu_ShortCardinal ScanServer::getScanVariable2 (Sct_SctApiStatus *_status) 
{
  return worker().getScanVariable2();
}

void ScanServer::setNTrigs (Sct_SctApiStatus *_status, ilu_Cardinal nTrigs) 
{
  worker().setNTrigs(nTrigs);
}

ilu_Cardinal ScanServer::getNTrigs (Sct_SctApiStatus *_status) 
{
  return worker().getNTrigs();
}

Sct_SctApi_T_TrigPoints ScanServer::getVariableTrigs (Sct_SctApiStatus *_status) 
{
  Sct_SctApi_T_TrigPoints result;

  result = _Sct_SctApi_T_TrigPoints_sequence::Create(4, NULL);
  result->Clear(0);

  for(unsigned long i=0; i<worker().getVariableTrigs().size(); i++) {
    result->Append(worker().getVariableTrigs()[i]);
  }

  return result;
}

static void enumerateToLongList(ilu_Cardinal val, void *list) {
  SctApi::Scan::TrigPoints &myList = *(SctApi::Scan::TrigPoints *)list;

  myList.push_back(val);
}

void ScanServer::setVariableTrigs (Sct_SctApiStatus *_status, Sct_SctApi_T_TrigPoints trigs) 
{
  SctApi::Scan::TrigPoints newPoints;

  trigs->Enumerate(enumerateToLongList, &newPoints);

  worker().setVariableTrigs(newPoints);
}

void ScanServer::setVariableTrigRange (Sct_SctApiStatus *_status, ilu_ShortCardinal start, ilu_ShortCardinal end, ilu_Cardinal value)
{
  worker().setVariableTrigRange(start, end, value);
}

Sct_SctApi_T_ScanPoints ScanServer::getScanPoints1 (Sct_SctApiStatus *_status) 
{
  Sct_SctApi_T_ScanPoints result;

  result = _Sct_SctApi_T_ScanPoints_sequence::Create(4, NULL);
  result->Clear(0);

  for(unsigned long i=0; i<worker().getScanPoints1().size(); i++) {
    result->Append(worker().getScanPoints1()[i]);
  }

  return result;
}

Sct_SctApi_T_ScanPoints ScanServer::getScanPoints2 (Sct_SctApiStatus *_status)
{
  Sct_SctApi_T_ScanPoints result;

  result = _Sct_SctApi_T_ScanPoints_sequence::Create(4, NULL);
  result->Clear(0);

  for(unsigned long i=0; i<worker().getScanPoints2().size(); i++) {
    result->Append(worker().getScanPoints2()[i]);
  }

  return result;
}

static void enumerateToRealList(ilu_Real val, void *list) {
  SctApi::Scan::ScanPoints &myList = *(SctApi::Scan::ScanPoints *)list;

  myList.push_back(val);
}

void ScanServer::setScanPoints1 (Sct_SctApiStatus *_status, Sct_SctApi_T_ScanPoints scans) 
{
  SctApi::Scan::ScanPoints newPoints;

  scans->Enumerate(enumerateToRealList, &newPoints);

  worker().setScanPoints1(newPoints);
}
 
void ScanServer::setScanPoints2 (Sct_SctApiStatus *_status, Sct_SctApi_T_ScanPoints scans) 
{
  SctApi::Scan::ScanPoints newPoints;

  scans->Enumerate(enumerateToRealList, &newPoints);

  worker().setScanPoints2(newPoints);
}

ilu_Integer ScanServer::getOption (Sct_SctApiStatus *_status, Sct_SctApi_T_Scan_ScanOptions opt) 
{
  return worker().getOption((SctApi::Scan::ScanOptions)opt);
}

void ScanServer::setOption (Sct_SctApiStatus *_status, Sct_SctApi_T_Scan_ScanOptions opt, ilu_Integer val) 
{
  worker().setOption((SctApi::Scan::ScanOptions)opt, val);
}

ilu_Cardinal ScanServer::getScanNumber (Sct_SctApiStatus *_status) 
{
  return worker().getScanNumber();
}
 
ilu_Cardinal ScanServer::getRunNumber (Sct_SctApiStatus *_status) 
{
  return worker().getRunNumber();
}


sctConf_T_ModuleList ScanServer::getModuleList (Sct_SctApiStatus *_status, ilu_Cardinal group) 
{
  sctConf_T_ModuleList result;

  result = _sctConf_T_ModuleList_sequence::Create(4, NULL);
  result->Clear(0);

  std::list<std::string> myList = worker().getModuleList(group);
  for(std::list<std::string> :: const_iterator m = myList.begin();
      m!=myList.end();
      m++) {
    char *copy = (char *)malloc((m->length()+1) * sizeof(char));
    strncpy(copy, m->c_str(), m->length());
    copy[m->length()] = 0;
    result->Append(copy);
  }

  return result;
}

ilu_Cardinal ScanServer::getNGroups (Sct_SctApiStatus *_status) 
{
  return worker().getNGroups();
}

SctApi::Scan &ScanServer::worker() 
{
  return *m_worker;
}


ilu_T_CString ScanServer::getStartTime(Sct_SctApiStatus *_status)
{
  string startTime = to_iso_string(worker().getStartTime()).substr(0, 15); 
  return copyStringToILU(startTime);
}

ilu_T_CString ScanServer::getEndTime(Sct_SctApiStatus *_status)
{
  string endTime = to_iso_string(worker().getEndTime()).substr(0, 15); 
  return copyStringToILU(endTime);    
}


/******************* TriggerServer **********************/

TriggerServer::TriggerServer(IPCServer* server) 
  : IPCObject(getServerName().c_str(), server), m_worker(new SctApi::TriggerImpl())
{
  mapServer(m_worker, this);
}

TriggerServer::~TriggerServer()
{
}

void TriggerServer::destroy(ipcStatus *_status) 
{
  decRef();
}

std::map<boost::shared_ptr< ::SctApi::Trigger>, TriggerServer *> TriggerServer::serverMap;

TriggerServer *TriggerServer::getServer(boost::shared_ptr< ::SctApi::Trigger> trigger) {
  cout << "Looking for server for " << trigger.get() << endl;

  TriggerServer *result = serverMap[trigger];
  cout << " ... Found " << result << endl;

  return result;
  //  return dynamic_cast<TriggerServer *>(trigger.get());
}

void TriggerServer::mapServer(boost::shared_ptr< ::SctApi::Trigger> trigger, TriggerServer *server) {
  cout << "Mapping trigger " << trigger.get() << " to  " << server << endl;
  server->incRef();
  serverMap.insert(make_pair(trigger, server));
}

int TriggerServer::nextNumber = 0;

std::string TriggerServer::getServerName() {
  std::ostringstream serverName;
  serverName << "TriggerServer" << nextNumber++;
  std::cout << "Trigger name: " << serverName.str() << std::endl;;
  return serverName.str();
}

void TriggerServer::print (Sct_SctApiStatus *_status) 
{
  std::cout << "Try to print a trigger\n";
  worker().print();
  std::cout << "Succeeded in printing a trigger\n";
}

void TriggerServer::singleL1A (Sct_SctApiStatus *_status) 
{
  worker().singleL1A();
}

void TriggerServer::doubleL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().doubleL1A(delay);
}

void TriggerServer::delayedL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().delayedL1A(delay);
}

void TriggerServer::calL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().calL1A(delay);
}

void TriggerServer::pulseL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().pulseL1A(delay);
}

void TriggerServer::softL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().softL1A(delay);
}

void TriggerServer::softCalL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay, ilu_ShortCardinal delay2) 
{
  worker().softCalL1A(delay, delay2);
}

void TriggerServer::softPulseL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay, ilu_ShortCardinal delay2) 
{
  worker().softPulseL1A(delay, delay2);
}

void TriggerServer::bcL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay) 
{
  worker().bcL1A(delay);
}

void TriggerServer::bcCalL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay, ilu_ShortCardinal delay2) 
{
  worker().bcCalL1A(delay, delay2);
}

void TriggerServer::bcPulseL1A (Sct_SctApiStatus *_status, ilu_ShortCardinal delay, ilu_ShortCardinal delay2) 
{
  worker().bcPulseL1A(delay, delay2);
}

void TriggerServer::setCommIncr (Sct_SctApiStatus *_status, ilu_ShortCardinal command, ilu_ShortCardinal incr)
{
  worker().setCommIncr(command, incr);
}

void TriggerServer::getCommIncr (Sct_SctApiStatus *_status, ilu_ShortCardinal * command, ilu_ShortCardinal * incr)
{
  worker().getCommIncr(*command, *incr);
}

SctApi::Trigger &TriggerServer::worker() 
{
  return *m_worker;
}

SctApi::BankType myTranslateBank(Sct_SctApi_T_BankType bank) {
  SctApi::BankType result;
  switch(bank) {
  case Sct_SctApi_T_BankType_PHYSICS_CONFIG:
    result = SctApi::SCTAPI_BANK_PHYSICS;
    break;
  case Sct_SctApi_T_BankType_CALIBRATION_CONFIG:
    result = SctApi::SCTAPI_BANK_CALIBRATION;
    break;
  case Sct_SctApi_T_BankType_SCAN_CONFIG:
    result = SctApi::SCTAPI_BANK_SCAN;
    break;
  default:
    result = SctApi::SCTAPI_BANK_PHYSICS;
    break;
  }
  return result;
}

typedef Sct_SctApi_T_ilu__prefix_idlExceptionType_SctApiException iluSctApiException;

void myRaise(Sct_SctApiStatus *_status, SctApi::SctApiException &c) 
{
  iluSctApiException *e = new iluSctApiException;
  char *buffer = new char[strlen(c.what())+1];

  cout << "Raising exception " << c.what() << endl;

  strcpy(buffer, c.what());
  buffer[strlen(c.what())] = 0;
  e->detail = buffer;
  Sct_SctApi_G::RaiseException(_status, Sct_SctApi_E_SctApiException, e);
}
