#include <iostream>

#include <TFile.h>
#include <TH1F.h>
#include <TH2F.h>

using namespace std;

void ReportHistogram(char* infname);

int main(int argc, char **argv) {
  for(int i=0; i<argc-1; i++) {
    ReportHistogram(argv[i+1]);
  }
}

unsigned short GetWord(FILE* file){

  // Need to fit EOF
  int lo_char, hi_char;
  unsigned short word;

  if(file==NULL){
    cout << "bad file\n";
    return 0;
  }

  // Read two bytes
  lo_char = fgetc(file);
  hi_char = fgetc(file);

  if(lo_char == EOF || hi_char == EOF) {
    cout << "end of file\n";
    return 0;
  }

  // Swap bytes to form 16 bit word
  word = (hi_char<<8)+lo_char;

  return word;
}

unsigned long GetLWord(FILE* file){
  // Fit EOF
  int lo_char, hi_char;
  int lo_char2, hi_char2;

  // Result
  unsigned long word;

  if(file==NULL){
    cout << "bad file\n";
    return 0;
  }

  // Read four bytes
  lo_char = fgetc(file);
  hi_char = fgetc(file);
  lo_char2 = fgetc(file);
  hi_char2 = fgetc(file);

  if(lo_char == EOF || hi_char == EOF) {
    cout << "end of file\n";
    return 0;
  }

  // Swap bytes to form 16 bit word
  word = (hi_char2<<24)+(lo_char2<<16) + (hi_char<<8)+lo_char;

  return word;
}

void ReportHistogram(char* infname) {
  cout << "File: " << infname;

  // Convert ROD histograms to SCTDAQ format

  char htitle[128];
  char hname[32];

  int i,j;
  unsigned short header[8];
  unsigned short trailer[8];
  unsigned long data[1024*2];
  unsigned short chan,chip;

  TH1F* h_scan_evcnt;
  TH2F* h_scan0;
  TH2F* h_scan1;

  FILE* infile;

  infile = fopen(infname,"rb");

  if(infile==NULL) {
    cout << "Could not open file\n";
    return;
  }  

  int dontSkip = 0;

  short version = GetWord(infile);          // version
  short length = GetWord(infile);
  dontSkip += 1;

  int run = GetLWord(infile);
  int scan = GetLWord(infile);
  dontSkip += 2;

  char moduleName[16];
  for(int i=0; i<16; i++) {
    moduleName[i] = fgetc(infile);
  }
  dontSkip +=4;

  if(version == 1) {
    char startTime[16];
    for(int i=0; i<16; i++) {
      startTime[i] = fgetc(infile);
    }
    dontSkip +=4;
//     printf("Started: %s\n", startTime);

    char endTime[16];
    for(int i=0; i<16; i++) {
      endTime[i] = fgetc(infile);
    }
    dontSkip +=4;

//     printf("Finished: %s\n", endTime);
  }

  short scanType = GetWord(infile);           // scanType

  cout << " Type: ";
  cout.width(3);
  cout << scanType;

  short nBins = GetWord(infile);
  dontSkip ++;

  int size = GetLWord(infile);
  dontSkip ++;
  short type = GetWord(infile);
  short width = GetWord(infile);

  if(!(width == 32 || width == 16)) {
    cout << " Bad data width: " << width << endl;
    return;
  } else {
    cout << " Width: " << width;
  }

  dontSkip ++;

  int bursts = nBins;

  dontSkip +=4;            // Dont skip pointers

  for(int i=0; i<length/4-dontSkip; i++) {
    unsigned long dummy = GetLWord(infile);
  }

  unsigned long pointer = GetLWord(infile);     // Pointer to thresholds
  unsigned long pEvents = GetLWord(infile);
  unsigned long pErrors = GetLWord(infile);
  unsigned long pData = GetLWord(infile);       // Pointer to data

  for(unsigned int i=0; i<(pointer-length)/4; i++) {   // Presumably 0
    unsigned long dummy = GetLWord(infile);
  }

  cout << " nbins = " << nBins;
  float *threshold = new float[bursts];

  // Scan points
  for(int i=0; i<bursts; i++) {
    unsigned long word = GetLWord(infile);
    threshold[i] = *((float*)(&word));
  }

  // Event counts
  for(int i=0; i<bursts; i++) {
    unsigned long events = GetLWord(infile);
  }

  // Error counts
  for(int i=0; i<bursts; i++) {
    unsigned long errors = GetLWord(infile);
  }

  double hist_start;
  double hist_stop;
  float fist_start;
  float fist_stop;

  //  hist_start = start-(step/2);
  //  hist_stop =  start+(112*step)+(step/2);
  float step = threshold[1] - threshold[0];
  if(threshold[0] < threshold[bursts-1]) {
    hist_start = threshold[0]; // start;
    hist_stop =  threshold[bursts-1] + step; // start+(bursts*step);
  } else {
    hist_start =  threshold[bursts-1] + step; // start+(bursts*step);
    hist_stop = threshold[0]; // start;
  }

  fist_start = hist_start;
  fist_stop =  hist_stop;

  sprintf(htitle,"ROD Events Requested");
  sprintf(hname,"h_scan_evcnt");
  h_scan_evcnt = new TH1F(hname, htitle, bursts, hist_start, hist_stop);
  h_scan_evcnt->SetFillColor(46);

  sprintf(htitle, "%s Stream 0 THRESHOLD Scan", moduleName);
  sprintf(hname,"h_scan0_0");
  h_scan0 = new TH2F(hname,htitle, 768,0.5,768.5, bursts, hist_start, hist_stop);
  h_scan0->SetXTitle("Channel Number");
  h_scan0->SetYTitle("THRESHOLD (mV)");

  sprintf(htitle, "%s Stream 1 THRESHOLD Scan", moduleName);
  sprintf(hname,"h_scan1_0");
  h_scan1 = new TH2F(hname,htitle, 768,0.5,768.5, bursts, hist_start, hist_stop);
  h_scan1->SetXTitle("Channel Number");
  h_scan1->SetYTitle("THRESHOLD (mV)");

  for(j=0;j<bursts;j++) {
    h_scan_evcnt->Fill(threshold[j],50);  // hardcoded to 50 events
  }

  int slice = 1;
  if(slice) {
    for(j=0;j<bursts;j++){
      int words, skip;
      if(type == 16 || type == 6) {
        words = 768*2; skip = 0;
      } else {
        words = 1024*2; skip = 1024-768;
      }
      for(i=0;i<words;i++){
        // data
        if(width == 32) {
          data[i]=GetLWord(infile);
        } else if(width == 16) {
          data[i]=GetWord(infile);
        }

        if(i>0    && i<768)
		  h_scan0->Fill(i,threshold[j],data[i]);
        if(i>(768 + skip) && i<(words-skip))
		  h_scan1->Fill(i-(768 + skip),threshold[j],data[i]);
      }
    }
  } else {
    for(i=0;i<512;i++){

      // For each channel we get 100 hex data words:
      //    
      // Event format
      // Header - 8 16 bit words
      // 00 - Chan Chip
      // 04 - 0000 0000
      // 08 - BBBB BBBB
      // 0C - BBBB BBBB
      //
      // Now the real data
      //
      // Trailer - 8 16 bit words
      // F0 - FFFF FFFF 
      // F4 - FFFF FFFF 
      // F8 - FFFF FFFF 
      // FC - FFFF FFFF

      // header
      for(j=0;j<8;j++){
        header[j]=GetWord(infile);
      }

      // data
      for(j=0;j<bursts;j++){
        data[j]=GetWord(infile);
      }

      // trailer
      for(j=0;j<8;j++){
        trailer[j]=GetWord(infile);
      }

      // Fill SCTDAQ scan Histogram
      chan = (header[1]*128)+header[0];

      for(j=0;j<bursts;j++){
        h_scan0->Fill(chan,threshold[j],data[j]);
      }
    
    } 

  }

  fclose(infile);

  Stat_t stat0[4];
  Stat_t stat1[4];
  for(int i=0; i<4; i++) { stat0[i] = stat1[i] = 0; }

  h_scan0->GetStats(stat0);
  h_scan1->GetStats(stat1);

  cout << " Sum link 0:  " << stat0[0];
  cout << " link 1:  " << stat1[0];
  cout << endl;
}
