#ifndef SCTAPI_PRIMLISTWRAPPER_HEADER
#define SCTAPI_PRIMLISTWRAPPER_HEADER

#include <memory>
#include <list>

namespace SctPixelRod {
  class RodPrimList;
  class RodPrimitive;
}

namespace SctApi {
  /**
     Wrap the SctPixelRod::RodPrimList class

     Why? When a RodPrimitive is created a buffer containing the 
     primitive is passed to the constructor. This remains under 
     the ownership of the caller, but is used by RodPrimitive, 
     and especially RodPrimList in the call buildList.

     PrimListWrapper allows the caller to pass ownership, so the 
     memory buffer isn't leaked.
   */
  struct PrimListWrapper {
    /**
       Create a default RodPrimList
     */
    PrimListWrapper();
    /**
       A RodPrimList with a particular index
     */
    PrimListWrapper(unsigned int index);
    /**
       Clear all datastructures

       This deletes the contents of the RodPrimList as well as all the data 
       buffers it owns.
     */
    void clear();

    /// Add primitive to list
    /**
       If data is non zero then take ownership of the provided memory block

       @param prim Prebuilt primitive to add to the list
       @param data Memory block to take ownership of (will be deleted by delete[])
     */
    void addPrimitive(const SctPixelRod::RodPrimitive &prim, long *data = 0);

    /// Add primitive to the list, using constant data
    /**
       Copy and keep data to end of life.

       @param length Length of data (ie primLength - 4)
       @param index  Id of primitive in primList
       @param prim   Primitive identifier
       @param rev    Primitive revision number
       @param data   Contents of primitive. Ownership remains with the caller.
     */
    void addPrimitive(long length, long index, long prim, long rev, const long *data);

    ~PrimListWrapper();

    std::auto_ptr<SctPixelRod::RodPrimList> list;
    std::list<long *> data;
  };
}

#endif
