// #define IDRAM_BASE 0x80000000     // Rev C
#define IDRAM_BASE 0x10000           // Rev E
#define MDSP_IDRAM_BASE 0x80000000   // Both

#define RW_REG_REV   105  // Was 103
#define POLL_REG_REV 105  // Was 103

void printBinary(unsigned int data, int length, int divisions = 0);

void printSlave(int s, int add, int words, bool usePrim = true) {
  unsigned long *mem = readSlave(s, add, words, usePrim);

  if(mem) {
    int lines = words/8;
    int skipping = 0;
    for(int i=0; i<lines; i++) {
      int val = 0;
      for(int j=0; j<8; j++) {
        val+=mem[i*8+j];
      }
      if(val == 0) {
        if(!skipping) 
          printf(".");
        skipping++;
      } else {
        if(skipping) printf(" Skipped %d\n", skipping);
        skipping = 0;

        for(int j=0; j<8; j++) {
          printf("%08x ", mem[i*8+j]);
        }
        printf("\n");
      }
    }
    if(skipping) printf(" Skipped %d\n", skipping);
    if(lines*8 < words) {
      for(int i=lines*8; i<words; i++) {
        printf("%08x ", mem[i]);
      }
      printf("\n");
    }
  }
}

void printMaster(int add, int words) {
  tapi.dspBlockDump(0, 0, 0, add, words, -1);
}

void printConfig(int mid, int bank = 0) {
  // Print module configuration
  tapi.getABCDModule(mid, bank);

  unsigned long *config = tapi.retrieveModule(mid);

  for(int i=0; i<585; i++) {
    if(i && ((i%8) == 0)) printf("\n");
    printf("%08x ", config[i]);
  }
  printf("\n");
}

void printHostMem(unsigned long *add, int hostLength) {
  for(int i=0; i<hostLength; i++) {
    if(i && ((i%8) == 0)) printf("\n");
    printf("%08x ", add[i]);
  }
  printf("\n");
}

void readSlaveFile(int s, int add, int words, char *fileName, bool usePrim = true) {
  unsigned long *mem = readSlave(s, add, words, usePrim); 

  if(mem) {
    ofstream fout(fileName, ios::binary);
    fout.write((char*)&mem[0], words*4);
    // Clear prim list for next time
    tapi.createDebugPrimList();
  }
}

void readMasterFile(int add, int words, char *fileName) {
  unsigned int readlength
  unsigned long *mem = tapi.dspBlockRead(0, 0, 0, add, words, -1, &readlength);

//    unsigned long *mem = readSlave(s, add, words, usePrim); 

  if(mem) {
    ofstream fout(fileName, ios::binary);
    fout.write((char*)&mem[0], words*4);
  }
}

void loadAndStartSlave(int s, bool usePrim = true) {
  writeSlaveFile(s, "../SlaveDspCode/slaveRun_IPRAM.bin", 0, 16384, usePrim);
  readSlaveFile(s, 0, 16384, "ipramS.bin", usePrim);
  writeSlaveFile(s, "../SlaveDspCode/slaveRun_IDRAM.bin", 0x80000000, 16384, usePrim);
  readSlaveFile(s, 0x80000000, 16384, "idramS.bin", usePrim);
  writeSlaveFile(s, "../SlaveDspCode/slaveRun_xcode.bin", 0x2000000, 65536, usePrim);
  readSlaveFile(s, 0x2000000, 65536, "xcodeS.bin", usePrim);
  startSlave(s);
}

void startSlave(int s) {
  unsigned long data[4];
  data[0] = s;     // Slave
  data[1] = 1;     // Comm on
  data[2] = 2;     // Mode
  data[3] = 0x2000000;    // Timeout

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(8, 1, 8194, 104, data);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);
  // Clear debug list
  tapi.createDebugPrimList();

  unsigned long *result = tapi.getResponse(0, 0, 0);

  if(result) {
    unsigned int repLength;
    repLength = result[0];
    if(repLength > 8) {
      if(result[8] == s) {
        cout << "Reply correct\n";
      } else {
        cout << "Slave started = " << result[8] << " not " << s << endl;
      } 
    } else {
      cout << "Response too short!\n"; 
    }
  } else {
    cout << "No response to slave start\n"; 
  }
}

void writeSlaveFile(int s, char *fileName, long add, int words, bool usePrim = true) {
  ifstream fin(fileName, ios::binary);
  unsigned long *buffer = new unsigned long[5+words];

  fin.read((char *)(buffer + 5), words*4);

  if(usePrim) {
    buffer[0] = s;
    buffer[1] = 0;
    buffer[2] = add;
    buffer[3] = 0xffffffff;
    buffer[4] = words;

    tapi.createDebugPrimList();
    tapi.addDebugPrimList(4 + 5 + words, 1, 8192, 100, buffer);
    tapi.sendDebugPrimList(0, 0, 0);
    tapi.awaitResponse(0, 0, 0);
    // Clear prim list for next time
    tapi.createDebugPrimList();
  } else {
    tapi.dspBlockWrite(0, 0, 0, buffer, add, words, s);
  }
 
  cout << "Slave data sent\n";
}

void clearSlaveMemory(int s, long add, int words) {
  unsigned long *buffer = new unsigned long[3];

  buffer[0] = add;
  buffer[1] = words;
  buffer[2] = 0;

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 3, 1, 4, 100, buffer);
  tapi.sendDebugSlavePrimList(0, 0, 0, s, 1, 0);
  tapi.awaitResponse(0, 0, 0);
  // Clear prim list for next time
  tapi.createDebugPrimList();

  cout << "Slave data sent\n";
}

void clearScratch() {
  for(int i=0; i<10000; i++) {
    scratch[i] = 0;
  }
}

void checkInmem(int delay, int units) {
  // Put formatters into raw data mode
  writeRegister(0x60, 2, 1, 1);
  writeRegister(0x61, 2, 1, 1);

  // Reset INMEM fifo
  writeRegister(0x1c6, 3, 1, 1);
  writeRegister(0x1c6, 3, 1, 0);

  // Test bench mode select (output only)
  //  writeRegister(0x1d6, 12, 6, 0x18);

  // Setup fifo. 22 is delay in clock cycles. 1000 is length
  unsigned long val = (delay << 16) + units;
  writeRegister(0x1d5, 0, 32, val);

  // Calibration Trigger Signal Decoder Enable
  writeRegister(0x1c7, 18, 1, 1);

  // Not sure about this, some bits don't have meanings
  writeRegister(0x1c7, 24, 4, 0xa);

  // Configuration readback enable (I added this)
  writeRegister(0x1c7, 19, 1, 1);

  // Do something that might trigger a response
  sendL1A();

  // Poll "Configuration read back" done
  pollRegister(0x1c9, 3, 1, 1);

  int status = readRegister(0x1d7, false);
  cout << "Inmem status: " << hex << status << dec << endl;

  readFifo(0, 0, units);
  readFifo(0, 1, units);

  // Turn off configuration read back
  writeRegister(0x1c7, 19, 1, 0);

  // Put formatters back to normal data mode
  writeRegister(0x60, 2, 1, 0);
  writeRegister(0x61, 2, 1, 0);
}

// This one works
void newLinkRead(int delay, int units) {
  // Reset INMEM fifo
  writeRegister(0x1c6, 3, 1, 1);
  writeRegister(0x1c6, 3, 1, 0);

  // Disable formatters
  for(int i=0; i<96; i++) {
    writeRegister(0x60 + i, 0, 4, 0x1);     // ???
  }

  // Enable FE Command Outputs with SP0
  writeRegister(0x1c7, 0, 2, 0x1);

  // Load masks
  writeRegister(0x1ca, 0, 32, 0xffffffff);
  writeRegister(0x1cb, 0, 16, 0xffff);
  writeRegister(0x1cc, 0, 32, 0xffffffff);
  writeRegister(0x1cd, 0, 16, 0xffff);

  // Setup fifo. 22 is delay in clock cycles. 1000 is length
  unsigned long val = (delay << 16) + units;

  writeRegister(0x1d5, 0, 32, val);

  // Load enable and load
  writeRegister(0x1c7, 20, 1, 0x1);
  writeRegister(0x1c7,  2, 1, 0x1);

  // Configuration read back mode
  writeRegister(0x1c7, 25, 3, 0x7);

  // Enable formatters
  for(int i=0; i<96; i++) {
    writeRegister(0x60 + i, 0, 4, 0x0);    //  ????
  }

  // Do something that might trigger a response
  sendL1A();

  // Poll "Configuration read back" done
  pollRegister(0x1c9, 3, 1, 1);

  int status = readRegister(0x1d7, false);
  cout << "Inmem status: " << hex << status << dec << endl;

  // Set MUX to read out
  writeRegister(0x1c7, 25, 3, 0x1);

  unsigned long *bufferA = readFifo(0, 0, units, false);
  unsigned long *bufferB = readFifo(0, 1, units, false);

  cout << hex;
  cout.fill('0');
  for(int i=0; i<units/2; i++) {
    cout.width(8); cout << bufferA[i*3 + 0] << " ";
    cout.width(8); cout << bufferA[i*3 + 1] << " ";
    cout.width(8); cout << bufferA[i*3 + 2] << " ";
    cout.width(8); cout << bufferB[i*3 + 0] << " ";
    cout.width(8); cout << bufferB[i*3 + 1] << " ";
    cout.width(8); cout << bufferB[i*3 + 2] << " ";
    cout << endl;
  }
  cout << dec;
  cout.fill(' ');

//   // Turn off configuration read back
//   writeRegister(0x1c7, 19, 1, 0);

//   // Put formatters back to normal data mode
//   writeRegister(0x60, 2, 1, 0);
//   writeRegister(0x61, 2, 1, 0);
}

void readConfiguration(int delay, int units) {
  // Reset INMEM fifo
  writeRegister(0x1c6, 3, 1, 1);
  writeRegister(0x1c6, 3, 1, 0);

  // Disable formatters
  for(int i=0; i<96; i++) {
    writeRegister(0x60 + i, 0, 4, 0x1);
  }

  writeRegister(0x1c7, 0, 2, 0x1);

  // Set and load masks

  // Configuration read back mode
  writeRegister(0x1c7, 25, 3, 0x5);

  // Trans_serial_data

  // Do something that might trigger a response
  sendL1A();

  // Poll "Configuration read back" done
  pollRegister(0x1c9, 3, 1, 1);

  // Setup fifo. 22 is delay in clock cycles. 1000 is length
  unsigned long val = (delay << 16) + units;

  writeRegister(0x1d5, 0, 32, val);

  int status = readRegister(0x1d7, false);
  cout << "Inmem status: " << hex << status << dec << endl;

  readFifo(0, 0, units);
  readFifo(0, 1, units);

  // Turn off configuration read back
  writeRegister(0x1c7, 19, 1, 0);

  // Put formatters back to normal data mode
  writeRegister(0x60, 2, 1, 0);
  writeRegister(0x61, 2, 1, 0);
}

void readConfiguration2(int delay, int units) {
  // Reset INMEM fifo
  writeRegister(0x216, 3, 1, 1);
  writeRegister(0x216, 3, 1, 0);

  // Disable formatters
//   for(int i=0; i<96; i++) {
//     writeRegister(0x60 + i, 0, 4, 0x1);
//   }

//   0x22d?

  writeRegister(0x217, 0, 2, 0x1);

  // Setup fifo. 22 is delay in clock cycles. 1000 is length
  unsigned long val = (delay << 16) + units;

  writeRegister(0x225, 0, 32, val);

  // Configuration read back mode
//   writeRegister(0x217, 25, 3, 0x5);
  writeRegister(0x217, 24, 3, 0xA);

  // Trans_serial_data
  // Do something that might trigger a response
  sendL1A(1);

  // Poll "Configuration read back" done
  pollRegister(0x218, 3, 1, 1);

  int status = readRegister(0x1d7, false);
  cout << "Inmem status: " << hex << status << dec << endl;

  readFifo(0, 0, units);
  readFifo(0, 1, units);

  // Turn off configuration read back
  writeRegister(0x217, 19, 1, 0);

//   // Put formatters back to normal data mode
//   writeRegister(0x60, 2, 1, 0);
//   writeRegister(0x61, 2, 1, 0);
}

void readLinksToFifos(int delay, int units) {
  // Enable Test Bench IO!!!
  writeRegister(0x1c7, 15, 2, 2);

  // Test bench mode select (output only)
  writeRegister(0x1d6, 12, 6, 0x18);

  unsigned long val = (delay << 16) + units;

  //  cout << "val = " << hex << val << dec << endl;

  //  // Setup fifo. 22 is delay in clock cycles. 1000 is length
  writeRegister(0x1d3, 0, 32, val);

//    readRegister(0x1d7);

  // Set up data link MUX       (Test bench)
  writeRegister(0x1c7, 25, 3, 0x2);
  // Test bench run bit
  writeRegister(0x1c7, 17, 1, 0x1);

  // Poll "Configuration read back" done
  pollRegister(0x1d7, 6, 2, 0x3);

  // Test bench idle
  writeRegister(0x1c7, 17, 1, 0x0);
  // Give DSP R/W control of debug FIFO
  writeRegister(0x1c7, 25, 3, 0x1);

  int status = readRegister(0x1d7, false);
  cout << "Inmem status: " << hex << status << dec << endl;

  cout << "Bank A\n";
  readFifo(0, 0, units);
  cout << "Bank B\n";
  readFifo(0, 1, units);

  cout << "TIM FIFO\n";
  readFifo(3, 0, units);
}

unsigned long *readFifo(int id, int bank, int elems, bool dumpFifo = true) {
  unsigned long fifo[] = {id, bank, 1, elems, 0xffffffff};

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 5, 1, 8198, 104, fifo);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

  unsigned long *result = tapi.getResponse(0, 0, 0);

  if(result) {
    unsigned int fifoLength = result[0];
    cout << "Got " << result[8] << " bytes Length is " << fifoLength << endl;

    // Clear prim list for next time
    tapi.createDebugPrimList();

    if(dumpFifo) {
      cout << hex;
      cout.fill('0');
      for(int i=0; i<fifoLength-11; i++) {
        if(i && i%8 == 0) cout << endl;
        cout.width(8);
        cout << result[i + 9] << " ";
      }
      cout << endl;
      cout << dec;
      cout.fill(' ');
    }

    return result + 9;
  }
}

void pollRegister(int r, int off, int width, int val) {
  // Timeout in 1000000 usec
  unsigned long poll[] = {r, off, width, val, 1000000};

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 5, 1, 8197, POLL_REG_REV, poll);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

  unsigned long *result = tapi.getResponse(0, 0, 0);

  if(result) {
    unsigned int pollLength = result[0];
    //    cout << "Got " << result[8] << " reply\n";
    if(pollLength>8) {
      if(result[8] == 1) {
        cout << "Success\n";
      } else {
        cout << "Failure (timeout)\n";
      }
    } else {
      cout << "Poll response too short\n";
    }
  }

  // Clear prim list for next time
  tapi.createDebugPrimList();
}

void taskOp(int typ, int op, int data, int dsp) {
  unsigned long tkOp[] = {typ, op, data};

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(7, 1, 13, 100, tkOp);
  if(dsp == -1) {
    tapi.sendDebugPrimList(0, 0, 0);
  } else {
    tapi.sendDebugSlavePrimList(0, 0, 0, dsp, 1, 0);
  }
  tapi.awaitResponse(0, 0, 0);
}

unsigned int readRegister(int r, bool output = true) {
  unsigned long d[] = {r, 0, 32, 1, 0};

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(9, 5, 8196, RW_REG_REV, d);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

  unsigned long *result = tapi.getResponse(0, 0, 0);
  unsigned int val = 0;
  if(result) {
    unsigned long myLength = result[0];
    if(output) {
      cout << "Result is 0x" << hex << result[8] << dec << endl;
    }
    val = result[8];
  } else {
    cout << "ReadRegister " << hex << r << dec << " failed\n";
  }

  // Clear prim list for next time
  tapi.createDebugPrimList();

  return val;
}

void writeRegister(int r, int off, int wid, int value) {
  unsigned long d[] = {r, off, wid, 0, value};

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(9, 5, 8196, RW_REG_REV, d);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

  // Clear prim list for next time
  tapi.createDebugPrimList();
}

void sendL1A(bool capture = 0) {//1   0     3   2     5   4    0    1    2    3    4    5  
  unsigned long l1aStream[] = {0x640065, 0x640064, 0x640064, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 2, 1, 63, 3, 0}; // 0x0, 0x0, 0x0, 0x0, 2};
  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 14, 2, 8202, 102, l1aStream);

  unsigned long sendStream[] = {2, capture};
  tapi.addDebugPrimList(4 + 2, 3, 8203, 100, sendStream);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);
}

void sendSR_L1A() {  //           1   0     3   2     5   4    0    1    2    3    4    5  
  unsigned long l1aStream[] = {0x650066, 0x640064, 0x640064, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 2, 1, 63, 3, 0}; // 0x0, 0x0, 0x0, 0x0, 2};
  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 14, 2, 8202, 102, l1aStream);

  unsigned long sendStream[] = {2, 0};
  tapi.addDebugPrimList(4 + 2, 3, 8203, 100, sendStream);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);
}

// Modify this one and don't expect it to be the same next time
void sendTrigger() { //           1   0     3   2     5   4    0     1    2    3    4    5  
  unsigned long l1aStream[] = {0x700066, 0x670064, 0x640064, 0x0, 0x10, 0x0, 0x0, 0x0, 0x0, 0, 1, 63, 2, 0}; // 0x0, 0x0, 0x0, 0x0, 2};
  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 14, 2, 8202, 102, l1aStream);

  unsigned long sendStream[] = {2, 0};
  tapi.addDebugPrimList(4 + 2, 3, 8203, 100, sendStream);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);
}

void calib_init() {

  // Clear router transfer buffer
  //      80008000 2000 words of 0

  // Set link static mask on all inputs
  for(int a=0x60; a<=0xbf; a++) {
    writeRegister(a, 0, 4, 0x01);
  }

  // Mask FE Occ counters
  writeRegister(0x1df, 0, 32, 0xffffffff);
  writeRegister(0x1e0, 0, 32, 0xffffffff);
  writeRegister(0x1e1, 0, 32, 0xffffffff);

  // Set RRIF cmd 0&1
  writeRegister(0x1c6, 0, 8, 0xff);    // Reset everything
  writeRegister(0x1c6, 0, 8, 0x78);
  writeRegister(0x1c7, 0, 23, 0x494a9);

  writeRegister(0x1c7, 22, 1, 1);     // Set static L1

  // Read register 1c7

  writeRegister(0x229, 0, 24, 1);     // Set CAL_L1_ID_0
  writeRegister(0x22a, 0, 24, 1);     // Set CAL_L1_ID_0

  // Load mask
  writeRegister(0x1ca, 0, 32, 1);   // Unmask channel 1
  writeRegister(0x1cb, 0, 16, 0);
  writeRegister(0x1cc, 0, 32, 0);
  writeRegister(0x1cd, 0, 16, 0);
  writeRegister(0x1c7, 2, 1, 1);    // Load mask

  // Load default mode bits
  for(int a=0x1ec; a<=0x1f3; a++) {
    writeRegister(a, 0, 16, 0);
  }

  // Set ROD type
  writeRegister(0x20c, 0, 8, 0);

  // Load default dynamic mask
  for(int a=0x20d; a<=0x218; a++) {
    writeRegister(a, 0, 16, 0);
  }

  // "set_rb10xFC_htl0xF0"
  // Set header/trailer limit on all formatters
  for(int a=0xd0; a<=0xd7; a++) {
    writeRegister(a, 0, 8, 0xf0);
  }
  // Set ROD busy limit on all formatters
  for(int a=0xd8; a<=0xdf; a++) {
    writeRegister(a, 0, 8, 0xfc);
  }

  // Normal data path
  writeRegister(0x1c7, 25, 3, 0x4);

  // Set Form Link 01 on
  writeRegister(0x60, 0, 4, 2);
  writeRegister(0x61, 0, 4, 2);
  writeRegister(0x1df, 0, 32, 0xfffffffc);
  writeRegister(0x1e0, 0, 32, 0xffffffff);
  writeRegister(0x1e1, 0, 32, 0xffffffff);
}

// Different modes
// 0    Normal (trap everything)
// 1    Error (trap in error mode)
// 2    Router distribution (slave 2 matches remainder 2)
// 3    DSP distribution (slave 2 matches event type 2)
void setupEvTrap(int s, int mode = 0) {
//    long setup[24] = {0x1 << s, 0, 0x2000, 0, 
//                      0x3, 0x1, 0, 0,
//                      0, 0, 1, 2,
//                      0x3, 0, 0, 1,                    // Config, exclusion
//                      0x1, 0, s, 1,                    // function, match
//                      1,   0, 0, 0};                   // modulus, remainder

  unsigned long setup[24] = {0x1 << s, 0, 0x2000, 0, 
                    0x3, 0x1, 0, 0,
                    0, 0, 0, 0,
                    0x3, 0, 0, 1,                    // Config, exclusion
                    0x1, 0, s, 1,                    // function, match
                    1,   0, 0, 0};                   // modulus, remainder

  unsigned long *realSetup = new unsigned long[24];

  for(int i=0; i<24; i++) realSetup[i] = setup[i];

  realSetup[0] = 1<<s;

//    if(mode == 1) {
//      realSetup[9] = 1;
//    } else if(mode == 2) {
//      realSetup[20] == 4;
//      realSetup[22] == s;
//    } else if(mode == 3) {
//      realSetup[18] == s;
//      realSetup[20] == 1;
//      realSetup[22] == 0;
//    } 

//    realSetup[0] = 0x1 << s;
//    realSetup[18] = s;

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(28, 1, 3, 103, realSetup);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0, 1);

  unsigned long *result = tapi.getResponse(0, 0, 0);

  if(result) {
    cout << "ETS Error code " << result[8] << endl;
  } else {
    cout << "No response to ETS\n";
  }
}

void startEvTrap(int s) {
  unsigned long trap[0] = {};

  tapi.createDebugPrimList();
  
  tapi.addDebugPrimList(4, 0, 4096, 101, trap);
  tapi.sendDebugSlavePrimList(0, 0, 0, s, 1, 1);
  tapi.awaitResponse(0, 0, 0);
}

void stopEvTrap(int s) {
  unsigned long trap[0] = {};

  tapi.createDebugPrimList();
  
  tapi.addDebugPrimList(4, 0, 4097, 100, trap);
  tapi.sendDebugSlavePrimList(0, 0, 0, s, 1, 1);
  tapi.awaitResponse(0, 0, 0);
}

void decodeEvent(int sl, int index, bool extFlag = false, bool errorType = false) {
  const int eventSize = 0x400;
  long base;
  if(extFlag) 
    base = 0x2092000;
  else {
    // RevE
//     base = 0x18000;
    // Rev C
    base = 0x80008000;
  }

  long add = base + eventSize * index;
  const int eventWords = eventSize / 4;
  int frameCount = 0;

  // Storage for 32 frames
  unsigned long frameBuffer[eventWords * 32];

  while(frameCount < 16) {
    unsigned long *mem = readSlave(sl, base + eventSize * ((index + frameCount) % 32), eventWords);

    if(mem) {
      for(int i=0; i<eventWords; i++) {
        frameBuffer[eventWords*frameCount + i] = mem[i];
      }

      frameCount ++;
    } else {
      break;
    }

    if((frameBuffer[eventWords*frameCount - 1] & 0xff0e0000) == 0x400e0000) {
      break;
    } 
  }

  if(frameCount == 0) {
    cout << "Couldn't load memory\n";
    return;
  } else {
    cout << "Found " << frameCount << " frames of event data\n";
  }

  if((frameBuffer[eventWords*frameCount - 1] & 0xff0e0000) != 0x400e0000) {
    cout << "Event trailer marker not found " << frameBuffer[eventWords*frameCount - 1] << endl;
    return;
  } 

  if(frameBuffer[0] == 0xb0f00000) { 
    cout << "Found header\n";
  } else {
    cout << "Bad header 0x" << hex << frameBuffer[0] << dec << endl;
    return;
  }

  if(frameBuffer[1] == 0xee1234ee) {
    cout << "Valid header\n";
  } else {
    cout << "Bad check 0x" << hex << frameBuffer[1] << dec << endl;
  }

  int headerLength;     // Including bof
  int trailerLength;    // Excluding eof

  if(errorType) {
    headerLength = 6;
    trailerLength = 8;

    cout << "L1ID = " << frameBuffer[2] << " BCID = " << frameBuffer[3] << endl;
    cout << "Trigger type = " << frameBuffer[4] << " Event type = " << frameBuffer[5] << endl;
  } else {
    headerLength = 9;
    trailerLength = 5;

    if(frameBuffer[2] != headerLength) {
      cout << "Unknown header length (" << frameBuffer[2] << ")\n";
    } 

    cout << "Version: " << frameBuffer[3] << " ID: " << frameBuffer[4] << endl;
    cout << "L1ID = " << frameBuffer[5] << " BCID = " << frameBuffer[6] 
         << " TType = " << frameBuffer[7] << " det type = " << frameBuffer[8] << endl;

  } 

  int eofAt = headerLength;

  while(frameBuffer[eofAt] != 0xe0f00000 && eofAt< (eventWords * frameCount)) eofAt ++;

  if(eofAt < (eventWords * frameCount)) {
    cout << "Found eof!\n";
  } else {
    cout << "EOF not found\n";
    return;
  }

  int eventLength = eofAt-headerLength-trailerLength;

  unsigned long *rawEvent = frameBuffer+headerLength;

  for(int i=0; i<eventLength * 2; i++) {
    UINT16 currWord;
    if(i&1) {
      currWord = rawEvent[i/2] & 0x00ffff;
    } else {
      currWord = (rawEvent[i/2] & 0xffff0000) >> 16;
    }
    cout << hex << currWord << dec;

    switch((currWord & 0xe000) >> 13) {
    case 0:
      {
        if(currWord & 0x1f80) 
          cout << " INVALID" << endl;
        else {
          // Flagged error
          cout << " Flagged error on " << ((currWord & 0x78) >> 3) 
               << " (" << ((currWord & 0x7)) << ")\n";
        }
      }
      break;
    case 1:
      {
        // Header
        cout << " Header: ";
        if(errorType) {
          cout << "L1 " << ((currWord & 0x0f00) >> 8) 
               << " BCID " << ((currWord & 0xff)) 
               << ((currWord & 0x1000)?" Preamble err ":"")
               << endl;
        } else {
          cout << "Link " << (currWord & 0x7f) << " " 
             << ((currWord & 0x100)?"Condensed ":"") 
             << ((currWord & 0x200)?"BC err ":"") 
             << ((currWord & 0x400)?"L1 err ":"") 
             << ((currWord & 0x800)?"Time out err ":"") 
             << ((currWord & 0x1000)?"Preamble err ":"")
             << endl;
        }
      }
      break;
    case 2:
      {
        if(currWord & 0x3ff) 
          cout << " INVALID\n";
        else {
          // Trailer
          cout << " Trailer: " 
               << ((currWord & 0x400)?"Data overflow err ":"") 
               << ((currWord & 0x800)?"H/T limit err ":"") 
               << ((currWord & 0x1000)?"Trailer bit err ":"");
          cout << endl;
        }
      }
      break;
    case 3:
      {
        if(currWord & 0x300) 
          cout << " INVALID\n";
        else {
          // Raw data
          cout << " Raw: " 
               << ((currWord & 0x1c00) >> 10) + 1 << " bits " 
               << hex << (currWord & 0xff) << dec;
          cout << endl;
        }
      }
      break;
    default:
      // Everything else
      //          cout << " Cluster or condensed\n";
      {
        if((currWord & 0x2) == 0) {
          cout << "  Condensed: " ;
          cout << "Chip " << ((currWord & 0x7e00) >> 11)
               << " address 0x" << hex << ((currWord & 0x7f0) >> 4) << dec;
          if(currWord & 1) {
            cout << " hits " << ((currWord & 0x4)?"1":"0") 
                 << ((currWord & 0x8)?"1":"0");
          } else {
            cout << " hit " << ((currWord & 0x4)?"1":"0"); 
          }
          cout << endl;
        }

        if(currWord & 0x8 == 0) {
          cout << "  1st hit clust exp: " 
               << "Chip " << ((currWord & 0x7e00) >> 11)
               << " address 0x" << hex << ((currWord & 0x7f0) >> 4) << dec;
          cout << hex << " 0x" << (currWord & 0x7) << dec << endl;
        } else {
          if((currWord & 0x7f00) == 0) {
            cout << "  Clust exp: ";
            cout << hex << " 0x" << (currWord & 0x7) << dec;
            if(currWord & 0x80) {
              cout << hex << " 0x" << ((currWord & 0x70) >> 4) << dec;
            }
            cout << endl;
          }
        }
      }
    }
  }

  cout << "Error count = " << frameBuffer[eofAt-trailerLength] << endl;
  cout << "Error flags = 0x" << hex << frameBuffer[eofAt-trailerLength + 1] << dec << endl;

  int flags = frameBuffer[eofAt-trailerLength + 1];
  if(flags != 0) {
    if(flags & 0x1) {
      cout << "HEADER ";
    }
    if(flags & 0x2) {
      cout << "TRAILER ";
    }
    if(flags & 0x4) {
      cout << "FLAGGED ";
    }
    if(flags & 0x8) {
      cout << "SYNC ";
    }
    if(flags & 0x10) {
      cout << "HIT PATTERN ";
    }
    if(flags & 0x20) {
      cout << "L1ID ";
    }
    if(flags & 0x40) {
      cout << "BCID ";
    }
    if(flags & 0x80) {
      cout << "TIMEOUT ";
    }
    if(flags & 0x100) {
      cout << "Almost Full ";
    }
    if(flags & 0x200) {
      cout << "OVERFLOW ";
    }
    cout << endl;
  }

  if(errorType) {
    cout << "Error Status: \n" << hex;

    for(int es = 0; es < 6; es++) {
      UINT16 esWord = frameBuffer[eofAt-trailerLength + 2 + es];
      for(int i=0; i<8; i++) {
        cout << (esWord & 0x3) << " ";
        esWord >>= 2;
      }
    }

    cout << dec << endl;
  } else {
    cout << "nData = " << frameBuffer[eofAt-2] << " words found = " << (eofAt-9-5) << endl;
  }

  cout << "EOF = 0x" << hex << frameBuffer[eofAt] << dec << endl;

  // Print out the raw data afterwards
  int wordsPerLine = 8;
  int lines = eventLength/wordsPerLine;
  int skipping = 0;

  for(int i=0; i<lines; i++) {
    int val = 0;
    for(int j=0; j<wordsPerLine; j++) {
      val+=rawEvent[i*wordsPerLine+j];
    }
    if(val == 0) {
      if(!skipping) 
        printf(".");
      skipping++;
    } else {
      if(skipping) printf(" Skipped %d\n", skipping);
      skipping = 0;

      for(int j=0; j<wordsPerLine; j++) {
        printf("%08x ", rawEvent[i*wordsPerLine+j]);
      }
      printf("\n");
    }
  }
  if(skipping) printf(" Skipped %d\n", skipping);
  if(lines*wordsPerLine < eventLength) {
    for(int i=lines*wordsPerLine; i<eventLength; i++) {
      printf("%08x ", rawEvent[i]);
    }
    printf("\n");
  }

  printf("%d frames finished in frame %d with 0x%08x\n", 
         frameCount, frameCount + index, frameBuffer[frameCount * eventWords - 1]);
}

void histoStatus(int dsp) {
  if(dsp == -1) {
    // Histogram command stat 0
    long reg = tapi.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x20, -1);
    cout << "Bin: " << (reg & 0xff) 
         << " Cal: " << ((reg & 0xff00) >> 8) 
         << " Errs: " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram command stat 1
    reg = tapi.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x24, -1);
    cout << "avg. trans " << (reg & 0xff)
         << " avg. len " << ((reg & 0xff00) >> 8)
         << " avg. proc. " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram status 1
    reg = tapi.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x2c, -1);
    cout << "Slave 0: " << (reg & 0xffff)
         << " Slave 1: " << ((reg & 0xffff0000) >> 16) << endl;

    // Histogram status 0
    reg = tapi.dspSingleRead(0, 0, 0, MDSP_IDRAM_BASE + 0x28, -1);
    cout << "Slave 2: " << (reg & 0xffff)
         << " Slave 3: " << ((reg & 0xffff0000) >> 16) << endl;
  } else {
    unsigned long *regs = readSlave(dsp, IDRAM_BASE + 0x10, 20);

    if(regs) {
      long reg;

      cout << "TrapStatus: \n";

      // Trap stat 0
      reg = regs[1];
      cout << "Event word count: " << ((reg &0xffff) >> 0);
      cout << " IFrame tail: " << ((reg & 0xff0000) >> 16);
      cout << " XFrame tail: " << ((reg & 0xff000000) >> 24) << endl;

      // Trap stat 1
      reg = regs[2];
      cout << " IFrame head: " << ((reg & 0xff0000) >> 16);
      cout << " XFrame head: " << ((reg & 0xff000000) >> 24) << endl;

      // Trap cmd stat
      reg = regs[17];
      cout << "Trap Command/Status: " 
           << ((reg & 0x1)?"Trailer ":"")
           << ((reg & 0x2)?"Transmit ":"")
           << ((reg & 0x4)?"Header ":"")
           << ((reg & 0x4)?"ISR active ":"");
      cout << ((reg & 0x10)?"\"Data Error\" ":"")
           << ((reg & 0x20)?"\"Header Error\" ":"")
           << ((reg & 0x40)?"\"Trailer Error\" ":"")
           << ((reg & 0x80)?"\"Link Error\" ":"");
      cout << ((reg & 0x100)?"\"Error\" ":"")
           << ((reg & 0x200)?"\"Overflow(old)\" ":"")
           << ((reg & 0x800)?"\"ISR pending\" ":"");
      cout << ((reg & 0x4000)?"\"Overflow error\" ":"")
           << ((reg & 0x8000)?"\"Overflow\" ":"");
      cout << endl;

      cout << "\"Error count\": " << ((reg >> 16) & 0xff) << endl;
      cout << "\"Event count\": " << ((reg >> 24) & 0xff) << endl;

      // Hcmd stat reg 0
      reg = regs[4];
      cout << "Bin: " << (reg & 0xff) 
           << " Cal: " << ((reg & 0x1f00) >> 8) 
           << ((reg & 0x800)?" Cal enabled":"")
           << ((reg & 0x1000)?" New bin":"")
           << ((reg & 0x20000)?" New cal":"") << endl;

      // Hcmd stat reg 1
      reg = regs[5];
      cout << "Num events: " << reg << endl;

      // H stat reg 0
      reg = regs[6];
      cout << "Histogram status: " << ((reg & 0x1)?"Ready ":"")
           << ((reg & 0x2)?"Expecting ":"")
           << ((reg & 0x4)?"Processing ":"")
           << ((reg & 0x8)?"Done ":"") << endl;
      cout << "Bin err: " << ((reg >> 8) & 0xff) 
           << " proc time: " << ((reg >> 16) & 0xffff) << endl;

      // H stat reg 1
      reg = regs[7];
      cout << "Events recvd: " << reg << endl;
    }
  }
}

unsigned long *readSlave(int s, int add, int words, bool usePrim = true) {
  unsigned long *mem = 0;

  if(usePrim) {
    // Read slave memory        address                 numwords
    unsigned long rwSlaveMem[5] = {s, 1, add, 0xffffffff, words};

    printf("Read slave %d: 0x%x\n", s, add);
    tapi.createDebugPrimList();
    tapi.addDebugPrimList(4 + 5, 1, 8192, 100, rwSlaveMem);
    tapi.sendDebugPrimList(0, 0, 0);
    tapi.awaitResponse(0, 0, 0);

    mem = tapi.getResponse(0, 0, 0);
    // Clear prim list for next time
    tapi.createDebugPrimList();
    
    if(mem) {
      return mem+8;
    }
  } else {
    // Turn off dsp accesses 
    tapi.dspSingleWrite(0, 0, 0, MDSP_IDRAM_BASE + 0xc, 0x400, -1);

    unsigned int readLength;
    unsigned long *block = tapi.dspBlockRead(0, 0, 0, add, words, s, &readLength);
    tapi.dspSingleWrite(0, 0, 0, MDSP_IDRAM_BASE + 0xc, 0x0, -1);
    return block;
  }

  return 0;
}

// Send config but turn off ROD processing of data stream
// So it doesn't get confused by "L1A"s 
void sendConfig(int module = 0, int bank = 0) {
  writeRegister(0x1c7, 18, 1, 0);
  tapi.sendABCDModule(module, bank);
  writeRegister(0x1c7, 18, 1, 1);
}

// Read histogram buffer to file
// How to handle multiple slaves?
void readHisto(int bins, char *fname = "histogram.bin", bool headers=false) {
  ofstream histoout(fname, ios::binary);

  // Not right
  if(headers) {
    unsigned int length = 22*4 + 584 * 4;
    histoout.put(1); histoout.put(0);                                           // Version
    histoout.put(length & 0xff); histoout.put((length >> 8) & 0xff);            // length 
    histoout.put(123); histoout.put(0); histoout.put(0); histoout.put(0);       // run
    histoout.put(221); histoout.put(0); histoout.put(0); histoout.put(0);       // Scan
    histoout << "Module 28......"; histoout.put(0);                             // 16 char name
    histoout << "Module 28......"; histoout.put(0);                             // 16 char start
    histoout << "Module 28......"; histoout.put(0);                             // 16 char end
    histoout.put(1); histoout.put(0);                                           // Type
    histoout.put(bins&0xff); histoout.put((bins&0xff00)>>8);                    // Points

    unsigned int size = 0x800 * bins * 2;
    histoout.put(size&0xff); histoout.put((size&0xff00)>>8);                   
            histoout.put((size&0xff0000)>>16); histoout.put((size&0xff000000)>>24);                    // Size
    histoout.put(2); histoout.put(0);                                           // Datatype
    histoout.put(16); histoout.put(0);                                          // width

    // Module config
    for(int i=0; i<584; i++) {
      histoout.put(i);
      histoout.put(i);
      histoout.put(i);
      histoout.put(i);
    }

    unsigned int pointer = length;

    histoout.put(((pointer) & 0xff));
    histoout.put(((pointer) & 0xff00) >> 8);
    histoout.put(((pointer) & 0xff0000) >> 16);
    histoout.put(((pointer) & 0xff000000) >> 24);

    // Other pointers
    for(int i=0; i<12; i++) {
      histoout.put(0);
    }

    for(float j=0.0; j<(float)bins; j+=1.0) {
      // X
      histoout.put(((*(int*)&j) & 0xff));
      histoout.put(((*(int*)&j) & 0xff00) >> 8);
      histoout.put(((*(int*)&j) & 0xff0000) >> 16);
      histoout.put(((*(int*)&j) & 0xff000000) >> 24);
    }
    for(int i=0; i<bins; i++) {
       // Events
      histoout.put(((1000) & 0xff));
      histoout.put(((1000) & 0xff) >> 8);
      histoout.put(((1000) & 0xff) >> 16);
      histoout.put(((1000) & 0xff) >> 24);
    }
    for(int i=0; i<bins; i++) {
      // Errors
      histoout.put(0);
      histoout.put(0);
      histoout.put(0);
      histoout.put(0);
    }
  }

  // The data!
  for(int i=0; i<bins/2; i++) {
    unsigned long *chunk = readSlave(0, 0xa00e2000 + 0x1000*4*i, 0x1000);
    histoout.write((char*)&chunk[0], 0x1000*4);
  }
}

void print_calib_old() {
  cout << "Formatters\n";
  for(int a=0x0; a<0x60; a++) {
    int data = readRegister(a, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";

    if(a % 12 == 11) cout << endl;
  }

  cout << "Link masks\n";
  for(int a=0x0; a<0x60; a++) {
    int link = readRegister(0x60 + a, false) & 0xf;
    cout.width(2);
    cout << link << " ";

    if(a % 12 == 11) cout << endl;

    // Status is 16 bits
//      link = readRegister(a, false) & 0xffff;
//      cout << link << " ";
  }
//    cout << endl;

  cout << "Occupancy counter mask\n";
  readRegister(0x1df);
  readRegister(0x1e0);
  readRegister(0x1e1);

  cout << "Status regs\n";
  readRegister(0x1c6);
  readRegister(0x1c7);
  readRegister(0x1c8);
  readRegister(0x1c9);

  cout << "Output mask\n";
  readRegister(0x1ca);
  readRegister(0x1cb);
  readRegister(0x1cc);
  readRegister(0x1cd);

  cout << "Mode bits\n";
  for(int a=0x1ec; a<=0x1f3; a++) {
    int mode = readRegister(a, false) & 0xfff;
    cout.width(3);
    cout << hex << mode << dec << " ";
  }
  cout << endl;

  cout << "ROD event type\n";
  readRegister(0x20c);

  cout << "Dynamic mask\n";
  for(int a=0x20d; a<=0x218; a++) {
    int mask = readRegister(a, false) & 0xffff;
    cout.width(4);
    cout << hex << mask << dec << " ";
  }
  cout << endl;
}

void print_calib() {
  cout << "Formatters\n";
  cout << " Enables\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Expanded\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+8, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Config mode\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x10, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Edge mode\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x18, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Time out\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x20, false) & 0xff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Overflow limit\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x28, false) & 0x7ff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " H/T limit\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x30, false) & 0x3ff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " ROD busy limit\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x38, false) & 0x3ff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Diagnostic link to LEMO\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x70, false) & 0xf;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Diagnostic mode bits read enable\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x78, false) & 0x1;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << "Link occupancies\n";
  for(int a=0x0; a<96; a++) {
    int link = readRegister(0x80 + a, false) & 0xfff;
    cout.width(3);
    cout << link << " ";

    if(a % 12 == 11) cout << endl;

  }
  cout << endl;

  cout << " Timeout error\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0xe0, false) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Overflow error\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0xe8, false) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " H/T error\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0xf0, false) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " ROD busy error\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0xf8, false) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Parsing mode (0 = Raw, 1 = normal) \n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x100, false) & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Formatter status\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x108, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Formatter versions\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x110, false) & 0xffff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
  }
  cout << endl;

  cout << " Formatter mode bits\n";
  for(int a=0; a<8; a++) {
    int data = readRegister(a+0x118, false); // & 0xfff;
    cout.width(4);
    cout << hex;
    cout << data << " ";
//     data = readRegister(a+0x120, false); //  & 0xfff;
//     cout.width(4);
//     cout << data << " ";
//     cout << dec;
  }
  cout << endl;

  cout << "Formatter to front panel\n";
  readRegister(0x1a4);

  cout << "EFB version\n";
  readRegister(0x1b2);

  cout << "RTR version\n";
  readRegister(0x205);

  cout << "RRIF version\n";
  readRegister(0x216); // 0x27d);
  cout << "RRIF command 0\n";
  readRegister(0x218);
  cout << "RRIF command 1\n";
  readRegister(0x217);
  cout << "RRIF status 0\n";
  readRegister(0x21c);
  cout << "RRIF status 1\n";
  readRegister(0x21b);

  cout << "Output masks\n";
  readRegister(0x21f);
  readRegister(0x220);
  readRegister(0x221);
  readRegister(0x222);

  cout << "L1ID for ports 0 and 1\n";
  readRegister(0x229);
  readRegister(0x22a);
  cout << "BCID for ports 0 and 1\n";
  readRegister(0x22b);
}

void print_mode_bits() {
  cout << "Mode bits LUT select\n";

  int data = tapi.dspSingleRead(0, 0, 0, 0x40441c, -1);
  cout.width(2);
  cout << data << " ";
  cout << endl;

  for(int m=0; m<8; m++) {
    cout << "LUT set: " << m << endl;
    for(int f=0; f<8; f++) {
      cout << "Formatter: " << f << endl;
      // Default mode bit 0
      data = tapi.dspSingleRead(0, 0, 0, 0x404800 + 8 * f + 0x80 * m, -1);
      printBinary(data, 12); cout << " ";

      // Default mode bit 1
      data = tapi.dspSingleRead(0, 0, 0, 0x404804 + 8 * f + 0x80 * m, -1);
      printBinary(data, 12); cout << " ";


      // Corrective mode bit 0
      data = tapi.dspSingleRead(0, 0, 0, 0x404840 + 8 * f + 0x80 * m, -1);
      printBinary(data, 12); cout << " ";

      // Corrective mode bit 1
      data = tapi.dspSingleRead(0, 0, 0, 0x404844 + 8 * f + 0x80 * m, -1);
      printBinary(data, 12); cout << " ";
      cout << endl;
    }

    // Mask bits 0-31
    data = tapi.dspSingleRead(0, 0, 0, 0x404600 + 0x10 * m, -1);
    printBinary(data, 32, 16);

    // SP0 bits 32-47
    data = tapi.dspSingleRead(0, 0, 0, 0x404604 + 0x10 * m, -1);
    printBinary(data, 16); cout << " ";

    // SP1 bits 0-31
    data = tapi.dspSingleRead(0, 0, 0, 0x404608 + 0x10 * m, -1);
    printBinary(data, 32, 16);

    data = tapi.dspSingleRead(0, 0, 0, 0x40460c + 0x10 * m, -1);
    printBinary(data, 16); cout << " ";
    cout << endl;
  }
}

void print_dsp_mode_bits(unsigned long mCD) {
  // CmdMask    4 * 2
  // DynMask    8 * 2
  // FmtMask    16
  // valid      1 * 2 * 2
  // lemo       4
  // DeltaMask  2 * 8

  int delta = 64;

  unsigned int readlength;
  unsigned long *mem = tapi.dspBlockRead(0, 0, 0, mCD, delta * 8, -1, &readlength);

  for(int m=0; m<8; m++) {
    cout << "Mask set: " << m << endl;

    // CmdMask

    cout << " CMD Mask 0: ";

    // Mask bits 0-31
    int data = mem[0 + delta * m]; printBinary(data, 32, 16);

    // SP0 bits 32-47
    data = mem[1 + delta * m]; printBinary(data, 16);
    cout << endl;

    cout << " CMD Mask 1: ";
    // SP1 bits 0-31
    data = mem[4 + delta * m]; printBinary(data, 32, 16);

    // SP1 bits 32-47
    data = mem[5 + delta * m]; printBinary(data, 16);
    cout << " \n";

    // DynMask
    for(int f=0; f<8; f++) {
      cout << " Formatter: " << f << " ";
      // Default mode bit 0 and 1
      data = mem[8 + f + delta * m];
      printBinary(data & 0xfff, 12); cout << " ";
      printBinary((data >> 16) & 0xfff, 12); cout << " ";
 
      // Corrective mode bit 0
      data = mem[16 + f + delta * m];
      printBinary(data & 0xfff, 12); cout << " ";
      printBinary((data >> 16) & 0xfff, 12); cout << " ";

      cout << endl;

      // FmtMask
      // Skip fmt Encable ( 24 )
      // Skip fmt Cfg     ( 28 )
    }
 
    // DataLinkMask
    cout << " Data link mask: ";
    data = mem[32 + delta * m];
    printBinary(data, 32, 16);// cout << " ";
    data = mem[33 + delta * m];
    printBinary(data, 32, 16);// cout << " ";
    data = mem[34 + delta * m];
    printBinary(data, 32, 16); cout << " \n";

    // Valid modules
    cout << " Modules:        ";
    data = mem[40 + delta * m];
    printBinary(data, 32, 16);
    data = mem[41 + delta * m];
    printBinary(data, 16); cout << " ";
    data = mem[42 + delta * m];
    printBinary(data, 32, 16);
    data = mem[43 + delta * m];
    printBinary(data, 16); cout << " ";
    cout << endl;

    // Lemo
    data = mem[44 + delta * m];
    cout << " Lemo link: " << data;
    data = mem[45 + delta * m];
    cout << " formatter: " << data << endl;

    // Delta masks
  }
}

void print_dsp_moduleConfig(int dsp) {
  unsigned int readlength;

  unsigned int address;
  if(dsp == -1) address = 0x02029b00;
  else address = 0xa0013420;

  unsigned long *mem = tapi.dspBlockRead(0, 0, 0, address, 2 * 48, dsp, &readlength);
  cout << hex;

  for(int i=0; i<48; i++) {
    cout << "tx: " << (mem[0 + i * 2] & 0xff) << " rx: ";
    int data = mem[1 + i * 2];
    for(int r=0; r<2; r++) {
      cout << ((data >> (r * 8)) & 0xff) << " ";
    }
    cout << endl;
  }
  cout << dec;
}

void print_rod_Status() {
  unsigned int reg;

  // Really a command register
//   reg = readRegister(0x218);

  reg = readRegister(0x217);

  cout << "FE Command output data streams " << ((reg&1)?"on":"off") << endl;
  cout << "Command input source for mask 1 " << ((reg&2)?"TIM":"DSP SP0") << endl;

  cout << "Calibration Trigger Signal Decoder " << ((reg&0x40000)?"Enable":"Idle") << endl;
  cout << "Configuration readback " << ((reg&0x80000)?"Enable":"Idle") << endl;
  cout << "Config/Cal Mask Load " << ((reg&0x100000)?"Enable Load":"Idle") << endl;
  cout << "Static Callibration BCID " << ((reg&0x200000)?"Static BCID":"Dynamic BCID") << endl;

  cout << "ROD type " << ((reg&0x80000000)?"Pixel":"SCT") << endl;

  reg = readRegister(0x21c);

  cout << "ECR ID Counter Value " << ((reg & 0xff)) << endl;
  cout << "FE Trigger FIFO Counter Value " << ((reg & 0x7ff00) >> 8) << endl;

  reg = readRegister(0x21b);

  cout << "TIM Clock " << ((reg&1)?"OK":"***FAIL***") << endl;
  cout << "BOC Clock " << ((reg&2)?"OK":"***FAIL***") << endl;
  cout << "BOC " << ((reg&4)?"Busy":"Ready") << endl;
  cout << "Configuration read back/Trap link data " << ((reg&8)?"Done":"Ready") << endl;

  cout << "Cal test ready " << ((reg&0x10)?"Ready":"Running") << endl;
  cout << "FE Trigger FIFO " << ((reg&0x20)?"Empty":"Not Empty") << endl;
  cout << "FE Trigger FIFO " << ((reg&0x40)?"Full":"Not Full") << endl;
  cout << "Formatter A Mode bits FIFO " << ((reg&0x80)?"Empty":"Not Empty") << endl;

  cout << "Formatter A Mode bits FIFO " << ((reg&0x100)?"Full":"Not Full") << endl;
  cout << "Formatter B Mode bits FIFO " << ((reg&0x200)?"Empty":"Not Empty") << endl;
  cout << "Formatter B Mode bits FIFO " << ((reg&0x400)?"Full":"Not Full") << endl;
  cout << "Header/Trailer limit formatter bank A " << ((reg&0x800)?"Limit":"OK") << endl;

  cout << "Header/Trailer limit formatter bank B " << ((reg&0x1000)?"Limit":"OK") << endl;
  cout << "ROD Busy limit Formatter Bank A " << ((reg&0x2000)?"Busy":"OK") << endl;
  cout << "ROD Busy limit Formatter Bank B " << ((reg&0x4000)?"Busy":"OK") << endl;
  cout << "EFB Dynamic Mask Bits FIFO " << ((reg&0x8000)?"Empty":"Not Empty") << endl;

  cout << "EFB Dynamic Mask Bits FIFO " << ((reg&0x10000)?"Full":"Not Full") << endl;
  cout << "EFB Event ID Empty " << ((reg&0x20000)?"Error":"OK") << endl;
  cout << "Event memory A FIFO " << ((reg&0x40000)?"Empty":"Not Empty") << endl;
  cout << "Event memory A FIFO " << ((reg&0x80000)?"Full":"Not Full") << endl;

  cout << "Event memory B FIFO " << ((reg&0x100000)?"Empty":"Not Empty") << endl;
  cout << "Event memory B FIFO " << ((reg&0x200000)?"Full":"Not Full") << endl;

  cout << "FE Command Pulse Counter 0x" << hex << ((reg&0x3fc00000) >> 22) << dec << endl;

  cout << "FE Occupancy counter " << ((reg&0x40000000)?"All Zero":"Not Zero") << endl;
  cout << "Mode Bits " << ((reg&0x80000000)?"Error":"OK") << endl;

}

void changeVariable(int module, int bank, int chip, int type, float val) {
  cout << "Change variable macro\n";
  tapi.createDebugPrimList();

  unsigned long *buffer = new unsigned long[7];

  buffer[0] = 0;         // _not_ write
  buffer[1] = bank;
  buffer[2] = 8;         // ALL_GROUPS
  buffer[3] = module;
  buffer[4] = chip;
  buffer[5] = type;
  buffer[6] = 1;         // Send text message confirming what was done
  buffer[7] = 1;
  unsigned int intVal = (unsigned int*)(&val);
  buffer[8] = (MDAT32*)intVal;

  tapi.addDebugPrimList(4 + 8, 1, 8208, 102, buffer);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

//    body = tapi.getResponse(0, 0, 0);
  // Clear prim list for next time
  tapi.createDebugPrimList();
}

void setModuleMask(int port, int cfg, unsigned long mask[2], int type, int storage, int maskSet) {
  cout << "Set module mask variable macro\n";
  tapi.createDebugPrimList();

  unsigned long *buffer = new unsigned long[16];
  buffer[0] = 0;
  buffer[1] = port;
  buffer[2] = 0;
  buffer[3] = 0;
  buffer[4] = 0;
  buffer[5] = 0;
  buffer[6] = 0;
  buffer[7] = 0;
  buffer[8] = 0;
  buffer[9] = 0;
  buffer[10] = cfg;
  buffer[11] = mask[0];
  buffer[12] = mask[1];
  buffer[13] = type;
  buffer[14] = storage;
  buffer[15] = maskSet;

  tapi.addDebugPrimList(4 + 16, 1, 10, 101, buffer);
  tapi.sendDebugPrimList(0, 0, 0);
  tapi.awaitResponse(0, 0, 0);

//    body = tapi.getResponse(0, 0, 0);
  // Clear prim list for next time
  tapi.createDebugPrimList();
}

void initMaskSet(int maskSet) {
  unsigned long masks[2] = {0, 0};
  setModuleMask(2, 1, masks, 0xfff, 0, maskSet); 
  // int port, int cfg, unsigned long mask[2], int type, int storage, int maskSet) {
}

void doEyePlot() {
  int histo[25][2][2];

  for(int delay = 0; delay < 25; delay++) {
    // Set BOC delay

    for(int event = 0; event < 100; event++) {
      readLinksToFifos(25, 100);

      unsigned long mask = 0x1;

      // Examine clock by two
      for (int i=0; i<100/2; i++) {
        if(scratch[i*6 + 0] & mask) {
          histo[delay][0][0] ++;
        } else {
          histo[delay][0][1] ++;
        }

        if(scratch[i*6 + 3] & mask) {
          histo[delay][1][0] ++;
        } else {
          histo[delay][1][1] ++;
        }
      }
    }
    cout << histo[delay][0][0] << " ";
    cout << histo[delay][0][1] << " ";
    cout << histo[delay][1][0] << " ";
    cout << histo[delay][1][1] << endl;
  }
}

void rawScan() {
  int histo[25][1536];

  for(int delay = 0; delay < 25; delay++) {
    tapi.modifyBOCParam(0); // Set BOC delay

    for(int event = 0; event < 20; event++) {
      readLinksToFifos(25, 1536);

      unsigned long mask = 0x1;

      // Examine clock by two
      for (int i=0; i<1536; i++) {
        if(scratch[i*3 + 0] & mask) {
          histo[delay][i] ++;
        }
      }
    }
  }

  for(int i=0; i<25; i++) {
    cout << histo[i][0] << " ";
    cout << histo[i][1] << " ";
    cout << histo[i][2] << " ";
    cout << histo[i][3] << endl;
  }
}

void printEventTrap() {
  cout << "Trap command:\n";
  for(int i=0; i<4; i++) {
    int command = readRegister(0x172+i, false) & 0x7f;
    cout << command << " ";
    command = readRegister(0x176+i, false) & 0x7;
    cout << command << "  ";
  }
  cout << endl;

  cout << "Trap status:\n";
  for(int i=0; i<4; i++) {
    int status = readRegister((0x17e)+i, false) & 0xff;
    cout << status << " ";
  }
  cout << endl;

  cout << "Trap match:\n";
  for(int i=0; i<4; i++) {
    int match = readRegister(0x182+i, false) & 0xff;
    cout << match << " ";
    match = readRegister(0x18a+i, false) & 0xff;
    cout << match << " ";
  }
  cout << endl;

  cout << "Trap rem/mod:\n";
  for(int i=0; i<4; i++) {
    int modrem = readRegister(0x186+i, false) & 0xffff;
    cout << modrem << " ";
    modrem = readRegister((0x18e)+i, false) & 0xffff;
    cout << modrem << " ";
  }
  cout << endl;

  cout << "Trap FIFO count:\n";
  for(int i=0; i<4; i++) {
    int count = readRegister(0x196+i, false) & 0x3ff;
    cout << count << " ";
  }
  cout << endl;

  cout << "Trap mask delay:\n";
  for(int i=0; i<4; i++) {
    int delay = readRegister(0x1a2+i, false) & 0x3f;
    cout << delay << " ";
  }
  cout << endl;
}

void printEventTrap2() {
  cout << "Trap command:\n";
  for(int i=0; i<4; i++) {
    int command = readRegister(0x1c2+i, false) & 0x7f;
    cout << command << " ";
    command = readRegister(0x1c6+i, false) & 0x7;
    cout << command << "  ";
  }
  cout << endl;

  cout << "Trap status:\n";
  for(int i=0; i<4; i++) {
    int status = readRegister((0x1ce)+i, false) & 0xff;
    cout << status << " ";
  }
  cout << endl;

  cout << "Trap match:\n";
  for(int i=0; i<4; i++) {
    int match = readRegister(0x1d2+i, false) & 0xff;
    cout << match << " ";
    match = readRegister(0x1da+i, false) & 0xff;
    cout << match << " ";
  }
  cout << endl;

  cout << "Trap rem/mod:\n";
  for(int i=0; i<4; i++) {
    int modrem = readRegister(0x1d6+i, false) & 0xffff;
    cout << modrem << " ";
    modrem = readRegister((0x1de)+i, false) & 0xffff;
    cout << modrem << " ";
  }
  cout << endl;

  cout << "Trap FIFO count:\n";
  for(int i=0; i<4; i++) {
    int count = readRegister(0x1e6+i, false) & 0x3ff;
    cout << count << " ";
  }
  cout << endl;

  cout << "Trap mask delay:\n";
  for(int i=0; i<4; i++) {
    int delay = readRegister(0x1f2+i, false) & 0x3f;
    cout << delay << " ";
  }
  cout << endl;
}

void rodModeCapture(int delay, int units) {
  tapi.rodMode(0, 0, 0, 0x00040002, 0, 1, units, delay, 1);
  sendL1A();

  unsigned long *bufferA = readFifo(0, 0, units, false);
  unsigned long *bufferB = readFifo(0, 1, units, false);

  cout << hex;
  cout.fill('0');
  for(int i=0; i<units/2; i++) {
    cout.width(8); cout << bufferA[i*3 + 0] << " ";
    cout.width(8); cout << bufferA[i*3 + 1] << " ";
    cout.width(8); cout << bufferA[i*3 + 2] << " ";
    cout.width(8); cout << bufferB[i*3 + 0] << " ";
    cout.width(8); cout << bufferB[i*3 + 1] << " ";
    cout.width(8); cout << bufferB[i*3 + 2] << " ";
    cout << endl;
  }
  cout << dec;
  cout.fill(' ');
}

void decodeConfig() {
  unsigned int readlength;
  unsigned long *config = tapi.dspBlockRead(0, 0, 0, 0x2102000, 270 * 1, -1, &readlength);

  int pos = 0;
  while(pos < readlength) {
    if((config[pos] & 0x5760000) != 0x5700000) {
      cout << "Strange data: ";
      cout << hex << " " << pos << " " << config[pos] << dec << endl;
      break;
    } 

    unsigned int f3 = (config[pos] & 0xff000) >> 12;
    unsigned int f4 = (config[pos] & 0x00fc0) >> 6;
    unsigned int f5 = (config[pos] & 0x0003f) >> 0;

    pos ++;

    cout << "Chip address " << f4 << ": ";
    cout << hex << f3 << "/" << f5 << dec << ": ";

    switch(f3) {
    case 0x1c:
      unsigned short data = (0xffff0000 & config[pos]) >> 16;
      switch(f5) {
      case 0:
        // Configuration reg
        cout << "Config: " << hex << data << dec;
        cout << " comp " << (data & 3) 
             << " cal "  << ((data & 0xc) >> 2)
             << " trim "  << ((data & 0x30) >> 4)
             << " edge "  << ((data & 0x40) >> 6)
             << " mask "  << ((data & 0x80) >> 7)
             << " acc "  << ((data & 0x100) >> 8)
             << " in_b " << ((data & 0x200) >> 9)
             << " out_b " << ((data & 0x400) >> 10)
             << " mast " << ((data & 0x800) >> 11)
             << " end " << ((data & 0x1000) >> 12)
             << " feed " << ((data & 0x2000) >> 13)
             << endl;
        break;
      case 0x10:
        cout << "Strobe delay: " << hex 
             << (data & 0x3f) << dec << endl;
        break;
      case 0x18:
        cout << "Threshold/Cal: " << hex
             << ((data & 0xff00) >> 8) << "/" 
             << (data & 0xff) << dec << endl;
        break;
      case 0x38:
        cout << "preamp/shaper: " << hex
             << ((data & 0xff00) >> 8) << "/" 
             << (data & 0xff) << dec << endl;
        break;
      case 0x04:
        cout << "\r";
//         cout << "TRIM DAC: " << hex
//              << (data & 0xf) << " " 
//              << ((data & 0x7f0) >> 4) << dec << endl;
        break;
      }
      pos ++;
      break;
    case 0x8c:
      // Mask reg
      cout << "Mask ";
      cout << hex;
      for(int i=0; i<4; i++) {
        cout << config[pos++] << " ";
      }
      cout << dec;
      cout << endl;
      pos++;
      break;
    case 0x0c:
      cout << "Something else\n";
      break;
    }
  }
}

void testConfigVariable(float start, float stop, float step, int var) {
  for(float val = start; val < stop; val += step) {
    tapi.modifyABCDVarROD(var, val, 0);
    tapi.sendABCDModule(0, 0);
    tapi.decodeConfig(0, 0, 0, true, false);
  }
}

void setSlaveMemory(int s, long add, int words, int value) {
  unsigned long *buffer = new unsigned long[3];

  buffer[0] = add;
  buffer[1] = words;
  buffer[2] = value;

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 3, 1, 4, 100, buffer);
  tapi.sendDebugSlavePrimList(0, 0, 0, s, 1, 1);
  tapi.awaitResponse(0, 0, 0);
  // Clear prim list for next time
  tapi.createDebugPrimList();

  cout << "Slave data sent\n";
}

void setNmask(int n) {
  // Put all modules in send all mask mode
  tapi.modifyABCDVar(11, 1.0);
  tapi.modifyABCDVar(13, 0.0);
  tapi.modifyABCDVar(14, 1.0);   
  tapi.modifyABCDVar(9, (double)n);
  tapi.setABCDModules(0);
  tapi.sendABCDModules(0);
}

void printBinary(unsigned int data, int length, int divisions) {
  // Actually -ve length not used...
  if(length < 0) {
    for(int i=-length-1; i>=0; i--) {
      int val = data & (1<<i);
      cout << (val?'1':'0');
      if(divisions && ((i%divisions) == 0)) cout << " ";
    }
  } else {
    for(int i=0; i<length; i++) {
      int val = data & (1<<i);
      cout << (val?'1':'0');
      if(divisions && ((i%divisions) == divisions-1)) cout << " ";
    }
  }
}

void setTrigger(int bin, int slave) {
  unsigned long *buffer = new unsigned long[30];

  buffer[18] = 0xf;  // slvbits
  
  buffer[19] = 0;
  buffer[20] = 0;
  buffer[21] = bin;
  buffer[22] = 0;  // Set
  buffer[23] = 0;  // Reps
  buffer[24] = 0;  // Interval
  buffer[25] = 0;  // preBDO
  buffer[26] = 0;
  buffer[27] = 0;  // incCmd
  buffer[28] = 0;  // incData
  buffer[29] = 0;  // incData

  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 30, 1, 11, 102, buffer);
  tapi.sendDebugSlavePrimList(0, 0, 0, slave, 1, 0);
  tapi.awaitResponse(0, 0, 0);
  // Clear prim list for next time
  tapi.createDebugPrimList();

  cout << "Set bin to " << bin << endl;
}

void sendData(int type, int dsp) {
  unsigned long data[] = {type, 0, 0, 0};
  tapi.createDebugPrimList();
  tapi.addDebugPrimList(4 + 4, 2, 9, 103, data);

  if(dsp == -1) {
    tapi.sendDebugPrimList(0, 0, 0);
  } else {
    tapi.sendDebugSlavePrimList(0, 0, 0, dsp, 1, 1);
  }
  tapi.awaitResponse(0, 0, 0);

  unsigned long *result = tapi.getResponse(0, 0, 0);

  if(result) {
    if(dsp == -1) {
      cout << "pointer " << (unsigned long)(result[8]) << endl;
      cout << "length " << (unsigned long)(result[9]) << endl;

      tapi.dspBlockDumpFile(0, 0, 0, (unsigned long)(result[8]), (unsigned long)(result[9]), -1, "SendDataDump.bin");
    } else {
      cout << "pointer " << (unsigned long)(result[16]) << endl;
      cout << "length " << (unsigned long)(result[17]) << endl;
      tapi.dspBlockDumpFile(0, 0, 0, (unsigned long)(result[16]), (unsigned long)(result[17]), dsp, "SendDataDump.bin");
    }
  }
}
