#ifndef SCTAPI_CRATE_IMPL
#define SCTAPI_CRATE_IMPL

// Make sure all compilations have same Crate structure!
#include "config.h"

#include <list>
#include <map>
#include <memory>
#include <mrs/message.h>

#include <boost/thread.hpp>
#include <boost/shared_ptr.hpp>

#include "crate.h"

#if USE_THREADS
enum RodPrimState { MYPRIM_STATE_LOADED, MYPRIM_STATE_EXECUTING, MYPRIM_STATE_COMPLETE, MYPRIM_STATE_IDLE };
#endif

namespace SctPixelRod {
  class RodOutList;
  class RodModule;
  class TimModule;
  class VmeInterface;
}

namespace SctApi {
  class PrimListWrapper;

  struct RodStatus {
#if USE_THREADS
    std::list<boost::shared_ptr<PrimListWrapper> > listQueue;
    boost::mutex mutex;
    RodPrimState currState;
#endif
    int slaves;

    std::map<BankType, std::vector<SctConfiguration::BOCChannelConfig> > savedBocSetup;
    std::map<BankType, boost::shared_ptr<SctConfiguration::BOCGlobalConfig> > savedBocRegisters;
  };

/**
   A representation of the crate controller.

   This is used by SCTAPI to send commands to RodModule
   objects. These are stored in crate memory. 

   Module configuration will eventually be stored here 
   also?
 */
class CrateImpl : public Crate {
  // Don't use copy constructor or assignment operator
  CrateImpl(const CrateImpl&);
  CrateImpl &operator=(const CrateImpl&);
 public:
  /// The constructor. 
  /**
     Tell this crate which partition it belongs to.
     This creates a VmeInterface.
  */
  CrateImpl(unsigned int partition, unsigned int crate, 
            boost::shared_ptr<SctConfiguration::Configuration> newConf = boost::shared_ptr<SctConfiguration::Configuration>() );
  /// The destructor
  /**
     Delete all the RodModules and VmeInterface 
  */
  ~CrateImpl();

  /************** Interaction with Configuration **************/

  /// Initialise TIM, RODs and BOCs in this crate
  /**
     Return number of RODs initialised
   */
  int initialiseCrate();

  /// Initialise the TIM from the configuration.
  /**
     Return 0 for success
   */ 
  int initialiseTim();

  /// Initialise a ROD from the configuration.
  /**
     Store object in local map.
     Return 0 for success
   */ 
  int initialiseRod(unsigned int rod);

  /// Initialise a BOC from the configuration.
  /**
     Return 0 for success
  */ 
  int initialiseBOC(unsigned int rod);

  /// Configure BOC parameters from the configuration
  /**
     Return 0 for success
   */
  int configureBOC(unsigned int rod);

  /*************** Primlists *********************/

  /// Send a primitive list to a ROD
  /**
     Return 0 for success
  */
  int sendPrimList(unsigned int rod, boost::shared_ptr<PrimListWrapper> prim);

  /// Send a primitive list to all RODs in a crate
  /**
     Return 0 for success
  */
  int sendPrimListAll(boost::shared_ptr<PrimListWrapper> prim);

  /// Await a response from a ROD
  /**
     Await response from a ROD within this crate.
     Handle text messages sent by the ROD (print to stdout)

     If waiting longer than timeout seconds then return anyway
  */
  int awaitResponse(unsigned int rod, int timeout);

  /// Await a response from all RODs in a crate
  /**
     Handle text messages sent by the RODs (print to stdout)

     If waiting longer than timeout seconds then return anyway
  */
  int awaitResponseAll(int timeout);

  /// Get the response from a ROD
  /**
     Return the response to the last primlist (a wrapped copy)
   */
  boost::shared_ptr<SctPixelRod::RodOutList> getResponse(unsigned int rod);

  void setMrsStream(MRSStream *stream);

  /***************** Messages ******************/

  /// Get a message in the text buffer
  /**
     @param rod Index of ROD
     @param buffer Buffer to write text into
     @param length Length of buffer allocated, on return contains amount of data copied.

     Return message from ROD if present
   */
  bool getRodMessage(unsigned int rod, char *buffer, unsigned long &length);

  /************** Misc read/writes *****************/

  /// Read data from the master DSP's memory
  /**
     Proxy for RodModule method
  */
  int mdspBlockRead(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords);

  /// Read data from a slave DSP's memory
  /**
     Proxy for RodModule method
  */
  int slvBlockRead(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords, long slaveNumber);

  /// Write data direct to a slave DSP's memory
  /**
     Proxy for RodModule method
  */
  int slvBlockWrite(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords, long dspNumber);

  /// Write block of master dsp memory
  /**
     Proxy for RodModule method
   */
  int mdspBlockWrite(unsigned int rod, long dspStart, unsigned long *buffer, unsigned long numWords);

  /// Read Status register 
  /**
     Proxy for RodModule method
  */
  unsigned long readRodStatusReg(unsigned int rod, long regNumber);


  /// Read command register
  /**
     Proxy for RodModule method
  */
  unsigned long readRodCommandReg(unsigned int rod, long regNumber);

  /// Read single word from ROD
  /**
     Proxy for RodModule method
  */
  unsigned long dspSingleRead(unsigned int rod, const unsigned long dspAddr, long dspNumber);

  /// Write single word
  /**
     Proxy for RodModule method
  */
  int dspSingleWrite(unsigned int rod, unsigned long dspAddr, unsigned long buffer, long dspNumber);

  /// Print status of rods on crate 
  void status();

  /// Control the primitive list running on a rod (Untested)
  void primListControl(unsigned int rod, PLControl function);

  /// Print all BOC paramters as set up
  void printBOCSetup(unsigned int rod);

  /// Return channel parameters as stored in BOC
  std::vector<SctConfiguration::BOCChannelConfig> currentBOCSetup(unsigned int rod);

  /// Print all BOC registers
  void printBOCRegisters(unsigned int rod);

  /// Return global registers as stored in BOC
  SctConfiguration::BOCGlobalConfig currentBOCRegisters(unsigned int rod);

  /// Save BOC channel setup
  void saveBOCSetup(unsigned int rod, BankType bank);

  /// Save BOC register setup
  void saveBOCRegisters(unsigned int rod, BankType bank);

  /// Restore BOC channel setup
  void restoreBOCSetup(unsigned int rod, BankType bank);

  /// Restore BOC register setup
  void restoreBOCRegisters(unsigned int rod, BankType bank);

  /// Check BOC laser interlock
  bool checkBOCLasersOn(unsigned int rod);

  /// Setup BOC to sample at 20MHz
  void enterBOCClockBy2Mode(unsigned int rod);

  /// Restore BOC to 40MHz mode
  void leaveBOCClockBy2Mode(unsigned int rod);

  /// Return 12 monitor values from BOC ADCs
  std::vector<double> getBOCMonitorArray(unsigned int rod);

  /// Modify parameter on all BOC channels
  /**
     @param raw If false then map module channel to BOC, otherwise channel refers to BOC channel
   */
  void modifyBOCParam(unsigned int type, unsigned int val, bool raw);

  /// Modify parameter on BOC channel
  void modifyBOCParam(unsigned int rod, unsigned int channel, unsigned int type, unsigned int val, bool raw);

  /// Turn all BOC laser currents to 0
  void lasersOff();

  /* ******************* TIM Functions ***************************/

  /// Set the TIM trigger frequency
  /**
     @param trigFrequency Frequency of triggers
     @param rstFrequency Frequency of resets
   */
  void timSetFrequency(double trigFreq, double rstFreq);

  /// Start the TIM generating regular triggers 
  void freeTriggers();

  /// Stop the TIM generating regular triggers
  void stopTriggers();

  /// Tell the TIM to send an L1A 
  void timL1A();

  /// Tell TIM to send Cal + L1A
  void timCalL1A(int delay);

  void timECR();

  void timBCR();

  void timFER();

  /// Tell TIM to send a burst of triggers
  void sendTimBurst(int count);

  /// Print all the Tim registers
  void timVerbose();

  /// Debug, load TIM register
  void timRegLoad(int reg, UINT16 val);

  /// Debug, read TIM register
  UINT16 timRegRead(int reg);

  /// Check if a slave has been started
  bool slavePresent(int rod, int index) const;

  /// Notify that slave was started externally...
  void slaveStarted(int rod, int slave);

  /// Check if a ROD has been configured
  bool RODPresent(int rod) const;

  long getRodSlot(int rod) const;

  /// Get ROD Revision
  int getRodRevision(int rod) const;

  /// Stop the primitive list polling 
  void stopPolling();

  /// Restart the primitive list polling
  void resumePolling();

 private:
  void rodInitialiseThread(int rod);

  /// Turn off all tx channels except channel 
  void setupScanTx(unsigned int rod, unsigned int channel);

  /// Turn off all tx channels except channel (Don't map the channel numbers)
  void setupScanRawTx(unsigned int rod, unsigned int channel);

  /// Lookup RodModule object
  /**
     Return 0 if not found
  */
  SctPixelRod::RodModule *getRod(unsigned int rod) const;

  const RodStatus &getRodData(unsigned int rod) const;
  RodStatus &getRodData(unsigned int rod);

  /// Get a text buffer from a ROD
  void doTextBuffer(SctPixelRod::RodModule &rod0);

#if USE_THREADS
  void pollingLoop();
#endif

  /// Shortcut for rod map
  typedef std::map<unsigned int, std::pair<RodStatus *, SctPixelRod::RodModule *> > RodMap;

  /// Map to lookup RodModule objects
  RodMap rodMap;

  /// TIM module
  SctPixelRod::TimModule *tim;

  /// The configuration
  boost::shared_ptr<SctConfiguration::Configuration> config;

  /// The vmeinterface that is used by all rods and tims
  SctPixelRod::VmeInterface *vme;

  /// Which partition this crate is in
  const unsigned int partition;

  /// The id of this crate
  const unsigned int crate;

  MRSStream *mrs;

  // Leave this in for no threads (its nice and small...)
  bool m_stopPolling;

  // Whether polling should be done
  bool m_enablePolling;

#if USE_THREADS
  /// Queue for prim lists to go to all RODs
  std::list<boost::shared_ptr<PrimListWrapper> > listQueueAll;

  /* **********  Mutexes  ********** */
  boost::mutex vmeMutex;

  boost::condition primQueue_notEmpty;
  boost::mutex primQueue_mutex;

  /* ******* Threads ******/
  std::auto_ptr<boost::thread> pollThread;

  // So ROD initialisation threads can count RODs independently
  int rodCounter;
#endif
};

}  // End of namespace SctApi
#endif
