#include <unistd.h>
#include <time.h>

#include <iostream>
#include <fstream>

#include "Sct/SctNames.h"
#include "Sct/Env.h"

#include "RodCrate/RodModule.h"
#include "RodCrate/BocCard.h"
#include "RodCrate/TimModule.h"              // For TimMaskFrequency
// #include "VmeInterface/VmeInterface.h"

#include "sctConf/configXMLImpl.h"

#include "../VmeInterface/DummyVmeInterface.h"
#ifndef USE_DUMMY_VME
#include "../VmeInterface/RCCVmeInterface.h"
#endif

#include "VmeDecode.h"

#include "SctApiDebug.h"

#include "SctApiConsts.h"
#include "crateImpl.h"

#include "extraScans.h"

#ifdef USE_THREADS
#include <boost/bind.hpp>
#endif

// For modify BOC params
#include "ABCD/ABCDscans.h"

using namespace SctPixelRod;
using namespace SctConfiguration;
// All std namespace are qualified explicitly

namespace SctApi {
/*
The constructor. Tell this crate which partition it belongs to.
This creates a VmeInterface.
*/
CrateImpl::CrateImpl(unsigned int partition, unsigned int crate, 
                     boost::shared_ptr<Configuration> newConf) : 
  rodMap(), tim(0), config(newConf), vme(0), 
  partition(partition), crate(crate), mrs(0), 
  m_stopPolling(false), m_enablePolling(true)
{
  // Create VME interface
  std::cout << "Initialising VME interface ... \n";

  if(config->isDummyCrate(partition, crate)) { 
    vme = new SctPixelRod::DummyVmeInterface();

    if(mrs) {
      *mrs << "CRATE_DUMMY"  << MRS_WARNING << MRS_QUALIF("SCTAPI") << MRS_TEXT("Using DUMMY interface") << ENDM;
    }
#ifdef USE_DUMMY_VME
  } else {
    if(mrs) {
      *mrs << "CRATE_DUMMY"  << MRS_ERROR << MRS_QUALIF("SCTAPI") << MRS_TEXT("Non dummy crate requested, support not compiled!") << ENDM;
    }

    throw CrateException("RCC support not compiled");
  }
#else
  } else {
    vme = new SctPixelRod::RCCVmeInterface();
    if(vme->getLastErrcode() == 515) {
      // Failed to open interface (probably not running on an SBC!)
      std::cerr << "Interface not opened error, check host and VME drivers. Crate initialisation failed!!\n";

      if(mrs) {
        *mrs << "CRATEINIT_FAILED" << MRS_ERROR << MRS_QUALIF("SCTAPI") << MRS_TEXT("Crate initialisation failed (VME interface)") << ENDM;
      }

      throw CrateException("Interface not found");
    }
    //    std::cout << "Last error " << vme->getLastErrcode() << " (" << vme->getErrorMessage(vme->getLastErrcode()) << std::endl;

    if(mrs) {
      *mrs << "CRATEINIT_SUCCESS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") << MRS_TEXT("Successfully initialised VME interface") << ENDM;
    }
  }
#endif

  std::cout << " ... Done\n";

  if(config.get() == 0) {
    std::cout << "Load configuration\n";
    config.reset(new ConfigurationXMLImpl());
  }

#if USE_THREADS
  pollThread.reset(new boost::thread(boost::bind(&CrateImpl::pollingLoop, this)));
#endif
}

/*
The destructor. Delete all the RodModules and VmeInterface
*/
CrateImpl::~CrateImpl() {
  // First make sure the polling is not using them
#if USE_THREADS
  m_stopPolling = true;
  pollThread->join();
#endif

  for(RodMap::const_iterator i=rodMap.begin();i!=rodMap.end();i++){
    std::cout << "RodModule on shutdown " << *(i->second.second) << std::endl;
    delete i->second.first;    //RodStatus
    delete i->second.second;   //RodModule
  }
  delete tim;

  delete vme;
}

void CrateImpl::rodInitialiseThread(int rod) {
  if(initialiseRod(rod) != 0) {
    std::cout << "Initialisation of rod  " << rod << " failed!\n";
    return;
  }

  rodCounter ++;

  if(initialiseBOC(rod) == 0) {
    if(Debug::getInstance()->checkDebugOption(DEBUG_DIAG)) {
      std::cout << "Should have initialised BOC now" << std::endl;
    }
  } // Else unsuccessful
}

int CrateImpl::initialiseCrate() {
  if(!initialiseTim()) {
    std::cout << "TIM initialise successful\n";
  }

  std::list<unsigned int> rods;
  try {
    rods = config->listRodsInCrate(partition, crate);
  } catch(ConfigurationException &c) {
    std::cout << "No RODs in this crate\n";
    std::cout << c.what() << std::endl;

    return 0;
  }

  std::cout << "Found configuration for " << rods.size() << " rods\n";

  std::list<boost::shared_ptr<boost::thread> > rodInitThreads;

  rodCounter = 0;

  for(std::list<unsigned int>::const_iterator r=rods.begin(); 
      r!=rods.end(); 
      r++) {

    rodInitThreads.push_back(boost::shared_ptr<boost::thread>(new boost::thread(boost::bind(&CrateImpl::rodInitialiseThread, this, *r))));
  }

  for(std::list<boost::shared_ptr<boost::thread> >::const_iterator thread = rodInitThreads.begin();
      thread != rodInitThreads.end(); 
      thread ++) {

    if(*thread) {
      (*thread)->join();
    } else {
      std::cout << " Null pointer to thread!\n";
    }
  }

  int totalRods = rodCounter;

  return totalRods;
}


/*
  Create a TimModule object using information from the Configuration.
*/
int CrateImpl::initialiseTim() {
  std::cout << "Load configuration for tim\n";

  TimConfig timConf;
  try {
    timConf = config->getTimConfig(partition, crate);
  } catch(ConfigurationException &c) {
    tim = 0;
    return -1;
  }

  try {
    // Create TimModule and initialize it
    std::cout << "Initialise TIM\n";

    tim = new SctPixelRod::TimModule(timConf.baseAddress, timMapSize, *vme);

    std::cout << "Constructed\n";

    tim->reset();
    tim->initialize();

    std::cout << "Initialised\n";

    std::cout << "Started TIM\n";
    tim->status();

    // Switch TIM trigger frequency to something not the default (600kHz!)
    timSetFrequency(timConf.trigFrequency, timConf.resetFrequency);
  } catch (SctPixelRod::TimException &r) {
    tim = 0;

    std::cerr << "While initialising TIM " << partition << " " << crate << ":\n";
    std::cerr << "Got TimException: " << std::endl;
    std::cerr << r.getDescriptor() << ", " << r.getData1() << ", " << r.getData2() << std::endl;
    std::cerr << "Continuing...\n";
  } catch (VmeException &v) {
    tim = 0;

    std::cout << "VmeException initialising TIM:\n";
    Utility::decodeVme(v);

    return -9;
  } catch (...) {
    tim = 0;

    std::cerr << "Unknown exception initialising TIM\n";

    return -6;
  }

  return 0;
}

/*
  Create a RodModule object using information from the Configuration.
*/
int CrateImpl::initialiseRod(unsigned int rod) {
  int returnValue = 0;

  char *returnReason = "none  specified";

  int slave = 0;

  // 1 is not a valid address!
  unsigned long rodAddress=1;

  std::cout << "Load configuration for rod\n";

  if(config->isDummyCrate(partition, crate)) {
    // Don't create a RODModule!
    //  But return success so a ROD is counted
    return 0;
  }


  SctPixelRod::RodModule* rodObject;

  try {
    SctConfiguration::RodConfig rodConf;

    rodConf = config->getRodConfig(partition, crate, rod);

    rodAddress = rodConf.baseAddress;

    // Create RodModule and initialize it
    std::cout << "Initialise ROD " << rod << " at 0x" << std::hex << rodConf.baseAddress << std::dec << std::endl;

    rodObject = new SctPixelRod::RodModule(rodConf.baseAddress, 
                                           rodMapSize, 
                                           *vme, 
                                           numSlaves);

    std::cout << "Constructed\n";

    if(rodConf.resetLevel > 1) {
      std::cout << "*** ROD reset level " << rodConf.resetLevel << " (not reset) **** \n";
      std::cout << "*** Debug, ROD not reset **** \n";
      rodObject->initialize(false);  // Don't reset the ROD!
    } else {
      std::cout << "Doing full ROD reset...\n";
      // The true is a new feature to reset the ROD (what it always used to do!)
      rodObject->initialize(true);   // Use true to reset the ROD!
      std::cout << " ... done\n";
    }

    std::cout << "Initialised\n";

    std::cout << "Started ROD\n";

    rodObject->status();

    while(rodObject->textHandler() == TEXT_RQ_SET) {
      doTextBuffer(*rodObject);
      // textstate now TEXT_IDLE, check for multiple text buffers in while statement
    }

    RodStatus *currStatus = new RodStatus;
    currStatus->slaves = 0;
#if USE_THREADS
    currStatus->currState = MYPRIM_STATE_IDLE;
    //     std::cout << "STATE: Initialising state to IDLE\n";
#endif

#if USE_ROD_MODULE_SLAVES
    int startedSlaves = 0;

    // Initialize slaves requested by config
    try {
      for(slave=0;  slave < rodConf.numSlaves; slave++) {
        if(rodConf.slaves[slave].ipramFile != "") {
          string ip(rodConf.slaves[slave].ipramFile);
          ip = Sct::Env::substituteVariables(ip);

          string id(rodConf.slaves[slave].idramFile);
          id = Sct::Env::substituteVariables(id);

          string ext(rodConf.slaves[slave].extFile);
          ext = Sct::Env::substituteVariables(ext);

          std::cerr << "Starting slave " << slave << std::endl;
          std::cerr << "((" << ip << ", " << id << ", " << ext << ")\n";

          rodObject->initSlaveDsp(ip, id, ext, slave, 'v');

          RodOutList *outList = rodObject->getOutList();
          if(outList) {
            unsigned long* outBody = outList->getBody();

            if(outBody) {
              if(outBody[3] != PRIM_LIST_REVISION) {
                std::cout << "Prim list revision disagrees\n";
              }

              if(outBody[6] != START_SLAVE_EXECUTING) {
                std::cout << "Prim ID not start slave!\n";
              }

              if(outBody[8] != slave) {
                std::cout << "Start slave says slave 0x" << std::hex << outBody[8] << std::dec << " started not " << slave << std::endl;
              }
            } else {
              std::cout << "No body from start slave outlist!\n";
            }
          } else {
            std::cout << "No response from start slave!\n";
          }
          startedSlaves ++;
          currStatus->slaves |= 1<<slave;
        }
      }
    } catch (SctPixelRod::NoImageFile &f) {
      std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << " slave " << slave << ":\n";
      std::cerr << "No image file exception " << f.getFileName() << std::endl;
      std::cerr << "Continuing...\n";

      if(startedSlaves < 1) {
        *mrs << "ROD Slave Initialisation Failure" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
             << MRS_PARAM<const char *>("Filename", f.getFileName().c_str())
             << ENDM;
      }
    } catch (SctPixelRod::RodException &r) {
      std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << " slave " << slave << ":\n";
      std::cerr << "Bad ROD exception: " << std::endl;
      std::cerr << r.getDescriptor() << ", " << r.getData1() << ", " << r.getData2() << std::endl;
      std::cerr << "Continuing...\n";
    }

    std::cout << "Started " << startedSlaves << " ROD slave DSPs\n";
#endif

    rodMap.insert(std::make_pair(rod, std::make_pair(currStatus, rodObject)));

    std::cout << "ROD " << rod << " initialised\n";

    if(mrs) {
      *mrs << "ROD_INIT" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("number", rod)
           << MRS_PARAM<int>("slot", rodObject->getSlot())
           << MRS_TEXT("ROD Initialised")
           << ENDM;
    }
  } catch(ConfigurationException &c) {
    std::cout << "Configuration error for rod " << rod << "\n";
    std::cout << "\t" << c.what() << std::endl;
    returnValue = -1;
    returnReason = "Error in configuration database";
  } catch (SctPixelRod::PrimListException &p) {
    std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << ":\n";
    std::cerr << "Exception sending primlist:\n";
    std::cerr << p.getDescriptor() << " " << p.getData1() << ", " << p.getData2() << std::endl;

    returnReason = "PrimList exception";
    returnValue = -2;
  } catch (SctPixelRod::HpiException &h) {    
    std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << ":\n";
    std::cerr << "HpiException:\n";
    std::hex(std::cerr);
    std::cerr << h.getDescriptor() << '\n';
    std::cerr << "calcAddr: " << h.getCalcAddr() << ", readAddr: " << 
      h.getReadAddr() << '\n';
    std::dec(std::cerr);

    returnReason = "HPI exception";
    returnValue = -3;
  } catch (SctPixelRod::RodException &r) {
    std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << ":\n";
    std::cerr << "Rod Exception\n";
    std::cerr << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
              << '\n';

    returnReason = "ROD exception";
    returnValue = -4;
  } catch (SctPixelRod::NoImageFile &f) { // This will be caught in the slave section!
    std::cerr << "While initialising ROD " << partition << " " << crate << " " << rod << " slave " << slave << ":\n";
    std::cerr << "No image file exception " << f.getFileName() << std::endl;

    returnReason = "NoImageFile exception";
    returnValue = -5;
  } catch (VmeException &v) {
    std::cout << "VmeException initialising ROD:\n";
    Utility::decodeVme(v);

    returnReason = "VmeException";

    returnValue = -9;
  } catch (...) {
    std::cerr << "Unknown exception initialising ROD\n";
    returnValue = -6;
  }

  if(returnValue < 0) {
    if(mrs) {
      *mrs << "RODINIT_FAIL" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_TEXT("ROD init failure")
           << MRS_PARAM<const char *>("reason", returnReason)
           << MRS_PARAM<int>("number", rod);
      if(rodAddress != 1) 
        *mrs << MRS_PARAM<int>("address", rodAddress);
      *mrs << ENDM;
    }
  }

  return returnValue;
}

/*
  Initialise a BOC using information from the Configuration.
*/
int CrateImpl::initialiseBOC(unsigned int rod) {
  BocCard* bocCard;

  try {
    //     SctConfiguration::RodConfig rodConf = config->getRodConfig(partition, crate, rod);

    //     if(!rodConf.bocPresent) {
    //       std::cout << "Skipping BOC (configuration says not present)\n";
    //       return -2;
    //     }
  } catch(ConfigurationException &c) {
    std::cout << "Request to initialise BOC with unconfigured rod\n";
    return -1;
  }

  if(RODPresent(rod)) {
    try {
      std::cout << "Initialise BOC " << rod << std::endl;

      bocCard = getRod(rod)->getBocCard();

      if(bocCard == 0) {
        // Jumper on ROD set for no BOC 
        return -2;
      }

      bocCard->initialize();
      bocCard->status(); // temporary

    } catch (VmeException &v) {
      std::cout << "VmeException initialising BOC:\n";
      Utility::decodeVme(v);
      return -9;
    } catch (BocException &b) {
      std::cout << "BocException initialising BOC:\n";
      std::cout << b << std::endl;
      return -9;
    } catch (...) {
      std::cerr << "Unknown exception initialising BOC\n";
      return -6;
    }

    configureBOC(rod);
  } else {
    std::cout << "Request for non-existent BOC " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }

  return 0;
}

/*
  Set up a BOC parameters according to information from the Configuration.
*/
int CrateImpl::configureBOC(unsigned int rod) {
  if(!RODPresent(rod) || getRod(rod)->getBocCard() == 0) {
    std::cout << " configureBOC called on missing ROD/BOC " << rod << std::endl;
    throw CrateException("Trying to configure missing BOC");
  }

  BocCard *bocCard = getRod(rod)->getBocCard();


  std::cout << " *************  Turning lasers on ROD " << rod << " ON  ***************\n";

  BOCGlobalConfig globalConf = config->getBOCGlobalConfig(partition, crate, rod);
  for(int r=0; r<9; r++) {
    if(globalConf.validMask & 1<<r) {
      switch(r) {
      case 0:
        bocCard->setClockControl(globalConf.clockControl);
        break;
      case 1:
        bocCard->setRxDataMode(globalConf.rxDataMode);
        break;
      case 2:
        if(globalConf.rxDacClear)
          bocCard->clearRxDac();
        break;
      case 3:
        if(globalConf.txDacClear)
          bocCard->clearTxDac();
        break;
      case 4:
        bocCard->setVernierFinePhase(globalConf.vernierFinePhase);
        break;
      case 5:
        bocCard->setVernierClockPhase0(globalConf.vernierClockPhase0);
        break;
      case 6:
        bocCard->setVernierClockPhase1(globalConf.vernierClockPhase1);
        break;
      case 7:
        bocCard->setBpmClockPhase(globalConf.bpmClockPhase);
        break;
      case 8:
        bocCard->setBregClockPhase(globalConf.bregClockPhase);
        break;
      }
    }
  }

  char *mappings = config->getFibreMappings(partition, crate, rod);

  for(int channel=0;channel<48;channel++){
    // How far we got
    int step = 0;

    try {
      BOCChannelConfig channelConfig = config->getBOCConfig(partition, crate, rod, channel);
      std::cout << " Data for channel " << channel << std::endl;

      // NB - fibre mapping hopefully now not hard wired
      int txChannel = mappings[channel * 3 + 0];
      int rx0Channel = mappings[channel * 3 + 1];
      int rx1Channel = mappings[channel * 3 + 2];

      if(((rx0Channel / 2) != (rx1Channel / 2)) 
         || ((rx0Channel <= 95) && (rx1Channel <= 95) && (rx0Channel == rx1Channel))) {
        std::cout << "Illegal rx fibre mappings for channel " << channel << std::endl;
        std::cout << "  rx fibres must be mapped in consecutive pairs and low fibre must be even\n";
        if(mrs) {
          *mrs << "CRATE_FIBRES"  << MRS_WARNING << MRS_QUALIF("SCTAPI")
               << MRS_TEXT("RX fibres must be mapped in consecutive pairs and low fibre must be even") 
               << MRS_PARAM<int>("channel", channel)
               << MRS_PARAM<int>("fibre0", rx0Channel)
               << MRS_PARAM<int>("fibre1", rx1Channel)
               << MRS_PARAM<int>("partition", partition)
               << MRS_PARAM<int>("crate", crate)
               << MRS_PARAM<int>("rod", rod)
               << ENDM;
        }
      }

      //        std::cout << txChannel << " " << rx0Channel << " " << rx1Channel << std::endl;

      // Configure clock/command output
      bocCard->setLaserCurrent(txChannel,   channelConfig.current);
      step ++;
      bocCard->setBpmFineDelay(txChannel,   (channelConfig.delay&0xff) );
      step ++;
      bocCard->setBpmCoarseDelay(txChannel, ((channelConfig.delay>>8)&0xff));
      step ++;
      bocCard->setBpmMarkSpace(txChannel,   channelConfig.markSpace);
      step ++;

      // Configure data reception
      if((char)rx0Channel != (char)DATA_LINK_OFF) {
	bocCard->setRxThreshold(rx0Channel, channelConfig.threshold0);
        step ++;
	bocCard->setRxDataDelay(rx0Channel, channelConfig.delay0);
        step ++;
      }
      if((char)rx1Channel != (char)DATA_LINK_OFF) {
	bocCard->setRxThreshold(rx1Channel, channelConfig.threshold1);
        step ++;
	bocCard->setRxDataDelay(rx1Channel, channelConfig.delay1);
        step ++;
      }
    } catch(ConfigurationException &c) {
      // This would be worth saying if a module was attached
      //         std::cout << "No channel data for BOC " << rod << " channel " << channel << std::endl;
    } catch(BaseException &b) {
      std::cout << "Exception setting channel data for BOC step " << step << " ROD (" << rod << " channel " << channel << "): " << b << std::endl;
    }
  }

  delete [] mappings;

  return 0;
}

/*!
  Send a primitive list to a ROD within this crate
*/
int CrateImpl::sendPrimList(unsigned int rod, boost::shared_ptr<PrimListWrapper> prim) {
  if(RODPresent(rod)) {   // If there's a rod presumably there's a data structure with it...
#if USE_THREADS
    if(m_stopPolling) {
      std::cerr << "PRIM LIST: Send prim list called when polling stopped\n";
      throw(CrateException("Called sendPrimList with polling stopped"));
    }

    {
      RodStatus &stat = getRodData(rod);
      boost::mutex::scoped_lock lock(stat.mutex);

      getRod(rod)->deleteOutList();

      {
        boost::mutex::scoped_lock lock(primQueue_mutex);
        stat.listQueue.push_back(prim);
        primQueue_notEmpty.notify_one();
      }

      //       std::cout << "STATE: Changing state to LOADED\n";
      if(stat.currState == MYPRIM_STATE_IDLE) {
        stat.currState = MYPRIM_STATE_LOADED;
      } else {
#warning "Throw exception?"
        std::cerr << "PRIM LIST: Sending prim list when not idle! (" << stat.currState << ")!\n";
        stat.currState = MYPRIM_STATE_LOADED;
        return -10;
      }
    }  // Release lock
#else
    try {
      // So the result can't be confused with any previous
      getRod(rod)->deleteOutList();
      getRod(rod)->sendPrimList(prim->list.get());
    } catch (SctPixelRod::PrimListException &p) {
      std::cerr << "While sending primlist " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "Exception sending primlist:\n";
      std::cerr << p.getDescriptor() << " " << p.getData1() << ", " << p.getData2() << std::endl;
      return -6;
    } catch (SctPixelRod::HpiException &h) {    
      std::cerr << "While sending primlist " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "HpiException:\n";
      std::hex(std::cerr);
      std::cerr << h.getDescriptor() << '\n';
      std::cerr << "calcAddr: " << h.getCalcAddr() << ", readAddr: " << 
        h.getReadAddr() << '\n';
      std::dec(std::cerr);
      return -2;
    } catch (SctPixelRod::RodException &r) {
      std::cerr << "While sending primlist " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "Rod Exception\n";
      std::cerr << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
                << '\n';
      return -3;
    } catch (SctPixelRod::NoImageFile &f) {
      std::cerr << "While sending primlist " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "No image file exception " << f.getFileName() << std::endl;
      return -4;
    } catch (...) {
      std::cerr << "Unknown exception sending primlist to ROD\n";
      return -5;
    }
    //    std::cout << "Status after sendPrimList is: " << *(getRod(rod));

#endif
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }

  return 0;
}

/*!
  Send a primitive list to a ROD within this crate
*/
int CrateImpl::sendPrimListAll(boost::shared_ptr<PrimListWrapper> prim) {
#if USE_THREADS
  {
    boost::mutex::scoped_lock lock(primQueue_mutex);
    listQueueAll.push_back(prim);
    primQueue_notEmpty.notify_one();
  }

  return 0;
#else
  int result = 0;
  for(RodMap::const_iterator item = rodMap.begin();
      item != rodMap.end();
      item++) {
    int rod = item->first;

    int newResult = sendPrimList(rod, prim);
    if(result == 0) result = newResult;
  }

  return result;
#endif
}

void CrateImpl::primListControl(unsigned int index, PLControl function) {
  if(RODPresent(index)) {
    SctPixelRod::RodModule &rod = *(getRod(index));

    switch(function) {
    case C_PL_STOP:
      rod.setVmeCommandRegBit(CR_ABORT);
      break;
    case C_PL_PAUSE:
      rod.setVmeCommandRegBit(CR_PAUSE);
      break;
    case C_PL_RESUME:
      rod.setVmeCommandRegBit(CR_RESUME);
      break;
    case C_PL_CANCEL:
      rod.clearVmeCommandRegBit(CR_IN_LIST_RDY);
      break;
    }
  }
}

/*
  Await response from a ROD within this crate
*/
int CrateImpl::awaitResponse(unsigned int rod, int timeout) {
  // Preliminary implementation from EchoTest
  // Wait for ROD to begin executing and then wait for it to finish executing
  // Check for error messages in text buffer and read them out if they exist.

  using namespace SctPixelRod;

  bool timedout = false;

  time_t start_time = time(0);

  if(RODPresent(rod)) {
#if USE_THREADS
    //     RodPrimState returnPState;
    /* Using threads we let the polling thread handle the text buffers
       and all the communication with the ROD 
    */

    RodStatus &status = getRodData(rod);

    // Shouldn't be called in the idle state...
    //  but for sendPrimListAll can get here before changing state.

    try {
      RodPrimState localState;
      do {
        if((time(0) - start_time) > timeout) {
          timedout = true;
          std::cout << "Timeout from wait complete\n";

          SctPixelRod::RodModule &myRod = *getRod(rod);
          std::cout << "  Current state is ";
          switch(myRod.getPrimState()) {
          case PRIM_IDLE: std::cout << "IDLE"; break;
          case PRIM_LOADED: std::cout << "LOADED"; break;
          case PRIM_EXECUTING: std::cout << "EXECUTING"; break;
          case PRIM_WAITING: std::cout << "WAITING"; break;
          case PRIM_PAUSED: std::cout << "PAUSED"; break;
          }
          std::cout << std::endl;
          std::cout << "  Local state is ";
          switch(localState) {
          case MYPRIM_STATE_LOADED: std::cout << "LOADED"; break;
          case MYPRIM_STATE_EXECUTING: std::cout << "EXECUTING"; break;
          case MYPRIM_STATE_COMPLETE: std::cout << "COMPLETE"; break;
          case MYPRIM_STATE_IDLE: std::cout << "IDLE"; break;
          }
          std::cout << std::endl;
        }

        // Yield when haven't got the lock...
        sched_yield();

        {
          boost::mutex::scoped_lock statLock(status.mutex);
          localState = status.currState;
        }
      } while ((localState != MYPRIM_STATE_COMPLETE)
               &&(!timedout));
      //      std::cout << "AwaitResponse: done wait or idle " << status.currState << "\n";

      {
        boost::mutex::scoped_lock statLock(status.mutex);
        status.currState = MYPRIM_STATE_IDLE;
        //         std::cout << "STATE: Changing state to IDLE\n";

        //         std::cout << "Current length of queue " << status.listQueue.size() << std::endl;
        //         std::cout << "Current RodModule state  " << getRod(rod)->getPrimState() << std::endl;
      }

      if(timedout) {
        std::cout << "Timed out at " << timeout << " seconds\n";
        return -200;
#warning "If timeout then should abort primitive..."
      }
    } catch (BaseException &b) {
      std::cerr << "In await response " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "Exception: " << b << std::endl;
      return -6;
    } catch (...) {
      std::cerr << "Unknown exception in await response\n";
      return -2;
    }
#else  // No threads
    // Create a buffer for test messages
    PrimState returnPState;
    TextBuffState returnTState;

    SctPixelRod::RodModule* rod0 = getRod(rod);

    try {
      int exceptCount = 0;
      do {
        // If there's an exception here its probably OK to try again a bit later
        try {
          returnPState = rod0->primHandler();
          returnTState = rod0->textHandler();
          exceptCount = 0;

          while (returnTState == TEXT_RQ_SET) {
            doTextBuffer(*rod0);
            // textState is cleared to TEXT_IDLE
            // Check there isn't another text buffer
            returnTState = rod0->textHandler();
          }

          if((time(0) - start_time) > timeout) {
            timedout = true;
            std::cout << "Timeout from wait execute\n";
          }
        } catch(VmeException &e) {
          exceptCount ++;
          if(exceptCount < 2) {
            std::cout << "Caught an exception, trying ignoring\n";
            rod0->sleep(100);
          } else {
            std::cout << " Repeat of exception, throwing anyway\n";
            throw;
          }
        }
      } while ((returnPState != PRIM_EXECUTING) && (!timedout));
      do {
        returnPState = rod0->primHandler();

        if((time(0) - start_time) > timeout) {
          timedout = true;
          std::cout << "Timeout from wait finish\n";
        }
      } while ((returnPState != PRIM_WAITING)&&(returnPState != PRIM_IDLE) &&(!timedout));

      if(timedout) {
        std::cout << "Timed out at " << timeout << " seconds\n";
        return -200;
#warning "If timeout then should abort primitive..."
      }
    } catch (PrimListException &p) {
      std::cerr << "In await response " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "Exception sending primlist:\n";
      std::cerr << p.getDescriptor() << " " << p.getData1() << ", " << p.getData2() << std::endl;
      return -6;
    } catch (HpiException &h) {    
      std::cerr << "In await response " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "HpiException:\n";
      std::hex(std::cerr);
      std::cerr << h.getDescriptor() << '\n';
      std::cerr << "calcAddr: " << h.getCalcAddr() << ", readAddr: " << 
        h.getReadAddr() << '\n';
      std::dec(std::cerr);
      return -5;
    } catch (VmeException &v) {
      std::cout << "VmeException in await response:\n";
      Utility::decodeVme(v);

      return -9;
    } catch (RodException &r) {
      std::cerr << "In await response " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "Rod Exception\n";
      std::cerr << r.getDescriptor() <<", " << r.getData1() << ", " << r.getData2()
                << '\n';
      return -4;
    } catch (NoImageFile &f) {
      std::cerr << "In await response " << partition << " " << crate << " " << rod << ":\n";
      std::cerr << "No image file exception " << f.getFileName() << std::endl;
      return -3;
    } catch (...) {
      std::cerr << "Unknown exception in await response\n";
      return -2;
    }
    //    std::cout << "Status at end of awaitResponse: " << *rod0;
#endif
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;

    return -1;
  }

  return 0;
}

/*
  Await response from all RODs within this crate
*/
int CrateImpl::awaitResponseAll(int timeout) {
  int result = 0;
  for(RodMap::const_iterator item = rodMap.begin();
      item != rodMap.end();
      item++) {
    int rod = item->first;

    int newResult = awaitResponse(rod, timeout);
    if(result == 0) result = newResult;
  }

  return 0;
}

/*
  Get the last response from a ROD within this crate
*/
boost::shared_ptr<RodOutList> CrateImpl::getResponse(unsigned int rod) {
  RodOutList *result = 0;

  if(RODPresent(rod)) {
    // This doesn't actually talk to the ROD just copies memory...
    RodOutList *response = getRod(rod)->getOutList();

    if(response) {
      result = new RodOutList(response->getLength());

      unsigned long *oldBody = response->getBody();
      unsigned long *newBody = result->getBody();

      for(int i=0; i<result->getLength(); i++) {
        newBody[i] = oldBody[i];
      }

      getRod(rod)->deleteOutList();
    } else {
      std::cout << "**** No response found when expected\n";
      result = 0;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    result = 0;
  }

  return boost::shared_ptr<RodOutList>(result);
}

void CrateImpl::setMrsStream(MRSStream *stream) 
{
  mrs = stream;
}

/*
  Read the message buffer if there is anything there
*/
bool CrateImpl::getRodMessage(unsigned int rod, char *buffer, unsigned long &length) {
  bool complete = true;

  char myBuffer[TEXT_BUFF_SIZE];
  long myLength;

  TEXT_BUFFER_TYPE dummy;
  if(RODPresent(rod)) {
    if(getRod(rod)->textHandler() == TEXT_READOUT) {
      getRod(rod)->getTextBuffer(myBuffer, myLength, dummy);
      getRod(rod)->clearTextBuffer();

      // Copy across what's allowed
      for(unsigned int i=0; i<length || i<(unsigned long)myLength; i++) {
        buffer[i] = myBuffer[i];
      }

      if(length < (unsigned long)myLength) {
        complete = false;
      } else {
        length = myLength;
      }
    } else {
      buffer[0] = '\0';
      length = 0;
      complete = false;
    }
  }

  return complete;
}

/*
  Write a block of master dsp memory 
*/
int CrateImpl::mdspBlockWrite(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords) {
  if(RODPresent(rod)) {
    try {
      getRod(rod)->mdspBlockWrite(dspStart, buffer, numWords);
    } catch(BaseException &b) {
      std::cout << "Exception writing mdsp\n";
      std::cout << b;
      return -3;
    } catch(...) {
      std::cout << "Exception writing mdsp\n";
      return -2;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }

  return 0;
}

/*
  Read a block of master dsp memory 
*/
int CrateImpl::mdspBlockRead(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords) {
  if(RODPresent(rod)) {
    try {
//       std::cout << "Calling RodModule::mdspBlockRead(0x" << std::hex << dspStart << ", " << buffer << ", " << std::dec << numWords << ");\n";
      getRod(rod)->mdspBlockRead(dspStart, buffer, numWords);
    } catch(BaseException &b) {
      std::cout << "Exception reading mdsp\n";
      std::cout << b;
      std::cout << "0x" << std::hex << dspStart << ", " << std::dec << numWords << std::endl;
      return -3;
    } catch(...) {
      std::cout << "Unknown exception reading mdsp\n";
      std::cout << "0x" << std::hex << dspStart << ", " << std::dec << numWords << std::endl;
      return -2;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }

  return 0;
}

/*
  Read a block of slave dsp memory 
*/
int CrateImpl::slvBlockWrite(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords, long dspNumber) {
  if(RODPresent(rod)) {
    try {
      getRod(rod)->slvBlockWrite(dspStart, buffer, numWords, dspNumber);
    } catch(BaseException &b) {
      std::cout << "Exception writing to slave\n";
      std::cout << b;
      return -3;
    } catch(...) {
      std::cout << "Exception writing to slave\n";
      return -2;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }
  return -0;
}

/*
  Read a block of slave dsp memory 
*/
int CrateImpl::slvBlockRead(unsigned int rod, long dspStart, unsigned long* buffer, unsigned long numWords, long dspNumber) {
  if(RODPresent(rod)) {
    try {
      RodModule &r = *getRod(rod);
#warning "Get lock so nobody else comes in and does something"
      unsigned long val = r.readRodStatusReg(2);
      // Turn off mdsp polling of that slave
      r.mdspSingleWrite(STATUS_REG[2], val & (~(1<<dspNumber)));
      r.slvBlockRead(dspStart, buffer, numWords, dspNumber);
      r.mdspSingleWrite(STATUS_REG[2], val);
    } catch(BaseException &b) {
      std::cout << "Exception reading slave\n";
      std::cout << b;
      return -3;
    } catch(...) {
      std::cout << "Exception reading slave\n";
      return -2;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }
  return 0;
}

unsigned long CrateImpl::readRodStatusReg(unsigned int rod, long regNumber) {
  unsigned long result = 0;

  if(RODPresent(rod)) {
    try {
      result = getRod(rod)->readRodStatusReg(regNumber);
    } catch(BaseException &b) {
      std::cout << "Exception reading rod status\n";
      std::cout << b;
    } catch(...) {
      std::cout << "Exception reading rod status\n";
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
  }
  return result;
}

unsigned long CrateImpl::readRodCommandReg(unsigned int rod, long regNumber) {
  unsigned long result = 0;

  if(RODPresent(rod)) {
    try {
      result = getRod(rod)->readRodCommandReg(regNumber);
    } catch(BaseException &b) {
      std::cout << "Exception reading rod command\n";
      std::cout << b;
    } catch(...) {
      std::cout << "Exception reading rod command\n";
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
  }
  return result;
}

unsigned long CrateImpl::dspSingleRead(unsigned int rod, const unsigned long dspAddr, long dspNumber) {
  unsigned long result = 0;

  if(RODPresent(rod)) {
    try {
      if(dspNumber == -1) {
        result = getRod(rod)->mdspSingleRead(dspAddr);
      } else {
        result = getRod(rod)->slvSingleRead(dspAddr, dspNumber);
      } 
    } catch(BaseException &b) {
      std::cout << "Exception dsp single read\n";
      std::cout << b;
    } catch(...) {
      std::cout << "Exception dsp single read\n";
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
  }
  return result;
}

int CrateImpl::dspSingleWrite(unsigned int rod, unsigned long dspAddr, unsigned long buffer, long dspNumber) {
  if(RODPresent(rod)) {
    try {
      if(dspNumber == -1) {
        getRod(rod)->mdspSingleWrite(dspAddr, buffer);
      } else {
        getRod(rod)->slvSingleWrite(dspAddr, buffer, dspNumber);
      }
    } catch(BaseException &b) {
      std::cout << "Exception dsp single write\n";
      std::cout << b;
    } catch(...) {
      std::cout << "Exception dsp single write\n";
      return -2;
    }
  } else {
    std::cout << "Request for non-existent ROD " << partition << " " << crate << " " << rod << std::endl;
    return -1;
  }

  return 0;
}

void CrateImpl::status() {
  std::cout << "TIM status\n";
  if(tim) 
    tim->status();
  std::cout << std::endl;

  if(rodMap.size() == 0) {
    if(mrs) {
      *mrs << "CRATE_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("partition", partition) << MRS_PARAM<int>("crate", crate) 
           << MRS_TEXT("No RODs loaded") << ENDM;
    }
  }

  for(RodMap::const_iterator item = rodMap.begin();
      item != rodMap.end();
      item++) {
    RodModule &rod = *(item->second.second);

    try {
      //       SctConfiguration::RodConfig rodConf = config->getRodConfig(partition, crate, item->first);
      //       if(rodConf.bocPresent) {
      std::cout << "BOC status\n";
      if(rod.getBocCard() != 0) {
        rod.getBocCard()->status();

        if(mrs) {
          *mrs << "BOC_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
               << MRS_PARAM<int>("partition", partition) << MRS_PARAM<int>("crate", crate) 
               << MRS_PARAM<int>("slot", rod.getSlot()) 
               << MRS_TEXT("BOC loaded") << ENDM;
        }
      } else {
        std::cout << "*********** No BOC status (BOC not present!!!!)\n";
      }
    } catch(ConfigurationException &c) {
      std::cout << "Request for status on unconfigured ROD! (impossible)\n";
      return;
    }

    if(mrs) {
      *mrs << "ROD_STATUS" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("partition", partition) << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("slot", rod.getSlot()) 
           << MRS_TEXT("ROD loaded") << ENDM;
    }

    std::cout << "ROD status\n";
    rod.status();

    if(rod.readRodStatusBit(0, SR_RUNNING)) {
      std::cout << "Running ";
    }
    if(rod.readRodStatusBit(0, SR_BUSY)) {
      std::cout << "Busy ";
    }
    if(rod.readRodStatusBit(0, SR_PAUSED)) {
      std::cout << "Paused ";
    }
    if(rod.readRodStatusBit(0, SR_OUT_LIST_RDY)) {
      std::cout << "Out list ready ";
    }
    if(rod.readRodStatusBit(0, SR_EXECUTING)) {
      std::cout << "Executing \n";
      unsigned long sr2 = rod.readRodStatusReg(1);

      std::cout << "Current Primitive index = " << ((sr2 & 0xfffff0) >> 4) << std::endl;
      std::cout << "Current List index = " << (sr2 & 0xf);
    }

    std::cout << std::endl;
  }
}

/* 
   Lookup the RodModule object to send instructions to.
   Return 0 if not found.
*/
SctPixelRod::RodModule *CrateImpl::getRod(unsigned int rod) const {
  RodMap::const_iterator found;

  found = rodMap.find(rod);
  if(found == rodMap.end()) return 0;
  else return found->second.second;
}

const RodStatus &CrateImpl::getRodData(unsigned int rod) const {
  RodMap::const_iterator found;

  found = rodMap.find(rod);
  if(found == rodMap.end()) {
//     std::cerr << "AJB ERROR Hello world" 
//               <<  __FILE__ << ":" <<__LINE__ << std::endl;
    std::cerr << " getRodData called on missing ROD " << rod << std::endl;
    throw CrateException("Asked for data on non-existant ROD (const)");
  }
  else return *(found->second.first);
}

RodStatus &CrateImpl::getRodData(unsigned int rod) {
  RodMap::const_iterator found;

  found = rodMap.find(rod);
  if(found == rodMap.end()) {
//     std::cerr << "AJB ERROR Hello world" 
//               <<  __FILE__ << ":" <<__LINE__ << std::endl;
    std::cerr << " getRodData (const) called on missing ROD " << rod << std::endl;
    throw CrateException("Asked for data on non-existant ROD");
  }
  else return *(found->second.first);
}

void CrateImpl::doTextBuffer(SctPixelRod::RodModule &rod) {
  TEXT_BUFFER_TYPE myTextType;
  TextBuffState returnTState;

  std::cout << "Got some text\n";
  do {
    returnTState = rod.textHandler();
  } while (returnTState != TEXT_READOUT);

  if(mrs) {
    *mrs << "TEXTBUFF_ARRIVE" << MRS_DIAGNOSTIC << MRS_QUALIF("SCTAPI")
         << MRS_PARAM<int>("partition", partition)
         << MRS_PARAM<int>("crate", crate)
         << MRS_PARAM<int>("rodSlot", rod.getSlot());
  }

  char * myTextBuffer = new char[TEXT_BUFF_SIZE];
  long myTextLength;

  rod.getTextBuffer(myTextBuffer, myTextLength, myTextType);
  rod.clearTextBuffer();

  std::string type;
  switch(myTextType) {
  case TEXT_ERR: type = "ERROR"; break;
  case TEXT_INFO: type = "INFO"; break;
  case TEXT_DIAG: type = "DIAG"; break;
  case TEXT_XFER: type = "TRANSFER"; break;
  case TEXT_UNDEF: type = "**Undefined**"; break;
  }

  std::cout << "Text " << type << " : " << myTextLength << std::endl;
  if(mrs) {
    *mrs << MRS_PARAM<const char *>("Type", type.c_str());
  }

  if(myTextLength != 32768) {
    std::ostringstream message;    
    std::ostringstream short_message;
    for (int i=0; i<myTextLength; i++) {
      message << myTextBuffer[i];
      if (myTextBuffer[i]!='\n' && i<100) short_message << myTextBuffer[i];
    }
    std::cout << message.str();

    myTextBuffer[myTextLength] = 0;
    if(mrs) {
      *mrs << MRS_TEXT(short_message.str().c_str())
           << MRS_PARAM<const char *>("Text", myTextBuffer);
    }
  } else {
    std::cout << "Buffersize suspiciously precisely 32768...\n";
    if(mrs) {
      *mrs << MRS_TEXT("Buffersize suspiciously precisely 32768...");
    }

    std::string fileName = Sct::SctNames::getLogDir() + "/SctApi32k" + type + "BufferDump.txt";
    std::ofstream outFile(fileName.c_str(), std::ios::binary | std::ios::app);
    time_t currTime = time(0);
    outFile << "Rod: " << rod.getSerialNumber() << " Time: " << ctime(&currTime) << std::endl;
    outFile.write(myTextBuffer, myTextLength);
    outFile << "\nEnd of dump\n";
  }

  std::cout << std::endl;

#warning "This should use debug level instead"
#ifdef DEBUG
  std::ofstream outFile((type + "BufferDump.txt").c_str(), std::ios::binary | std::ios::app);
  outFile << "\nBuffer block length " << myTextLength << " rod " << rod.getSerialNumber() << std::endl;
  time_t currTime = time(0);
  outFile << "Time " << ctime(&currTime);
  outFile.write(myTextBuffer, myTextLength);
#endif

  if(mrs) {
    *mrs << ENDM;
  }

  delete [] myTextBuffer;
}

void CrateImpl::printBOCSetup(unsigned int rod) {
  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " printBOCSetup called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for print setup");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    std::cout.fill('0');
    for(int c=0; c<48; c++) {
      std::cout << "Channel ";
      std::cout.width(2);
      std::cout << c << ":  I 0x";
      std::cout << std::hex;
      std::cout.width(2);
      std::cout << bocCard.getLaserCurrent(c) << " Delay 0x";
      std::cout.width(2);
      std::cout << bocCard.getBpmFineDelay(c) << "/0x";
      std::cout.width(2);
      std::cout << bocCard.getBpmCoarseDelay(c)
                << " M/S 0x";
      std::cout.width(2);
      std::cout << bocCard.getBpmMarkSpace(c);
      std::cout << "  Input 0: Th 0x";
      std::cout.width(2);
      std::cout << bocCard.getRxThreshold(c*2) << " Delay 0x";
      std::cout.width(2);
      std::cout << bocCard.getRxDataDelay(c*2)
                << "  Input 1: Th 0x";
      std::cout.width(2);
      std::cout << bocCard.getRxThreshold(c*2+1) << " Delay 0x";
      std::cout.width(2);
      std::cout << bocCard.getRxDataDelay(c*2+1)
                << std::endl;
    }
    std::cout.fill(' ');
    std::cout << std::dec;
  }
}

std::vector<SctConfiguration::BOCChannelConfig> CrateImpl::currentBOCSetup(unsigned int rod) {
  std::vector<SctConfiguration::BOCChannelConfig> result;

  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " currentBOCSetup called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC to get setup from");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    for(int c=0; c<48; c++) {
      SctConfiguration::BOCChannelConfig chan;

      chan.current = bocCard.getLaserCurrent(c);
      chan.delay =  (bocCard.getBpmCoarseDelay(c) << 8) + bocCard.getBpmFineDelay(c);
      chan.markSpace = bocCard.getBpmMarkSpace(c);
      chan.threshold0 = bocCard.getRxThreshold(c*2);
      chan.delay0 = bocCard.getRxDataDelay(c*2);
      chan.threshold1 = bocCard.getRxThreshold(c*2+1);
      chan.delay1 = bocCard.getRxDataDelay(c*2+1);

      result.push_back(chan);
    }
  }

  return result;
}

void CrateImpl::printBOCRegisters(unsigned int rod) {
  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " printBOCRegisters called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for print registers");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    std::cout << std::hex;

    std::cout << "BOC register dump\n";

    std::cout << "Clock control  0x" << bocCard.getClockControl() << std::endl;
    std::cout << "RX data mode   0x" << bocCard.getRxDataMode() << std::endl;
    std::cout << "RX DAC clear   0x" << bocCard.getRxDacClear() << std::endl;
    std::cout << "TX DAC clear   0x" << bocCard.getTxDacClear() << std::endl;
    std::cout << "Vernier fine   0x" << bocCard.getVernierFinePhase() << std::endl;
    std::cout << "Vernier clk 0  0x" << bocCard.getVernierClockPhase0() << std::endl;
    std::cout << "Vernier clk 1  0x" << bocCard.getVernierClockPhase1() << std::endl;
    std::cout << "BPM clk phase  0x" << bocCard.getBpmClockPhase() << std::endl;
    std::cout << "BReg clk phase 0x" << bocCard.getBregClockPhase() << std::endl;
    std::cout << "BOC reset      0x" << bocCard.getBocReset() << std::endl;
    std::cout << "BPM reset      0x" << bocCard.getBpmReset() << std::endl;
    std::cout << "BOC status     0x" << bocCard.getBocStatusRegister() << std::endl;

    std::cout << std::dec;
  } else {
    std::cout << "No ROD found\n";
  }
}

SctConfiguration::BOCGlobalConfig CrateImpl::currentBOCRegisters(unsigned int rod) {
  BOCGlobalConfig result;

  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " currentBOCRegisters called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for current registers");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    result.validMask = 0x1ff;
    result.clockControl = bocCard.getClockControl();
    result.rxDataMode = bocCard.getRxDataMode();
    result.rxDacClear = bocCard.getRxDacClear();
    result.txDacClear = bocCard.getTxDacClear();
    result.vernierFinePhase = bocCard.getVernierFinePhase();
    result.vernierClockPhase0 = bocCard.getVernierClockPhase0();
    result.vernierClockPhase1 = bocCard.getVernierClockPhase1();
    result.bpmClockPhase = bocCard.getBpmClockPhase();
    result.bregClockPhase = bocCard.getBregClockPhase();

    // Also getBocReset, getBpmReset and getBocStatusRegister
  }

  return result;
}

void CrateImpl::saveBOCSetup(unsigned int rod, BankType bank) {
  getRodData(rod).savedBocSetup[bank] = currentBOCSetup(rod);
}

void CrateImpl::saveBOCRegisters(unsigned int rod, BankType bank) {
  getRodData(rod).savedBocRegisters[bank].reset(new BOCGlobalConfig(currentBOCRegisters(rod)));
}

void CrateImpl::restoreBOCSetup(unsigned int rod, BankType bank) {
  BocCard *bocCard = getRod(rod)->getBocCard();

  std::vector<SctConfiguration::BOCChannelConfig> &setup = getRodData(rod).savedBocSetup[bank];

//   char *mappings = config->getFibreMappings(partition, crate, rod);

  for(int channel=0;channel<48;channel++){
    try {
      BOCChannelConfig channelConfig = setup[channel];

      std::cout << " Restoring channel " << channel << std::endl;

      int txChannel = channel;
      int rx0Channel = channel * 2;
      int rx1Channel = channel * 2 + 1;

      // Configure clock/command output
      bocCard->setLaserCurrent(txChannel,   channelConfig.current);
      bocCard->setBpmFineDelay(txChannel,   (channelConfig.delay&0xff) );
      bocCard->setBpmCoarseDelay(txChannel, ((channelConfig.delay>>8)&0xff));
      bocCard->setBpmMarkSpace(txChannel,   channelConfig.markSpace);

      // Configure data reception
      // Don't need to check for DATA_LINK_OFF as rx?Channel is generated locally
      bocCard->setRxThreshold(rx0Channel, channelConfig.threshold0);
      bocCard->setRxDataDelay(rx0Channel, channelConfig.delay0);

      bocCard->setRxThreshold(rx1Channel, channelConfig.threshold1);
      bocCard->setRxDataDelay(rx1Channel, channelConfig.delay1);
    } catch(BaseException &b) {
      std::cout << "Exception setting channel data for BOC " << " ROD (" << rod << " channel " << channel << "): " << b << std::endl;
    }
  }
}

void CrateImpl::restoreBOCRegisters(unsigned int rod, BankType bank) {
  BocCard *bocCard = getRod(rod)->getBocCard();

  BOCGlobalConfig &globalConf = *getRodData(rod).savedBocRegisters[bank];
  for(int r=0; r<9; r++) {
    if(globalConf.validMask & 1<<r) {
      switch(r) {
      case 0:
        bocCard->setClockControl(globalConf.clockControl);
        break;
      case 1:
        bocCard->setRxDataMode(globalConf.rxDataMode);
        break;
      case 2:
        if(globalConf.rxDacClear)
          bocCard->clearRxDac();
        break;
      case 3:
        if(globalConf.txDacClear)
          bocCard->clearTxDac();
        break;
      case 4:
        bocCard->setVernierFinePhase(globalConf.vernierFinePhase);
        break;
      case 5:
        bocCard->setVernierClockPhase0(globalConf.vernierClockPhase0);
        break;
      case 6:
        bocCard->setVernierClockPhase1(globalConf.vernierClockPhase1);
        break;
      case 7:
        bocCard->setBpmClockPhase(globalConf.bpmClockPhase);
        break;
      case 8:
        bocCard->setBregClockPhase(globalConf.bregClockPhase);
        break;
      }
    }
  }
}

// Check that the BOC lasers are on
/*
  status = 0xe2
  status = 0x82 when crate interlock not off
  status = 0xa0 when on board interlock not off
*/
bool CrateImpl::checkBOCLasersOn(unsigned int rod) { 
  //   unsigned long status = getRod(rod)->getBocCard()->getBocStatusRegister();

  //   return status == 0xe2;

  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      return false; // Ie it won't work ... (but because the BOC's not there!...)
    }
    return getRod(rod)->getBocCard()->getInterlockStatus();
  } else {
    return false; // Ie it won't work ... (but because the ROD's not configured...)
  }
}

void CrateImpl::enterBOCClockBy2Mode(unsigned int rod) {
  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " enterBOCClockBy2Mode called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for enter clock/2");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    bocCard.setRxDataMode(1);
    bocCard.setClockControl(2);
  }
}

void CrateImpl::leaveBOCClockBy2Mode(unsigned int rod) {
  if(RODPresent(rod)) {
    if(getRod(rod)->getBocCard() == 0) {
      std::cerr << " leaveBOCClockBy2Mode called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for leave clock/2");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    bocCard.setRxDataMode(0);
    bocCard.setClockControl(0);
  }
}

std::vector<double> CrateImpl::getBOCMonitorArray(unsigned int rod) {
  if(!RODPresent(rod)) {
    std::cerr << " getBOCMonitorArray called on missing BOC " << rod << std::endl;
    throw CrateException("No BOC for getBOCMonitorArray");
  }

  if(getRod(rod)->getBocCard() == 0) {
    std::cerr << " getBOCMonitorArray called on missing BOC " << rod << std::endl;
    throw CrateException("No BOC for getBOCMonitorArray");
  }

  BocCard &bocCard = *(getRod(rod)->getBocCard());

  std::vector<double> result;

  for(int i=0; i<12; i++) {
    result.push_back(bocCard.getMonitorAdc(i));
  }

  return result;
}

void CrateImpl::modifyBOCParam(unsigned int type, unsigned int val, bool raw) {
  for(RodMap::const_iterator ri = rodMap.begin();
      ri!=rodMap.end();
      ri++) {
    for(int channel=0; channel<48; channel++) {
      modifyBOCParam(ri->first, channel, type, val, raw);
    }
  }
}

void CrateImpl::modifyBOCParam(unsigned int rod, unsigned int channel, unsigned int type, unsigned int val, bool raw) {
  if(RODPresent(rod)) {
    if(!getRod(rod)->getBocCard()) {
      std::cerr << " modifyBOCParam called on missing BOC " << rod << std::endl;
      throw CrateException("No BOC for modify BOC param");
    }

    BocCard &bocCard = *(getRod(rod)->getBocCard());

    // Raw channels don't need look up
    int txFibre = channel;
    int rxFibre1 = channel * 2 + 0;
    int rxFibre2 = channel * 2 + 1;

    if(!raw) {
      char *mappings;

      try { 
        mappings = config->getFibreMappings(partition, crate, rod);
      }  catch(ConfigurationException &c) {
        std::cout << "No fibre mappings for appropriate module: \n";
        std::cout << c.what() << std::endl;

        //     log << "  (Not found)\n";

        return;
      }

      txFibre = mappings[channel * 3];
      rxFibre1 = mappings[channel * 3 + 1];
      rxFibre2 = mappings[channel * 3 + 2];
    }

    try {
      switch(type) {
      case ST_RX_DELAY0:
        if(rxFibre1 != DATA_LINK_OFF) bocCard.setRxDataDelay(rxFibre1, val);
        break;
      case ST_RX_DELAY1:
        if(rxFibre2 != DATA_LINK_OFF) bocCard.setRxDataDelay(rxFibre2, val);
        break;
      case ST_RX_DELAY:
        if(rxFibre1 != DATA_LINK_OFF) bocCard.setRxDataDelay(rxFibre1, val);
        if(rxFibre2 != DATA_LINK_OFF) bocCard.setRxDataDelay(rxFibre2, val);
        break;

      case ST_RX_THRESHOLD0:
        if(rxFibre1 != DATA_LINK_OFF) bocCard.setRxThreshold(rxFibre1, val);
        break;
      case ST_RX_THRESHOLD1:
        if(rxFibre2 != DATA_LINK_OFF) bocCard.setRxThreshold(rxFibre2, val);
        break;
      case ST_RX_THRESHOLD:
        if(rxFibre1 != DATA_LINK_OFF) bocCard.setRxThreshold(rxFibre1, val);
        if(rxFibre2 != DATA_LINK_OFF) bocCard.setRxThreshold(rxFibre2, val);
        break;

      case ST_TX_CURRENT:
        if(txFibre != DATA_LINK_OFF) bocCard.setLaserCurrent(txFibre, val);
        break;
      case ST_TX_MARKSPACE:
        if(txFibre != DATA_LINK_OFF) bocCard.setBpmMarkSpace(txFibre, val);
        break;
      case ST_TX_DELAY:
        if(txFibre != DATA_LINK_OFF) bocCard.setBpmCoarseDelay(txFibre, (val&0xff00) >> 8);
        if(txFibre != DATA_LINK_OFF) bocCard.setBpmFineDelay(txFibre, (val&0xff));
        break;
      case ST_TX_COARSE:
        if(txFibre != DATA_LINK_OFF) bocCard.setBpmCoarseDelay(txFibre, val);
        break;
      case ST_TX_FINE:
        if(txFibre != DATA_LINK_OFF) bocCard.setBpmFineDelay(txFibre, val);
        break;

        // Extra scans
      case SCT_SCAN_BOC_BPM_PHASE:
        bocCard.setBpmClockPhase(val);
        break;
      case SCT_SCAN_BOC_BREG_PHASE:
        bocCard.setBregClockPhase(val);
        break;
      case SCT_SCAN_BOC_V0_PHASE:
        bocCard.setVernierClockPhase0(val);
        break;
      case SCT_SCAN_BOC_V1_PHASE:
        bocCard.setVernierClockPhase1(val);
        break;
      case SCT_SCAN_BOC_VRN_FINE:
        bocCard.setVernierFinePhase(val);
        break;

        // Useful scans for finding modules...
      case SCT_SCAN_TX_CHANNELS:
        setupScanTx(rod, val);
        break;

      case SCT_SCAN_RAW_TX_CHANNELS:
        setupScanRawTx(rod, val);
        break;

      default:
        break;
      }
    } catch(BocException &b) {
      std::cout << "BocException in modifyBOCParam: " << b << std::endl;
      throw CrateException("BOCException in modify");
    }
  }
}

void CrateImpl::lasersOff() {
  for(RodMap::const_iterator ri = rodMap.begin();
      ri!=rodMap.end();
      ri++) {
    if(ri->second.second->getBocCard() == 0) {
      std::cerr << " Missing BOC in lasersOff\n";
      continue;
//       throw CrateException("No BOC for lasers off");
    }

    BocCard &bocCard = *(ri->second.second->getBocCard());

    for(int i=0; i<48; i++) {
      bocCard.setLaserCurrent(i, 0);
    }
  }
}

void CrateImpl::timSetFrequency(double trigFreq, double rstFreq) {
  if(tim) {
    std::cout << "Setting TIM frequencies: " << trigFreq << "kHz " << rstFreq << "Hz\n";
    int highBits;

    // std:: needed because of ROOT (I think)
    int power = std::ceil(std::log10(trigFreq/8.0));
    std::cout << "trigPower " << power << std::endl;
    double trigBase;
    switch(power) {
    case 2: highBits = 0; trigBase = trigFreq / 10; break;
    case 1: highBits = 1; trigBase = trigFreq / 1; break;
    case 0: highBits = 2; trigBase = trigFreq / 0.1; break;
    case -1: highBits = 3; trigBase = trigFreq / 0.01; break;
    default:
      if(power<-1) { highBits = 3; trigBase = trigFreq / 0.01; }
      else { highBits = 0; trigBase = trigFreq / 10; }
    }

    std::cout << "trigBase " << trigBase << std::endl;
    int lowBits =  60.0 / trigBase;

    if(lowBits == 1) lowBits = 0;
    if(lowBits == 10) lowBits = 1;
    if(lowBits > 6) lowBits = 7;

    int trigNibble = (lowBits & 0x7) + ((highBits & 0xf) << 3);
    std::cout << "trigNibble " << trigNibble << " (" << lowBits << ", " << highBits << ")\n";

    // Decode reset frequency


    // std:: needed because of ROOT (I think)
    power = std::ceil(std::log10(rstFreq/8.0));
    std::cout << "rstPower " << power << std::endl;
    double rstBase;
    switch(power) {
    case 2: highBits = 0; rstBase = rstFreq / 100; break;
    case 1: highBits = 1; rstBase = rstFreq / 10; break;
    case 0: highBits = 2; rstBase = rstFreq / 1; break;
    case -1: highBits = 3; rstBase = rstFreq / 0.1; break;
    default:
      if(power<-1) { highBits = 3; rstBase = rstFreq / 0.1; }
      else { highBits = 0; rstBase = rstFreq / 100; }
    }

    std::cout << "rstBase " << rstBase << std::endl;

    lowBits =  (int)(6.0 / rstBase);

    if(lowBits == 1) lowBits = 0;
    if(lowBits == 10) lowBits = 1;
    if(lowBits > 6) lowBits = 7;

    int rstNibble = (lowBits & 0x7) + ((highBits & 0xf) << 3);
    std::cout << "rstNibble " << rstNibble << " (" << lowBits << ", " << highBits << ")\n";

    tim->regLoad(TIM_REG_FREQUENCY, trigNibble + (rstNibble << 8));
  }
}

void CrateImpl::freeTriggers() {
  if(tim) {
    // Enable L1As
    tim->loadBitSet(TIM_REG_ENABLES, SctPixelRod::TIM_BIT_EN_INT_TRIG);
  }
}

void CrateImpl::stopTriggers() {
  if(tim)
    tim->intTrigStop();
}

void CrateImpl::timL1A() {
  if(tim) {
    tim->issueCommand(SctPixelRod::TIM_VTRG);
    usleep(1);
    tim->loadBitClear(TIM_REG_COMMAND, SctPixelRod::TIM_VTRG);
  }
}

void CrateImpl::timCalL1A(int delay) {
  if(tim)
    tim->issueVCAL(delay);
}


void CrateImpl::timECR() {
  if(tim) {
    tim->issueCommand(SctPixelRod::TIM_VECR);
    // Manually send TIM ECR
    //     tapi.timRegLoad(0, 0, 2, 0x04); 
    timRegLoad(2, 0x0);
  }
}

void CrateImpl::timBCR() {
  if(tim) {
    tim->issueCommand(SctPixelRod::TIM_VBCR);
    // Manually send TIM BCR
    //    tapi.timRegLoad(0, 0, 2, 0x08); 
    timRegLoad(2, 0x0);
  }
}

void CrateImpl::timFER() {
  if(tim) {
    tim->issueCommand(SctPixelRod::TIM_VFER);
    // Manually send TIM FER
    // tapi.timRegLoad(0, 0, 2, 0x20); 
    timRegLoad(2, 0x0);
  }
}

void CrateImpl::sendTimBurst(int count) {
  if(tim) {
    // Load count
    tim->regLoad(TIM_REG_BURST_COUNT, count);
    // Enable burst mode
    tim->loadBitSet(TIM_REG_COMMAND, 0x200);

    // Use single L1A
    tim->loadBitSet(TIM_REG_ENABLES, SctPixelRod::TIM_BIT_EN_INT_TRIG);

    // Start burst
    tim->loadBitSet(TIM_REG_COMMAND, 0x400);

//     std::cout << "Status is " << std::hex << tim->regFetch(TIM_REG_STATUS) << std::dec << std::endl;

    // Wait for burst to finish
    while(tim->regFetch(TIM_REG_STATUS) & 0x10)
      ;

//     std::cout << "Status now " << std::hex << tim->regFetch(TIM_REG_STATUS) << std::dec << std::endl;

    // Stop burst
    tim->loadBitClear(TIM_REG_COMMAND, 0x400);

    // Stop triggers before leaving burst mode
    tim->loadBitClear(TIM_REG_ENABLES, SctPixelRod::TIM_BIT_EN_INT_TRIG);

    // Disable burst mode
    tim->loadBitClear(TIM_REG_COMMAND, 0x200);
  }
}

void CrateImpl::timVerbose() {
  for(int r = 0; r<0x34; r+=2) {
    std::cout.width(2);
    std::cout << std::hex << r << std::dec 
              << ": 0x" << std::hex << tim->regFetch((SctPixelRod::TimRegister)r) << std::dec << std::endl;
  }
}

void CrateImpl::timRegLoad(int reg, UINT16 val) {
  if(tim) {
    tim->regLoad(SctPixelRod::TimRegister(reg), val);
  }
}

UINT16 CrateImpl::timRegRead(int reg) {
  if(tim) {
    return tim->regFetch(SctPixelRod::TimRegister(reg));
  } else {
    return 0;
  }
}

bool CrateImpl::slavePresent(int rod, int index) const {
  bool present;

  if(!RODPresent(rod)) 
    present = false;   // The ROD's not present so the slave definitely isn't!
  else {
    present = (getRodData(rod).slaves & (1<<index))?true:false;
  }

  return present;
}

void CrateImpl::slaveStarted(int rod, int slave){
  getRodData(rod).slaves |= 1<<slave;
}

bool CrateImpl::RODPresent(int rod) const {
  return getRod(rod);
}

long CrateImpl::getRodSlot(int rod) const {
  return getRod(rod)->getSlot();
}

int CrateImpl::getRodRevision(int rod) const {
  return getRod(rod)->getRevision();
}

void CrateImpl::stopPolling() {
  m_enablePolling = false;
}

void CrateImpl::resumePolling() {
  m_enablePolling = true;
}

void CrateImpl::setupScanTx(unsigned int rod, unsigned int channel) {
  BocCard &bocCard = *(getRod(rod)->getBocCard());

  char *mappings = config->getFibreMappings(partition, crate, rod);

  // Turn all channels off
  for(unsigned int ch=0;ch<48;ch++){
    if(ch == channel) continue;
    int txChannel = mappings[ch * 3 + 0];
    bocCard.setLaserCurrent(txChannel, 0);
  }

  try {
    BOCChannelConfig channelConfig = config->getBOCConfig(partition, crate, rod, channel);

    int txChannel = mappings[channel * 3 + 0];

    // Configure clock/command output
    bocCard.setLaserCurrent(txChannel, channelConfig.current);
  } catch(ConfigurationException &c) {
    std::cout << "No current data for channel " << channel << " leaving at 0\n";
  }
}

void CrateImpl::setupScanRawTx(unsigned int rod, unsigned int channel) {
  BocCard &bocCard = *(getRod(rod)->getBocCard());

  int mappedChannel = 0; // -1

  // Turn all channels off
  for(unsigned int ch=0;ch<48;ch++){
    // Don't turn the current channel off...
    if(ch == channel) 
      continue;

    bocCard.setLaserCurrent(ch, 0);
  }

  if(mappedChannel >= 0) {
    try {
#warning "This shouldn't be a constant current! (where should it come from though? Configuration may not be right...)"
      // Configure clock/command output
      bocCard.setLaserCurrent(channel, 0xa0);
    } catch(ConfigurationException &c) {
      std::cout << "No current data for channel " << mappedChannel << " leaving at 0\n";
    }
  } else {
    std::cout << "No mapping for channel " << channel << " leaving at 0\n";
  }
}

} // End of namespace SctApi
