#include <algorithm>

#include <unistd.h>
#include <sys/stat.h>   // mkdir etc
#include <sys/types.h>

#include "primListWrapper.h"
#include "PrimBuilder.h"
#include "sctConf/configuration.h"
#include "SctApi.h"
#include "SctApiDebug.h"
#include "crate.h"
#include "ScanResultWriter/scan.h"
#include "utility.h"

#include "SctApiHisto.h" // For histogramming helper functions 

#ifdef USE_IS
#include "Sct/IoExceptions.h"
#include "ScanResultWriter/ScanResultWriter.h"
#endif

#include "Sct/SctNames.h"

#include "CommonWithDsp/primParams.h"
#include "CommonWithDsp/registerIndices.h"

#include <boost/date_time/posix_time/posix_time.hpp>

using namespace std;
using namespace SctConfiguration;
using namespace boost::posix_time;

namespace SctApi {

int SctApi::setupRawData(unsigned int partition, unsigned int crate, unsigned int rod, 
                         int delay, int units, bool setMask, const Trigger *trig) {
  int success = 1;

  PrimBuilder &builder = PrimBuilder::instance();

#if (R_SET_ROD_MODE < 101)
#error "Old form (update primParams.h)"
#else
  boost::shared_ptr<PrimListWrapper> setupList(new PrimListWrapper(2));

  // Configuration readback requires bit 19 of RRIF_CMND to be set to "trigger" the capture
  rodMode(partition, crate, rod, CALIBRATION_MODE + INMEM_EVT_CAPTURE_MODE, 0, 1, units+1, delay, 0);

  // Disable formatters (after ROD mode, which would reenable them...)
  for(int i=0; i<8; i++) {
    builder.writeRegister(setupList, FMT_LINK_EN(i), 0, 12, 0x0);
  }

//   // Enable FE Command Outputs with SP0
//   builder.writeRegister(0x1c7, 0, 2, 0x1, setupList);

  if(setMask) {
    // Set masks
    builder.writeRegister(setupList, FE_CMND_MASK_0_LO, 0, 32, 0xffffffff);
    builder.writeRegister(setupList, FE_CMND_MASK_0_HI, 0, 16, 0xffff);
    builder.writeRegister(setupList, FE_CMND_MASK_1_LO, 0, 32, 0xffffffff);
    builder.writeRegister(setupList, FE_CMND_MASK_1_HI, 0, 16, 0xffff);

    // Load enable?
//   builder.writeRegister(setupList, RRIF_CMND_1, 20, 1, 0x1);
    // Load masks
    builder.writeRegister(setupList, RRIF_CMND_1, 2, 1, 0x1);
  }

  sendPrimList(partition, crate, rod, setupList);
  int responseCode = awaitResponse(partition, crate, rod, 10);

  // Do something that might trigger a response
  if(trig) {
    sendTrigger(partition, crate, rod, trig);
  } else {
    sendL1A(partition, crate, rod);
  }

//   // Set MUX to read out
//   writeRODRegister(partition, crate, rod, 0x1c7, 25, 3, 0x1);

  // Leave formatters disabled, will be enabled by next scan

//   // Enable formatters
//   for(int i=0; i<12; i++) {
//     builder.writeRegister(0x0 + i, 0, 12, 0x1, setupList);
//   }
#endif

  return success;
}

void SctApi::dumpRawEvent(unsigned int partition, unsigned int crate, unsigned int rod, int units, 
                          unsigned long *bufferA, unsigned long *bufferB) {
  static int captureCount = 0;

  cout << "Dumping RAW FIFOs to file, count " << captureCount << endl;

  string dirName = Sct::SctNames::getTempDir() + "/Captures";

  // Ignore success (0) or failure
  mkdir(dirName.c_str(), S_IRUSR|S_IWUSR|S_IXUSR);
  
  char *saveDir = getcwd(NULL, 0);
  chdir(dirName.c_str());

  char fileName[100];
  sprintf(fileName, "RawCaptureA_%05d.bin", captureCount);

  ofstream foutA(fileName, ios::binary | ios::trunc);
  foutA.write((char*)&bufferA[0], units*4);

  // Only increment second time
  sprintf(fileName, "RawCaptureB_%05d.bin", captureCount++);

  ofstream foutB(fileName, ios::binary | ios::trunc);
  foutB.write((char*)&bufferB[0], units*4);

  chdir(saveDir);

  free(saveDir);
}

/// Is there a hit at a position
static int lookupFifoHit(UINT16 *bufA, UINT16 *bufB,
                         int channel, int pos) {
  int word = (channel / 16) % 3;
  int bit = channel % 16;

  int val = 0;
  if(channel<48) {
    if(bufA)
      val = ((bufA[pos*3 + word] >> bit) & 0x1);
  } else {
    if(bufB)
      val = ((bufB[pos*3 + word] >> bit) & 0x1);
  }

  return val;
}

void SctApi::rawData(unsigned int partition, unsigned int crate, unsigned int rod, 
                     int delay, int units, bool setMask, const Trigger *trig) {
  setupRawData(partition, crate, rod, delay, units, setMask, trig);
  
  unsigned long *bufferA = readFifo(partition, crate, rod, 0, 0, units);
  unsigned long *bufferB = readFifo(partition, crate, rod, 0, 1, units);

  for(int ch=0; ch<96; ch++) {
    cout << "Ch ";
    cout.width(2);
    cout << ch << ": ";
    for(int i=0; i<units; i++) {
      // Data is in sets of three 16bit words 
      cout.width(1);
      cout << lookupFifoHit((UINT16*)bufferA, (UINT16*)bufferB, ch, i);
    }
    cout << endl;
  }

  delete [] bufferA;
  delete [] bufferB;

  if(setMask) {
    // Restore to how it was
    calib_init();
  }
}

// Could add do all modules option but then where would the data go?!
void SctApi::doRawScan(boost::shared_ptr<Scan> scan, int delay, int width, bool configure, bool clkBy2) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "doRawScan\n";
  }

  scan->setStartTime(second_clock::universal_time());

  scan->setRunNumber(runNumber);
  scan->setScanNumber(scanNumber);

  scanNumber ++;

  scan->print();

  const int realUnits = width;
  int units;

  if(checkDebugOption(DEBUG_SAVE_RAW_CAPTURE)) {
    units = 0x8000;
  } else {
    units = realUnits;
  }

#warning "Raw scan only ROD 0"
  int partition = 0;
  int crate = 0;
  int rod = 0;

  if(!getCrate(partition, crate)->checkBOCLasersOn(rod)) {
    cout << "Continuing raw scan using BOC that has its lasers cut out\n";
    if(mrs) {
      *mrs << "BOC_INTERLOCKED" << MRS_WARNING << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Continuing raw scan (BOC may be interlocked)") << ENDM;
    }
  }

  int nBins = scan->getScanPoints1().size();

  map<UINT32, UINT32 *> dataMap;

  for(map<UINT32, ABCDModule> ::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {
    UINT32 *rawData = new UINT32[nBins * 768 * 2];

    for(int i=0; i<nBins*768*2; i++) {
      rawData[i] = 0;
    }

    dataMap.insert(make_pair(iter->first, rawData));
  }

  char *mappings;

  try { 
    mappings = config->getFibreMappings(partition, crate, rod);
  } catch(ConfigurationException &c) {
    cout << "No fibre mappings for appropriate module: \n";
    cout << c.what() << endl;

    {
      boost::mutex::scoped_lock lock(logMutex);
      log << "  (Not found)\n";
    }

    if(mrs) {
      *mrs << "SCAN_ABORT" << MRS_ERROR 
           << MRS_QUALIF("SCTAPI") << MRS_QUALIF("doRawScan") 
           << MRS_TEXT("Aborting scan due to missing fibre mappings") 
           << ENDM;
    }
    return;
  }

  if(clkBy2)
    getCrate(partition, crate)->enterBOCClockBy2Mode(rod);

  if(configure) {
    cout << "Configure modules for scan\n";
  } else {
    cout << "Don't configure module\n";
  }

  /**** Start scan ****/

  Scan::TrigPoints trigs = scan->getVariableTrigs();

  int bin = 0;
  vector<FLOAT32> scanPoints = scan->getScanPoints1();

  const boost::shared_ptr<Trigger> trigger = scan->getTrigger1();

#if USE_IS
    if(m_isDict) {
      ISInfoInt binNumber(0);
      m_isDict->insert("SCTAPIServer.currentBin", binNumber);

      ISInfoInt maxNumber(scanPoints.size()-1);
      m_isDict->insert("SCTAPIServer.maxBin", maxNumber);
    }
#endif

  for(vector<FLOAT32>::const_iterator point_iter = scanPoints.begin();
      point_iter != scanPoints.end();
      point_iter++, bin++) {

    cout << "Bin: " << bin << endl;

#if USE_IS
        if(m_isDict) {
          ISInfoInt binNumber(bin);
          m_isDict->update("SCTAPIServer.currentBin", binNumber);
        }
#endif

    modifyABCDVar(scan->getScanVariable1(), *point_iter);
    if(configure) {
      setABCDModules(SCTAPI_BANK_SCAN);
      sendAllABCDModules(SCTAPI_BANK_SCAN);
    }

    for(unsigned int t=0; t<trigs[bin]; t++) {
      setupRawData(partition, crate, rod, delay, units, false, trigger.get());

      unsigned long* bufferA(readFifo(partition, crate, rod, 0, 0, units));
      unsigned long* bufferB(readFifo(partition, crate, rod, 0, 1, units));

      if(!bufferA || !bufferB) {
        cout << "Failed to read FIFO\n";
        return;
      }

      if(checkDebugOption(DEBUG_SAVE_RAW_CAPTURE)) {
        dumpRawEvent(partition, crate, rod, units, bufferA, bufferB);
      }

      for(map<UINT32, ABCDModule> ::const_iterator iter = moduleMap.begin();
          iter != moduleMap.end();
          iter ++) {

        UINT32 mid = iter->first;

        unsigned int partition, crate, rod, channel;
        Utility::getpcrc(mid, partition, crate, rod, channel);

        UINT32 *rawData = dataMap.find(mid)->second;

        if(!rawData) {
          cout << "Can't find preallocated memory!\n";
          return;
        }

        for(int link=0; link<2; link++) {
          int fibre = mappings[channel * 3 + link + 1];  // Looking for offsets 1 and 2

          for(int i=0; i<realUnits; i++) {
            int val = lookupFifoHit((UINT16*)bufferA, (UINT16*)bufferB, fibre, i);
            rawData[bin * (realUnits*2) + (realUnits * link) + i] += val;
          }
        }
      }
    }
  }

  
#if USE_IS
    if(m_isDict) {
      m_isDict->remove("SCTAPIServer.currentBin");
      m_isDict->remove("SCTAPIServer.maxBin");
    }
#endif

  delete [] mappings;

  /**** Finished scan ****/

  scan->setEndTime(second_clock::universal_time());

  if(clkBy2)
    getCrate(partition, crate)->leaveBOCClockBy2Mode(rod);


  // Mostly the same data, just copy in the FIFOs for each module
  scan_result_ptrs scanResult;

  ScanHeader &header = scanResult.header;

  // Fill in common data
  header.version = CURRENT_SCANHEADER_VERSION;
  header.length = sizeof(ScanHeader);
  header.runNumber = scan->getRunNumber();
  header.scanNumber = scan->getScanNumber();

  strncpy(header.startTime, to_iso_string(scan->getStartTime()).c_str(), 16);
  strncpy(header.endTime, to_iso_string(scan->getEndTime()).c_str(), 16);
  header.startTime[15] = 0;    // Null terminate the string
  header.endTime[15] = 0;

  header.scanType = scan->getScanVariable1(); 
  header.npoints = nBins;
  header.size = nBins * 2 * 768 * 2;
  header.dataType = SR_DT_RAWHIST; // Was SLICE_COMPRESSED
#warning "Gratuitous memory hog (do raw scan)!"
  header.width = SR_WD_32;
  header.pntPoints = sizeof(ScanHeader);
  header.pntEvents = header.pntPoints + nBins * sizeof(FLOAT32);
  header.pntErrors = header.pntEvents + nBins * sizeof(UINT32);
  header.pntData = header.pntErrors + nBins * sizeof(UINT32);

  // The points
  FLOAT32 *points = new FLOAT32[nBins];
  for(int i=0; i<nBins; i++) {
    points[i] = scan->getScanPoints1()[i];
  }
  scanResult.points = points;

  // The event counts (trigs already copied)
  UINT32 *events = new UINT32[nBins];
  for(int i=0; i<nBins; i++) {
    events[i] = trigs[i];
  }
#warning "Could be counted (raw scan)?"
  scanResult.nEvents = events;

  // The error counts     
#warning "What errors should be recorded in raw scan?"
  //  (If complete failure then no histogram generated)
  UINT32 *errors = new UINT32[nBins];
  for(int i=0; i<nBins; i++) {
    errors[i] = 0;           // Of course! 
  }
  scanResult.nErrorEvents = errors;

  // Use the same time for all modules
  time_t secs;
  secs = time(NULL);
  struct tm broke;
  broke = *(gmtime(&secs));

  //  for each module
  for(map<UINT32, ABCDModule> ::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {

    // Fill in module specific information

    strncpy(header.moduleName, convertToString(iter->first).c_str(), 16);
    header.moduleName[15] = 0;                        // Null terminate the string
    header.config = iter->second;

    scanResult.data = dataMap.find(iter->first)->second;

    // What to do with it now!

    if(checkDebugOption(DEBUG_SAVE_HISTOGRAM)) {
      const int BUFF_SIZE = 50;
      char filename[BUFF_SIZE];

      int written = snprintf(filename, BUFF_SIZE, "RawHistogram_%s", convertToString(iter->first).c_str());
      //                                                  20021112143712
      strftime(filename + written, BUFF_SIZE - written, "_%Y%m%d%H%M%S.bin", &broke);

      cout << "Saving raw histogram: " << filename << endl;
      saveHistogramToFile(scanResult, filename);
    }

#if USE_IS
    cout << "Writing raw data to IS\n";

    try {
      SctData::ScanResultWriter::publish(scanResult);
    } catch(Sct::IoException &e) {
      cout << "ScanResult publish failed:\n" << e.what() << endl;
      e.sendToMrs(MRS_ERROR);
    }
#endif

    delete [] dataMap.find(iter->first)->second;
    // Make sure its not used again (duplicate module in moduleMap!)
    dataMap.find(iter->first)->second = 0;
  }

  delete [] scanResult.points; 
  delete [] scanResult.nEvents;
  delete [] scanResult.nErrorEvents;

  // Delete any unused memory
  for(map<UINT32, UINT32 *> ::const_iterator iter = dataMap.begin();
      iter != dataMap.end();
      iter ++) {
    if(iter->second) {
      delete [] iter->second;
    }
  }

  if(checkDebugOption(DEBUG_BOC_SETUP))
    printBOCSetup(partition, crate, rod);

  getCrate(partition, crate)->configureBOC(rod);
}

#warning "Should this stay, or become something better..."
// This needs trigger added, read all channels into one histogram...
void SctApi::readRawData(unsigned int partition, unsigned int crate, unsigned int rod, 
                         int delay, bool setMask) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "Raw histogram of all ROD data\n";
  }

  int units = 768 * 2;

  scan_result_ptrs scanResult;

  // When the scan started
  time_t secs;
  secs = time(NULL);
  struct tm broke;
  broke = *(gmtime(&secs));

  int nBins = 96; // scan->getScanPoints1().size();

  ScanHeader &header = scanResult.header;

  header.version = 0;
  header.length = sizeof(ScanHeader);
  header.runNumber = runNumber;
  header.scanNumber = scanNumber;
  strncpy(header.moduleName, "raw link data", 16);
  header.moduleName[15] = 0;                        // Null terminate the string
  header.scanType = 0; 
  header.npoints = 96;
  header.size = nBins * 2 * 1024 * 2;
#warning "Need new type for this (readRawData)"
  header.dataType = SR_DT_SLICE;

#warning  "Gratuitous memory hog! (readRawData)"
  header.width = SR_WD_32;

  // No module config for all modules!
  unsigned long *startAddress = (unsigned long*)&header.config;
  fill(startAddress, startAddress + sizeof(ABCDModule)/sizeof(unsigned long), 0);

  header.pntPoints = sizeof(ScanHeader);
  header.pntEvents = header.pntPoints + nBins * sizeof(FLOAT32);
  header.pntErrors = header.pntEvents + nBins * sizeof(UINT32);
  header.pntData = header.pntErrors + nBins * sizeof(UINT32);

  // The points
  FLOAT32 *points = new FLOAT32[nBins];
  for(int i=0; i<nBins; i++) {
    points[i] = i;
  }
  scanResult.points = points;

  // The event counts
  UINT32 *events = new UINT32[nBins];
  for(int i=0; i<nBins; i++) {
    events[i] = 1;
  }
  scanResult.nEvents = events;

  // The error counts
  UINT32 *errors = new UINT32[nBins];
  for(int i=0; i<nBins; i++) {
#warning "More 0 error counts (readRawData)"
    errors[i] = 0;
  }
  scanResult.nErrorEvents = errors;

  // The data!
  int binSize = 0x800*4;   // In bytes, so works for 16bit histograms

  setupRawData(partition, crate, rod, delay, units, setMask);

  unsigned long *bufferA = readFifo(partition, crate, rod, 0, 0, units);
  unsigned long *bufferB = readFifo(partition, crate, rod, 0, 1, units);

  char *data = new char[nBins * binSize];

  unsigned long *longData = (unsigned long *)data;

  for(int ch=0; ch<96; ch++) {
    for(int i=0; i<units; i++) {
      int val = lookupFifoHit((UINT16*)bufferA, (UINT16*)bufferB, ch, i);
      longData[ch * (units) + i] = val;
    }
    cout << endl;
  }

  delete [] bufferA;
  delete [] bufferB;

  scanResult.data = data;

  // Save histogram to file
  const int BUFF_SIZE = 50;
  char filename[BUFF_SIZE];

  int written = snprintf(filename, BUFF_SIZE, "RawHistogramFull");
  //                                                  20021112143712
  strftime(filename + written, BUFF_SIZE - written, "_%Y%m%d%H%M%S.bin", &broke);

  cout << "Saving raw histogram: " << filename << endl;

  saveHistogramToFile(scanResult, filename);
}

vector<char> SctApi::probeWithTrigger(unsigned int partition, unsigned int crate, unsigned int rod, 
                                      const Trigger *trigger, signed int harness) {
  int delay = 5;
  const int realUnits = 100;
  int units;

  if(checkDebugOption(DEBUG_SAVE_RAW_CAPTURE)) {
    units = 0x8000;
  } else {
    units = realUnits;
  }

  int startCh = 0;
  int endCh = 47;

  if(harness != -1) {
    startCh = harness*6;
    endCh = harness*6 + 5;
  }

  // Each channel has two links!
  int numLinks = ((endCh+1) - startCh) * 2;
  vector<char> result(numLinks);

  if(!getCrate(partition, crate)) {
    cout << "Aborting probe on nonexistant crate!\n";
    if(mrs) {
      *mrs << "CRATE_NOTEXIST" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", crate) 
           << MRS_TEXT("Aborting probe (Crate does not exist)") << ENDM;
    }
    return result;
  }

  if(!getCrate(partition, crate)->RODPresent(rod)) {
    cout << "No configured ROD ! Aborting probe\n";
    if(mrs) {
      *mrs << "ROD_NONEXIST" << MRS_ERROR << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Aborting probe (ROD not configured)") << ENDM;
    }

    return result;
  }

  if(!getCrate(partition, crate)->checkBOCLasersOn(rod)) {
    cout << "Continuing probe using BOC that has its lasers cut out\n";
    if(mrs) {
      *mrs << "BOC_INTERLOCKED" << MRS_WARNING << MRS_QUALIF("SCTAPI") 
           << MRS_PARAM<int>("crate", crate) 
           << MRS_PARAM<int>("rod", rod)
           << MRS_TEXT("Continuing probe (BOC interlocked)") << ENDM;
    }
  }

  // Always set up masks first
  int setupSuccess = setupRawData(partition, crate, rod, delay, units, true, trigger);

  if(!setupSuccess) {
    cout << "Probe failed!";
    for(int i = 0; i<numLinks; i++) {
      result[i] = 'X';
    }
    return result;
  }

  unsigned long *bufferA = readFifo(partition, crate, rod, 0, 0, units);
  unsigned long *bufferB = readFifo(partition, crate, rod, 0, 1, units);

  if(!bufferA && !bufferB) {
    cout << "No data refusing to analyse probe!\n";
    return vector<char>();
  }

  if(checkDebugOption(DEBUG_SAVE_RAW_CAPTURE)) {
    dumpRawEvent(partition, crate, rod, units, bufferA, bufferB);
  }

  // Now we've saved the data we can pretend it always was 100
  units = realUnits;

  char *data = new char[units];

  for(int index=0; index<numLinks; index++) {
    int link = index + startCh*2;
    bool same = true;
    bool clock2 = true;
    bool clock4 = true;
    signed char lastValue2 = -1;
    signed char lastValue = -1;

    int headerState = 0;

    for(int i=0; i<units; i++) {
      int val = lookupFifoHit((UINT16*)bufferA, (UINT16*)bufferB, link, i);

      if(i > 0) {   // not first pass
        if(same && lastValue != val) {
          same = false;
        }
	if(clock2 && lastValue == val) {
          clock2 = false;
        }
	if(clock4 && lastValue2 == val) {
          clock4 = false;
        }
        lastValue2 = lastValue;
      }

      lastValue = val;
      data[i] = val;

      switch(headerState) {
      case 0:
        if(val == 1) headerState = 1;
        break;
      case 1:  // 1xxxxxx
        if(val == 0) headerState = 10; // 10 
        else headerState = 2;          // 11
        break;
      case 2:  // 11xxxxx
        if(val == 0) headerState = 20; // 110 
        else headerState = 3;          // 111
        break;
      case 3:  // 111xxxx
        if(val == 0) headerState = 4;  // 1110 
        else headerState = 99;         // junk
        break;
      case 4:  // 1110xxx
        if(val == 1) headerState = 21; // 11101 - event header
        else headerState = 99;         // junk
        break;
      case 10: // 10xxxxx
        if(val == 1) headerState = 11; // 101 
        else headerState = 99;         // junk
        break;
      case 11: // 101xxxx
        if(val == 0) headerState = 12; // 1010 
        else headerState = 99;         // junk
        break;
      case 12: // 1010xxx
        if(val == 1) headerState = 13; // 10101 
        else headerState = 15;         // 10100
        break;
      case 13: // 10101xx
        if(val == 0) headerState = 14; // 101010 
        else headerState = 99;         // junk
        break;
      case 14: // 10101xx
        if(val == 0) headerState = 22; // 1010100 - soft reset 
        else headerState = 99;         // junk
        break;
      case 15: // 10100xx
        if(val == 1) headerState = 16; // 101001 
        else headerState = 99;         // junk
        break;
      case 16: // 10100xx
        if(val == 0) headerState = 23; // 1010010 - BC reset 
        else headerState = 99;         // junk
        break;
      case 20: // L1A 110
        // Any more "1"  - could be another L1A?
        if(val == 1) headerState = 31;
        break;
      case 22: // SR  1010100
      case 23: // BCR 1010010
        // Any more "1" and we assume junk
        if(val == 1) headerState = 99;
        break;
      case 31:  // 1xxxxxx
        if(val == 0) headerState = 99; // 10 
        else headerState = 32;         // 11
        break;
      case 32:  // 11xxxxx
        if(val == 0) headerState = 33; // 110 
        else headerState = 99;         // 111
        break;
      case 33: // second L1A 110
        // Any more "1" and we assume junk
        if(val == 1) headerState = 99;
        break;
      default:
        // including valid event header or junk
        // - do nothing 
        break;
      }
    }

    bool verbose = checkDebugOption(DEBUG_VERBOSE_PROBE);

    if(verbose) {
      cout.width(2);
      cout << (link/2) << " (" << link%2 << ") ";
    }

    if(same) {
      result[index] = int(lastValue) + '0';
      if(verbose) cout << "All " << int(lastValue) << "'s\n";
    } else if(clock2) {
      result[index] = '2';
      if(verbose) cout << "Clock by 2\n";
    } else if(clock4) {
      result[index] = '4';
      if(verbose) cout << "Clock by 4\n";
    } else {
      switch(headerState) {
      case 20:
        result[index] = 'L';
        if(verbose) cout << "L1A\n";
        break;
      case 21:
        result[index] = 'E';
        if(verbose) cout << "Event Header\n";
        break;
      case 22:
        result[index] = 'S';
        if(verbose) cout << "Soft Reset\n";
        break;
      case 23:
        result[index] = 'B';
        if(verbose) cout << "BC Reset\n";
        break;
      case 33:
        result[index] = 'l';
        if(verbose) cout << "2 * L1A\n";
        break;
      default:
        result[index] = 'J';
        if(verbose) cout << "Junk\n";
        break;
      }
      if(verbose) {
        cout << "       ";
        for(int i=0; i<min(units, 60); i++) {
          cout << int(data[i]);
        }
        cout << endl;
      }
    }
  }

  delete [] bufferA;
  delete [] bufferB;

  delete [] data;

  return result;
}

vector<char> SctApi::probe(unsigned int partition, unsigned int crate, unsigned int rod, 
                           signed int harness) {
  return probeWithTrigger(partition, crate, rod, 0, harness);
}

vector<vector<char> > SctApi::probeScan(unsigned int partition, unsigned int crate, unsigned int rod, 
                                        boost::shared_ptr<Scan> scan, signed int harness) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "probeScan\n";
  }

  scan->setRunNumber(runNumber);
  scan->setScanNumber(scanNumber);

  scan->print();

  vector<FLOAT32> scanPoints = scan->getScanPoints1();

  const boost::shared_ptr<Trigger> trigger = scan->getTrigger1();

  vector<vector<char> > result(scanPoints.size());

  if(scanPoints.size() == 0) {
    cout << "Asked for probeScan with 0 scan points\n";
  }

  for(unsigned int point = 0; point < scanPoints.size(); point ++) {
    FLOAT32 val = scanPoints[point];

    try {
      modifyABCDVar(scan->getScanVariable1(), val);
    } catch(SctApiException &s) {
      std::cout << "SctApi Exception thrown by modifyABCDVar in probeScan:\n";
      std::cout << s.what() << std::endl;
    }

    result[point] = probeWithTrigger(partition, crate, rod, trigger.get(), harness);
  }

  if(checkDebugOption(DEBUG_BOC_SETUP))
    printBOCSetup(partition, crate, rod);

  getCrate(partition, crate)->configureBOC(rod);

  return result;
}

bool SctApi::checkAllModulesProbe(string value) {
  bool result = true;

  for(list<RodLabel>::const_iterator rl = rodList.begin();
      rl!=rodList.end();
      rl++){
    vector<char> values = probe(rl->partition, rl->crate, rl->rod);

    // For each module on ROD
#warning "Slightly inefficient (all modules for each ROD)"
    for(map<UINT32, ABCDModule>::const_iterator mi = moduleMap.begin();
        mi!=moduleMap.end();
        mi ++) {
      UINT32 currMid = mi->first;

#warning "Only checks non redundant links"
      // To do "off ROD" modules need to trigger probe off a TIM trigger and read all RODs at the same time

      unsigned int partition, crate, rod, channel;
      ::SctApi::Utility::getpcrc(currMid, partition, crate, rod, channel);

      if(partition == rl->partition && crate == rl->crate && rod == rl->rod) {
        char *mappings = config->getFibreMappings(partition, crate, rod);

        if(mappings[channel * 3 + 1] != (char)DATA_LINK_OFF) {
          char link0 = values[mappings[channel * 3 + 1]];
          if(find(value.begin(), value.end(), link0) == value.end()) {
            result = false;
            cout << "Failed on link 0 of " << currMid << " (" << link0 << ")\n";

            if(mrs) {
              // Make it into a string
              char link0str[2];
              link0str[0] = link0;
              link0str[1] = 0;
              *mrs << "MODULE_STATE" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
                   << MRS_PARAM<int>("moduleID", currMid) 
                   << MRS_PARAM<const char *>("sn", convertToString(currMid).c_str())
                   << MRS_PARAM<const char *>("required", value.c_str())
                   << MRS_PARAM<const char *>("found", link0str)
                   << MRS_TEXT("Link0 of module not in required state") << ENDM;
            }

            break;
          }
        }

        if(mappings[channel * 3 + 2] != (char)DATA_LINK_OFF) {
          char link1 = values[mappings[channel * 3 + 2]];
          if(find(value.begin(), value.end(), link1) == value.end()) {
            result = false;
            cout << "Failed on link 1 of " << currMid << " (" << link1 << ")\n";
            if(mrs) {
              // Make it into a string
              char link1str[2];
              link1str[0] = link1;
              link1str[1] = 0;
              *mrs << "MODULE_STATE" << MRS_INFORMATION << MRS_QUALIF("SCTAPI") 
                   << MRS_PARAM<int>("moduleID", currMid) 
                   << MRS_PARAM<const char *>("sn", convertToString(currMid).c_str())
                   << MRS_PARAM<const char *>("required", value.c_str())
                   << MRS_PARAM<const char *>("found", link1str)
                   << MRS_TEXT("Link1 of module not in required state") << ENDM;
            }

            break;
          }
        }
      }
    }
  }

  return result;
}

}
