#include <iostream>
#include <cmath>

#include "processor.h"

#include "ABCD/ABCDscans.h"
#include "ABCD/ABCDchip.h"

#include "SctApi.h"
#include "crate.h"
#include "utility.h"

#include "extraScans.h"

using namespace std;

const static short tokens[20]={0x01, 0x03, 0x05, 0x07, 0x0b,
                               0x0d, 0x0f, 0x15, 0x17, 0x1b,
                               0x1d, 0x1f, 0x2b, 0x2d, 0x2f,
                               0x35, 0x37, 0x3b, 0x3d, 0x3f};

const char *chipRoles[] = {"MISSING", "DEAD", "END", "MASTER", "SLAVE", "LONELY", "PARANOID"};

static INT32 setChipRole(ABCDModule *module, UINT32 theChip, ABCD_ROLES role){
//   cout << "Set chip " << theChip << " role to " << chipRoles[role] << endl;

  ABCDChip*   chipPtr;
  ABCDConfig* configPtrLast;  /* chip (n-1) */ 
  ABCDConfig* configPtr;      /* chip n     */
  ABCDConfig* configPtrNext;  /* chip (n+1) */

  UINT32 lastChip, nextChip;

  if(theChip>N_SCT_CHIPS-1) return -1;  /* chip out of range */
  if(module->active==0) return -2;  /* module inactive */

  chipPtr = &module->chip[theChip];
  if(chipPtr->active==0) return -3; /* chip inactive */

  lastChip = theChip-1;
  nextChip = theChip+1;

  if(theChip==0)  lastChip = 11;
  if(theChip==11) nextChip =  0;

  configPtrLast = &module->chip[lastChip].basic.config;
  configPtr     = &module->chip[theChip].basic.config;
  configPtrNext = &module->chip[nextChip].basic.config;

  switch(role){
    case MISSING:
      /* who cares! */
      break;
    case DEAD:
      configPtrNext->outputBypass = 1;
      configPtrLast->inputBypass  = 1;
      break;
    case END:
      configPtr->master = 1;
      configPtr->end    = 1;
      configPtr->feedThrough = 1;
      break;
    case MASTER:
      configPtr->master = 0;
      configPtr->end    = 0;
      configPtr->feedThrough = 1;  /* active low */
      break;
    case SLAVE:
      configPtr->master = 1;
      configPtr->end    = 0;
      configPtr->feedThrough = 1;
      break;
    case LONELY:
      configPtr->master = 0;
      configPtr->end    = 1;
      configPtr->feedThrough = 1;
      break;
    case PARANOID:
      configPtrNext->outputBypass = 1;
      configPtrLast->inputBypass  = 1;
      configPtr->master = 0;
      configPtr->end    = 1;
      configPtr->feedThrough = 1;
      break;
    default:
      break;
  }

  return 0;
}

static void setToken(ABCDModule* module, unsigned token, int link0, int link1) {
  cout << "Set token to 0x" << hex << token << dec << ": " << link0 << " " << link1 << endl;

  // Set up defaults...
  for(unsigned int theChip=0; theChip<N_SCT_CHIPS; theChip++){
       module->chip[theChip].basic.config.outputBypass = 0;
       module->chip[theChip].basic.config.inputBypass = 0;
       module->chip[theChip].basic.config.master = 1;        /* active low */
       module->chip[theChip].basic.config.end = 0;           /* active high */
       module->chip[theChip].basic.config.feedThrough  = 1;  /* active low  */
  }

  for(unsigned int theChip=0; theChip<N_SCT_CHIPS; theChip++){
    
    if( (token >> theChip) & 0x1 ) { /* chip to be read out */

      if(theChip==0) {
        if(link0) {
          if((token&0x003F)==0x0001) { /* only M0 in use */
            setChipRole(module,theChip,LONELY);
          } else {
            setChipRole(module,theChip,MASTER);
          }
        } else {
          setChipRole(module,theChip,END);
        }

      } else if(theChip==6) {
        if (link1) { 
          if((token&0x0FC0)==0x0040) { /* only M8 in use */
            setChipRole(module,theChip,LONELY);
          } else {
            setChipRole(module,theChip,MASTER);
          }
        } else {
          setChipRole(module,theChip,END);
        }

      } else if(theChip<6) {
        /* if( (link0<2) && (pow(2,(theChip+1))>(token&0x3F)) ) { dpsf */
        if( (link0<2) && (1<<(theChip+1)>(token&0x3F)) ) {
          setChipRole(module,theChip,END);
        } else {
          setChipRole(module,theChip,SLAVE);
        }

      } else {
        /* if( (link1<2) && (pow(2,(theChip+1))>token) ) {  dpsf */
        if( (link1<2) && (1<<(theChip+1)>token) ) {
          setChipRole(module,theChip,END);
        } else {
          setChipRole(module,theChip,SLAVE);
        }
      }

    } else {
      setChipRole(module,theChip,DEAD);
    }
  }

//   if(debugLevel > 1) {
    for(unsigned int theChip=0; theChip<N_SCT_CHIPS; theChip++){
      cout << " OB: " << module->chip[theChip].basic.config.outputBypass;
      cout << " IB: " << module->chip[theChip].basic.config.inputBypass;
      cout << " bM: " << module->chip[theChip].basic.config.master;    
      cout << " END: " << module->chip[theChip].basic.config.end;       
      cout << " FT: " << module->chip[theChip].basic.config.feedThrough;
      cout << endl;
    }
//   }

  return;
}

namespace SctApi {

void SctApi::modifyABCDMask(UINT32 mid, UINT32* mask) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDMask on " << mid << "\n";
  }

  ABCDModule *module = lookupConfig(mid);

  if(module) {
    for(int c=0; c<12; c++) {
      for(int m=0; m<4; m++) {
        module->chip[c].basic.mask[m] = mask[c*4+m];
      }
    }
  } 
}

void SctApi::modifyABCDRC(UINT32 mid, UINT32 chip, UINT16 index, FLOAT32 p0, FLOAT32 p1, FLOAT32 p2){
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDRC on " << mid << " " << chip << "\n";
  }
  ABCDModule *module = lookupConfig(mid);

  if(module) {
    module->chip[chip].caldata.rc_function = index;
    module->chip[chip].caldata.rc_params[0]=p0;
    module->chip[chip].caldata.rc_params[1]=p1;
    module->chip[chip].caldata.rc_params[2]=p2;
  }
}

void SctApi::modifyABCDTrims(UINT32 mid, UINT8* trims) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDTrims on " << mid << "\n";
  }

  ABCDModule *module = lookupConfig(mid);

  if(module) {
    for(int c=0; c<12; c++) {
      for(int t=0; t<128; t++) {
        module->chip[c].trim[t] = trims[c*128+t];
      }
    }
  }
}

void SctApi::modifyABCDVar(UINT32 typ, FLOAT32 var) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVar (all)\n";
  }

  cout << "Modify var on all modules: setting " << typ << " to " << var << endl;

  for(map<UINT32, ABCDModule>::const_iterator iter = moduleMap.begin();
      iter != moduleMap.end();
      iter ++) {
    modifyABCDVar(iter->first, typ, var);
  }
}

#define fToInt(val) ((int)((val)+0.5))

void SctApi::modifyABCDVar(UINT32 mid, UINT32 typ, FLOAT32 var) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVar (module " << mid << ") " << typ << " " << var << "\n";
  }

  switch(typ) {
    // First some module parameters (BOC)
  case ST_RX_DELAY:
  case ST_RX_DELAY0:
  case ST_RX_DELAY1:
  case ST_RX_THRESHOLD:
  case ST_RX_THRESHOLD0:
  case ST_RX_THRESHOLD1:
  case ST_TX_CURRENT:
  case ST_TX_MARKSPACE:
  case ST_TX_DELAY:
  case ST_TX_COARSE:
  case ST_TX_FINE:

    // Global BOC registers
  case SCT_SCAN_BOC_BPM_PHASE:
  case SCT_SCAN_BOC_BREG_PHASE:
  case SCT_SCAN_BOC_V0_PHASE:
  case SCT_SCAN_BOC_V1_PHASE:
  case SCT_SCAN_BOC_VRN_FINE:

  case SCT_SCAN_TX_CHANNELS:
  case SCT_SCAN_RAW_TX_CHANNELS:
    {
      unsigned int partition, crate, rod, channel;
      Utility::getpcrc(mid, partition, crate, rod, channel);

      cout << "Calling BOCCard method for module " << mid << " " << typ << " " << var << endl;

      try {
        // Do channel mapping
        getCrate(partition, crate)->modifyBOCParam(rod, channel, typ, fToInt(var), false);
      } catch(CrateException &c) {
        cout << "CrateException: " << c.what() << endl;
      }
      break;
    }

    // Some module wide parameters

   case ST_BYPASS:
     {
       // If more then could be more general
       ABCDModule *mConf = lookupConfig(mid);

       if(!mConf) {
         cout << "No module configuration to change\n";
         return;
       }

       ABCDModule &module = *mConf;

       /*
        * A method used to select certain predetermined bypass configurations.
        * Made obsolete by ST_TOKEN (below) but included here for reasons of
        * backward compatibility. Recoded to use setToken function.
        */
       UINT8 int_val = (UINT8)((var+0.5)/1.0);
       switch(int_val){
       case 0: /* ME---- ME---- */
         setToken(&module, 0x03 +(0x03<<6),1,1);
         break;
       case 1: /* MSE--- MSE--- */
         setToken(&module, 0x07 +(0x07<<6),1,1);
         break;
       case 2: /* MSSE-- MSSE-- */
         setToken(&module, 0x0F +(0x0F<<6),1,1);
         break;
       case 3: /* MSSSE- MSSSE- */
         setToken(&module, 0x1F +(0x1F<<6),1,1);
         break;
         /* case 4 is the default configuration */
       case 5: /* MDE--- MDE--- */
         setToken(&module, 0x05 +(0x05<<6),1,1);
         break;
       case 6: /* MSDE-- MSDE-- */
         setToken(&module, 0x0b +(0x0b<<6),1,1);
         break;
       case 7: /* MSSDE- MSSDE- */
         setToken(&module, 0x17 +(0x17<<6),1,1);
         break;
       case 8: /* MSSSDE MSSSDE */
         setToken(&module, 0x2f +(0x2f<<6),1,1);
         break;
       case 9: /* MSSSSS DSSSSE */
         setToken(&module, 0x3f +(0x3e<<6),2,0);
         break;
       case 10: /* DSSSSE MSSSSS */
         setToken(&module, 0x3e +(0x3F<<6),0,2);
         break;
       case 11: /* MSSSSS SSSSSE */
         setToken(&module, 0x3f +(0x3f<<6),2,0);
         break;
       case 12: /* SSSSSE MSSSSS */
         setToken(&module, 0x3f +(0x3F<<6),0,2);
         break;
       default: /* MSSSSE MSSSSE */
         setToken(&module, 0x3F +(0x3F<<6),1,1);
         break;
       }
       break;
     }

  case ST_TOKEN:
     {
       // If more then could be more general
       ABCDModule *mConf = lookupConfig(mid);

       if(!mConf) {
         cout << "No module configuration to change\n";
         return;
       }

       ABCDModule &module = *mConf;

       /*
        * This variable iterates over a set of token/data passing
        * schemes to determine the functionality of each token link.
        * The sequence sets all the combinations to read-out
        * n chips by first testing the bypass token (chip n-2 to n) 
        * for all combinations of (n-2) chips. Secondly trying the 
        * direct token (n-1 to n) for all combinations of (n-1) chips.
        *
        * For configurations 0-19, both datalinks are exercised at once.
        * For the remaining configurations, only one datalink is
        * used at once, as outlined below:
        *
        * o Even Configurations 20-34 read out chip S9 via datalink 0
        *   using the bypass link between E5 and S9.
        * 
        * o Odd  Configurations 21-35 read out chip S1 via datalink 1
        *   using the bypass link between E13 and S1.
        *
        * o Even configurations 36-60 read out chip M8 via datalink 0
        *   using first the bypass link between S4 and M8 and then
        *   the direct link between E5 and M8.
        *
        * o Odd  configurations 37-61 read out chip M0 via datalink 1
        *   using first the bypass link between S12 and M0 and then
        *   the direct link between E13 and M0.
        *
        * Only cases 0-35 have been implememted in the design of the
        * barrel hybrid. The logical inconsistency in the order between 
        * cases 20-35 and 36-61 is done such that the subset of the 
        * schemes used by the barrel design form a consecutive series.
        *
        * Orignal code by Lars Eklund, March 2001
        */

       int myVal = fToInt(var);

       if(myVal<20) {              /* use both datalinks */
         setToken(&module, (tokens[myVal] + (tokens[myVal] << 6)),1,1);

       } else if(myVal<36) {       /* use datalink 0 */
         if(!((int)myVal%2)) {
           setToken(&module, (tokens[(myVal-20)/2+12] + (0x1 << 7)),2,0);
         } else {                /* use datalink 1 */
           setToken(&module, (0x2 + (tokens[(myVal-21)/2+12] << 6)),0,2);
         }
       } else { // only possible for forward modules
         if(!(myVal%2)) {
           setToken(&module, (tokens[((myVal-36)/2+7)] + (1 << 6)),2,0);
         } else {
           setToken(&module, (0x1 + (tokens[((myVal-37)/2+7)] << 6)),0,2);
         }
       }
       break;
   }

    // The rest are chip parameters
  default:
    {
      for(int c=0; c<12; c++) {
        modifyABCDVar(mid, c, typ, var);
      }
    }
    break;
  }
}

void SctApi::modifyABCDVar(UINT32 mid, UINT32 c, UINT32 typ, FLOAT32 val) {
  {
    boost::mutex::scoped_lock lock(logMutex);
    log << "modifyABCDVar (module " << mid << ": chip " << c  << ") " << typ << " " << val << "\n";
  }

  ABCDModule *mConf = lookupConfig(mid);

  if(!mConf) {
    cout << "No module configuration to change\n";
    return;
  }

  ABCDModule &module = *mConf;

  ABCDChip &chip = module.chip[c];

  switch(typ) {
  case ST_ELAPSED: 
    // Do nothing
    break;
  case ST_VTHR: 
    chip.basic.vthr = (char) fToInt(val/2.5);
    break;
  case ST_VCAL:
    chip.basic.vcal = (char) fToInt(val/0.625);
    break;
  case ST_STROBE_DELAY:
    chip.basic.delay = (char) fToInt(val);
    break;
  case ST_PREAMP:
    chip.basic.preamp = (char) fToInt(val/9.2);
    break;
  case ST_SHAPER:
    chip.basic.shaper = (char) fToInt(val/1.2);
    break;
  case ST_TRIM:
    for(int i=0; i<128; i++) {
      chip.trim[i] = (char) fToInt(val);
    }
    break;
  case ST_MASK:
    {
      UINT32 myval = (UINT32) fToInt(val);
      UINT32 mask = myval;
      for(int i=0; i<16; i++) {
        mask = (mask << 2) + myval;
      }

      for(int i=0; i<4; i++) {
        chip.basic.mask[i] = mask;
      }
    }
    break;
  case ST_ROLE:
    switch(fToInt(val)) {
    case END:
      chip.basic.config.master = 1;  // active low
      chip.basic.config.end = 1;     // active high
      chip.basic.config.feedThrough = 1; // active low
      break;
    case MASTER:
      chip.basic.config.master = 0;  // active low
      chip.basic.config.end = 0;     // active high
      chip.basic.config.feedThrough = 1;
      break;
    case SLAVE:
      chip.basic.config.master = 1;  // active low
      chip.basic.config.end = 0;     // active high
      chip.basic.config.feedThrough = 1;
      break;
    case LONELY:
      chip.basic.config.master = 0;  // active low
      chip.basic.config.end = 1;     // active high
      chip.basic.config.feedThrough = 1;
      break;
    case PARANOID:
      chip.basic.config.master = 0;  // active low
      chip.basic.config.end = 1;     // active high
      chip.basic.config.feedThrough = 1;
    case DEAD:
    default:
      module.chip[(c+1)%12].basic.config.outputBypass = 1;
      module.chip[(c+11)%12].basic.config.inputBypass = 1;
    case MISSING:
      break;
    } // End switch role

  case ST_NMASK:
    {
      int channels = fToInt(val);
      if(channels > 128) channels = 128;

      for(int i=0; i<4; i++) {
        // How many channels in this group
        int chan = channels - i*32;
        if(chan >= 32) {
          chip.basic.mask[i] = 0x00000000;
        } else {
          int value = 0;
          for(int b=0; b<chan; b++) {
            value <<= 1;
            value ++;
          }
          chip.basic.mask[i] = ~value;
        }
      }
    }
    break;
  case ST_CAL_MODE:
    chip.basic.config.calibMode = fToInt(val);
    break;
  case ST_COMPRESSION:
    chip.basic.config.readoutMode = fToInt(val);
    break;
  case ST_TRIM_RANGE:
    chip.basic.config.trimRange = fToInt(val);
    break;
  case ST_EDGE_DETECT:
    chip.basic.config.edgeDetect = fToInt(val);
    break;
  case ST_SEND_MASK:
    chip.basic.config.mask = fToInt(val)?1:0;
    break;
  case ST_ACCUMULATE:
    chip.basic.config.accumulate = fToInt(val);
    break;
  case ST_FEEDTHROUGH:
    chip.basic.config.feedThrough = fToInt(val);
    break;
  case ST_ACTIVE:
    chip.active = fToInt(val);
    break;

    //  Can't be set on a chip by chip basis!
//    case ST_BYPASS:
//    case ST_TOKEN:

#ifdef DO_TOKENS_BY_CHIP
  case ST_BYPASS:
    {
      /*
       * A method used to select certain predetermined bypass configurations.
       * Made obsolete by ST_TOKEN (below) but included here for reasons of
       * backward compatibility. Recoded to use setToken function.
       */
      UINT8 int_val = (UINT8)((val+0.5)/1.0);
      switch(int_val){
      case 0: /* ME---- ME---- */
        setToken(&module, 0x03 +(0x03<<6),1,1);
        break;
      case 1: /* MSE--- MSE--- */
        setToken(&module, 0x07 +(0x07<<6),1,1);
        break;
      case 2: /* MSSE-- MSSE-- */
        setToken(&module, 0x0F +(0x0F<<6),1,1);
        break;
      case 3: /* MSSSE- MSSSE- */
        setToken(&module, 0x1F +(0x1F<<6),1,1);
        break;
        /* case 4 is the default configuration */
      case 5: /* MDE--- MDE--- */
        setToken(&module, 0x05 +(0x05<<6),1,1);
        break;
      case 6: /* MSDE-- MSDE-- */
        setToken(&module, 0x0b +(0x0b<<6),1,1);
        break;
      case 7: /* MSSDE- MSSDE- */
        setToken(&module, 0x17 +(0x17<<6),1,1);
        break;
      case 8: /* MSSSDE MSSSDE */
        setToken(&module, 0x2f +(0x2f<<6),1,1);
        break;
      case 9: /* MSSSSS DSSSSE */
        setToken(&module, 0x3f +(0x3e<<6),2,0);
        break;
      case 10: /* DSSSSE MSSSSS */
        setToken(&module, 0x3e +(0x3F<<6),0,2);
        break;
      case 11: /* MSSSSS SSSSSE */
        setToken(&module, 0x3f +(0x3f<<6),2,0);
        break;
      case 12: /* SSSSSE MSSSSS */
        setToken(&module, 0x3f +(0x3F<<6),0,2);
        break;
      default: /* MSSSSE MSSSSE */
        setToken(&module, 0x3F +(0x3F<<6),1,1);
        break;
      }
      break;
    }

  case ST_TOKEN:
    {
      /*
       * This variable iterates over a set of token/data passing
       * schemes to determine the functionality of each token link.
       * The sequence sets all the combinations to read-out
       * n chips by first testing the bypass token (chip n-2 to n) 
       * for all combinations of (n-2) chips. Secondly trying the 
       * direct token (n-1 to n) for all combinations of (n-1) chips.
       *
       * For configurations 0-19, both datalinks are exercised at once.
       * For the remaining configurations, only one datalink is
       * used at once, as outlined below:
       *
       * o Even Configurations 20-34 read out chip S9 via datalink 0
       *   using the bypass link between E5 and S9.
       * 
       * o Odd  Configurations 21-35 read out chip S1 via datalink 1
       *   using the bypass link between E13 and S1.
       *
       * o Even configurations 36-60 read out chip M8 via datalink 0
       *   using first the bypass link between S4 and M8 and then
       *   the direct link between E5 and M8.
       *
       * o Odd  configurations 37-61 read out chip M0 via datalink 1
       *   using first the bypass link between S12 and M0 and then
       *   the direct link between E13 and M0.
       *
       * Only cases 0-35 have been implememted in the design of the
       * barrel hybrid. The logical inconsistency in the order between 
       * cases 20-35 and 36-61 is done such that the subset of the 
       * schemes used by the barrel design form a consecutive series.
       *
       * Orignal code by Lars Eklund, March 2001
       */

      int myVal = fToInt(val);

      if(myVal<20) {              /* use both datalinks */
        setToken(&module, (tokens[(int)myVal] + (tokens[(int)myVal] << 6)),1,1);
      } else if(myVal<36) {       /* use datalink 0 */
        if(!((int)myVal%2)) {
          setToken(&module, (tokens[(int)(myVal-20)/2+12] + (0x1 << 7)),2,0);
        } else {                /* use datalink 1 */
          setToken(&module, (0x2 + (tokens[((int)myVal-21)/2+12] << 6)),0,2);
        }
      } else { // only possible for forward modules
        if(!((int)myVal%2)) {
          setToken(&module, (tokens[(int)((myVal-36)/2+7)] + (1 << 6)),2,0);
        } else {
          setToken(&module, (0x1 + (tokens[(int)((myVal-37)/2+7)] << 6)),0,2);
        }
      }
      break;
    }
#endif
  case ST_QTHR: 
    {
      ABCDCaldata &calPtr = chip.caldata;
      float v, vv;

      switch(calPtr.rc_function){
      case 1: /* polynomial */
        v = calPtr.rc_params[0] + 
          val * (calPtr.rc_params[1] + 
                 (val * calPtr.rc_params[2]) );
        break; 
      case 2: /* grillo */
        if((val==0)|| (calPtr.rc_params[1]==0))
          vv = 0;
        else
          vv = calPtr.rc_params[2] / 
            (calPtr.rc_params[1] *val);
        v  = calPtr.rc_params[0] + 
          calPtr.rc_params[2] / (float) sqrt(1+(vv*vv)); 
        break; 
      case 3: /* exp */
        if(calPtr.rc_params[1]==0)
          v = 0;
        else
          v = calPtr.rc_params[2] + 
            calPtr.rc_params[0]/(1+ (float)exp(-val/calPtr.rc_params[1]));
        break;  
      case 4: /* linear fit */
        v = calPtr.rc_params[0] + 
          (val * calPtr.rc_params[1]);
        break; 
      default:
        v = 0;
        break;
      }

      UINT8 int_val = ((UINT8)((v+1.25)/(float)2.5));  /* 2.5mV / DAC bit */
      chip.basic.vthr = int_val;
      break;
    }

  case ST_QCAL:
    /*
     * calculate the actual vcal step size taking into
     * account the capacitor correction factor:
     * step size = (c_factor * 0.0625) fC / DAC bit 
     */
    {
      FLOAT32 step = (float) 0.0625 * chip.caldata.c_factor;
      UINT8 int_val = (UINT8)((val+(step/2))/step);
      chip.basic.vcal = int_val;
      break;
    }
  case ST_TARGET:
    {
      /*
       * record the target to which the chip has been trimmed
       * added PWP 07.01.03
       */
      UINT8 int_val = ((UINT8)((val+1.25)/(float)2.5));  /* 2.5mV / DAC bit */

      /* int_val must be in range if declared UINT8
         if(int_val<0x00) int_val = 0x00;
         if(int_val>0xff) int_val = 0xff;
      */
      chip.target = int_val;
      break;
    }

  case ST_TTHR:
    {
      /*
       * set the threshold in mV wrt the trim target
       * added PWP 07.01.03
       */
      UINT8 int_val = ((UINT8)((val+1.25)/(float)2.5));  /* 2.5mV / DAC bit */
      int_val += chip.target;

      /*
        if(int_val<0x00) int_val = 0x00;
        if(int_val>0xff) int_val = 0xff;
      */
      chip.basic.vthr = int_val;
      break;
    }

  case ST_RX_DELAY:
  case ST_RX_DELAY0:
  case ST_RX_DELAY1:
  case ST_RX_THRESHOLD:
  case ST_RX_THRESHOLD0:
  case ST_RX_THRESHOLD1:
  case ST_TX_CURRENT:
  case ST_TX_MARKSPACE:
  case ST_TX_DELAY:
  case ST_TX_COARSE:
  case ST_TX_FINE:
    {
      unsigned int partition, crate, rod, channel;
      Utility::getpcrc(mid, partition, crate, rod, channel);

      cout << "Calling BOCCard method for module " << channel << " " << typ << " " << val << endl;

      // Do channel mapping
      getCrate(partition, crate)->modifyBOCParam(rod, channel, typ, fToInt(val), false);
      break;
    }

  default:
    cout << "Unknown variable type " << typ << "\n";
    break;
  } // End switch var
}
}
