#include <iostream>

#include <vector>

#include "SctApiImpl.h"
#include "ABCD/ABCDchip.h"

using namespace SctApi;
using namespace boost;
using namespace boost::posix_time;

#include "crate.h"

SctApiException::SctApiException(const CrateException &c) throw() : desc(c.what()) {}

TriggerImpl::TriggerImpl() 
{
  incCmd = incData = 0;
}

TriggerImpl::TriggerImpl(const TriggerImpl &other) :
  trigSequence(other.trigSequence), incCmd(other.incCmd), incData(other.incData)
{
}

boost::shared_ptr<TriggerImpl> TriggerImpl::clone(const boost::shared_ptr< ::SctApi::Trigger> other) {
  boost::shared_ptr<TriggerImpl> result(new TriggerImpl());

  result->trigSequence = other->getRODTriggers();

  unsigned short command;
  unsigned short incr;
  other->getCommIncr(command, incr);

  result->incCmd = command;
  result->incData = incr;

  return result;
}

TriggerImpl::~TriggerImpl() 
{
}

void TriggerImpl::singleL1A() 
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::doubleL1A(unsigned short delay) 
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::delayedL1A(unsigned short delay) 
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::calL1A(unsigned short delay) 
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(CALIBRATION_PULSE, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::pulseL1A(unsigned short delay)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(PULSE_INPUT_REG, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::softL1A(unsigned short delay)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(SOFT_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::softCalL1A(unsigned short delay, unsigned short delay2)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(SOFT_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(CALIBRATION_PULSE, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay2));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}
 
void TriggerImpl::softPulseL1A(unsigned short delay, unsigned short delay2)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(SOFT_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(PULSE_INPUT_REG, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay2));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::bcL1A(unsigned short delay)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(BC_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::bcCalL1A(unsigned short delay, unsigned short delay2)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(BC_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(CALIBRATION_PULSE, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay2));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}
 
void TriggerImpl::bcPulseL1A(unsigned short delay, unsigned short delay2)
{
  trigSequence.clear();

  trigSequence.push_back(std::make_pair(BC_RESET, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay));
  trigSequence.push_back(std::make_pair(PULSE_INPUT_REG, 0));
  trigSequence.push_back(std::make_pair(DELAY, delay2));
  trigSequence.push_back(std::make_pair(L1_TRIGGER, 0));

  incCmd = incData = 0;
}

void TriggerImpl::setCommIncr(unsigned short command, unsigned short incr)
{
  incCmd = command;
  incData = incr;
}

void TriggerImpl::getCommIncr(unsigned short &command, unsigned short &incr) const
{
  command = incCmd;
  incr = incData;
}

const ::SctApi::Trigger::RODTriggers TriggerImpl::getRODTriggers() const 
{
  return trigSequence;

//   ::SctApi::Trigger::RODTriggers result;

  

//   return result;
}

#if 0
void TriggerImpl::copy(const ::SctApi::Trigger &trig) 
{
  short unsigned cmd, data;

  trig.getCommIncr(cmd, data);

  incCmd = cmd;
  incData = data;

  trigSequence = trig.getRODTriggers();
}
#endif

ScanDefImpl::ScanDefImpl() 
  : scanVariable(0),
    scanVariable2(0),
    trigSequence(new TriggerImpl), trigSequence2(new TriggerImpl),
    full(0),
    bits32(1),
    loopCalLine(0),
    distSlave(1),
    debug(0),
    tim(0),
    nth(0), nth_rem(0),
    scanNumber(0),
    runNumber(0),
    startTime(second_clock::universal_time()),
    endTime(second_clock::universal_time())
{
  trigPoints.clear();
  scanPoints.clear();
  scanPoints2.clear();
}

ScanDefImpl::~ScanDefImpl() 
{
//   delete trigSequence;
//   delete trigSequence2;
}

ScanDefImpl::ScanDefImpl(const ScanDefImpl &other) :
  scanPoints(other.scanPoints),
  scanPoints2(other.scanPoints2),
  trigPoints(other.trigPoints),
  scanVariable(other.scanVariable),
  scanVariable2(other.scanVariable2),

  trigSequence(TriggerImpl::clone(other.trigSequence)),
  trigSequence2(TriggerImpl::clone(other.trigSequence2)),

  full(other.full), 
  bits32(other.bits32),
  loopCalLine(other.loopCalLine),

  distSlave(other.distSlave),
  debug(other.debug),
  tim(other.tim),
  nth(other.nth),
  nth_rem(other.nth_rem),
  startTime(other.startTime),
  endTime(other.endTime)
{
}

boost::shared_ptr<ScanDefImpl> ScanDefImpl::clone(const boost::shared_ptr< ::SctApi::Scan> other) {
  boost::shared_ptr<ScanDefImpl> result(new ScanDefImpl);

  result->trigPoints = other->getVariableTrigs();
  result->scanVariable = other->getScanVariable1();
  result->scanVariable2 = other->getScanVariable2();
  result->scanPoints = other->getScanPoints1();
  result->scanPoints2 = other->getScanPoints2();

  result->trigSequence = TriggerImpl::clone(other->getTrigger1());
  result->trigSequence2 = TriggerImpl::clone(other->getTrigger2());

  result->full = other->getOption(Scan::FULL);
  result->bits32 = other->getOption(Scan::BITS32);
  result->loopCalLine = other->getOption(Scan::LOOPCALLINE);
  result->distSlave = other->getOption(Scan::DISTSLAVE);
  result->debug = other->getOption(Scan::DEBUG);
  result->tim = other->getOption(Scan::TIM);
  result->nth = other->getOption(Scan::NTH);
  result->nth_rem = other->getOption(Scan::NTH_REM);

  result->startTime = other->getStartTime();
  result->endTime = other->getEndTime();

  return result;
}

void ScanDefImpl::configure(unsigned short type, float start, float stop, float step)
{
  scanVariable = type;

  // Add 1 to be inclusive of the stop point, extra 0.1% (over kill) for floating point error 
  int npoints = int(((stop - start) / step) + 1.001);

  scanPoints.resize(npoints);

  for(int i=0; i<npoints; i++) {
    FLOAT32 point = start + i * step;
    scanPoints[i] = point;
  }

  if(allTrigsSame) {
    // Default to 0
    int newNTrig = trigPoints.size() > 0?trigPoints[0]:0;
    trigPoints.resize(std::max(1U, scanPoints.size()), newNTrig);
  }
}

void ScanDefImpl::configure2(unsigned short type, float start, float stop, float step)
{
  scanVariable2 = type;

  int npoints = int(((stop - start) / step) + 1.001);

  scanPoints2.resize(npoints);

  for(int i=0; i<npoints; i++) {
    FLOAT32 point = start + i * step; 
    scanPoints2[i] = point;
  }
}

const boost::shared_ptr< ::SctApi::Trigger> ScanDefImpl::getTrigger1() const
{
  return trigSequence;
}

const boost::shared_ptr< ::SctApi::Trigger> ScanDefImpl::getTrigger2() const
{
  return trigSequence2;
}

boost::shared_ptr< ::SctApi::Trigger> ScanDefImpl::getTrigger1()
{
  return trigSequence;
}

boost::shared_ptr< ::SctApi::Trigger> ScanDefImpl::getTrigger2()
{
  return trigSequence2;
}

void ScanDefImpl::setTrigger1(boost::shared_ptr< ::SctApi::Trigger> trigger)
{
  trigSequence = trigger;

//   trigSequence = TriggerImpl::clone(trigger); // dynamic_cast< ::SctApi::TriggerImpl *>(trigger);
//   trigSequence.copy(*trigger);
}

void ScanDefImpl::setTrigger2(boost::shared_ptr< ::SctApi::Trigger> trigger)
{
  trigSequence2 = trigger;

//   trigSequence2 = TriggerImpl::clone(trigger); // dynamic_cast< ::SctApi::TriggerImpl *>(trigger);
//   trigSequence2.copy(*trigger);
}

void ScanDefImpl::setScanVariable1(unsigned short var)
{
  scanVariable = var;
}

void ScanDefImpl::setScanVariable2(unsigned short var)
{
  scanVariable2 = var;
}

unsigned short ScanDefImpl::getScanVariable1() const
{
  return scanVariable;
}

unsigned short ScanDefImpl::getScanVariable2() const
{
  return scanVariable2;
}

void ScanDefImpl::setNTrigs(unsigned long nTrigs) 
{
  trigPoints.resize(std::max(1U, scanPoints.size()));

  for(unsigned int i=0; i<trigPoints.size(); i++) {
    trigPoints[i] = nTrigs;
  }

  allTrigsSame = true;
}

unsigned long ScanDefImpl::getNTrigs() const
{
  if(trigPoints.size() > 0) {
    return trigPoints[0];
  } else {
    return 0;
  }
}

const ::SctApi::Scan::TrigPoints ScanDefImpl::getVariableTrigs() const
{
  return trigPoints;
}

void ScanDefImpl::setVariableTrigs(const TrigPoints &trigs)
{
  allTrigsSame = false;
  trigPoints = trigs;
}

void ScanDefImpl::setVariableTrigRange(unsigned short start, unsigned short end, unsigned long value)
{
  allTrigsSame = false;
  for(int i=start; i<=end; i++) 
    trigPoints[i] = value;
}

const ::SctApi::Scan::ScanPoints ScanDefImpl::getScanPoints1() const
{
  return scanPoints;
}

const ::SctApi::Scan::ScanPoints ScanDefImpl::getScanPoints2() const
{
  return scanPoints2;
}

void ScanDefImpl::setScanPoints1(const ScanPoints &scans)
{
  scanPoints = scans;

  if(allTrigsSame) {
    // Default to 0
    int newNTrig = trigPoints.size() > 0?trigPoints[0]:0;
    trigPoints.resize(std::max(1U, scanPoints.size()), newNTrig);
  }
}

void ScanDefImpl::setScanPoints2(const ScanPoints &scans)
{
  scanPoints2 = scans;
}

int ScanDefImpl::getOption(enum ScanOptions opt) const
{
  switch(opt) {
  case FULL:
    return full;
  case BITS32:
    return bits32;
  case LOOPCALLINE:
    return loopCalLine;
  case DISTSLAVE:
    return distSlave;
  case DEBUG:
    return debug;
  case TIM:
    return tim;
  case NTH:
    return nth;
  case NTH_REM:
    return nth_rem;
  }
}

void ScanDefImpl::setOption(enum ScanOptions opt, int option) 
{
  switch(opt) {
  case FULL:
    full = option;
    break;
  case BITS32:
    bits32 = option;
    break;
  case LOOPCALLINE:
    loopCalLine = option;
    break;
  case DISTSLAVE:
    distSlave = option;
    break;
  case DEBUG:
    debug = option;
    break;
  case TIM:
    tim = option;
    break;
  case NTH:
    nth = option;
    break;
  case NTH_REM:
    nth_rem = option;
    break;
  }
}

void ScanDefImpl::print() const {
  std::cout << "Scan\n";
  std::cout << "Run: " << getRunNumber() << " Scan: " << getScanNumber() << std::endl;

  if(allTrigsSame) {
    std::cout << getNTrigs() << " triggers per scan point\n";

    TrigPoints points = getVariableTrigs();

    int initVal = points[0];

    for(TrigPoints::const_iterator iter = points.begin();
        iter != points.end();
        iter++) {
      if(*iter != initVal) std::cout << " Trigger count array not constant\n";
    }

    if(points.size() != getScanPoints1().size()) {
      std::cout << " Trigger count array not the same size as scan point array!!!\n";
    }

  } else {
    std::cout << "Triggers per scan point:\n";

    TrigPoints points = getVariableTrigs();

    int i=0;
    for(TrigPoints::const_iterator iter = points.begin();
        iter != points.end();
        i++, iter++) {
      if(i && (i%20 == 0)) std::cout << std::endl;
      std::cout.width(4);

      std::cout << *iter << " ";
    }
    std::cout << std::endl;
  }

  std::cout << "Binning mode " << (getOption(Scan::FULL)?"full":"condensed") << std::endl;
  std::cout << "Bin size " << (getOption(Scan::BITS32)?"32bit":"16bit") << std::endl;
  std::cout << "Cal line loop " << (getOption(Scan::LOOPCALLINE)?"yes":"no") << std::endl;
  std::cout << "Use TIM for triggers (synchronous) " << (getOption(Scan::TIM)?"yes":"no") << std::endl;

  std::cout << "Slave DSPs: ";
  switch(getOption(Scan::DISTSLAVE)) {
  case 0: 
    {
      std::cout << "All to Slave 0";
      if(getOption(Scan::NTH) > 1) {
        std::cout << ": Capture trigger%" << getOption(Scan::NTH) << "==" << getOption(Scan::NTH_REM);
      }
      break;
    }
  case 1: std::cout << "According to groups"; break;
  case 2: 
    {
      std::cout << "Router distribution";
      if(getOption(Scan::NTH) > 1) {
        std::cout << ": Capture trigger%" << getOption(Scan::NTH) << "==" << getOption(Scan::NTH_REM);
      }
      break;
    }
  default: std::cout << "Invalid type!"; break;
  }
  std::cout << std::endl;

  if(getOption(Scan::DEBUG)) {
    std::cout << "In debug mode\n";
  }

  if(getScanPoints2().size() > 0) {
    std::cout << "Module set 0\n";
  } else {
    std::cout << "Scan parameters\n";
  }

  std::cout << "Scan over variable number " << getScanVariable1() << std::endl;

  std::cout << getScanPoints1().size() << " scan points:\n";
  int i=0;

  std::vector<FLOAT32> points = getScanPoints1();

  for(std::vector<FLOAT32>::const_iterator iter=points.begin();
      iter != points.end();
      i++, iter++) {
    if(i && (i%20 == 0)) std::cout << std::endl;
    std::cout.width(4);

    std::cout << *iter << " ";
  }
  std::cout << std::endl;

  std::cout << "Trigger:\n";

  getTrigger1()->print();

  if(getScanPoints2().size() > 0) {
    std::cout << "Module set 1\n";

    std::cout << "Scan over variable number " << getScanVariable2() << std::endl;

    std::cout << getScanPoints2().size() << " scan points:\n";
    int i=0;

    std::vector<FLOAT32> points2 = getScanPoints2();

    for(std::vector<FLOAT32>::const_iterator iter=points2.begin();
        iter != points2.end();
        i++, iter++) {
      if(i && (i%20 == 0)) std::cout << std::endl;
      std::cout.width(4);

      std::cout << *iter << " ";
    }
    std::cout << std::endl;

    std::cout << "Trigger2:\n";

    getTrigger2()->print();
  } else {
    std::cout << "Second trigger not defined\n";
  }

  for(int i=0; i<4; i++) {
    std::cout << "Modules in group " << i << std::endl;

    std::list<std::string> groupList = getModuleList(i);

    for(std::list<std::string>::const_iterator iter = groupList.begin();
        iter != groupList.end();
        iter++) {
      std::cout << " " << *iter << std::endl;
    }
  }
}


void ScanDefImpl::setScanNumber(unsigned int scan) 
{
  scanNumber = scan;
}
 
unsigned int ScanDefImpl::getScanNumber() const 
{
  return scanNumber;
}

void ScanDefImpl::setRunNumber(unsigned int run)
{
  runNumber = run;
}

unsigned int ScanDefImpl::getRunNumber() const 
{
  return runNumber;
}

std::list<std::string> ScanDefImpl::getModuleList(unsigned int group) const 
{
  std::list<std::string> result;

  if(group<groupLists.size()) {
    result = groupLists[group];
  }

  return result;
}
 
void ScanDefImpl::setModuleList(unsigned int group, std::list<std::string> newList)
{
  if(group+1 > groupLists.size()) {
    groupLists.resize(group + 1);
  }
  groupLists[group] = newList;
}

unsigned int ScanDefImpl::getNGroups() const
{
  return m_maxGroup;
}

void ScanDefImpl::setNGroups(unsigned int val)
{
  m_maxGroup = val;
}

void TriggerImpl::print() const 
{
  Trigger::RODTriggers trigs = getRODTriggers();

  std::cout << "Printing trigger: length " << trigs.size() << std::endl;

  if(trigs.size() > 6) {
    std::cout << "Trigger sequence over size\n";
  }

  unsigned short trigData, trigCmd;
  getCommIncr(trigData, trigCmd);

  for(unsigned int i = 0; i < trigs.size(); i++) {
    std::cout.width(4);

    std::cout << trigs[i].first << ": " << trigs[i].second << "   ";
    if(trigData != 0 && trigCmd == i) {
      std::cout << "+" << trigData;
    }
    std::cout << std::endl;
  }
}

ptime ScanDefImpl::getStartTime() const {
    return startTime;
}

void ScanDefImpl::setStartTime(ptime t) {
    startTime = t;
}
  
ptime ScanDefImpl::getEndTime() const {
    return endTime;
}

void ScanDefImpl::setEndTime(ptime t) {
    endTime = t;
}

bool RodLabel::operator==(const RodLabel &rhs) const {
  return partition == rhs.partition && crate == rhs.crate && rod == rhs.rod;
}

bool RodLabel::operator<(const RodLabel &rhs) const {
  if(partition < rhs.partition) return true;
  if(crate < rhs.crate) return true;
  if(rod < rhs.rod) return true;

  return false;
}
