#include <algorithm>

#include "CommonWithDsp/primParams.h"

#include "RodCrate/RodPrimList.h"

#include "PrimBuilder.h"
#include "utility.h"

using namespace SctApi;

PrimBuilder *PrimBuilder::singleton = 0;

PrimBuilder::PrimBuilder() {}

void PrimBuilder::slavePrimList(boost::shared_ptr<PrimListWrapper> parentList, 
                                boost::shared_ptr<PrimListWrapper> primList, 
                                unsigned int slaveNumber, bool await, bool response) {
  try {
    primList->list->bufferBuild();
  } catch (SctPixelRod::PrimListException &p) {
    std::cout << "Exception building prim list for slave " << slaveNumber << std::endl;
    std::cout << p.getDescriptor() << " ";
    std::cout << p.getData1() << ", " << p.getData2() << "\n";
    return;
  }

  // Get a copy of buffer and use SEND_SLAVE_LIST/START_SLAVE_LIST primitives
  // to have MDSP send it to the slave
  long buffLength = primList->list->getBufferLength();
  unsigned long* bufferStart = primList->list->getBuffer();

  // Create Slave List primitives
  int offset = sizeof(SEND_SLAVE_LIST_IN)/4;

  long *slaveData = new long[offset+buffLength];

  SEND_SLAVE_LIST_IN &slaveList = *(SEND_SLAVE_LIST_IN *)slaveData;

#if (R_SEND_SLAVE_LIST == 103)
  slaveList.slaveNumber = slaveNumber;
  slaveList.listLength = buffLength;
  slaveList.slavePrimList = (unsigned int*)DEFAULT;
  slaveList.slaveRepData = (unsigned int*)DEFAULT;
#else 
#error "SEND_SLAVE_LIST not compiled (new revision)"
#endif

  // Copy prim list as send data
  for (int i=0; i< buffLength; i++) {
    slaveData[offset + i] = bufferStart[i];
  }

  parentList->addPrimitive(4+offset+buffLength, 1, SEND_SLAVE_LIST, R_SEND_SLAVE_LIST, slaveData);

  delete [] slaveData;

  startSlaveList(parentList, slaveNumber, await, response);
}

void PrimBuilder::startSlaveList(boost::shared_ptr<PrimListWrapper> primList, 
                                 unsigned int slaveNumber, bool await, bool response) {
  START_SLAVE_LIST_IN startPrim;
#if (R_START_SLAVE_LIST == 103)
  startPrim.slaveNumber = slaveNumber;
  startPrim.pauseMasterList = await?1:0;
  startPrim.getSlaveReply = response?1:0;
#else
#error "START_SLAVE_LIST not compiled (new revision)"
#endif

  primList->addPrimitive(sizeof(START_SLAVE_LIST_IN)/4, 2, START_SLAVE_LIST, R_START_SLAVE_LIST, (long*)&startPrim);
}

void PrimBuilder::echo(boost::shared_ptr<PrimListWrapper> primList, Utility::MemoryBlock block) {
#if (R_ECHO != 100) 
#error "ECHO revision changed!"
#endif

  long *myData = new long[block.size()];
  std::copy(block.address(), block.address() + block.size(), myData);
  primList->addPrimitive(4+block.size(), 1, ECHO, R_ECHO, myData);

  delete [] myData;
}

void PrimBuilder::startSlave(boost::shared_ptr<PrimListWrapper> prim, int s) {
  START_SLAVE_EXECUTING_IN data;
  data.slaveNumber = s;              // Slave
  data.commOnOff = SLV_DSP_COMM_ON;  // Comm on
  data.slaveType = SLAVE_MONITOR;    // Mode
  data.timeoutInUsec = 0x200000;     // Timeout

  prim->addPrimitive(sizeof(START_SLAVE_EXECUTING_IN)/4, 1, START_SLAVE_EXECUTING, R_START_SLAVE_EXECUTING, (long *)&data);
}

void PrimBuilder::readFifo(boost::shared_ptr<PrimListWrapper> prim, int id, int bank, int elems) {
  RW_FIFO_IN fifo;
#if (R_RW_FIFO == 104)
  fifo.fifoId = id;
  fifo.bank = bank;
  fifo.readNotWrite = 1;
  fifo.numElements = elems;
  fifo.dataBaseAdr = (UINT32 *)0xffffffff;
#else
#error "RW_FIFO revision change"
#endif

  prim->addPrimitive(5, 1, RW_FIFO, R_RW_FIFO, (long *)&fifo);
}

void PrimBuilder::pollRegister(boost::shared_ptr<PrimListWrapper> prim, int r, int off, int width, int val, int timeout) {
  POLL_REG_FIELD_IN poll;

#if (R_POLL_REG_FIELD == 105)
  poll.registerID = r;
  poll.offset = off;
  poll.width = width;
  poll.desiredValue = val;
  poll.timeoutInUsec = timeout;
#else
#error "POLL_REG_FIELD revision changed"
#endif

  prim->addPrimitive(5, 1, POLL_REG_FIELD, R_POLL_REG_FIELD, (long*)&poll);
}

void PrimBuilder::taskOp(boost::shared_ptr<PrimListWrapper> prim, int typ, int op, int data) {
  TASK_OPERATION_IN taskOp;
#if (R_TASK_OPERATION == 100) 
  taskOp.taskType = typ;
  taskOp.taskOperation = op;
  taskOp.data = data;
#else 
#error "TASK_OPERATION revision change"
#endif

  prim->addPrimitive(3, 1, TASK_OPERATION, R_TASK_OPERATION, (long*)&taskOp);
}

void PrimBuilder::readRegister(boost::shared_ptr<PrimListWrapper> prim, int r) {
  RW_REG_FIELD_IN rField;
#if (R_RW_REG_FIELD == 103) || (R_RW_REG_FIELD == 104) || (R_RW_REG_FIELD == 105)
  rField.registerID = r;
  rField.offset = 0;
  rField.width = 32;
  rField.readNotWrite = 1;
  rField.dataIn = 0; // Unused!
#else
#error "RW_REG_FIELD revision change"
#endif

  prim->addPrimitive(5, 5, RW_REG_FIELD, R_RW_REG_FIELD, (long *)&rField);
}

void PrimBuilder::writeRegister(boost::shared_ptr<PrimListWrapper> prim, int r, int off, int wid, int value) {
  RW_REG_FIELD_IN rField;
#if (R_RW_REG_FIELD == 103) || (R_RW_REG_FIELD == 104) || (R_RW_REG_FIELD == 105)
  rField.registerID = r;
  rField.offset = off;
  rField.width = wid;
  rField.readNotWrite = 0;
  rField.dataIn = value;
#else
#error "RW_REG_FIELD revision change"
#endif

  prim->addPrimitive(5, 5, RW_REG_FIELD, R_RW_REG_FIELD, (long *)&rField);
}

void PrimBuilder::sendStream(boost::shared_ptr<PrimListWrapper> prim, int cmdBuff, bool capture) {
  SEND_STREAM_IN sendStream;

#if (R_SEND_STREAM == 100)
  sendStream.cmdBuff = cmdBuff;
  sendStream.captureSerOn = capture;
#else
#error "SEND_STREAM revision change"
#endif

  prim->addPrimitive(4 + sizeof(SEND_STREAM_IN)/4, 3, SEND_STREAM, R_SEND_STREAM, (long *)&sendStream);
}

void PrimBuilder::startEvTrap(boost::shared_ptr<PrimListWrapper> prim) {
  long trap[0] = {};

#if (R_START_EVENT_TRAPPING != 101)
#error "START_EVENT_TRAPPING revision change"
#endif

  prim->addPrimitive(0, 0, START_EVENT_TRAPPING, R_START_EVENT_TRAPPING, trap);
}

void PrimBuilder::stopEvTrap(boost::shared_ptr<PrimListWrapper> prim) {
  long trap[0] = {};

#if (R_STOP_EVENT_TRAPPING != 100)
#error "STOP_EVENT_TRAPPING revision change"
#endif

  prim->addPrimitive(0, 0, STOP_EVENT_TRAPPING, R_STOP_EVENT_TRAPPING, trap);
}

void PrimBuilder::setMemory(boost::shared_ptr<PrimListWrapper> primList, unsigned long address, unsigned long words, long value) {
  SET_MEMORY_IN prim;

#if (R_SET_MEMORY == 100)
  prim.start = (UINT32 *)address;
  prim.size = words;
  prim.val = value;
#else 
#error "SET_MEMORY revision changed"
#endif

  primList->addPrimitive(sizeof(SET_MEMORY_IN)/4, 1, SET_MEMORY, R_SET_MEMORY, (const long *)&prim);
}

void PrimBuilder::readSlaveMemory(boost::shared_ptr<PrimListWrapper> primList, unsigned long slave, unsigned long address, unsigned long words) {
  RW_SLAVE_MEMORY_IN rwPrim;

#if (R_RW_SLAVE_MEMORY == 100)
  rwPrim.slaveNumber = slave;
  rwPrim.readNotWrite = 1;
  rwPrim.slaveAddress = (UINT32 *)address;
  rwPrim.masterAddress = (UINT32 *)0xffffffff;
  rwPrim.numWords = words;
#else
#error "RW_SLAVE_MEMORY revision change"
#endif

  primList->addPrimitive(sizeof(RW_SLAVE_MEMORY_IN)/4, 1, RW_SLAVE_MEMORY, R_RW_SLAVE_MEMORY, (long*)&rwPrim);
}

void PrimBuilder::writeSlaveMemory(boost::shared_ptr<PrimListWrapper> primList, 
                                   unsigned long slave, unsigned long address, unsigned long length, unsigned long *data) {
  long *primData = new long[sizeof(RW_SLAVE_MEMORY_IN)/4 + length];

  RW_SLAVE_MEMORY_IN &rwPrim = *(RW_SLAVE_MEMORY_IN*)primData;

#if (R_RW_SLAVE_MEMORY == 100)
  rwPrim.slaveNumber = slave;
  rwPrim.readNotWrite = 0;
  rwPrim.slaveAddress = (UINT32 *)address;
  rwPrim.masterAddress = (UINT32 *)0xffffffff;
  rwPrim.numWords = length;
#else
#error "RW_SLAVE_MEMORY revision change"
#endif

  std::copy(data, data + length, primData + 5);

  primList->addPrimitive(sizeof(RW_SLAVE_MEMORY_IN)/4 + length, 
                       1, RW_SLAVE_MEMORY, R_RW_SLAVE_MEMORY, primData);
}

void PrimBuilder::rodMode(boost::shared_ptr<PrimListWrapper> rodModeList, 
                          int mode, int flag, int fifoSetup, int nBits, int delay, int message) {
#ifndef SET_ROD_MODE
  throw SctApiException("Set Rod Mode wasn't compiled");
#else
  SET_ROD_MODE_IN rodPrim;

#if (R_SET_ROD_MODE == 101)
  rodPrim.mode = mode;
  rodPrim.flag = flag;
  rodPrim.fifoSetup = fifoSetup;
  rodPrim.nBits = nBits;
  rodPrim.delay = delay;
  rodPrim.evtsPerL1A = 1;         // This is a pixel thing!
  rodPrim.message = message;
#else
  throw SctApiException("Set Rod Mode wasn't compiled (wrong prim version)");
// #error "SET_ROD_MODE revision change"
#endif

  rodModeList->addPrimitive(4 + sizeof(SET_ROD_MODE_IN)/4, 
                            1, SET_ROD_MODE, R_SET_ROD_MODE, (long *)&rodPrim);
#endif
}

void PrimBuilder::moduleMask(boost::shared_ptr<PrimListWrapper> maskList, 
                             int module, int port, int useStructSet, int passOn, int slvs,
                             int cmdLine, int dataLine0, int dataLine1, // 2 and 3 only for pixels
                             int cfg, int modMask0, int modMask1, int maskType, int storage, int maskSet) { 
#if (R_MODULE_MASK == 101)
  // Load module masks
  MODULE_MASK_IN maskPrim;
  maskPrim.moduleNum = module;
  maskPrim.port = port;
  maskPrim.useStructSet = useStructSet;
  maskPrim.passOn = passOn;
  maskPrim.slvBits = slvs;
  maskPrim.cmdLine = cmdLine;
  maskPrim.dataLine[0] = dataLine0;
  maskPrim.dataLine[1] = dataLine1;
  maskPrim.dataLine[2] = 0;   // Pixels only 
  maskPrim.dataLine[3] = 0;   // Pixels only 
  maskPrim.cfg = cfg;               // Don't do manual mask bits set up
  maskPrim.modMask[0] = modMask0;
  maskPrim.modMask[1] = modMask1;
  maskPrim.maskType = maskType;
  maskPrim.storage = storage;
  maskPrim.maskSet = maskSet;

  maskList->addPrimitive(4 + sizeof(MODULE_MASK_IN)/4,
                         1, MODULE_MASK, R_MODULE_MASK, (long *)&maskPrim);
#elif (R_MODULE_MASK == 100) 
#warning "MODULE_MASK method not compiled"
#else 
#error "MODULE_MASK revision change"
#endif
}

void PrimBuilder::masksFromConfig(boost::shared_ptr<PrimListWrapper> maskList, int port) { 
  // Everything ignored except          this
  moduleMask(maskList, ALL_MODULES, port, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
}

void PrimBuilder::masksToSlaves(boost::shared_ptr<PrimListWrapper> maskList, int slvBits) { 
  // Use struct set and pass on...
  moduleMask(maskList, ALL_MODULES, 0, 0, 1, slvBits,
             // The rest is ignored
             0, 0, 0, 0, 0, 0, 0, 0, 0);
}

void PrimBuilder::readModuleData(boost::shared_ptr<PrimListWrapper> primList, int bank, int channel) {
  RW_MODULE_DATA_IN primData;

#if (R_RW_MODULE_DATA == 101) || (R_RW_MODULE_DATA == 102)
  primData.readNotWrite = 1;
  primData.structId = bank;
  primData.moduleNum = channel;
#elif (R_RW_MODULE_DATA == 100)
  primData.readNotWrite = 1;
  primData.structID = bank;
  primData.moduleNum = channel;
#else
#error "RW_MODULE_DATA not compiled (primitive change)"
#endif

  // Ignore configdata as this is a read
  primList->addPrimitive(sizeof(RW_MODULE_DATA_IN)/4, 1, RW_MODULE_DATA, R_RW_MODULE_DATA, (long*)&primData);
}

void PrimBuilder::writeModuleData(boost::shared_ptr<PrimListWrapper> primList, const ABCDModule* module, int bank, int channel) {
  const unsigned int longsInConfig = sizeof(ABCDModule) / 4;

  long *configData = new long[sizeof(RW_MODULE_DATA_IN)/4+longsInConfig];

  RW_MODULE_DATA_IN &params = *(RW_MODULE_DATA_IN*)configData;
#if (R_RW_MODULE_DATA == 101) || (R_RW_MODULE_DATA == 102)
  params.readNotWrite = 0;
  params.structId = bank;
  params.moduleNum = channel;
  params.configData = (ABCDModule *)0xffffffff;
#else
#error "RW_MODULE_DATA not compiled (primitive change)"
#endif

  // Overwrite params.configData with the first word of the config
  std::copy((long *)module, ((long*)module)+longsInConfig, configData+3);

  primList->addPrimitive(sizeof(RW_MODULE_DATA_IN)/4+longsInConfig-1, 1, RW_MODULE_DATA, R_RW_MODULE_DATA, configData);

  delete [] configData;
}

void PrimBuilder::sendConfig(boost::shared_ptr<PrimListWrapper> primList, 
                             int port, int capture, int module0, int module1, int chip, int setLinks, int restore, 
                             int bank, int group, int data, int active, int enableData) {
  SEND_CONFIG_IN primData;

#if (R_SEND_CONFIG == 104) || (R_SEND_CONFIG == 105)
  primData.port = port;
  primData.captureSerOn = capture;
  primData.moduleNum[0] = module0;
  primData.moduleNum[1] = module1;
  primData.chipNum = chip;
  primData.setLinks = setLinks;             // Set links 
  primData.restore = restore;               // Restore links 
  primData.structId = bank;
  primData.groupId = group;
  primData.dataType = data;
  primData.activeOnly = active;
  primData.enableDataTaking = enableData;
#else
#error "Send config not compiled! (new prim version)" 
#endif

  primList->addPrimitive(sizeof(primData)/4, 1, SEND_CONFIG, R_SEND_CONFIG, (long *)&primData);
}

void PrimBuilder::writeModuleVariable(boost::shared_ptr<PrimListWrapper> primList, int bank, int group, int channel, int chip, int type, float value) {
#if (defined(RW_MODULE_VARIABLE) && defined(R_RW_MODULE_VARIABLE)) 
  RW_MODULE_VARIABLE_IN prim;

#if (R_RW_MODULE_VARIABLE == 101) 
#warning "This one doesn't work"
#elif (R_RW_MODULE_VARIABLE == 102) 
  prim.read = 0;
  prim.structId = bank;
  prim.groupId = group;
  prim.module = channel;
  prim.chip = chip;
  prim.varType = type;
  prim.info = 1;         // Send text message confirming what was done
  prim.dataLen = 1;
  prim.data = (MDAT32*)(*(unsigned int*)(&value));
#else 
#warning "Modify config (ROD) not compiled! (new prim version)" 
#endif

  primList->addPrimitive(sizeof(RW_MODULE_VARIABLE_IN)/4, 1, RW_MODULE_VARIABLE, R_RW_MODULE_VARIABLE, (long*)&prim);
#else 
#warning "Modify config (ROD) not compiled! (primitive doesn't exist)" 
#endif
}

void PrimBuilder::bocHistogram(boost::shared_ptr<PrimListWrapper> primList, int samples, int numLoops) {
#ifdef R_BOC_HISTOGRAM
  BOC_HISTOGRAM_IN bocPrim;

#if (R_BOC_HISTOGRAM == 100)
  bocPrim.numSamples = samples;
  bocPrim.numLoops = numLoops;
#else
#warning "BOC_HISTOGRAM revision change"
#endif

  primList->addPrimitive(sizeof(BOC_HISTOGRAM_IN)/4, 1, BOC_HISTOGRAM, R_BOC_HISTOGRAM, (long*)&bocPrim);
#else 
  std::cout << "BOC_HISTOGRAM not compiled\n";
#endif
}

void PrimBuilder::setTrigger(boost::shared_ptr<PrimListWrapper> primList, int currentBin, int histogramSet) 
{
  SET_TRIGGER_IN triggerPrim;

#if (R_SET_TRIGGER == 102)
  triggerPrim.bin = currentBin;
  triggerPrim.set = histogramSet;
#else
#warning "SET_TRIGGER revision change"
#endif

  primList->addPrimitive(sizeof(SET_TRIGGER_IN)/4, 1, SET_TRIGGER, R_SET_TRIGGER, (long*)&triggerPrim);
}

PrimBuilder& PrimBuilder::instance() {
  if(!singleton) {
    singleton = new PrimBuilder;
  }

  return *singleton;
}

