#include "SctDaqRootFile.h"
#include "Exceptions.h"
#include "Sct/SctParameters.h"
#include "CommonWithDsp/ABCD/ABCDscans.h"

#include <boost/scoped_ptr.hpp>

#include <TFile.h>
#include <TH1.h>
#include <TH2.h>
#include <TObjString.h>

#include <sstream>

using namespace std;
using namespace boost;
using namespace Sct;

namespace SctTest {

SctDaqRootFile::SctDaqRootFile(string name) : file(name.c_str()){
    if (!file.IsOpen()) throw NoDataFileException("No datafile: " + name, __FILE__, __LINE__);
}

int SctDaqRootFile::getCycleNum(string serialNum) {
    for (int i=1; i<=6; ++i) {
        ostringstream oss;
        oss << "ModuleName;" << i;
        scoped_ptr<TObjString> s ((TObjString*)file.Get(oss.str().c_str()));
        if (serialNum == s->String().Data()) return i;
    }
    throw NoDataFileException("Module " + serialNum + " not found in file", __FILE__, __LINE__);
}

auto_ptr<TH1F> SctDaqRootFile::getHistData(int cycleNum, int id) {
    ostringstream oss;
    oss << "h_scan" << id << ";" << cycleNum;
    auto_ptr<TH1F> hist ((TH1F*)file.Get(oss.str().c_str()));
    if (!hist.get()) throw DataNotValidException("Couldn't get histogram: " + oss.str(), __FILE__, __LINE__);
    return hist;
}

auto_ptr<TH1F> SctDaqRootFile::getTriggerData() {
    auto_ptr<TH1F> hist ((TH1F*)file.Get("h_scan_tsent"));
    if (!hist.get()) {
        //cout << "File contents:" << endl;
        //f.ls();
        ostringstream oss;
        oss << "Couldn't get histogram: h_scan_tsent in file: " << file.GetName();
        throw DataNotValidException(oss.str(), __FILE__, __LINE__);
    }
    return hist;
}

void SctDaqRootFile::dealWithOccupancy(TH1F& hist0Data, TH1F& hist1Data, TH1F& triggerData, unsigned short scanVar, bool forceConstTrigger) {
    bool bOccupancy = false;
    if (hist0Data.GetMaximum() <= 1)
        bOccupancy = true;

    //cout << "Occupancy hist? " << bOccupancy << endl;

    //Factor of 4 is for consistency with other scans
    for (int i=1; i<=triggerData.GetNbinsX(); ++i) {
        if (!bOccupancy && scanVar != ST_TOKEN)
            triggerData.SetBinContent(i, triggerData.GetBinContent(i)/4);
    }

    for (int i=0; i<hist0Data.GetNbinsX(); ++i) {
        for (int j=0; j<hist0Data.GetNbinsY(); ++j) {
            if (bOccupancy) {
		if (forceConstTrigger) {
		    //Force triggers to all be the same....
		    hist0Data.SetBinContent(i, j+1, triggerData.GetBinContent(1)*hist0Data.GetBinContent(i,j+1));
		    hist1Data.SetBinContent(i, j+1, triggerData.GetBinContent(1)*hist1Data.GetBinContent(i,j+1));
		} else {
		    hist0Data.SetBinContent(i, j+1, triggerData.GetBinContent(j+1)*hist0Data.GetBinContent(i,j+1));
		    hist1Data.SetBinContent(i, j+1, triggerData.GetBinContent(j+1)*hist1Data.GetBinContent(i,j+1));
		}
            }
        }
    }
}

char* SctDaqRootFile::fillData16(const TH1F& hist0Data, const TH1F& hist1Data) {
    int nBins = hist0Data.GetNbinsY();
    unsigned short* data = new unsigned short[2*1024*nBins];

    for (int i=0; i<2048; ++i) {
        for (int j=0; j<nBins; ++j) {
            if (i<nChannelLink)
                data[j*2048+i] = (unsigned short)hist0Data.GetBinContent(i, j+1);
            else if (i>=1024 && i<1024+nChannelLink)
                data[j*2048+i] = (unsigned short)hist1Data.GetBinContent(i-1024, j+1);
            else
                data[j*2048+i] = (unsigned short)0;
        }
    }

    return reinterpret_cast<char*>(data);
}
char* SctDaqRootFile::fillData32(const TH1F& hist0Data, const TH1F& hist1Data) {
    int nBins = hist0Data.GetNbinsY();
    unsigned int* data = new unsigned int[2*1024*nBins];

    for (int i=0; i<2048; ++i) {
        for (int j=0; j<nBins; ++j) {
            if (i<nChannelLink) {
                data[j*2048+i] = (unsigned int)hist0Data.GetBinContent(i, j+1);
            } else if (i>=1024 && i<1024+nChannelLink) {
                data[j*2048+i] = (unsigned int)hist1Data.GetBinContent(i-1024, j+1);
            } else {
                data[j*2048+i] = (unsigned int)0;
            }
        }
    }

    return reinterpret_cast<char*>(data);
}

auto_ptr<TH2D> SctDaqRootFile::fillData(const TH1F& histData) {
    auto_ptr<TH2D> link (new TH2D("link", "Link data", histData.GetNbinsX(), 0, 100, histData.GetNbinsY(), 0, 100));

    for (int i=0; i<histData.GetNbinsX(); ++i) {
        for (int j=0; j<histData.GetNbinsY(); ++j) {
            link->SetBinContent(i+1, j+1, histData.GetBinContent(i,j+1));
        }
    }
    return link;
}

}
