#include "ScanMap.h"
#include "ModuleDataFile.h"
#include "Exceptions.h"
#include "Sct/IoExceptions.h"
#include <iostream>
#include <fstream>
#include <sstream>

using namespace std;

namespace SctTest {
    
ScanMap& ScanMap::instance() {
    static ScanMap sm;
    return sm;
}

string ScanMap::convertSerial(string serialNum) {
    //cout << serialNum << " converted to : " << flush;
    string s = serialNum.replace(0, 2, "20");
    //cout << s << endl;
    return s;
}
    
string ScanMap::getDataFile(string serialNum, const ABCDModule& config, unsigned short scanVar) const {
    serialNum = convertSerial(serialNum);

    if (scanVar == ST_QCAL) scanVar = ST_VCAL;
    if (scanVar == ST_QTHR) scanVar = ST_VTHR;
    
    if (scanVar == ST_VTHR) return getThresholdDataFile(serialNum, config);
    if (scanVar == ST_STROBE_DELAY) return getStrobeDataFile(serialNum, config);
    
    //OK, it must have been a digital scan - return the first one that matches
    
    ScanInfo si;
    si.serialNum = serialNum;
    si.scanVar = scanVar;
    
    DigitalMap::const_iterator i = digitalMap.find(si);
    if (i == digitalMap.end()) {
        cout << "Failed to find digital scan for " << serialNum << " Scan var: " << scanVar << endl;
        throw NoDataFileException("No data file for: " + serialNum, __FILE__, __LINE__);
    }
    else return (*i).second;
}
    
string ScanMap::getThresholdDataFile(string serialNum, const ABCDModule& config) const {
    ThresholdScanDetails tsd(config);
    
    //Find all 
    pair<ThresholdMap::const_iterator, ThresholdMap::const_iterator> its = thresholdMap.equal_range(serialNum);
    for (ThresholdMap::const_iterator it = its.first; it!=its.second; ++it) {
        if ((*it).second == tsd) {
            return (*it).second.fileName;
        }
    }
    cout << "Failed to find threshold scan for " << serialNum << ". Details: charge " << tsd.charge << " Range " << tsd.range << " Trims " << tsd.trims << endl;
    throw NoDataFileException("No data file for: " + serialNum, __FILE__, __LINE__);
}

string ScanMap::getStrobeDataFile(string serialNum, const ABCDModule& config) const {
    StrobeDelayScanDetails tsd(config);
    
    //Find all 
    pair<StrobeMap::const_iterator, StrobeMap::const_iterator> its = strobeMap.equal_range(serialNum);
    for (StrobeMap::const_iterator it = its.first; it!=its.second; ++it) {
        if ((*it).second == tsd) {
            return (*it).second.fileName;
        }
    }
    cout << "Failed to find strobe delay scan for " << serialNum << ". Details: charge " << tsd.charge << " threshold " << tsd.threshold << endl;
    throw NoDataFileException("No data file for: " + serialNum, __FILE__, __LINE__);
}

void ScanMap::initialize(string fileName) {
    ModuleDataFile mdf(fileName);

    digitalMap.clear();
    strobeMap.clear();
    thresholdMap.clear();

    while (mdf.hasMoreRecords()) {
        ModuleDataRecord mdr = mdf.getNextRecord();
        if (mdr.testName == "FullBypassTest")
            addFullBypassTest(mdr);
        if (mdr.testName == "PipelineTest")
            addPipelineTest(mdr);
        if (mdr.testName == "StrobeDelay")
            addStrobeDelay(mdr);
        if (mdr.testName == "ThreePointGain")
            addThreePointGain(mdr);
        if (mdr.testName == "Trim")
            addTrim(mdr);
        if (mdr.testName == "ResponseCurve")
            addResponseCurve(mdr);
        if (mdr.testName == "NO")
            addNoise(mdr);
        if (mdr.testName == "Timewalk")
            addTimewalk(mdr);
    }
}

void ScanMap::addFullBypassTest(const ModuleDataRecord& mdr) {
    ScanInfo si;
    si.serialNum = mdr.serialNumber;
    si.scanVar = ST_TOKEN;
    for (unsigned int i=0; i<6; ++i) {
        digitalMap.insert(make_pair(si, constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber+i)));
    }
}

void ScanMap::addPipelineTest(const ModuleDataRecord& mdr) {
    //Not implemented atm.
}

void ScanMap::addStrobeDelay(const ModuleDataRecord& mdr) {
    StrobeDelayScanDetails sdsd;
    sdsd.charge = 4.0;
    sdsd.threshold = 2.0;
    sdsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber);
    strobeMap.insert(make_pair(mdr.serialNumber, sdsd));
}

void ScanMap::addThreePointGain(const ModuleDataRecord& mdr) {
    ThresholdScanDetails tsd;
    tsd.range = 0;
    tsd.trims = 0;
    double charges[] = {1.5, 2.0, 2.5};
    for (unsigned int i=0; i<3; ++i) {
        tsd.charge = charges[i];
        tsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber+i);
        thresholdMap.insert(make_pair(mdr.serialNumber, tsd));
    }
}

void ScanMap::addTrim(const ModuleDataRecord& mdr) {
    ThresholdScanDetails tsd;
    tsd.charge = 1.0;
    for (unsigned int i=0; i<28; ++i) {
        tsd.range = i<16 ? 0 : (i-16)/4+1;
        tsd.trims = i<16 ? i : ((i-16)%4+1)*4-1;
        tsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber+i);
        thresholdMap.insert(make_pair(mdr.serialNumber, tsd));
    }
}

void ScanMap::addResponseCurve(const ModuleDataRecord& mdr) {
    ThresholdScanDetails tsd;
    tsd.range = -1;
    tsd.trims = -1;
    double charges[] = {0.5, 0.75, 1.0, 1.25, 1.5, 2.0, 3.0, 4.0, 6.0, 8.0};
    for (unsigned int i=0; i<10; ++i) {
        tsd.charge = charges[i];
        tsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber+i);
        thresholdMap.insert(make_pair(mdr.serialNumber, tsd));
    }
}

void ScanMap::addNoise(const ModuleDataRecord& mdr) {
    ThresholdScanDetails tsd;
    tsd.range = -1;
    tsd.trims = -1;
    tsd.charge = 0;
    tsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber);
    thresholdMap.insert(make_pair(mdr.serialNumber, tsd));
}

void ScanMap::addTimewalk(const ModuleDataRecord& mdr) {
    StrobeDelayScanDetails sdsd;
    sdsd.threshold = 1.0;
    double charges[] = {1.25, 1.5, 1.75, 2, 3, 4.0, 5.0, 6.0, 8.0, 10.0};
    for (unsigned int i=0; i<10; ++i) {
        sdsd.charge = charges[i];
        sdsd.fileName = constructFileName(mdr.dataPath, mdr.runNumber, mdr.scanNumber+i);
        strobeMap.insert(make_pair(mdr.serialNumber, sdsd));
    }
}

string ScanMap::constructFileName(string path, unsigned int runNumber, unsigned int scanNumber) const {
    ostringstream oss;
    oss << path << "/strun" << runNumber << "_" << scanNumber << ".root";
    return oss.str();
}

}

