#ifndef SCTDATA_TESTRESULT_H
#define SCTDATA_TESTRESULT_H

#include <string>
#include <vector>
#include <boost/shared_ptr.hpp>
#include "Sct/LogicErrors.h"
#include "Sct/IoExceptions.h"
#include "Sct/Exception.h"
#include "ModuleConfiguration.h"
#include "DefectList.h"
#include "Result.h"

using namespace std;
using namespace Sct;
using boost::shared_ptr;

namespace SctData {
class FitScanResult;
class RawScanResult;
class ConfigurationVariable;
class DcsData;

/**
This class is the Abstract Base Class for all TestResults.  Examples are Response Curves, Trim Range Tests and StrobeDelay Test.
It provides useful data and functions to the sub-classes.  For instance, identifying information such as the list of 
scan numbers, the run number, module name and the test points.
@author Matthew Palmer
@date 12 March 2003
*/
class TestResult:public virtual Result {
public:
    /// Constructor is in protected
    ~TestResult() throw() {};
    
    ///Serializable override
    ///@throw LogicError if TestResult is empty.
    virtual string getUniqueID() const throw(LogicError) ;    
    
    ///@return the run number
    unsigned int getRunNumber() const throw();

    void setRunNumber(unsigned int runNumber) throw();

    ///@return the module name
    string getModuleName() const throw();

    void setModuleName(string name) throw();

    /**
      Get the configuration variable associated with all the scans in this test.
      */
    const ConfigurationVariable& getScanVariable() const throw();

    void setScanVariable(const ConfigurationVariable& scanVariable) throw();
    
    /**
      Gets the header
      @note Nothrow
      */
    virtual const ResultHeader& getHeader() const;
    
    /**
      Gets the header
      @note Nothrow
      */
    virtual ResultHeader& getHeader();
    

    /**
      Get the configuration variable associated with this test.
      */
    const ConfigurationVariable& getTestVariable() const throw();

    void setTestVariable(const ConfigurationVariable& testVariable) throw();

    /// gets the scan number at a particular postition `i' within the data.
    unsigned getScanNumberAt(unsigned int i) const throw(LogicError);

    /// gets the test point at a particular position `i' within the data.
    double getTestPointAt(unsigned int i) const throw(LogicError);

    /**
    Get the defects found in the TestResult.
    @return The DefectList class representing all the defects found in this TestResult.
    */
    const DefectList& getDefects() const throw() ;
    DefectList& getDefects() throw() ;

    /**
      Return true if the module passed the Test
      */
    bool getPassed() const throw();
    
    /**
      Set whether the module passed the scan
      */
    void setPassed(bool passed) throw();
    
    /**
      Return true if the module had a problem
      */
    bool getProblem() const throw();
    
    /**
      Set whether the module had a problem
      */
    void setProblem(bool problem) throw();
    
    /**
      Return all the comments.
      */
    vector<string> getComments() const throw(); 
    
    /**      
      Add a comment - must be a single line (no newlines)
      */
    void addComment(string comment) throw();
    
    /// Add a scan to this test.
    virtual void addScan(const unsigned int scanNumber, const double testPoint) throw(LogicError);


    /**
    Returns an iterator that is a model of (at least) const forward iterator that can be used to
    traverse all the scanNumbers.
    */
    //ScanNumberIterator& getScanNumbers();

    /**
     * @return the number of scans in this test.
     */
    const unsigned int getNScans() const throw() {
        return data.size();
    }

    /**
    * Find the index for a particular scanNumber.
    * @throw LogicError if not found.
    */
    unsigned int getIndex(const unsigned int scanNumber) const throw(LogicError) ;

    /**
       Get DcsData pointer. Null if none available.
     */
    shared_ptr<const DcsData> getDcsData() const;
    /**
       Set the DcsData object.
    */
    void setDcsData(boost::shared_ptr<const DcsData> dcs);
protected:
    /**
      Default constructor.
      */
    TestResult() throw();

    /**
     * Create a TestResult from all the stuff needed to initialize it.
     */
    TestResult(const unsigned int runNumber, const string & moduleName, const ConfigurationVariable& testVariable,
	       const ConfigurationVariable& scanVariable) throw ();

    /**
    * small internal class contains all the data relevant to a particular scan.
    */
    class ScanData {
    public:

        /**
	   constructor of ScanData
	   @param scanNumber : the scan number
	   @param testPoint the value of the variable changed between scans.
        */
        ScanData(const unsigned int scanNumber, const double testPoint) throw();

        //@{
        /// Operators needed for sorting
        bool operator<(const ScanData & s) const throw();
        bool operator>(const ScanData & s) const throw();
        bool operator==(const unsigned int aScanNumber) const throw();
        //@}

        unsigned int scanNumber;	///< The scan number
        double testPoint;	///< The value of the variable changed between scans
    };

    vector <ScanData> data;	/// vector of scanData (internal to this class) objects: one for each scan in the Test.
    ResultHeader header;				///< The header
    const ConfigurationVariable* testVariable;		///< The test variable
    DefectList defects;	                                ///< The Defects found in producing this TestResult.
    bool passed;					///< True if the module passed the Test
    bool problem;					///< True if there was a problem with the Module
    vector<string> comments;				///< Comments
    boost::shared_ptr<const DcsData> m_dcsData;         ///< Dcs Data if available.
};

}				// end of namespace SctData
#endif				//SCTDATA_TESTRESULT_H
