#ifndef SCTDATA_RESULTHEADER_H
#define SCTDATA_RESULTHEADER_H

#include <string>
#include <boost/shared_ptr.hpp>
#include "Sct/Serializable.h"
#include "Sct/Streamable.h"

using namespace std;
using boost::shared_ptr;

class TH1;
class TH2;

namespace boost {
    namespace posix_time {
	class ptime;
    }
}

namespace SctData {

class ConfigurationVariable;
class DataStatus;

/**
 * This class contains the identifying information about a scan result.
 * It is entirely defined by a run number, a scan number, a module name and a scan type.
 * The combination of the first 3 of those should be guarenteed to be unique.
 * @author Matthew Palmer
 */
class ResultHeader : public virtual Sct::Serializable {
public:
    /**
      Construct a scan from it's identifying information.
      @note Nothrow
    */
    ResultHeader(unsigned int scanNumber, unsigned int runNumber, string moduleName, const ConfigurationVariable& variable) ;

    /**
       Should be used for IO only. @note Nothrow
    */
    ResultHeader() ;		
    
    //Streamable and Serializable overrides
    virtual string getClassName() const throw() ;
    virtual string getUniqueID() const throw();

    static string getUniqueID(unsigned int runNumber, unsigned int scanNumber, const string& moduleName);

    /**
      @return the scan number.
      @note Nothrow
    */
    unsigned int getScanNumber() const {
        return scanNumber;
    }  		
    
    /**
      Set the scan number
      @note Nothrow
      */
    void setScanNumber(unsigned int scanNumber) {
	this->scanNumber = scanNumber;
    }
    
    /** @return the run number 
        @note Nothrow
      */
    unsigned int getRunNumber() const {
        return runNumber;
    } 
    
    /**
      Set the run number
      @note Nothrow
      */
    void setRunNumber(unsigned int runNumber) {
	this->runNumber = runNumber;
    }
    
    /** @return the module name 
      @note Nothrow
    */
    const string& getModuleName() const {
        return moduleName;
    }
    
    /**      
      Set the module name
      @note Nothrow
      */
    void setModuleName(string name) {
	moduleName = name;
    }
    
    /** @return the scan type
      @note Nothrow
    */
    const ConfigurationVariable& getVariable() const {
        return *variable;
    }
 
    /**
      Set the scan variable
      @note Nothrow
      */
    void setVariable(const ConfigurationVariable& var) {
	variable = &var;
    }
    
    /**
      Get the time the operation that resulted in this object started
      @note Nothrow
      */
    boost::posix_time::ptime getStartTime() const;
    
    /**
      Set the time the operation that resulted in this object started
      @note Nothrow
      */
    void setStartTime(boost::posix_time::ptime start);    
    
    /**
      Get the time the operation that resulted in this object ended
      @note Nothrow
      */
    boost::posix_time::ptime getEndTime() const;
    
    /**
      Set the time the operation that resulted in this object started
      @note Nothrow
      */
    void setEndTime(boost::posix_time::ptime start);  
    
    /**
      Gets the start time as a string
      @note Nothrow
      */
    string getStartTimeString() const {return startTime;}

    /**
       Sets the start time as a string
       @note Nothrow
       */
    void setStartTimeString(string start) {startTime = start;}
    
    /**
      Gets the end time as a string
      @note Nothrow
      */
    string getEndTimeString() const {return endTime;}
    
    /**
       Sets the end time as a string
       @note Nothrow
       */
    void setEndTimeString(string end) {endTime = end;}
    

    /**
     * Equality operator.
     * @return true if scan has the same scan number, run number, module number and scan type as this one.
     * @note Nothrow
     */
    bool operator==(const ResultHeader& scan) const;


protected:
    unsigned int scanNumber; 			///<The scan number;
    unsigned int runNumber; 			///<The run number
    string moduleName; 			   	///<The moduleName.
    string startTime;				///<The time the operation that resulted in this object started in ISO format, UTC
    string endTime;				///<The time the operation that resulted in this object ended in ISO format, UTC
    const ConfigurationVariable* variable; 		///<The variable that was modified in this scan.
};
}
#endif //#ifndef SCTDATA_RESULTHEADER_H
