/*
  This is the implementation of the ObjectPool class
  Necessarily included in the header, but split up here
  */

namespace SctData {  
    
template<typename T>
class ObjectPoolDeleter {
public:
    ObjectPoolDeleter(ObjectPool<T>& pool) : pool(pool) {}
    
    void operator() (T* p) {
	//Put a shared_ptr back on the front of the list
	//Make sure we stay with an ObjectPoolDeleter - faster get(), slower clear()
	boost::mutex::scoped_lock scoped_lock(pool.poolMutex);
	pool.pool.push_front(p);
    }
    
private:
    ObjectPool<T>& pool;
};	

template<typename T> 
ObjectPool<T>::~ObjectPool() {
    clear();
}
    
template<typename T> 
shared_ptr<T> ObjectPool<T>::get() {
    boost::mutex::scoped_lock scoped_lock(poolMutex);
    if (pool.empty()) return wrap(create());
    T* ptr = pool.front();
    ptr->reset();
    pool.pop_front();
    return wrap(ptr);
}
    
template<typename T> 
void ObjectPool<T>::clear() {
    boost::mutex::scoped_lock scoped_lock(poolMutex);
    while (!pool.empty()) {
	T* ptr = pool.front();
	pool.pop_front();
	delete ptr;
    }
}


template<typename T> 
unsigned int ObjectPool<T>::poolSize() const {
    return pool.size();
}
    
template<typename T> 
void ObjectPool<T>::resize(unsigned int n) {
    boost::mutex::scoped_lock scoped_lock(poolMutex);
    for (unsigned int i=pool.size(); i<n; ++i) {
	pool.push_front(create());
    }
}
    
template<typename T> 
shared_ptr<T> ObjectPool<T>::wrap(T* ptr) {
    return shared_ptr<T>(ptr, ObjectPoolDeleter<T>(*this));
}

template<typename T> 
T* ObjectPool<T>::create() {
    return new T();
}

}

