#ifndef SCTDATA_OBJECTPOOL_H
#define SCTDATA_OBJECTPOOL_H

#include <list>
#include <boost/shared_ptr.hpp>
#include <boost/thread/mutex.hpp>

using boost::shared_ptr;
using boost::mutex;
using std::list;

namespace SctData {
 
template<typename T> class ObjectPoolDeleter;
    
/**
  This class is an object pool.  It requires that its type, T is default constructible
  and has a reset() method.  It will maintain a list of created but unused objects then return 
  shared_ptr to them when asked.  If there are no objects, then it will create one.  All the nothrow
  guarentees are based on the default constructor and reset methods not throwing.
  
  The ObjectPool must outlive all the objects that it is used to create.  It is probably best if it is
  statically created!
  
  This class is thread-safe.
  
  @note That it isn't possible to reliably find out the total assigned and un-assigned pool size - is this a problem?
  
  @author Matthew Palmer
  */   
template<typename T>
    class ObjectPool {
public:
    ~ObjectPool();
    
    /**
      Gets an object from the pool.  Will call reset on the object first
      @note nothrow
      */
    shared_ptr<T> get();
    
    /**
      Releases all unused objects.  Note that objects currently in use will
      still be returned to the pool
      @note nothrow
      */
    void clear();
    
    /**
      Returns the number of elements in the Pool
      Not thread safe - should be used as an indication only!
      */
    unsigned int poolSize() const;
    
    /**
      Ensures there are at least n objects in the pool
      @note nothrow
      */
    void resize(unsigned int n);
    
private:    
    /**
      Wraps a ptr with an appropriate shared_ptr 
      @note nothrow
      */
    shared_ptr<T> wrap(T* ptr);
    
    /**
      Creates a T
      */
    T* create();
    
    friend class ObjectPoolDeleter<T>;	///< The custom deleter class
    list<T*> pool;		///< The pool of unused objects
    mutex poolMutex;		///< Used to lock access to the pool
};
    
    
}
#include "ObjectPool.inl"

#endif //SCTDATA_OBJECTPOOL_H
