#ifndef NPTGAINTESTRESULT_H
#define NPTGAINTESTRESULT_H

#include "TestResult.h"
#include "FitObject.h"
#include "Sct/RangedVector.h"

#include <boost/utility.hpp>
#include <boost/scoped_ptr.hpp>
#include <boost/shared_ptr.hpp>
#include <memory>

#include <TF1.h>
#include <TGraph.h>

using std::auto_ptr;
using boost::scoped_ptr;
using boost::shared_ptr;

namespace SctData {

class ResponseCurve;
class CalChargeVariable;
class ThresholdVariable;

/**
Holds the result of the fit for each channel
Used shared_ptr to allow this to be stored in stl containers.
*/
class NPtGainTestResultData {
public:
    NPtGainTestResultData() throw() : gain(0), noise(0), offset(0) {}
    shared_ptr<ResponseCurve> rc;   ///< Fitted Response Curves
    shared_ptr<TGraph> graph;        ///< Graph of the points that were fitted
    double gain;                     ///< Gain at the special point
    double noise;                    ///< Noise at the special point
    double offset;                   ///< Offset
};


/**
This class represents the result of doing either a 3 pt Gain Test or a Response Curve Test.
The difference is of course, N.	
@author Matthew Palmer
*/
class NPtGainTestResult : public TestResult {
public:
    NPtGainTestResult() throw();

    ~NPtGainTestResult() throw();
    
    /**
     * Contructor.  Pass on all the necessary stuff to TestResult.
     * NB the NPtGainTestResult owns the response curve.
     */
    NPtGainTestResult(unsigned int runNumber, const string& moduleName, 
		      const CalChargeVariable& testVariable, const ThresholdVariable& scanVariable) throw();

    //Override
    virtual string getClassName() const throw();

    /**
     * @return the Response Curve
     */
    const ResponseCurve& getResponseCurve() const throw();

    void setChannelDataSize(unsigned int i) throw();
    void setChipDataSize(unsigned int i) throw();
    
    /**
    Get the N Pt Gain data for a channel.
    @param chip The channel number [0,11]
    @return The data.
    @throw LogicError if out of range
    */
    const NPtGainTestResultData& getChipData(unsigned int chip) const throw(LogicError);
    NPtGainTestResultData& getChipData(unsigned int chip) throw(LogicError);

    /**
    Get the N Pt Gain data for a channel.
    @param chip The channel number [0,1535]
    @return The data.
    @throw LogicError if out of range
    */
    const NPtGainTestResultData& getChannelData(unsigned int channel) const throw (LogicError);
    NPtGainTestResultData& getChannelData(unsigned int channel) throw (LogicError);

    /// get the number of chip fits, 0 or 12;
    unsigned int getNChipData() const throw() {
        return chipData.size();
    }

    /// get the number of channel fits, 0 or 1535;
    unsigned int getNChannelData() const throw() {
        return channelData.size();
    }
    
    /**
       @return the special point
       @throw LogicError if the special point hasn't been initialized.
    */	
    double getSpecialScanPointValue() const throw(LogicError) {
        return getTestPointAt(specialScanIndex);
    }

    unsigned int getSpecialScanIndex() const throw() {
        return specialScanIndex;
    }
    
    /**
      Set the value of the special point.
      @param pointValue the value of the testVariable for the special scan
      @throw LogicError if no scan with such a value exists.
      */
    void setSpecialScanPointValue(double pointValue) throw(LogicError);

private:    
    //Data
    unsigned int specialScanIndex;			///< The index of the special scan!
    RangedVector <NPtGainTestResultData> chipData;      ///< The chip fits
    RangedVector <NPtGainTestResultData> channelData ;	///< The channel fits
};


inline const NPtGainTestResultData& NPtGainTestResult::getChipData(unsigned int chip) const throw(LogicError) {
    return chipData[chip];
}

inline NPtGainTestResultData& NPtGainTestResult::getChipData(unsigned int chip) throw(LogicError) {
    return chipData[chip];
}

inline const NPtGainTestResultData& NPtGainTestResult::getChannelData(unsigned int channel) const throw(LogicError) {
    return channelData[channel];
}

inline NPtGainTestResultData& NPtGainTestResult::getChannelData(unsigned int channel) throw(LogicError) {
    return channelData[channel];
}
}
#endif	//#ifndef NPTGAINTESTRESULT_H
