#ifndef FITSCANRESULT_H
#define FITSCANRESULT_H

#include "Sct/SctParameters.h"
#include "Sct/RangedVector.h"
#include "ScanResult.h"
#include "DefectList.h"

#include <boost/shared_ptr.hpp>
#include <iostream>
#include <vector>

using boost::shared_ptr;
using std::string;
using std::ostream;
using Sct::RangedVector;

class TH1D;

namespace SctData {
class FitObject;
class ModuleConfiguration;
class RawScanResult;
class FitScanResultIOHelper;


/**
 * Class for the various fitted ROD scan result objects
 * @author Alan Barr
 * @date 6 December 2002
 */

class FitScanResult : public ScanResult {
public:
    /// Create a FitScanResult from the objects that are needed to initialize the base class.  Is this really necessary?
    FitScanResult(const ResultHeader& header, const ModuleConfiguration& config, const ScanPoints& points) throw() ;

    /// Create a FitScanResult object from a given RawScanResult Object.
    FitScanResult(const RawScanResult& raw) throw();

    /// Destroy the FitScanResult object and all of its FitObject members.
    virtual ~FitScanResult() throw();
    
    //@{
    ///Serializable overrides
    virtual string getClassName() const throw();
    virtual string getUniqueID() const throw();
    //@}

    /// Returns the uniqueID a FitScanResult would have if it included the given ScanHeader.
    static string getUniqueID(const ResultHeader& header) throw();

    //@{
    ///Get the DefectList associated with this FitScanResult.
    const DefectList& getDefects() const throw() {
        return defects;
    }
    DefectList& getDefects() throw() {
        return defects;
    }
    //@}

    //@{
    /**
    * Functions to retrieve the FitObjects, by link, chip or channel
    * get link fit by : link 0 or 1
    */
    FitObject& getLinkFit (const unsigned ilink) const throw(LogicError) ;

    /**
    * get chip fit by : link 0 or 1, chip 0 -> 5
    */
    FitObject& getChipFit (const unsigned ilink,
                           const unsigned ichip) const throw(LogicError) ;

    /**
    * get chip fit by : chip 0 -> 11, with 6->11 on link 1
    */
    FitObject& getChipFit (const unsigned ichip) const throw(LogicError) ;

    /**
    * get channel fit by : channel 0 -> 1535 (with 768->1535 on link 1)
    */
    FitObject& getChannelFit (const unsigned ichannel) const throw(LogicError) ;

    //@}

    /**
    * get channel fit by : link 0 or 1, channel 0 -> 767
    */
    FitObject& getChannelFit (const unsigned ilink,
                              const unsigned ichip) const throw(LogicError) ;

    /// get number of summary histograms
    unsigned getNHist() const throw() {
        return m_summaryHist.size();
    }

    /// get an existing summary histogram
    const TH1D& getHist(unsigned i) const throw(LogicError) ;

    ///@{
    /**
    * Largely included for bug-checking,
    * the following functions should return 2
    * if the object has been correctly initialized.
    */
    unsigned getNLinkFits () const throw();

    /**
    * Largely included for bug-checking,
    * the following functions should return 12
    * if the object has been correctly initialized.
    */
    unsigned getNChipFits () const throw();

    /**
    * Largely included for bug-checking,
    * the following functions should return 1536
    * if the object has been correctly initialized.
    */
    unsigned getNChannelFits () const throw();
    
    /**
      Set the number of channel fits that are stored
      */
    void setNChannelFits(unsigned int n=Sct::nChannelModule) throw();
    
    /**
      Set the number of chip fits that are stored
      */
    void setNChipFits(unsigned int n=Sct::nChipModule) throw();
    
    /**
      Set the number of channel fits that are stored
      */
    void setNLinkFits(unsigned int n=Sct::nLinkModule) throw();
    
    /**
      Initializes this so that it can store channel fits
      @param prototype A prototypical FitObject that will be copied.
      */
    void initializeChannelFits(const FitObject& prototype) throw();
    ///  set the number of chip fits
    void initializeChipFits(const FitObject& prototype) throw();
    ///  set the number of link fits
    void initializeLinkFits(const FitObject& prototype) throw();

    /**
      Set a channel fit.  
      Must first call enableChannelFits
      */
    void setChannelFit(unsigned int channel, shared_ptr<FitObject> fit) throw(LogicError);
    
    /**
      Set a chip fit.  
      Must first call enableChipFits
      */
    void setChipFit(unsigned int chip, shared_ptr<FitObject> fit) throw(LogicError);
    
    
    /**
      Set a link fit.  
      Must first call enableLinkFits
      */
    void setLinkFit(unsigned int link, shared_ptr<FitObject> fit) throw(LogicError);
    
    ///@}

protected:
    RangedVector <shared_ptr<FitObject> > m_channelFits;	///< fits for each channel
    RangedVector <shared_ptr<FitObject> > m_chipFits;	///< fits for each chip
    RangedVector <shared_ptr<FitObject> > m_linkFits;	///< fits for each link
    RangedVector <TH1D*> m_summaryHist;        ///< summary histograms
    DefectList defects;				///< Contains all the defects discovered in doing the fits

private:
    //IOHelper stuff
    template <class T>
    static void resizeAndFill( std::vector<shared_ptr<T> >& v, unsigned newSize, const T& prototype) throw(); ///< utility function;
    FitScanResult() throw();		///<Default constructor - only for IO purposes
    friend class FitScanResultIOHelper; ///< The IOHelper class.
};

// INLINES:
template <class T>
void FitScanResult::resizeAndFill( std::vector<shared_ptr<T> >& v, unsigned newSize, const T& prototype) throw(){
    unsigned oldsize=v.size(); v.resize(newSize);
    for (unsigned i=oldsize; i<v.size(); ++i){
	v[i]=prototype.clone();
    }
}

}// end of namespace SctData

ostream & operator<< (ostream &, const SctData::FitScanResult &) throw() ;

#endif // #ifndef FITSCANRESULT_H
